#define ENABLE_V17
/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t38_ifppacket.c - Encode and decode the ASN.1 of a T.38 IFP message
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t38_ifppacket.c,v 1.17 2006/05/18 14:15:36 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <inttypes.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <tgmath.h>
#include <math.h>
#include <assert.h>
#include <memory.h>
#include <tiffio.h>

#include "spandsp/telephony.h"
#include "spandsp/logging.h"
#include "spandsp/bit_operations.h"
#include "spandsp/queue.h"
#include "spandsp/power_meter.h"
#include "spandsp/complex.h"
#include "spandsp/tone_generate.h"
#include "spandsp/async.h"
#include "spandsp/hdlc.h"
#include "spandsp/fsk.h"
#include "spandsp/v29rx.h"
#include "spandsp/v29tx.h"
#include "spandsp/v27ter_rx.h"
#include "spandsp/v27ter_tx.h"
#if defined(ENABLE_V17)
#include "spandsp/v17rx.h"
#include "spandsp/v17tx.h"
#endif
#include "spandsp/t4.h"

#include "spandsp/t30_fcf.h"
#include "spandsp/t35.h"
#include "spandsp/t30.h"

#include "spandsp/t38.h"

const char *t38_indicator(int indicator)
{
    const char *type;

    switch (indicator)
    {
    case T38_IND_NO_SIGNAL:
        type = "T38_IND_NO_SIGNAL";
        break;
    case T38_IND_CNG:
        type = "T38_IND_CNG";
        break;
    case T38_IND_CED:
        type = "T38_IND_CED";
        break;
    case T38_IND_V21_PREAMBLE:
        type = "T38_IND_V21_PREAMBLE";
        break;
    case T38_IND_V27TER_2400_TRAINING:
        type = "T38_IND_V27TER_2400_TRAINING";
        break;
    case T38_IND_V27TER_4800_TRAINING:
        type = "T38_IND_V27TER_4800_TRAINING";
        break;
    case T38_IND_V29_7200_TRAINING:
        type = "T38_IND_V29_7200_TRAINING";
        break;
    case T38_IND_V29_9600_TRAINING:
        type = "T38_IND_V29_9600_TRAINING";
        break;
    case T38_IND_V17_7200_SHORT_TRAINING:
        type = "T38_IND_V17_7200_SHORT_TRAINING";
        break;
    case T38_IND_V17_7200_LONG_TRAINING:
        type = "T38_IND_V17_7200_LONG_TRAINING";
        break;
    case T38_IND_V17_9600_SHORT_TRAINING:
        type = "T38_IND_V17_9600_SHORT_TRAINING";
        break;
    case T38_IND_V17_9600_LONG_TRAINING:
        type = "T38_IND_V17_9600_LONG_TRAINING";
        break;
    case T38_IND_V17_12000_SHORT_TRAINING:
        type = "T38_IND_V17_12000_SHORT_TRAINING";
        break;
    case T38_IND_V17_12000_LONG_TRAINING:
        type = "T38_IND_V17_12000_LONG_TRAINING";
        break;
    case T38_IND_V17_14400_SHORT_TRAINING:
        type = "T38_IND_V17_14400_SHORT_TRAINING";
        break;
    case T38_IND_V17_14400_LONG_TRAINING:
        type = "T38_IND_V17_14400_LONG_TRAINING";
        break;
    case T38_IND_V8_ANSAM:
        type = "T38_IND_V8_ANSAM";
        break;
    case T38_IND_V8_SIGNAL:
        type = "T38_IND_V8_SIGNAL";
        break;
    case T38_IND_V34_CNTL_CHANNEL_1200:
        type = "T38_IND_V34_CNTL_CHANNEL_1200";
        break;
    case T38_IND_V34_PRI_CHANNEL:
        type = "T38_IND_V34_PRI_CHANNEL";
        break;
    case T38_IND_V34_CC_RETRAIN:
        type = "T38_IND_V34_CC_RETRAIN";
        break;
    case T38_IND_V33_12000_TRAINING:
        type = "T38_IND_V33_12000_TRAINING";
        break;
    case T38_IND_V33_14400_TRAINING:
        type = "T38_IND_V33_14400_TRAINING";
        break;
    default:
        type = "???";
        break;
    }
    return type;
}
/*- End of function --------------------------------------------------------*/

const char *t38_data_type(int data_type)
{
    const char *type;

    switch (data_type)
    {
    case T38_DATA_V21:
        type = "T38_DATA_V21";
        break;
    case T38_DATA_V27TER_2400:
        type = "T38_DATA_V27TER_2400";
        break;
    case T38_DATA_V27TER_4800:
        type = "T38_DATA_V27TER_4800";
        break;
    case T38_DATA_V29_7200:
        type = "T38_DATA_V29_7200";
        break;
    case T38_DATA_V29_9600:
        type = "T38_DATA_V29_9600";
        break;
    case T38_DATA_V17_7200:
        type = "T38_DATA_V17_7200";
        break;
    case T38_DATA_V17_9600:
        type = "T38_DATA_V17_9600";
        break;
    case T38_DATA_V17_12000:
        type = "T38_DATA_V17_12000";
        break;
    case T38_DATA_V17_14400:
        type = "T38_DATA_V17_14400";
        break;
    case T38_DATA_V8:
        type = "T38_DATA_V8";
        break;
    case T38_DATA_V34_PRI_RATE:
        type = "T38_DATA_V34_PRI_RATE";
        break;
    case T38_DATA_V34_CC_1200:
        type = "T38_DATA_V34_CC_1200";
        break;
    case T38_DATA_V34_PRI_CH:
        type = "T38_DATA_V34_PRI_CH";
        break;
    case T38_DATA_V33_12000:
        type = "T38_DATA_V33_12000";
        break;
    case T38_DATA_V33_14400:
        type = "T38_DATA_V33_14400";
        break;
    default:
        type = "???";
        break;
    }
    return type;
}
/*- End of function --------------------------------------------------------*/

const char *t38_field_type(int field_type)
{
    const char *type;

    switch (field_type)
    {
    case T38_FIELD_HDLC_DATA:
        type = "T38_FIELD_HDLC_DATA";
        break;
    case T38_FIELD_HDLC_SIG_END:
        type = "T38_FIELD_HDLC_SIG_END";
        break;
    case T38_FIELD_HDLC_FCS_OK:
        type = "T38_FIELD_HDLC_FCS_OK";
        break;
    case T38_FIELD_HDLC_FCS_BAD:
        type = "T38_FIELD_HDLC_FCS_BAD";
        break;
    case T38_FIELD_HDLC_FCS_OK_SIG_END:
        type = "T38_FIELD_HDLC_FCS_OK_SIG_END";
        break;
    case T38_FIELD_HDLC_FCS_BAD_SIG_END:
        type = "T38_FIELD_HDLC_FCS_BAD_SIG_END";
        break;
    case T38_FIELD_T4_NON_ECM_DATA:
        type = "T38_FIELD_NON_ECM_DATA";
        break;
    case T38_FIELD_T4_NON_ECM_SIG_END:
        type = "T38_FIELD_NON_ECM_SIG_END";
        break;
    case T38_FIELD_CM_MESSAGE:
        type = "T38_FIELD_CM_MESSAGE";
        break;
    case T38_FIELD_JM_MESSAGE:
        type = "T38_FIELD_JM_MESSAGE";
        break;
    case T38_FIELD_CI_MESSAGE:
        type = "T38_FIELD_CI_MESSAGE";
        break;
    case T38_FIELD_V34RATE:
        type = "T38_FIELD_V34RATE";
        break;
    default:
        type = "???";
        break;
    }
    return type;
}
/*- End of function --------------------------------------------------------*/

int t38_rx_ifp_packet(t38_state_t *s, int seq_no, const uint8_t *buf, int len)
{
    int i;
    int t30_indicator;
    int t30_data;
    int ptr;
    int numocts;
    const uint8_t *msg;
    uint8_t type;
    unsigned int value;
    unsigned int count;
    uint8_t data_field_present;
    uint8_t field_data_present;
    unsigned int field_type;

    if (span_log_test(&s->logging, SPAN_LOG_FLOW))
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "Rx: ");
        for (i = 0;  i < len;  i++)
            span_log(&s->logging, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "%02x ", buf[i]);
        span_log(&s->logging, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "\n");
    }
    span_log(&s->logging, SPAN_LOG_FLOW, "Expecting %d, got %d\n", s->rx_expected_seq_no, seq_no);
    if (seq_no != s->rx_expected_seq_no)
    {
        if (seq_no < s->rx_expected_seq_no)
        {
            /* We are already beyond this packet. It could be a repeat. It could be too late. Whatever,
               the cause, we just ignore this packet. */
            return 0;
        }
        if (s->gateway)
        {
            for (i = s->rx_expected_seq_no;  i < seq_no;  i++)
                t38_gateway_process_rx_missing(s, i);
        }
        else
        {
            for (i = s->rx_expected_seq_no;  i < seq_no;  i++)
                t38_terminal_process_rx_missing(s, i);
        }
        s->rx_expected_seq_no = seq_no;
    }
    s->rx_expected_seq_no++;
    
    data_field_present = (buf[0] >> 7) & 1;
    type = (buf[0] >> 6) & 1;
    ptr = 0;
    switch (type)
    {
    case T38_TYPE_OF_MSG_T30_INDICATOR:
        /* Indicators should never have a data field */
        if (data_field_present)
            return -1;
        if ((buf[0] & 0x20))
        {
            /* Extension */
            if (len != 2)
                return -1;
            value = ((buf[0] << 2) & 0x3C) | ((buf[1] >> 6) & 0x3);
            if (value > T38_IND_V33_14400_TRAINING - T38_IND_V8_ANSAM)
                return -1;
            t30_indicator = T38_IND_V8_ANSAM + value;
        }
        else
        {
            if (len != 1)
                return -1;
            value = (buf[0] >> 1) & 0xF;
            if (value > T38_IND_V17_14400_LONG_TRAINING)
                return -1;
            t30_indicator = value;
        }
        if (s->gateway)
            t38_gateway_process_rx_indicator(s, t30_indicator);
        else
            t38_terminal_process_rx_indicator(s, t30_indicator);
        break;
    case T38_TYPE_OF_MSG_T30_DATA:
        if ((buf[0] & 0x20))
        {
            /* Extension */
            if (len < 2)
                return -1;
            value = ((buf[0] << 2) & 0x3C) | ((buf[1] >> 6) & 0x3);
            if (value > T38_DATA_V33_14400 - T38_DATA_V8)
                return -1;
            t30_data = T38_DATA_V8 + value;
            ptr = 2;
        }
        else
        {
            if (len < 1)
                return -1;
            value = (buf[0] >> 1) & 0xF;
            if (value > T38_DATA_V17_14400)
                return -1;
            t30_data = value;
            ptr = 1;
        }
        if (data_field_present)
        {
#if 0
            stat2 = asn1_per_decode_length(&ctxt, &count);
            if (stat2 != ASN1_OK  &&  stat2 != ASN1_OK_FRAG)
                return stat2;
#else
            count = buf[ptr++];
            //printf("Count is %d\n", count);
#endif
            for (i = 0;  i < (int) count;  i++)
            {
                field_data_present = (buf[ptr] >> 7) & 1;
                /* Decode field_type */
                if (s->t38_version == 0)
                {
                    /* Original version of T.38 with a typo */
                    value = (buf[ptr++] >> 4) & 0x7;
                    if (value > T38_FIELD_T4_NON_ECM_SIG_END)
                        return -1;
                    field_type = value;
                }
                else
                {
                    if ((buf[ptr] & 0x40))
                    {
                        value = ((buf[ptr] << 2) & 0x3C) | ((buf[ptr + 1] >> 6) & 0x3);
                        ptr++;
                        if (value > T38_FIELD_V34RATE - T38_FIELD_CM_MESSAGE)
                            return -1;
                        field_type = T38_FIELD_CM_MESSAGE + value;
                    }
                    else
                    {
                        value = (buf[ptr++] >> 3) & 0x7;
                        if (value > T38_FIELD_T4_NON_ECM_SIG_END)
                            return -1;
                        field_type = value;
                    }
                }
                /* Decode field_data */
                if (field_data_present)
                {
                    numocts = ((buf[ptr] << 8) | buf[ptr + 1]) + 1;
                    msg = buf + ptr + 2;
                    ptr += numocts + 2;
                }
                else
                {
                    numocts = 0;
                    msg = NULL;
                }
                if (s->gateway)
                    t38_gateway_process_rx_data(s, t30_data, field_type, msg, numocts);
                else
                    t38_terminal_process_rx_data(s, t30_data, field_type, msg, numocts);
            }
        }
        break;
    }

    return 0;
}
/*- End of function --------------------------------------------------------*/

static int t38_encode_data(t38_state_t *s, uint8_t buf[], int data_type, int field_type, const uint8_t *msg, int msglen)
{
    int len;
    int i;
    int enclen;
    int multiplier;
    int data_field_no;
    int data_field_count;
    unsigned int value;
    data_field_element_t data_field_seq[10];
    data_field_element_t *q;
    unsigned int encoded_len;
    unsigned int fragment_len;
    uint8_t data_field_present;

    span_log(&s->logging, SPAN_LOG_FLOW, "%5d: Tx data type %s/%s\n", s->tx_seq_no, t38_data_type(data_type), t38_field_type(field_type));

    /* Build the IFP packet */
    data_field_present = TRUE;

    data_field_seq[0].field_data_present = (uint8_t) (msglen > 0);
    data_field_seq[0].field_type = field_type;
    data_field_seq[0].field_data.numocts = msglen;
    data_field_seq[0].field_data.data = msg;
    data_field_count = 1;

    len = 0;
    /* Data field present */
    /* Data packet */
    /* Type of data */
    if (data_type <= T38_DATA_V17_14400)
    {
        buf[len++] = (uint8_t) ((data_field_present << 7) | 0x40 | (data_type << 1));
    }
    else if (data_type <= T38_DATA_V33_14400)
    {
        buf[len++] = (uint8_t) ((data_field_present << 7) | 0x60 | (((data_type - T38_DATA_V8) & 0xF) >> 2));
        buf[len++] = (uint8_t) (((data_type - T38_DATA_V8) << 6) & 0xFF);
    }
    else
    {
        return -1;
    }
    if (data_field_present)
    {
        encoded_len = 0;
        data_field_no = 0;
        do
        {
            value = data_field_count - encoded_len;
            if (value < 0x80)
            {
                /* 1 octet case */
                buf[len++] = (uint8_t) value;
                enclen = value;
            }
            else if (value < 0x4000)
            {
                /* 2 octet case */
                buf[len++] = (uint8_t) (0x80 | ((value >> 8) & 0xFF));
                buf[len++] = (uint8_t) (value & 0xFF);
                enclen = value;
            }
            else
            {
                /* Fragmentation case */
                multiplier = (value/0x4000 < 4)  ?  value/0x4000  :  4;
                buf[len++] = (uint8_t) (0xC0 | multiplier);
                enclen = 0x4000*multiplier;
            }

            fragment_len = enclen;
            encoded_len += fragment_len;
            /* Encode the elements */
            for (i = 0;  i < (int) encoded_len;  i++)
            {
                q = &data_field_seq[data_field_no];
                /* Encode field_type */
                if (s->t38_version == 0)
                {
                    /* Original version of T.38 with a typo */
                    if (q->field_type > T38_FIELD_T4_NON_ECM_SIG_END)
                        return -1;
                    buf[len++] = (uint8_t) ((q->field_data_present << 7) | (q->field_type << 4));
                }
                else
                {
                    if (q->field_type <= T38_FIELD_T4_NON_ECM_SIG_END)
                    {
                        buf[len++] = (uint8_t) ((q->field_data_present << 7) | (q->field_type << 3));
                    }
                    else if (q->field_type <= T38_FIELD_V34RATE)
                    {
                        buf[len++] = (uint8_t) ((q->field_data_present << 7) | 0x40 | (((q->field_type - T38_FIELD_CM_MESSAGE) & 0x1F) >> 1));
                        buf[len++] = (uint8_t) (((q->field_type - T38_FIELD_CM_MESSAGE) << 7) & 0xFF);
                    }
                    else
                    {
                        return -1;
                    }
                }
                /* Encode field_data */
                if (q->field_data_present)
                {
                    if (q->field_data.numocts < 1  ||  q->field_data.numocts > 65535)
                        return -1;
                    buf[len++] = (uint8_t) (((q->field_data.numocts - 1) >> 8) & 0xFF);
                    buf[len++] = (uint8_t) ((q->field_data.numocts - 1) & 0xFF);
                    memcpy(buf + len, q->field_data.data, q->field_data.numocts);
                    len += q->field_data.numocts;
                }
                data_field_no++;
            }
        }
        while (data_field_count != (int) encoded_len  ||  fragment_len >= 16384);
    }

    if (span_log_test(&s->logging, SPAN_LOG_FLOW))
    {
        int i;

        span_log(&s->logging, SPAN_LOG_FLOW, "Tx: ");
        for (i = 0;  i < len;  i++)
            span_log(&s->logging, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "%02x ", buf[i]);
        span_log(&s->logging, SPAN_LOG_FLOW | SPAN_LOG_SUPPRESS_LABELLING, "\n");
    }
    
    return len;
}
/*- End of function --------------------------------------------------------*/

static int t38_encode_indicator(t38_state_t *s, uint8_t buf[], int indicator)
{
    int len;

    span_log(&s->logging, SPAN_LOG_FLOW, "%5d: Tx indicator %s\n", s->tx_seq_no, t38_indicator(indicator));

    /* Build the IFP packet */
    /* Data field not present */
    /* Indicator packet */
    /* Type of indicator */
    if (indicator <= T38_IND_V17_14400_LONG_TRAINING)
    {
        buf[0] = (uint8_t) (indicator << 1);
        len = 1;
    }
    else if (indicator <= T38_IND_V33_14400_TRAINING)
    {
        buf[0] = (uint8_t) (0x20 | (((indicator - T38_IND_V8_ANSAM) & 0xF) >> 2));
        buf[1] = (uint8_t) (((indicator - T38_IND_V8_ANSAM) << 6) & 0xFF);
        len = 2;
    }
    else
    {
        len = -1;
    }
    return len;
}
/*- End of function --------------------------------------------------------*/

int t38_send_data(t38_state_t *s, int data_type, int field_type, const uint8_t *msg, int msglen)
{
    uint8_t buf[100];
    int len;

    if ((len = t38_encode_data(s, buf, data_type, field_type, msg, msglen)) > 0)
        s->tx_packet_handler(s, s->tx_packet_user_data, buf, len, 1);
    else
        span_log(&s->logging, SPAN_LOG_FLOW, "T.38 data len is %d\n", len);
    return 0;
}
/*- End of function --------------------------------------------------------*/

int t38_send_indicator(t38_state_t *s, int indicator, int count)
{
    uint8_t buf[100];
    int len;

    if ((len = t38_encode_indicator(s, buf, indicator)) > 0)
    {
        s->tx_packet_handler(s, s->tx_packet_user_data, buf, len, count);
        s->current_tx_indicator = indicator;
    }
    else
    {
        span_log(&s->logging, SPAN_LOG_FLOW, "T.38 indicator len is %d\n", len);
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

void t38_set_data_rate_management_method(t38_state_t *s, int method)
{
    s->data_rate_management_method = method;
}
/*- End of function --------------------------------------------------------*/

void t38_set_data_transport_protocol(t38_state_t *s, int data_transport_protocol)
{
    s->data_transport_protocol = data_transport_protocol;
}
/*- End of function --------------------------------------------------------*/

void t38_set_fill_bit_removal(t38_state_t *s, int fill_bit_removal)
{
    s->fill_bit_removal = fill_bit_removal;
}
/*- End of function --------------------------------------------------------*/

void t38_set_mmr_transcoding(t38_state_t *s, int mmr_transcoding)
{
    s->mmr_transcoding = mmr_transcoding;
}
/*- End of function --------------------------------------------------------*/

void t38_set_jbig_transcoding(t38_state_t *s, int jbig_transcoding)
{
    s->jbig_transcoding = jbig_transcoding;
}
/*- End of function --------------------------------------------------------*/

void t38_set_max_buffer_size(t38_state_t *s, int max_buffer_size)
{
    s->max_buffer_size = max_buffer_size;
}
/*- End of function --------------------------------------------------------*/

void t38_set_max_datagram_size(t38_state_t *s, int max_datagram_size)
{
    s->max_datagram_size = max_datagram_size;
}
/*- End of function --------------------------------------------------------*/

void t38_set_t38_version(t38_state_t *s, int t38_version)
{
    s->t38_version = t38_version;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
