#define ENABLE_V17
/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t38.h - An implementation of T.38, less the packet exchange part
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t38.h,v 1.23 2006/05/19 12:23:32 steveu Exp $
 */

/*! \file */

#if !defined(_T38_H_)
#define _T38_H_

/*! \page t38_page T.38 real time FAX over IP handling
\section t38_page_sec_1 What does it do?
The T.38 real time FAX over IP module provides two forms of T.38 operation:
    - It provides a T.38 termination point facility.
    - It provides a T.38 gateway facility, to link traditional analogue FAX machines
      into the T.38 world.

The T.38 gateway facility provides a robust interface between T.38 IP packet streams and
and 8k samples/second audio streams. It provides the buffering and flow control features needed
to maximum the tolerance of jitter and packet loss on the IP network.

\section t38_page_sec_2 How does it work?

Timing differences and jitter between two T.38 entities can be a serious problem, if one of those
entities is a PSTN gateway.

Flow control for non-ECM image data takes advantage of several features of the T.30 specification.
First, an unspecified number of 0xFF octets may be sent at the start of transmission. This means we
can add endless extra 0xFF bytes at this point, without breaking the T.30 spec. In practice, we
cannot add too many, or we will affect the timing tolerance of the T.30 protocol by delaying the
response at the end of each image. Secondly, just before an end of line (EOL) marker we can pad
with zero bits. Again, the number is limited only by need to avoid upsetting the timing of the
step following the non-ECM data.
*/

enum t30_indicator_types_e
{
    T38_IND_NO_SIGNAL = 0,
    T38_IND_CNG,
    T38_IND_CED,
    T38_IND_V21_PREAMBLE,
    T38_IND_V27TER_2400_TRAINING,
    T38_IND_V27TER_4800_TRAINING,
    T38_IND_V29_7200_TRAINING,
    T38_IND_V29_9600_TRAINING,
    T38_IND_V17_7200_SHORT_TRAINING,
    T38_IND_V17_7200_LONG_TRAINING,
    T38_IND_V17_9600_SHORT_TRAINING,
    T38_IND_V17_9600_LONG_TRAINING,
    T38_IND_V17_12000_SHORT_TRAINING,
    T38_IND_V17_12000_LONG_TRAINING,
    T38_IND_V17_14400_SHORT_TRAINING,
    T38_IND_V17_14400_LONG_TRAINING,
    T38_IND_V8_ANSAM,
    T38_IND_V8_SIGNAL,
    T38_IND_V34_CNTL_CHANNEL_1200,
    T38_IND_V34_PRI_CHANNEL,
    T38_IND_V34_CC_RETRAIN,
    T38_IND_V33_12000_TRAINING,
    T38_IND_V33_14400_TRAINING
};

enum t38_data_types_e
{
    T38_DATA_V21 = 0,
    T38_DATA_V27TER_2400,
    T38_DATA_V27TER_4800,
    T38_DATA_V29_7200,
    T38_DATA_V29_9600,
    T38_DATA_V17_7200,
    T38_DATA_V17_9600,
    T38_DATA_V17_12000,
    T38_DATA_V17_14400,
    T38_DATA_V8,
    T38_DATA_V34_PRI_RATE,
    T38_DATA_V34_CC_1200,
    T38_DATA_V34_PRI_CH,
    T38_DATA_V33_12000,
    T38_DATA_V33_14400
};

enum t38_field_types_e
{
    T38_FIELD_HDLC_DATA = 0,
    T38_FIELD_HDLC_SIG_END,
    T38_FIELD_HDLC_FCS_OK,
    T38_FIELD_HDLC_FCS_BAD,
    T38_FIELD_HDLC_FCS_OK_SIG_END,
    T38_FIELD_HDLC_FCS_BAD_SIG_END,
    T38_FIELD_T4_NON_ECM_DATA,
    T38_FIELD_T4_NON_ECM_SIG_END,
    T38_FIELD_CM_MESSAGE,
    T38_FIELD_JM_MESSAGE,
    T38_FIELD_CI_MESSAGE,
    T38_FIELD_V34RATE
};

enum
{
    T38_TYPE_OF_MSG_T30_INDICATOR = 0,
    T38_TYPE_OF_MSG_T30_DATA
};

typedef struct
{
    int                 numocts;
    const uint8_t       *data;
} asn1_dyn_oct_str_t;

typedef struct
{
    uint8_t             field_data_present;
    unsigned int        field_type;
    asn1_dyn_oct_str_t  field_data;
} data_field_element_t;

typedef struct t38_state_s t38_state_t;

typedef int (t38_tx_packet_handler_t)(t38_state_t *s, void *user_data, const uint8_t *buf, int len, int count);

enum
{
    T38_TRANSPORT_UDPTL,
    T38_TRANSPORT_RTP,
    T38_TRANSPORT_TCP
};

#define T38_RX_BUF_LEN  2048
#define T38_TX_BUF_LEN  16384

#include <sys/time.h>

struct t38_state_s
{
    int gateway;

    /*! Method 1: Local generation of TCF (required for use with TCP).
        Method 2: Transfer of TCF is required for use with UDP (UDPTL or RTP).
        Method 2 is not recommended for use with TCP. */
    int data_rate_management_method;
    
    /*! The emitting gateway may indicate a preference for either UDP/UDPTL, or
        UDP/RTP, or TCP for transport of T.38 IFP Packets. The receiving device
        selects the transport protocol. */
    int data_transport_protocol;

    /*! Indicates the capability to remove and insert fill bits in Phase C, non-ECM
        data to reduce bandwidth in the packet network. Optional. See Note. */
    int fill_bit_removal;

    /*! Indicates the ability to convert to/from MMR from/to the line format to
        improve the compression of the data, and reduce the bandwidth, in the
        packet network. Optional. See Note. */
    int mmr_transcoding;

    /*! Indicates the ability to convert to/from JBIG to reduce bandwidth. Optional.
        See Note. */
    int jbig_transcoding;

    /*! For UDP (UDPTL or RTP) modes, this option indicates the maximum
        number of octets that can be stored on the remote device before an overflow
        condition occurs. It is the responsibility of the transmitting application to
        limit the transfer rate to prevent an overflow. The negotiated data rate
        should be used to determine the rate at which data is being removed from
        the buffer. */
    int max_buffer_size;

    /*! This option indicates the maximum size of a UDPTL packet or the
        maximum size of the payload within an RTP packet that can be accepted by
        the remote device. */
    int max_datagram_size;

    /*! This is the version number of ITU-T Rec. T.38. New versions shall be
        compatible with previous versions. */
    int t38_version;
    
    /*! Internet aware FAX (IAF) mode. */
    int iaf;

    /*! NOTE - Bandwidth reduction shall only be done on suitable Phase C data, i.e., MH, MR and - in the case
        of transcoding to JBIG - MMR. MMR and JBIG require reliable data transport such as that provided by
        TCP. When transcoding is selected, it shall be applied to every suitable page in a call. */

    uint8_t hdlc_buf[256];
    int hdlc_len;
    uint8_t hdlc_buf2[256];
    int hdlc_len2;
    uint8_t hdlc_buf3[256];
    int hdlc_len3;
    int hdlc_ptr;
    int timed_step;

    uint8_t rx_data[T38_RX_BUF_LEN];
    int rx_data_bytes;

    uint8_t tx_data[T38_TX_BUF_LEN];
    int tx_in_bytes;
    int tx_out_bytes;
    /*! \brief The location of the most recent EOL marker in the non-ECM data buffer */
    int tx_latest_eol_byte;
    unsigned int bit_stream;
    uint8_t flow_control_fill_octet;

    int bit_no;
    int current_byte;
    int flow_control_fill_octets;

    int current_rx_indicator;
    int next_rx_indicator;

    int current_tx_indicator;
    int next_tx_indicator;
    int current_tx_data;

    int tx_seq_no;
    int rx_seq_no;
    int rx_expected_seq_no;

    /*! \brief TRUE if we are at the TCF exchange stage */
    int at_tcf;
    /*! \brief TRUE if we are in the initial all ones part of non-ECM transmission */
    int at_initial_all_ones;
    /*! \brief The current bit rate for the fast modem. */
    int bit_rate;
    /*! \brief The current fast modem. */
    int fast_modem;
    /*! \brief Use talker echo protection when transmitting. */
    int use_tep;    
    /*! \brief TRUE is a carrier is presnt. Otherwise FALSE. */
    int rx_signal_present;

    /*! \brief A tone generator context used to generate supervisory tones during
               FAX handling. */
    tone_gen_state_t tone_gen;
    /*! \brief An HDLC context used when receiving HDLC over V.21 messages. */
    hdlc_rx_state_t hdlcrx;

    /*! \brief HDLC data used when transmitting HDLC over V.21 messages. */
    int hdlctx_num_bits;
    int hdlctx_idle_byte;
    int hdlctx_len;
    uint8_t hdlctx_buffer[HDLC_MAXFRAME_LEN + 2];
    int hdlctx_pos;
    int hdlctx_byte;
    int hdlctx_bits;

    int hdlc_underflow_reported;
    /*! \brief A V.21 FSK modem context used when transmitting HDLC over V.21
               messages. */
    fsk_tx_state_t v21tx;
    /*! \brief A V.21 FSK modem context used when receiving HDLC over V.21
               messages. */
    fsk_rx_state_t v21rx;

#if defined(ENABLE_V17)
    /*! \brief A V.17 modem context used when sending FAXes at 7200bps, 9600bps
               12000bps or 14400bps*/
    v17_tx_state_t v17tx;
    /*! \brief A V.29 modem context used when receiving FAXes at 7200bps, 9600bps
               12000bps or 14400bps*/
    v17_rx_state_t v17rx;
#endif

    /*! \brief A V.29 modem context used when sending FAXes at 7200bps or
               9600bps */
    v29_tx_state_t v29tx;
    /*! \brief A V.29 modem context used when receiving FAXes at 7200bps or
               9600bps */
    v29_rx_state_t v29rx;

    /*! \brief A V.27ter modem context used when sending FAXes at 2400bps or
               4800bps */
    v27ter_tx_state_t v27ter_tx;
    /*! \brief A V.27ter modem context used when receiving FAXes at 2400bps or
               4800bps */
    v27ter_rx_state_t v27ter_rx;

    int v21_rx_active;
    int fast_rx_active;
    int since_last_tx_packet;
    int silence;
    int short_train;
    int data_final;

    t30_state_t t30_state;
    t38_tx_packet_handler_t *tx_packet_handler;
    void *tx_packet_user_data;

    /*! \brief TRUE if we need to corrupt the HDLC frame in progress, so the receiver cannot
               interpret it. */
    int corrupt_the_frame;
    
    int current_rx_type;
    int current_tx_type;

    int octets_per_non_ecm_packet;

    int32_t samples;
    int32_t next_send_samples;
    
    logging_state_t logging;
};

#ifdef __cplusplus
extern "C" {
#endif

/*! \brief Convert the code for an indicator to a short text name.
    \param indicator The type of indicator.
    \return A pointer to a short text name for the indicator. */
const char *t38_indicator(int indicator);

/*! \brief Convert the code for a type of data to a short text name.
    \param data_type The data type.
    \return A pointer to a short text name for the data type. */
const char *t38_data_type(int data_type);

/*! \brief Convert the code for a type of data field to a short text name.
    \param field_type The field type.
    \return A pointer to a short text name for the field type. */
const char *t38_field_type(int field_type);

int t38_send_data(t38_state_t *s, int data_type, int field_type, const uint8_t *msg, int msglen);

int t38_send_indicator(t38_state_t *s, int indicator, int count);

int t38_gateway_process_rx_indicator(t38_state_t *s, int indicator);

int t38_gateway_process_rx_data(t38_state_t *s, int data_type, int field_type, const uint8_t *buf, int len);

int t38_gateway_process_rx_missing(t38_state_t *s, int seq_no);

int t38_terminal_process_rx_indicator(t38_state_t *s, int indicator);

int t38_terminal_process_rx_data(t38_state_t *s, int data_type, int field_type, const uint8_t *buf, int len);

int t38_terminal_process_rx_missing(t38_state_t *s, int seq_no);

int t38_send_timeout(t38_state_t *s, int samples);

/*! \brief Initialise a termination mode T.38 context.
    \param s The T.38 context.
    \param calling_party TRUE if the context is for a calling party. FALSE if the
           context is for an answering party.
    \param tx_packet_handler ???.
    \param tx_packet_user_data An opaque pointer passed to the tx_packet_handler routine.
    \return A pointer to the termination mode T.38 context, or NULL if there was a problem. */
t38_state_t *t38_terminal_init(t38_state_t *s,
                               int calling_party,
                               t38_tx_packet_handler_t *tx_packet_handler,
                               void *tx_packet_user_data);

/*! \brief Initialise a gateway mode T.38 context.
    \param s The T.38 context.
    \param calling_party TRUE if the context is for a calling party. FALSE if the
           context is for an answering party.
    \param tx_packet_handler ???.
    \param tx_packet_user_data An opaque pointer passed to the tx_packet_handler routine.
    \return A pointer to the termination mode T.38 context, or NULL if there was a problem. */
t38_state_t *t38_gateway_init(t38_state_t *s,
                              int calling_party,
                              t38_tx_packet_handler_t *tx_packet_handler,
                              void *tx_packet_user_data);

/*! \brief Process a received T.38 IFP packet.
    \param s The T.38 context.
    \param seq_no The packet sequence number.
    \param buf The packet contents.
    \param len The length of the packet contents.
    \return 0 for OK, else -1. */
int t38_rx_ifp_packet(t38_state_t *s, int seq_no, const uint8_t *buf, int len);

/*! Process a block of received FAX audio samples.
    \brief Process a block of received FAX audio samples.
    \param s The T.38 context.
    \param amp The audio sample buffer.
    \param len The number of samples in the buffer.
    \return The number of samples unprocessed. */
int t38_rx(t38_state_t *s, const int16_t *amp, int len);

/*! Generate a block of FAX audio samples.
    \brief Generate a block of FAX audio samples.
    \param s The T.38 context.
    \param amp The audio sample buffer.
    \param max_len The number of samples to be generated.
    \return The number of samples actually generated.
*/
int t38_tx(t38_state_t *s, int16_t *amp, int max_len);

/*! Set the method to be used for data rate management, as per the T.38 spec.
    \param s The T.38 context.
    \param method 1 for pass TCF across the T.38 link, 2 for handle TCF locally.
*/
void t38_set_data_rate_management_method(t38_state_t *s, int method);

/*! Set the data transport protocol.
    \param s The T.38 context.
    \param data_transport_protocol UDPTL, RTP or TPKT.
*/
void t38_set_data_transport_protocol(t38_state_t *s, int data_transport_protocol);

/*! Set the non-ECM fill bit removal mode.
    \param s The T.38 context.
    \param fill_bit_removal TRUE to remove fill bits across the T.38 link, else FALSE.
*/
void t38_set_fill_bit_removal(t38_state_t *s, int fill_bit_removal);

/*! Set the MMR transcoding mode.
    \param s The T.38 context.
    \param mmr_transcoding TRUE to transcode to MMR across the T.38 link, else FALSE.
*/
void t38_set_mmr_transcoding(t38_state_t *s, int mmr_transcoding);

/*! Set the JBIG transcoding mode.
    \param s The T.38 context.
    \param jbig_transcoding TRUE to transcode to JBIG across the T.38 link, else FALSE.
*/
void t38_set_jbig_transcoding(t38_state_t *s, int jbig_transcoding);

void t38_set_max_buffer_size(t38_state_t *s, int max_buffer_size);

void t38_set_max_datagram_size(t38_state_t *s, int max_datagram_size);

/*! Set the T.38 version to be emulated.
    \param s The T.38 context.
    \param t38_version Version number, as in the T.38 spec.
*/
void t38_set_t38_version(t38_state_t *s, int t38_version);

#ifdef __cplusplus
}
#endif

#endif
/*- End of file ------------------------------------------------------------*/
