#define ENABLE_V17
//#define LOG_FAX_AUDIO
/*
 * SpanDSP - a series of DSP components for telephony
 *
 * fax.c - Analogue line ITU T.30 FAX transfer processing
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2003, 2005 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: fax.c,v 1.26 2006/05/19 16:43:46 steveu Exp $
 */

/*! \file */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <inttypes.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <tgmath.h>
#include <assert.h>
#include <fcntl.h>
#include <time.h>
#if defined(LOG_FAX_AUDIO)
#include <unistd.h>
#endif

#include <tiffio.h>

#include "spandsp/telephony.h"
#include "spandsp/logging.h"
#include "spandsp/queue.h"
#include "spandsp/dc_restore.h"
#include "spandsp/power_meter.h"
#include "spandsp/complex.h"
#include "spandsp/tone_generate.h"
#include "spandsp/async.h"
#include "spandsp/hdlc.h"
#include "spandsp/fsk.h"
#include "spandsp/v29rx.h"
#include "spandsp/v29tx.h"
#include "spandsp/v27ter_rx.h"
#include "spandsp/v27ter_tx.h"
#if defined(ENABLE_V17)
#include "spandsp/v17rx.h"
#include "spandsp/v17tx.h"
#endif
#include "spandsp/t4.h"

#include "spandsp/t30_fcf.h"
#include "spandsp/t35.h"
#include "spandsp/t30.h"

#include "spandsp/fax.h"

#define ms_to_samples(t)            (((t)*SAMPLE_RATE)/1000)

static void fax_send_hdlc(void *user_data, const uint8_t *msg, int len)
{
    fax_state_t *s;

    s = (fax_state_t *) user_data;
    /* The spec says 1s +-15% of preamble. So, the minimum is 32 octets. */
    if (s->first_tx_hdlc_frame)
        hdlc_tx_preamble(&(s->hdlctx), 32);
    else
        hdlc_tx_preamble(&(s->hdlctx), 2);
    hdlc_tx_frame(&(s->hdlctx), msg, len);
    hdlc_tx_preamble(&(s->hdlctx), 1);
    s->first_tx_hdlc_frame = FALSE;
}
/*- End of function --------------------------------------------------------*/

static int rx_dummy(void *s, int16_t *amp, int len)
{
    return 0;
}
/*- End of function --------------------------------------------------------*/

int fax_rx(fax_state_t *s, int16_t *amp, int len)
{
    int i;

#if defined(LOG_FAX_AUDIO)
    if (s->fax_audio_rx_log >= 0)
        write(s->fax_audio_rx_log, amp, len*sizeof(int16_t));
#endif
    for (i = 0;  i < len;  i++)
        amp[i] = dc_restore(&(s->dc_restore), amp[i]);
    s->rx_handler(s->rx_user_data, amp, len);
    t30_timer_update(&(s->t30_state), len);
    return  0;
}
/*- End of function --------------------------------------------------------*/

static int silence_gen(fax_state_t *s, int16_t *amp, int max_len)
{
    int len;
    
    len = (max_len > s->silent_samples)  ?  s->silent_samples  :  max_len;
    s->silent_samples -= len;
    memset(amp, 0, len*sizeof(int16_t));
    return len;
}
/*- End of function --------------------------------------------------------*/

static int tx_dummy(void *s, int16_t *amp, int max_len)
{
    return 0;
}
/*- End of function --------------------------------------------------------*/

int fax_tx(fax_state_t *s, int16_t *amp, int max_len)
{
    int len;
#if defined(LOG_FAX_AUDIO)
    int required_len;
    
    required_len = max_len;
#endif
    len = s->tx_handler(s->tx_user_data, amp, max_len);
    if (len < max_len)
    {
        /* Allow for one change of tx handler within a block */
        if (s->next_tx_handler)
        {
            s->tx_handler = s->next_tx_handler;
            s->tx_user_data = s->next_tx_user_data;
            s->next_tx_handler = NULL;
            len += s->tx_handler(s->tx_user_data, amp + len, max_len - len);
        }
        else
        {
            /* If there is nothing else to change to, use the dummy routine */
            s->tx_handler = &tx_dummy;
            s->tx_user_data = s;
        }
        if (len < max_len  &&  s->current_tx_type != T30_MODEM_NONE  &&  s->current_tx_type != T30_MODEM_DONE)
            t30_send_complete(&(s->t30_state));
    }
#if defined(LOG_FAX_AUDIO)
    if (s->fax_audio_tx_log >= 0)
    {
        if (len < required_len)
            memset(amp + len, 0, (required_len - len)*sizeof(int16_t));
        write(s->fax_audio_tx_log, amp, required_len*sizeof(int16_t));
    }
#endif
    return len;
}
/*- End of function --------------------------------------------------------*/

static void fax_set_rx_type(void *user_data, int type, int short_train)
{
    fax_state_t *s;

    s = (fax_state_t *) user_data;
    span_log(&s->logging, SPAN_LOG_FLOW, "Set rx type %d\n", type);
    if (s->current_rx_type == type)
        return;
    s->current_rx_type = type;
    switch (type)
    {
    case T30_MODEM_V21:
        if (s->flush_handler)
            s->flush_handler(s, s->flush_user_data, 3);
        hdlc_rx_init(&(s->hdlcrx), FALSE, FALSE, 1, t30_hdlc_accept, &(s->t30_state));
        fsk_rx_init(&(s->v21rx), &preset_fsk_specs[FSK_V21CH2], TRUE, (put_bit_func_t) hdlc_rx_bit, &(s->hdlcrx));
        s->rx_handler = (fax_rx_handler_t *) &(fsk_rx);
        s->rx_user_data = &(s->v21rx);
        break;
    case T30_MODEM_V27TER_2400:
        v27ter_rx_restart(&(s->v27ter_rx), 2400, FALSE);
        s->rx_handler = (fax_rx_handler_t *) &(v27ter_rx);
        s->rx_user_data = &(s->v27ter_rx);
        break;
    case T30_MODEM_V27TER_4800:
        v27ter_rx_restart(&(s->v27ter_rx), 4800, FALSE);
        s->rx_handler = (fax_rx_handler_t *) &(v27ter_rx);
        s->rx_user_data = &(s->v27ter_rx);
        break;
    case T30_MODEM_V29_7200:
        v29_rx_restart(&(s->v29rx), 7200, FALSE);
        s->rx_handler = (fax_rx_handler_t *) &(v29_rx);
        s->rx_user_data = &(s->v29rx);
        break;
    case T30_MODEM_V29_9600:
        v29_rx_restart(&(s->v29rx), 9600, FALSE);
        s->rx_handler = (fax_rx_handler_t *) &(v29_rx);
        s->rx_user_data = &(s->v29rx);
        break;
#if defined(ENABLE_V17)
    case T30_MODEM_V17_7200:
        v17_rx_restart(&(s->v17rx), 7200, short_train);
        s->rx_handler = (fax_rx_handler_t *) &(v17_rx);
        s->rx_user_data = &(s->v17rx);
        break;
    case T30_MODEM_V17_9600:
        v17_rx_restart(&(s->v17rx), 9600, short_train);
        s->rx_handler = (fax_rx_handler_t *) &(v17_rx);
        s->rx_user_data = &(s->v17rx);
        break;
    case T30_MODEM_V17_12000:
        v17_rx_restart(&(s->v17rx), 12000, short_train);
        s->rx_handler = (fax_rx_handler_t *) &(v17_rx);
        s->rx_user_data = &(s->v17rx);
        break;
    case T30_MODEM_V17_14400:
        v17_rx_restart(&(s->v17rx), 14400, short_train);
        s->rx_handler = (fax_rx_handler_t *) &(v17_rx);
        s->rx_user_data = &(s->v17rx);
        break;
#endif
    case T30_MODEM_DONE:
        span_log(&s->logging, SPAN_LOG_FLOW, "FAX exchange complete\n");
    default:
        s->rx_handler = &(rx_dummy);
        s->rx_user_data = s;
        break;
    }
}
/*- End of function --------------------------------------------------------*/

static void fax_set_tx_type(void *user_data, int type, int short_train)
{
    fax_state_t *s;
    tone_gen_descriptor_t tone_desc;

    s = (fax_state_t *) user_data;
    span_log(&s->logging, SPAN_LOG_FLOW, "Set tx type %d\n", type);
    if (s->current_tx_type == type)
        return;
    s->current_tx_type = type;
    s->first_tx_hdlc_frame = TRUE;
    switch (type)
    {
    case T30_MODEM_PAUSE:
        s->silent_samples += ms_to_samples(short_train);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = NULL;
        break;
    case T30_MODEM_CED:
        /* 0.2s of silence, then 2.6s to 4s of 2100Hz tone, then 75ms of silence. */
        s->silent_samples += ms_to_samples(200);
        make_tone_gen_descriptor(&tone_desc,
                                 2100,
                                 -11,
                                 0,
                                 0,
                                 2600,
                                 75,
                                 0,
                                 0,
                                 FALSE);
        tone_gen_init(&(s->tone_gen), &tone_desc);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(tone_gen);
        s->next_tx_user_data = &(s->tone_gen);
        break;
    case T30_MODEM_CNG:
        /* 0.5s of 1100Hz + 3.0s of silence repeating */
        make_tone_gen_descriptor(&tone_desc,
                                 1100,
                                 -11,
                                 0,
                                 0,
                                 500,
                                 3000,
                                 0,
                                 0,
                                 TRUE);
        tone_gen_init(&(s->tone_gen), &tone_desc);
        s->tx_handler = (fax_tx_handler_t *) &(tone_gen);
        s->tx_user_data = &(s->tone_gen);
        s->next_tx_handler = NULL;
        break;
    case T30_MODEM_V21:
        if (s->current_tx_type > T30_MODEM_V21)
        {
            /* Pause before switching from phase C, as per T.30. If we omit this, the receiver
               might not see the carrier fall between the high speed and low speed sections. */
            s->silent_samples += ms_to_samples(75);
            fsk_tx_init(&(s->v21tx), &preset_fsk_specs[FSK_V21CH2], (get_bit_func_t) hdlc_tx_getbit, &(s->hdlctx));
            s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
            s->tx_user_data = s;
            s->next_tx_handler = (fax_tx_handler_t *) &(fsk_tx);
            s->next_tx_user_data = &(s->v21tx);
        }
        else
        {
            fsk_tx_init(&(s->v21tx), &preset_fsk_specs[FSK_V21CH2], (get_bit_func_t) hdlc_tx_getbit, &(s->hdlctx));
            s->tx_handler = (fax_tx_handler_t *) &(fsk_tx);
            s->tx_user_data = &(s->v21tx);
            s->next_tx_handler = NULL;
        }
        break;
    case T30_MODEM_V27TER_2400:
        s->silent_samples += ms_to_samples(75);
        v27ter_tx_restart(&(s->v27ter_tx), 2400, FALSE);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(v27ter_tx);
        s->next_tx_user_data = &(s->v27ter_tx);
        break;
    case T30_MODEM_V27TER_4800:
        s->silent_samples += ms_to_samples(75);
        v27ter_tx_restart(&(s->v27ter_tx), 4800, FALSE);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(v27ter_tx);
        s->next_tx_user_data = &(s->v27ter_tx);
        break;
    case T30_MODEM_V29_7200:
        s->silent_samples += ms_to_samples(75);
        v29_tx_restart(&(s->v29tx), 7200, FALSE);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(v29_tx);
        s->next_tx_user_data = &(s->v29tx);
        break;
    case T30_MODEM_V29_9600:
        s->silent_samples += ms_to_samples(75);
        v29_tx_restart(&(s->v29tx), 9600, FALSE);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(v29_tx);
        s->next_tx_user_data = &(s->v29tx);
        break;
#if defined(ENABLE_V17)
    case T30_MODEM_V17_7200:
        s->silent_samples += ms_to_samples(75);
        v17_tx_restart(&(s->v17tx), 7200, FALSE, short_train);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(v17_tx);
        s->next_tx_user_data = &(s->v17tx);
        break;
    case T30_MODEM_V17_9600:
        s->silent_samples += ms_to_samples(75);
        v17_tx_restart(&(s->v17tx), 9600, FALSE, short_train);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(v17_tx);
        s->next_tx_user_data = &(s->v17tx);
        break;
    case T30_MODEM_V17_12000:
        s->silent_samples += ms_to_samples(75);
        v17_tx_restart(&(s->v17tx), 12000, FALSE, short_train);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(v17_tx);
        s->next_tx_user_data = &(s->v17tx);
        break;
    case T30_MODEM_V17_14400:
        s->silent_samples += ms_to_samples(75);
        v17_tx_restart(&(s->v17tx), 14400, FALSE, short_train);
        s->tx_handler = (fax_tx_handler_t *) &(silence_gen);
        s->tx_user_data = s;
        s->next_tx_handler = (fax_tx_handler_t *) &(v17_tx);
        s->next_tx_user_data = &(s->v17tx);
        break;
#endif
    case T30_MODEM_DONE:
        span_log(&s->logging, SPAN_LOG_FLOW, "FAX exchange complete\n");
    default:
        s->tx_handler = &(tx_dummy);
        s->tx_user_data = s;
        s->next_tx_handler = NULL;
        break;
    }
}
/*- End of function --------------------------------------------------------*/

int fax_init(fax_state_t *s, int calling_party, void *user_data)
{
    memset(s, 0, sizeof(*s));
    span_log_init(&s->logging, SPAN_LOG_NONE, NULL);
    span_log_set_protocol(&s->logging, "FAX");
    t30_init(&(s->t30_state), calling_party, user_data);

    s->t30_state.set_rx_type_handler = fax_set_rx_type;
    s->t30_state.set_rx_type_user_data = (void *) s;
    s->t30_state.set_tx_type_handler = fax_set_tx_type;
    s->t30_state.set_tx_type_user_data = (void *) s;
    s->t30_state.send_hdlc_handler = fax_send_hdlc;
    s->t30_state.send_hdlc_user_data = (void *) s;

    hdlc_rx_init(&(s->hdlcrx), FALSE, FALSE, 1, t30_hdlc_accept, &(s->t30_state));
    fsk_rx_init(&(s->v21rx), &preset_fsk_specs[FSK_V21CH2], TRUE, (put_bit_func_t) hdlc_rx_bit, &(s->hdlcrx));
    hdlc_tx_init(&(s->hdlctx), FALSE, t30_send_complete, &(s->t30_state));
    s->first_tx_hdlc_frame = TRUE;
    fsk_tx_init(&(s->v21tx), &preset_fsk_specs[FSK_V21CH2], (get_bit_func_t) hdlc_tx_getbit, &(s->hdlctx));
#if defined(ENABLE_V17)
    v17_rx_init(&(s->v17rx), 14400, t30_non_ecm_putbit, &(s->t30_state));
    v17_tx_init(&(s->v17tx), 14400, FALSE, t30_non_ecm_getbit, &(s->t30_state));
#endif
    v29_rx_init(&(s->v29rx), 9600, t30_non_ecm_putbit, &(s->t30_state));
    v29_tx_init(&(s->v29tx), 9600, FALSE, t30_non_ecm_getbit, &(s->t30_state));
    v27ter_rx_init(&(s->v27ter_rx), 4800, t30_non_ecm_putbit, &(s->t30_state));
    v27ter_tx_init(&(s->v27ter_tx), 4800, FALSE, t30_non_ecm_getbit, &(s->t30_state));
    dc_restore_init(&(s->dc_restore));
    t30_restart(&(s->t30_state));
#if defined(LOG_FAX_AUDIO)
    {
        char buf[100 + 1];
        struct tm *tm;
        time_t now;

        time(&now);
        tm = localtime(&now);
        sprintf(buf,
                "/tmp/fax-rx-audio-%x-%02d%02d%02d%02d%02d%02d",
                user_data,
                tm->tm_year%100,
                tm->tm_mon + 1,
                tm->tm_mday,
                tm->tm_hour,
                tm->tm_min,
                tm->tm_sec);
        s->fax_audio_rx_log = open(buf, O_CREAT | O_TRUNC | O_WRONLY, 0666);
        sprintf(buf,
                "/tmp/fax-tx-audio-%x-%02d%02d%02d%02d%02d%02d",
                user_data,
                tm->tm_year%100,
                tm->tm_mon + 1,
                tm->tm_mday,
                tm->tm_hour,
                tm->tm_min,
                tm->tm_sec);
        s->fax_audio_tx_log = open(buf, O_CREAT | O_TRUNC | O_WRONLY, 0666);
    }
#endif
    return 0;
}
/*- End of function --------------------------------------------------------*/

int fax_release(fax_state_t *s)
{
    t30_release(&s->t30_state);
    return 0;
}
/*- End of function --------------------------------------------------------*/

void fax_set_flush_handler(fax_state_t *s, fax_flush_handler_t *handler, void *user_data)
{
    s->flush_handler = handler;
    s->flush_user_data = user_data;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
