/** @file scim_hangul_imengine.cpp
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (C) 2004-2006 Choe Hwanjin
 * Copyright (c) 2004-2006 James Su <suzhe@tsinghua.org.cn>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * $Id: scim_hangul_imengine.cpp,v 1.8 2006/02/05 15:17:17 hwanjin Exp $
 */

#define Uses_SCIM_UTILITY
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_LOOKUP_TABLE
#define Uses_SCIM_CONFIG_BASE

#ifdef HAVE_CONFIG_H
  #include <config.h>
#endif

#include <scim.h>
#include "scim_hangul_imengine.h"

#ifdef HAVE_GETTEXT
  #include <libintl.h>
  #define _(String) dgettext(GETTEXT_PACKAGE,String)
  #define N_(String) (String)
#else
  #define _(String) (String)
  #define N_(String) (String)
  #define bindtextdomain(Package,Directory)
  #define textdomain(domain)
  #define bind_textdomain_codeset(domain,codeset)
#endif

#define scim_module_init hangul_LTX_scim_module_init
#define scim_module_exit hangul_LTX_scim_module_exit
#define scim_imengine_module_init hangul_LTX_scim_imengine_module_init
#define scim_imengine_module_create_factory hangul_LTX_scim_imengine_module_create_factory

#define SCIM_CONFIG_IMENGINE_HANGUL_SHOW_CANDIDATE_COMMENT      "/IMEngine/Hangul/ShowCandidateComment"
#define SCIM_CONFIG_IMENGINE_HANGUL_HANGUL_HANJA_KEY            "/IMEngine/Hangul/HangulHanjaKey"


#ifndef SCIM_HANGUL_ICON_FILE
    #define SCIM_HANGUL_ICON_FILE           (SCIM_ICONDIR "/scim-hangul.png")
#endif

struct CandidateItem {
    ucs4_t      ch;
    const char *comment;
};

#include "candidatetable.h"

static HangulFactoryData hangul_factory_data [] = {
    {
        "d75857a5-4148-4745-89e2-1da7ddaf70a9",
        N_("2bul"),
        HANGUL_KEYBOARD_2
    },
    {
        "353a81d2-a065-4b3b-bf67-ee457f7b72a8",
        N_("3bul 2bul-shifted"),
        HANGUL_KEYBOARD_32
    },
    {
        "c0d227c4-c70f-413a-8a35-a72155cca459",
        N_("3bul 390"),
        HANGUL_KEYBOARD_390
    },
    {
        "a32d6d4f-2aef-40e7-95e0-8ff73c6d855a",
        N_("3bul Final"),
        HANGUL_KEYBOARD_3FINAL
    },
    {
        "56f62c53-ef31-457c-86bf-9ff692db5209",
        N_("3bul No-Shift"),
        HANGUL_KEYBOARD_3NOSHIFT
    },
    {
        "888b8e80-3419-4fcc-9110-92de45774053",
        N_("3bul Yetgeul"),
        HANGUL_KEYBOARD_3YETGUL
    }
};

#define HANGUL_NUM_OF_FACTORY (sizeof(hangul_factory_data)/sizeof(hangul_factory_data[0]))

static ConfigPointer _scim_config (0);

extern "C" {
    void scim_module_init (void)
    {
        bindtextdomain (GETTEXT_PACKAGE, SCIM_HANGUL_LOCALEDIR);
        bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
    }

    void scim_module_exit (void)
    {
        _scim_config.reset ();
    }

    uint32 scim_imengine_module_init (const ConfigPointer &config)
    {
        SCIM_DEBUG_IMENGINE(1) << "Initialize Hangul Engine: " << HANGUL_NUM_OF_FACTORY << "\n";

        _scim_config = config;

        return HANGUL_NUM_OF_FACTORY;
    }

    IMEngineFactoryPointer scim_imengine_module_create_factory (uint32 engine)
    {
        if (engine >= HANGUL_NUM_OF_FACTORY)
            return 0;

        HangulFactory *factory = 0;

        try {
            factory = new HangulFactory (_scim_config, hangul_factory_data [engine]);
        } catch (...) {
            delete factory;
            factory = 0;
        }

        return factory;
    }
}

HangulFactory::HangulFactory (const ConfigPointer     &config,
                              const HangulFactoryData &data)
                        
    : m_uuid                   (data.uuid),
      m_name                   (_(data.name)),
      m_keyboard_type          (data.keyboard_type),
      m_show_candidate_comment (true)
{
    if (!config.null ()) {
        m_show_candidate_comment =
            config->read (String (SCIM_CONFIG_IMENGINE_HANGUL_SHOW_CANDIDATE_COMMENT),
                          m_show_candidate_comment);

        String str;

        str = config->read (String (SCIM_CONFIG_IMENGINE_HANGUL_HANGUL_HANJA_KEY),
                            String ("Hangul_Hanja,F9"));

        scim_string_to_key_list (m_hangul_hanja_keys, str);
    }

    if (!m_hangul_hanja_keys.size ()) {
        m_hangul_hanja_keys.push_back (KeyEvent (SCIM_KEY_Hangul_Hanja, 0));
        m_hangul_hanja_keys.push_back (KeyEvent (SCIM_KEY_F9, 0));
    }

    set_languages ("ko");
}

HangulFactory::~HangulFactory ()
{
}

WideString
HangulFactory::get_name () const
{
    return utf8_mbstowcs (m_name);
}

WideString
HangulFactory::get_authors () const
{
    return WideString ();
}

WideString
HangulFactory::get_credits () const
{
    return WideString ();
}

WideString
HangulFactory::get_help () const
{
    return WideString ();
}

String
HangulFactory::get_uuid () const
{
    return m_uuid;
}

String
HangulFactory::get_icon_file () const
{
    return String (SCIM_HANGUL_ICON_FILE);
}

IMEngineInstancePointer
HangulFactory::create_instance (const String &encoding, int id)
{
    SCIM_DEBUG_IMENGINE(1) << "create_instance: HangulInstance.\n";
    return new HangulInstance (this, encoding, id);
}

HangulInstance::HangulInstance (HangulFactory *factory,
                                const String  &encoding,
                                int            id)
    : IMEngineInstanceBase (factory, encoding, id),
      m_factory (factory),
      m_prev_key (0,0),
      m_input_mode (INPUT_MODE_HANGUL),
      m_output_mode (OUTPUT_MODE_SYLLABLE)
{
    m_hic = hangul_ic_new(factory->m_keyboard_type);
    hangul_ic_reset(m_hic);

    char label[16];
    std::vector <WideString> labels;

    for (int i = 1; i < 10; ++i) {
        snprintf (label, sizeof(label), "%d", i);
        labels.push_back (utf8_mbstowcs (label));
    }

    m_lookup_table.set_candidate_labels (labels);
}

HangulInstance::~HangulInstance ()
{
}

bool
HangulInstance::candidate_key_event (const KeyEvent &key)
{
    switch (key.code) {
        case SCIM_KEY_Return:
        case SCIM_KEY_KP_Enter:
            select_candidate (m_lookup_table.get_cursor_pos_in_current_page ());
            break;
        case SCIM_KEY_BackSpace:
	    hangul_ic_backspace(m_hic);
            hangul_update_preedit_string ();
	    update_candidates ();
            break;
        case SCIM_KEY_Left:
        case SCIM_KEY_h:
        case SCIM_KEY_KP_Subtract:
            m_lookup_table.cursor_up ();
            update_lookup_table (m_lookup_table);
            hangul_update_aux_string ();
            break;
        case SCIM_KEY_Right:
        case SCIM_KEY_l:
        case SCIM_KEY_space:
        case SCIM_KEY_KP_Add:
            m_lookup_table.cursor_down ();
            update_lookup_table (m_lookup_table);
            hangul_update_aux_string ();
            break;
        case SCIM_KEY_Up:
        case SCIM_KEY_Page_Up:
        case SCIM_KEY_k:
            m_lookup_table.page_up ();
            update_lookup_table (m_lookup_table);
            hangul_update_aux_string ();
            break;
        case SCIM_KEY_Down:
        case SCIM_KEY_Page_Down:
        case SCIM_KEY_j:
        case SCIM_KEY_KP_Tab:
            m_lookup_table.page_down ();
            update_lookup_table (m_lookup_table);
            hangul_update_aux_string ();
            break;
        case SCIM_KEY_Escape:
            delete_candidates ();
            break;
        case SCIM_KEY_1: 
        case SCIM_KEY_2: 
        case SCIM_KEY_3: 
        case SCIM_KEY_4: 
        case SCIM_KEY_5: 
        case SCIM_KEY_6: 
        case SCIM_KEY_7: 
        case SCIM_KEY_8: 
        case SCIM_KEY_9: 
            select_candidate (key.code - SCIM_KEY_1);
            break;
        default:
            break;
    }

    return true;
}

bool
HangulInstance::process_key_event (const KeyEvent& rawkey)
{
    SCIM_DEBUG_IMENGINE(1) << "process_key_event.\n";
    KeyEvent key = rawkey.map_to_layout(SCIM_KEYBOARD_Default);

    m_prev_key = key;

    /* ignore key release. */
    if (key.is_key_release ())
        return false;

    if (is_backspace_key(key)) {
        bool ret = hangul_ic_backspace(m_hic);
        if (ret) {
	    if (m_lookup_table.number_of_candidates ()) {
		WideString wstr = get_preedit_string();
		if (wstr.empty())
		    delete_candidates ();
		else
		    update_candidates ();
	    }
	    hangul_update_preedit_string ();
	}
        return ret;
    }

    /* toggle candidate table */
    if (is_hangul_hanja_key (key)) {
        if (m_lookup_table.number_of_candidates ())
            delete_candidates ();
        else
            update_candidates ();

        return true;
    }

    /* ignore shift keys */
    if (key.code == SCIM_KEY_Shift_L || key.code == SCIM_KEY_Shift_R)
        return false;

    /* reset on modifier-on keys */
    if (key.is_control_down() || key.is_alt_down()) {
	reset ();
        return false;
    }

    /* candidate keys */
    if (m_lookup_table.number_of_candidates ()) {
        return candidate_key_event (key);
    }

    if (key.code >= SCIM_KEY_exclam && key.code <= SCIM_KEY_asciitilde) {
	/* main hangul composing process */
	int ascii = key.get_ascii_code();
	if (key.is_caps_lock_down()) {
	    if (isupper(ascii))
		ascii = tolower(ascii);
	    else if (islower(ascii))
		ascii = toupper(ascii);
	}

	bool ret = hangul_ic_filter(m_hic, ascii);

	WideString wstr;
	wstr = get_commit_string ();
	if (wstr.length ()) {
	    /* Before commit, we set preedit string to null to work arround
	     * some buggy IM implementation, ex) Qt, Evolution */
	    hide_preedit_string ();
	    commit_string(wstr);
	}

	hangul_update_preedit_string ();

	return ret;
    }

    reset ();
    return false;
}

void
HangulInstance::move_preedit_caret (unsigned int pos)
{
}

void
HangulInstance::select_candidate (unsigned int index)
{
    SCIM_DEBUG_IMENGINE(2) << "select_candidate.\n";

    if ((int)index < m_lookup_table.get_current_page_size ()) {
        WideString wstr = m_lookup_table.get_candidate_in_current_page (index);
        hangul_ic_reset (m_hic);
        commit_string (wstr);
        delete_candidates ();
        hangul_update_preedit_string ();
    }
}

void
HangulInstance::update_lookup_table_page_size (unsigned int page_size)
{
    SCIM_DEBUG_IMENGINE(2) << "update_lookup_table_page_size.\n";

    m_lookup_table.set_page_size (page_size);
}

void
HangulInstance::lookup_table_page_up ()
{
    if (!m_lookup_table.number_of_candidates () || !m_lookup_table.get_current_page_start ())
        return;

    SCIM_DEBUG_IMENGINE(2) << "lookup_table_page_up.\n";

    m_lookup_table.page_up ();

    update_lookup_table (m_lookup_table);

    hangul_update_aux_string ();
}

void
HangulInstance::lookup_table_page_down ()
{
    if (m_lookup_table.number_of_candidates () <= 0 ||
        m_lookup_table.get_current_page_start () + m_lookup_table.get_current_page_size () >=
          (int)m_lookup_table.number_of_candidates ())
        return;

    SCIM_DEBUG_IMENGINE(2) << "lookup_table_page_down.\n";

    m_lookup_table.page_down ();

    update_lookup_table (m_lookup_table);

    hangul_update_aux_string ();
}

void
HangulInstance::reset ()
{
    SCIM_DEBUG_IMENGINE(2) << "reset.\n";

    hangul_ic_reset(m_hic);

    hide_preedit_string();

    WideString wstr = get_commit_string();
    if (wstr.length())
        commit_string(wstr);

    delete_candidates ();
}

void
HangulInstance::focus_in ()
{
    SCIM_DEBUG_IMENGINE(2) << "focus_in.\n";

    if (m_lookup_table.number_of_candidates ()) {
        update_lookup_table (m_lookup_table);
        show_lookup_table ();
    } else {
        hide_lookup_table ();
    }

    hangul_update_aux_string ();
}

void
HangulInstance::focus_out ()
{
    SCIM_DEBUG_IMENGINE(2) << "focus_out.\n";
    reset ();
}

void
HangulInstance::trigger_property (const String &property)
{
    SCIM_DEBUG_IMENGINE(2) << "trigger_property.\n";
}

static int
get_index_of_candidate_table (ucs4_t ch)
{
    int first, last, mid;

    /* binary search */
    first = 0;
    last = sizeof (candidate_table)/sizeof (candidate_table [0]) - 1;

    while (first <= last) {
        mid = (first + last) / 2;

        if (ch == candidate_table[mid][0].ch)
            return mid;

        if (ch < candidate_table[mid][0].ch)
            last = mid - 1;
        else
            first = mid + 1;
    }

    return -1;
}

void
HangulInstance::update_candidates ()
{
    WideString wstr = get_preedit_string ();
    if (wstr.empty())
        return; 

    ucs4_t ch = wstr[0];

    m_lookup_table.clear ();
    m_candidate_comments.clear ();

    if (ch > 0) {
        int index = get_index_of_candidate_table (ch);
        if (index != -1) {
            for (const CandidateItem *it = candidate_table [index] + 1; it->ch; ++it) {
                m_lookup_table.append_candidate (it->ch);
                m_candidate_comments.push_back (String (it->comment));
            }

            m_lookup_table.set_page_size (9);
            m_lookup_table.show_cursor ();

            update_lookup_table (m_lookup_table);
            show_lookup_table ();

            hangul_update_aux_string ();
        }
    }
}

void
HangulInstance::delete_candidates ()
{
    m_lookup_table.clear ();
    m_candidate_comments.clear ();
    hide_lookup_table ();
    hide_aux_string ();
}

void
HangulInstance::hangul_update_aux_string ()
{
    if (!m_factory->m_show_candidate_comment || !m_lookup_table.number_of_candidates ()) {
        hide_aux_string ();
        return;
    }

    size_t cursor = m_lookup_table.get_cursor_pos ();

    if (cursor >= m_candidate_comments.size () || !m_candidate_comments [cursor].length ()) {
        hide_aux_string ();
        return;
    }

    update_aux_string (m_lookup_table.get_candidate (cursor) + utf8_mbstowcs (String (" : ") + m_candidate_comments [cursor]));
    show_aux_string ();
}

void
HangulInstance::hangul_update_preedit_string ()
{
    WideString wstr = get_preedit_string ();

    if (wstr.length ()) {
        AttributeList attrs;
        attrs.push_back (Attribute (0, wstr.length (), SCIM_ATTR_DECORATE, SCIM_ATTR_DECORATE_REVERSE));
        show_preedit_string ();
        update_preedit_string (wstr, attrs);
    } else {
        hide_preedit_string ();
    }
}

bool
HangulInstance::match_key_event (const KeyEventList &keys, const KeyEvent &key) const
{
    KeyEventList::const_iterator kit; 

    for (kit = keys.begin (); kit != keys.end (); ++kit) {
        if (key.code == kit->code && key.mask == kit->mask)
            if (!(key.mask & SCIM_KEY_ReleaseMask) || m_prev_key.code == key.code)
                return true;
    }
    return false;
}

//vim: ts=4:nowrap:ai:expandtab:
