# webapp/webrick-servlet.rb provides an adaptor for a webapp and WEBrick.
#
# webapp/webrick-servlet.rb registers a handler for .webrick extension
# using WEBrick::HTTPServlet::FileHandler.add_handler.
# So WEBrick based HTTP server which require webapp/webrick-servlet.rb
# can run a web application using webapp which filename has .webrick
# extension.
#
# Such HTTP server can be defined as follows.
#
#   require 'webapp/webrick-servlet'
#   httpd = WEBrick::HTTPServer.new(:Port => 10080, :DocumentRoot => Dir.getwd)
#   trap(:INT){ httpd.shutdown }
#   httpd.start 
#
# Apart from that, webapp/webrick-servlet.rb provides 
# WebApp::WEBrickServletHandler.load_servlet which load WEBrick servlet
# using webapp.
#
# WebApp::WEBrickServletHandler.load_servlet can be used for constructing
# a web site with a single web application as follows.
#
#   require 'webapp/webrick-servlet'
#   servlet = ARGV.shift
#   httpd = WEBrick::HTTPServer.new(:Port => 10080)
#   trap(:INT){ httpd.shutdown }
#   httpd.mount("/", WebApp::WEBrickServletHandler.load_servlet(servlet))
#   httpd.start
#
# When above server accept a request to http://host:10080/foo/bar,
# the servlet takes /foo/bar as a path_info.
#
# Note that the servlet loading mechanism may be used without webapp
# because the mechanism itself is not webapp dependent.

require 'webrick'

module Hobix
class WebApp
  class WEBrickServletHandler
    LoadedServlets = {}
    def WEBrickServletHandler.get_instance(config, name)
      unless LoadedServlets[name]
        LoadedServlets[name] = load_servlet(name)
      end
      LoadedServlets[name]
    end

    # load a WEBrick servlet written using webapp.
    # WEBrickServletHandler.load_servlet returns a WEBrick servlet
    # generated by WEBrick::HTTPServlet::ProcHandler.
    def WEBrickServletHandler.load_servlet(path)
      begin
        Thread.current[:webrick_load_servlet] = true
        load path, true
        unless Thread.current[:webrick_load_servlet].respond_to? :call
          raise "WEBrick servlet is not registered: #{path}"
        end
        procedure = Thread.current[:webrick_load_servlet]
        return WEBrick::HTTPServlet::ProcHandler.new(procedure)
      ensure
        Thread.current[:webrick_load_servlet] = nil
      end
    end
  end
end
end
WEBrick::HTTPServlet::FileHandler.add_handler('webrick',
  Hobix::WebApp::WEBrickServletHandler)

if $0 == __FILE__
  # usage: [-p port] [docroot|servlet]
  require 'optparse'
  port = 10080
  ARGV.options {|q|
    q.def_option('--help', 'show this message') {puts q; exit(0)}
    q.def_option('--port=portnum', 'specify server port (default: 10080)') {|num| port = num.to_i }
    q.parse!
  }
  docroot = ARGV.shift || Dir.getwd
  httpd = WEBrick::HTTPServer.new(:Port => port)
  trap(:INT){ httpd.shutdown }
  if File.directory? docroot
    httpd.mount("/", WEBrick::HTTPServlet::FileHandler, docroot, WEBrick::Config::HTTP[:DocumentRootOptions])
    httpd.start
  else
    httpd.mount("/", Hobix::WebApp::WEBrickServletHandler.load_servlet(docroot))
    httpd.start
  end
end
