{- Copyright (c) 2007 John Goerzen <jgoerzen@complete.org>
   Please see the COPYRIGHT file -}

module Utils where
import Types
import HSH
import System.Directory
import Control.Monad
import Data.String
import Text.Regex.Posix
import System.Log.Logger

hglibdir = "/usr/lib/hg-buildpackage"

v = debugM "hg-buildpackage"

setverbose = do
  updateGlobalLogger "hg-buildpackage" (setLevel DEBUG)
  updateGlobalLogger "HSH.Command.Run" (setLevel DEBUG)

{- | Extract all UPSTREAM_ or DEBIAN_ tags from the current working
directory -}
gettags :: IO [String]
gettags = run $ "hg -q tags" -|- egrep "^(UPSTREAM|DEBIAN)_" -|- egrepV "_TAG$"

{- | Check the given version.  Return False if any UPSTREAM or DEBIAN
tag is newer than or equal to the givern version. -}
istagok v = gettags >>= (checktags (tag2ver v) . map (tag2ver . parsetag))
    where checktags _ [] = return True
          checktags ver (x:xs) = 
              do r <- run $ ("dpkg", ["--compare-versions", x, "ge", ver])
                 if r
                    then return False
                    else checktags ver xs

hastag :: Tag -> IO Bool
hastag t = gettags >>= (return . elem (show t))

{- | These need control on stdin -}
getcontrolversion = "grep-dctrl -n -sVersion ."
getcontrolsource = "grep-dctrl -n -sSource ."

getwdversion = runSL $ "dpkg-parsechangelog" -|- getcontrolversion  
getwdpackage = runSL $ "dpkg-parsechangelog" -|- getcontrolsource
getcontroltag = 
    do ver <- getwdversion
       pkg <- getwdpackage
       return $ Tag "DEBIAN" pkg ver

isvalidhg fp = doesDirectoryExist (fp ++ "/.hg")

assertvalidhg fp = 
    do isvalid <- isvalidhg fp
       unless isvalid (fail $ fp ++ " is not a valid hg working directory")
       return ()

dsc2control =
    delBetween "^-----BEGIN PGP SIGNED MESSAGE-----" "^Hash: " .
    delBetween "^-----BEGIN PGP SIGNATURE-----" "^^-----END PGP SIGNATURE-----"

-- | Take files lines from DSC
dsc2files =
    dsc2control -|- "grep-dctrl -n -sFiles ." -|- "sed 's/^ \\+//'" -|-
    egrepV "^$"

-- | Take file names from file lines from DSC (expects DSC as input)
dscfiles2fn = map (\x -> split " " x !! 2)

-- | Does this DSC filename have an upstream component?
dschasupstream :: FilePath -> IO Bool
dschasupstream fn = run $
    catFrom [fn] -|- dsc2files -|- dscfiles2fn -|- "egrep -q '\\.diff\\.gz$'"

-- | Get specific files from the DSC
dscgetfile fn pat = catFrom [fn] -|- dsc2files -|- dscfiles2fn -|- egrep pat
-- This should probably go into MissingH

delBetween :: String -> String -> [String] -> [String]
delBetween start end inp =
    case break (=~ start) inp of
      (_, []) -> inp            -- No start divider found; just return
      (before, (x:xs)) -> before ++ case break (=~ end) xs of
                                      (rest, []) -> x : rest -- No end divider
                                      (_, (x:xs)) -> xs

die :: Int -> String -> IO a
die code msg =
    do putStrLn msg
       exit code
