-- | Implement XSLT transformations using xsltproc
module HAppS.Protocols.XSLT
    (xsltFile, xsltString, xsltElem, xsltFPS, xsltFPSIO, XSLPath
    ) where


import System.Log.Logger

import HAppS.Protocols.MinHaXML
import HAppS.Util.Common(runCommand)
import Control.Exception(bracket,try)
import qualified Data.ByteString.Char8 as P
import System.Directory(removeFile)
import System.Environment(getEnv)
import System.IO
import System.IO.Unsafe(unsafePerformIO)
import Text.XML.HaXml.Verbatim(verbatim)

logMX = logM "HAppS.Protocols.XSLT"

type XSLPath = FilePath

xsltElem :: XSLPath -> Element -> String
xsltElem xsl = xsltString xsl . verbatim

xsltFPS :: XSLPath -> [P.ByteString] -> [P.ByteString]
xsltFPS xsl inp = unsafePerformIO $ xsltFPSIO xsl inp

xsltFPSIO xsl inp = 
    withTempFile "happs-src.xml" $ \sfp sh -> do
    withTempFile "happs-dst.xml" $ \dfp dh -> do
    mapM_ (P.hPut sh) inp
    hClose sh
    hClose dh
    xsltFile xsl sfp dfp
    s <- P.readFile dfp
    logMX DEBUG (">>> XSLT: result: "++ show s)
    return [s]

xsltString :: XSLPath -> String -> String
xsltString xsl inp = unsafePerformIO $
    withTempFile "happs-src.xml" $ \sfp sh -> do
    withTempFile "happs-dst.xml" $ \dfp dh -> do
    hPutStr sh inp
    hClose sh
    hClose dh
    xsltFile xsl sfp dfp
    s <- readFileStrict dfp
    logMX DEBUG (">>> XSLT: result: "++ show s)
    return s

-- | Note that the xsl file must have .xsl suffix.
xsltFile   :: XSLPath -> FilePath -> FilePath -> IO ()
xsltFile xsl src dst = do
    logMX DEBUG (">>> XSLT: Starting xsltproc " ++ unwords ["-o",dst,xsl,src])
    runCommand "xsltproc" ["-o",dst,xsl,src]
    logMX DEBUG (">>> XSLT: xsltproc done")

-- Utilities

withTempFile :: String -> (FilePath -> Handle -> IO a) -> IO a
withTempFile str hand = bracket (openTempFile tempDir str) (removeFile . fst) (uncurry hand)

readFileStrict :: FilePath -> IO String
readFileStrict fp = do
    let fseqM [] = return [] 
        fseqM xs = last xs `seq` return xs
    fseqM =<< readFile fp

{-
hGetContentsStrict :: Handle -> IO String
hGetContentsStrict h = flip finally (hClose h) $ do
    r <- hGetContents h
    let fseq [] = []; fseq xs = last xs `seq` xs
    return $! fseq r
-}

{-# NOINLINE tempDir #-}
tempDir :: FilePath
tempDir = unsafePerformIO $ tryAny [getEnv "TEMP",getEnv "TMP"] err
    where err = return "/tmp"

tryAny :: [IO a] -> IO a -> IO a
tryAny [] c     = c
tryAny (x:xs) c = either (\_ -> tryAny xs c) return =<< try x
