/* 
 * File: gxmms2.c
 *
 * gxmms2 - A gtk2 frontend for XMMS2
 * Copyright (c) 2005-2006 Johannes Heimansberg
 *
 * requires GTK+ 2.6 or better and XMMS2
 *
 * Version: 0.6.4
 *
 * Released under the GNU General Public License v2
 */

#define VERSION_NUMBER "0.6.4"

#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <xmmsclient/xmmsclient-glib.h>
#include <gdk/gdkkeysyms.h>
#include "xmms2ctrl/xmms2ctrl.h"
#include "gtrackinfo/trackinfo.h"
#include "gtrackinfo/gtrackinfo.h"
#include "gmedialib/gmedialib.h"
#include "sdecode/sdecode.h"
#include "wejpscroller.h"
#include "statusdisplay.h"
#include "wejpvolume.h"
#include "wejpconfig.h"
#include "gxmms2_mini.xpm"
#include "playlist.xpm"
#include "eggtrayicon.h"
#define LABEL_WIDTH 19
#define TITLE_STR_SIZE 512
#define FONT_SIZE 12

static GtkWidget    *label_time, *eventbox_time, *scroller;
static GtkWidget    *scale;
static GtkWidget    *playlist = NULL;
static GtkWidget    *window = NULL;
static EggTrayIcon  *docklet = NULL;
static GtkTooltips  *docklet_tooltip;
static gboolean      update_scale = TRUE;
static gboolean      flag_time_remaining = FALSE;
xmmsc_connection_t  *connection;
static trackinfo     track;
static gchar         title_str[TITLE_STR_SIZE] = "gxmms2 (not connected)";
static StatusDisplay sd;
static GtkWidget    *wv;
static ConfigFile    cf;
static gboolean      local_xmms2d = FALSE;

static int xmms2_connect(void);

enum { PREV_BUTTON = 0, PLAY_BUTTON, PAUSE_BUTTON, NEXT_BUTTON, STOP_BUTTON,
       PLAYLIST_BUTTON, INFO_BUTTON, ABOUT_BUTTON };

static void update_title_label(gchar     *title_str, 
                               trackinfo *track,
                               gboolean   reset_scroller)
{
	const gchar *current_str = wejpscroller_get_text(WEJPSCROLLER(scroller));

	trackinfo_get_full_title(title_str, TITLE_STR_SIZE, track, FALSE);
	if (!current_str) {
		wejpscroller_set_text(WEJPSCROLLER(scroller), title_str, TRUE);
	} else if (strcmp(current_str, title_str)) {
		wejpscroller_set_text(WEJPSCROLLER(scroller), title_str, reset_scroller);
	}
	if (GTK_IS_WIDGET(docklet))
		gtk_tooltips_set_tip(docklet_tooltip, GTK_WIDGET(docklet),
		                     title_str, NULL);
}

static gboolean cb_scale_press_event(GtkWidget      *widget,
                                     GdkEventButton *event,
                                     gpointer        user_data)
{
	update_scale = FALSE;
	return FALSE;
}

static gboolean cb_scale_release_event(GtkWidget      *widget,
                                       GdkEventButton *event,
                                       gpointer        user_data)
{
	xmmsc_result_t *res;

	if (xmms2ctrl_get_error() != ERROR_CONNECTION) {
		res = xmmsc_playback_seek_ms(connection, 
	          	(guint)gtk_range_get_value(GTK_RANGE(widget)));
		xmmsc_result_unref(res);
	}
	update_scale = TRUE;
	return FALSE;
}

/* Set initial_volume to -1 except for initialising (in that case 
   volume_change should be 0) */
static void change_volume(gint volume_change, gint initial_volume)
{
	static gint volume = 0;
	if (initial_volume >= 0) volume = initial_volume;

	if (volume < 100 && volume_change > 0) volume+=2;
	if (volume > 0   && volume_change < 0) volume-=2;
	xmms2ctrl_volume_set(connection, volume);
}

static void bc_handle_volume_change(xmmsc_result_t *res, void *userdata)
{
	guint32 volume;
	gchar   buf[64];
	
	xmmsc_result_get_dict_entry_uint32(res, "left", &volume);
	wejpvolume_set_volume(WEJPVOLUME(wv), volume);
	snprintf(buf, 63, "Volume: %d %%", volume);
	wejpscroller_shortmessage_show(WEJPSCROLLER(scroller), buf, 50);
}

static void n_volume_init(xmmsc_result_t *res, void *userdata)
{
	guint32 volume;
	
	xmmsc_result_get_dict_entry_uint32(res, "left", &volume);
	wejpvolume_set_volume(WEJPVOLUME(wv), volume);
	change_volume(0, volume);
	xmmsc_result_unref(res);
}

static void cb_volume_changed(GtkWidget *widget, GdkEvent *e, gpointer data)
{
	change_volume(-1, wejpvolume_get_volume(WEJPVOLUME(widget)));
}

static void show_gxmms2_about_dialog(void)
{
	GtkWidget *dialog;

	dialog = gtk_about_dialog_new();
	gtk_about_dialog_set_version(GTK_ABOUT_DIALOG(dialog), 
	                             VERSION_NUMBER);
	gtk_about_dialog_set_copyright(GTK_ABOUT_DIALOG(dialog),
	                               "Copyright (c) 2005-2006 " \
	                               "Johannes Heimansberg");
	gtk_about_dialog_set_comments(GTK_ABOUT_DIALOG(dialog),
	                               "A XMMS2 frontend written in C " \
	                               "using GTK+ 2.6. Released under " \
	                               "the GNU General Public License v2.");
	gtk_about_dialog_set_website(GTK_ABOUT_DIALOG(dialog),
	                             "http://wejp.k.vu/");
	gtk_dialog_run(GTK_DIALOG(dialog));
}

static void show_trackinfo(void)
{
	GtkWidget *dialog;

	dialog = gtrackinfo_new();
	gtrackinfo_set_info(GTRACKINFO(dialog), &track);
	gtk_widget_show(dialog);
}

static void load_medialib_config(ConfigFile *cf, GMedialib *playlist)
{
		gml_set_font_size(GMEDIALIB(playlist),
		                  atoi(cfg_get_key_value(*cf, "MediaLibFontSizePt")));
		if (strncasecmp(cfg_get_key_value(*cf, "RestoreMediaLibSize"), 
		                "yes", 3) == 0) {
			gtk_window_resize(GTK_WINDOW(playlist),
			                  atoi(cfg_get_key_value(*cf, "MediaLibWindowWidth")),
			                  atoi(cfg_get_key_value(*cf, "MediaLibWindowHeight")));
		}
		if (strncasecmp(cfg_get_key_value(*cf, "RestoreMediaLibPosition"), 
		                "yes", 3) == 0) {
			gtk_window_move(GTK_WINDOW(playlist),
			                atoi(cfg_get_key_value(*cf, "MediaLibWindowPosX")),
			                atoi(cfg_get_key_value(*cf, "MediaLibWindowPosY")));
		}
		if (gml_notebook_is_mlib_search_visible(GMEDIALIB(playlist))) {
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibSearchColumnID"),
			                "hide", 4) == 0) {
				gml_search_set_column_id_visible(GMEDIALIB(playlist), FALSE);
			}
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibSearchColumnArtist"),
			                "hide", 4) == 0) {
				gml_search_set_column_artist_visible(GMEDIALIB(playlist), FALSE);
			}
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibSearchColumnTrack"),
			                "hide", 4) == 0) {
				gml_search_set_column_track_visible(GMEDIALIB(playlist), FALSE);
			}
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibSearchColumnAlbum"),
			                "hide", 4) == 0) {
				gml_search_set_column_album_visible(GMEDIALIB(playlist), FALSE);
			}
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibSearchColumnRating"),
			                "hide", 4) == 0) {
				gml_search_set_column_rating_visible(GMEDIALIB(playlist), FALSE);
			}
		}
		if (gml_notebook_is_playlist_visible(GMEDIALIB(playlist))) {
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibPlaylistColumnPos"),
			                "hide", 4) == 0) {
				gml_pl_set_column_pos_visible(GMEDIALIB(playlist), FALSE);
			}
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibPlaylistColumnID"),
			                "hide", 4) == 0) {
				gml_pl_set_column_id_visible(GMEDIALIB(playlist), FALSE);
			}
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibPlaylistColumnArtist"),
			                "hide", 4) == 0) {
				gml_pl_set_column_artist_visible(GMEDIALIB(playlist), FALSE);
			}
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibPlaylistColumnAlbum"),
			                "hide", 4) == 0) {
				gml_pl_set_column_album_visible(GMEDIALIB(playlist), FALSE);
			}
			if (strncasecmp(cfg_get_key_value(*cf, "MediaLibPlaylistColumnRating"),
			                "hide", 4) == 0) {
				gml_pl_set_column_rating_visible(GMEDIALIB(playlist), FALSE);
			}
		}
		gml_set_album_tracknr_digits(GMEDIALIB(playlist), 
		                             atoi(cfg_get_key_value(*cf, 
		                             "MediaLibAlbumTrackNrDigits")));
}

static void show_playlist_editor(void)
{
	if (!GTK_IS_WIDGET(playlist)) {
		playlist = gmedialib_new(TRUE, TRUE, local_xmms2d, TRUE, TRUE);
		load_medialib_config(&cf, GMEDIALIB(playlist));
	} else {
		gtk_window_move(GTK_WINDOW(playlist), 
		                gml_window_get_pos_x(GMEDIALIB(playlist)),
		                gml_window_get_pos_y(GMEDIALIB(playlist)));
	}
	gtk_window_present(GTK_WINDOW(playlist));
}

static void rate_current_track(guint rating)
{
	if (rating >= 0 && rating <= 5) {
		gchar           buf[64];
		xmmsc_result_t *res;

		snprintf(buf, 63, "Rated track: %d", rating - 1);
		wejpscroller_shortmessage_show(WEJPSCROLLER(scroller), buf, 50);

		res = xmmsc_medialib_entry_property_set_int_with_source(connection,
		                                                        trackinfo_get_id(&track), 
		                                                        "client/generic",
		                                                        "rating", rating);
		xmmsc_result_unref(res);
	}
}

static gboolean reconnect(void)
{
	gboolean result = TRUE;

	/* try to reconnect */
	if (xmms2_connect() != ERROR_NONE) {
		strncpy(title_str, xmmsc_get_last_error(connection), TITLE_STR_SIZE);
		wejpscroller_set_text(WEJPSCROLLER(scroller), title_str, TRUE);
		printf("ERROR: %s\n", xmmsc_get_last_error(connection));
		result = FALSE;
	} else if (playlist) {
		gml_pl_setup_xmms_callbacks((GMedialib *)playlist);
		gml_setup_xmms_callbacks((GMedialib *)playlist);
	}
	return result;
}

static void cb_button_clicked(GtkWidget *widget, gpointer data)
{
	gint button_pressed = GPOINTER_TO_INT(data);

	if (xmms2ctrl_get_error() == ERROR_CONNECTION)
		if (!reconnect())
			if (button_pressed != ABOUT_BUTTON) button_pressed = -1;;

	switch(button_pressed) {
		case PREV_BUTTON:
			xmms2ctrl_do_reljump(connection, -1);
			break;
		case PLAY_BUTTON:
			xmms2ctrl_play(connection);
			break;
		case PAUSE_BUTTON:
			xmms2ctrl_toggle_pause(connection, FALSE);
			break;
		case NEXT_BUTTON:
			xmms2ctrl_do_reljump(connection, 1);
			break;
		case STOP_BUTTON:
			xmms2ctrl_stop(connection);
			break;
		case INFO_BUTTON:
			show_trackinfo();
			break;
		case PLAYLIST_BUTTON:
			show_playlist_editor();
			break;
		case ABOUT_BUTTON:
			show_gxmms2_about_dialog();
			break;
	}
}

static gboolean cb_key_press(GtkWidget   *widget,
                             GdkEventKey *event,
                             gpointer     userdata)
{
	gboolean result = FALSE;

	if (event->type == GDK_KEY_PRESS) {
		switch (event->keyval) {
			case GDK_b:
				cb_button_clicked(widget, (gpointer)NEXT_BUTTON);
				result = TRUE;
				break;
			case GDK_v:
				cb_button_clicked(widget, (gpointer)STOP_BUTTON);
				result = TRUE;
				break;
			case GDK_c:
				cb_button_clicked(widget, (gpointer)PAUSE_BUTTON);
				result = TRUE;
				break;
			case GDK_x:
				cb_button_clicked(widget, (gpointer)PLAY_BUTTON);
				result = TRUE;
				break;
			case GDK_z:
				cb_button_clicked(widget, (gpointer)PREV_BUTTON);
				result = TRUE;
				break;
			case GDK_j:
				cb_button_clicked(widget, (gpointer)PLAYLIST_BUTTON);
				result = TRUE;
				break;
			case GDK_t:
				flag_time_remaining = !flag_time_remaining;
				break;
			case GDK_0:
				rate_current_track(1);
				break;
			case GDK_1:
				rate_current_track(2);
				break;
			case GDK_2:
				rate_current_track(3);
				break;
			case GDK_3:
				rate_current_track(4);
				break;
			case GDK_4:
				rate_current_track(5);
				break;
			case GDK_plus:
				change_volume(1, -1);
				break;
			case GDK_minus:
				change_volume(-1, -1);
				break;
		}
	}
	return result;
}

static void n_media_lib_get_info(xmmsc_result_t *res, void *arg)
{
	trackinfo_update(res, &track);
	gtk_range_set_value(GTK_RANGE(scale), 0);

	if (trackinfo_get_playtime(&track) > 0) {
		gtk_range_set_range(GTK_RANGE(scale), 
		                    0, trackinfo_get_playtime(&track));
		gtk_widget_set_sensitive(GTK_WIDGET(scale), TRUE);
		update_scale = TRUE;
	} else {
		gtk_range_set_range(GTK_RANGE(scale), 0, 1);
		gtk_widget_set_sensitive(GTK_WIDGET(scale), FALSE);
		update_scale = FALSE;
	}
	update_title_label(title_str, &track, TRUE);
	xmmsc_result_unref(res);
}

static guint current_id;

static void bc_handle_medialib_entry_changed(xmmsc_result_t *res, void *userdata)
{
	guint id;
	if (xmmsc_result_get_uint(res, &id)) { 
	    if (current_id == id && xmms2ctrl_get_error() != ERROR_CONNECTION) {
			xmmsc_result_t *res2;

			res2 = xmmsc_medialib_get_info(connection, current_id);
			xmmsc_result_notifier_set(res2, n_media_lib_get_info, &current_id);
			xmmsc_result_unref(res2);
		}
	}
}

static void bc_handle_current_id(xmmsc_result_t *res, void *userdata)
{
	if (xmmsc_result_get_uint(res, &current_id) && 
		xmms2ctrl_get_error() != ERROR_CONNECTION) {
		xmmsc_result_t *res2;

		res2 = xmmsc_medialib_get_info(connection, current_id);
		xmmsc_result_notifier_set(res2, n_media_lib_get_info, &current_id);
		xmmsc_result_unref(res2);
	}
}

static void sig_handle_current_id(xmmsc_result_t *res, void *userdata)
{
	if (xmmsc_result_get_uint(res, &current_id) && 
		xmms2ctrl_get_error() != ERROR_CONNECTION) {
		xmmsc_result_t *res2;

		res2 = xmmsc_medialib_get_info(connection, current_id);
		xmmsc_result_notifier_set(res2, n_media_lib_get_info, &current_id);
		xmmsc_result_unref(res2);
	}
	xmmsc_result_unref(res);
}

static void sig_handle_playtime(xmmsc_result_t *res, void *userdata)
{
	xmmsc_result_t *res2;
	guint          ptime;
	gchar          buffer[20];
	static guint   cnt = 0;
	guint          seconds_elapsed, seconds_remaining, seconds_total;

	if (xmms2ctrl_get_error() != ERROR_CONNECTION) {
		if (xmmsc_result_iserror(res))
			return;
		if (!xmmsc_result_get_uint(res, &ptime))
			return;

		res2 = xmmsc_result_restart(res);
		xmmsc_result_unref(res2);
		xmmsc_result_unref(res);
	}
	if (cnt > 10 && GTK_IS_RANGE(scale)) {
		cnt = 0;

		seconds_elapsed = ptime / 1000;
		seconds_total = trackinfo_get_minutes(&track) * 60 + 
		                trackinfo_get_seconds(&track);
		seconds_remaining = seconds_total - seconds_elapsed;

		if (flag_time_remaining == TRUE && trackinfo_has_playtime(&track) &&
		    seconds_total > seconds_elapsed)
			snprintf(buffer, sizeof(buffer), "-%02d:%02d",
			         seconds_remaining / 60 % 100, seconds_remaining % 60);
		else
			snprintf(buffer, sizeof(buffer), " %02d:%02d",
			         seconds_elapsed / 60 % 100, seconds_elapsed % 60);

			gtk_label_set_text(GTK_LABEL(label_time), buffer);
			if (update_scale)
				gtk_range_set_value(GTK_RANGE(scale), ptime);
	}
	cnt++;
}

static void bc_handle_playback_status_change(xmmsc_result_t *res, void *userdata)
{
	guint pb_status = 0;

	if (xmmsc_result_iserror(res))
		return;
	if (!xmmsc_result_get_uint(res, &pb_status))
		return;

	switch (pb_status) {
		case XMMS_PLAYBACK_STATUS_PAUSE:
			status_display_set_status(&sd, STATUS_PAUSE);
			break;
		case XMMS_PLAYBACK_STATUS_STOP:
			status_display_set_status(&sd, STATUS_STOP);
			if (GTK_IS_RANGE(scale))
				gtk_range_set_value(GTK_RANGE(scale), 0);
			break;
		default:
			status_display_set_status(&sd, STATUS_PLAY);
	}
}

static void n_handle_playback_status_change(xmmsc_result_t *res, void *userdata)
{
	guint pb_status = 0;

	if (!xmmsc_result_iserror(res) && xmmsc_result_get_uint(res, &pb_status)) {
		switch (pb_status) {
			case XMMS_PLAYBACK_STATUS_PAUSE:
				status_display_set_status(&sd, STATUS_PAUSE);
				break;
			case XMMS_PLAYBACK_STATUS_STOP:
				status_display_set_status(&sd, STATUS_STOP);
				gtk_range_set_value(GTK_RANGE(scale), 0);
				break;
			default:
				status_display_set_status(&sd, STATUS_PLAY);
		}
	}
	
	xmmsc_result_unref(res);
}

static gboolean cb_timer_connection_lost(gpointer userdata)
{
	gboolean result = TRUE;
	printf("Reconnect timer cb.\n");
	if (xmms2ctrl_get_error() == ERROR_CONNECTION) {
		if (reconnect()) result = FALSE;   /* do not restart timer on success */
	} else {
		result = FALSE; /* do not restart timer if there is no connection prob*/
	}
	return result;
}

static void connection_lost(void *data)
{
	strcpy(title_str, "ERROR: Connection to xmms2d lost.");
	wejpscroller_set_text(WEJPSCROLLER(scroller), title_str, TRUE);
	xmms2ctrl_set_error(ERROR_CONNECTION);
	xmmsc_unref(connection);
	printf("ERROR: Connection to xmms2d lost.\n");
	if (strncasecmp(cfg_get_key_value(cf, "AutoReconnect"), "yes", 3) == 0)
		g_timeout_add(6000, &cb_timer_connection_lost, NULL);
}

static int xmms2_connect(void)
{
	gchar          *path = NULL;
	xmmsc_result_t *res;
	gint            conn_status, autostart = FALSE;

	path = getenv("XMMS_PATH"); /* Try XMMS_PATH environment variable first */
	if (path == NULL) path = cfg_get_key_value(cf, "IPCPath");
	xmms2ctrl_set_error(ERROR_NONE);

	if (!path || g_ascii_strncasecmp(path, "unix://", 7) == 0)
		local_xmms2d = TRUE;

	if (strncasecmp(cfg_get_key_value(cf, "StartLocalXmms2dOnStartup"), "yes", 3) == 0)
		autostart = TRUE;

	conn_status = xmmsc_connect(connection, path);
	if (!conn_status && autostart) {
		if (!path || g_ascii_strncasecmp(path, "unix://", 7) == 0) {
			if (!system("xmms2-launcher")) {
				conn_status = xmmsc_connect(connection, path);
			}
		}
	}

	if (!conn_status) {
		strncpy(title_str, xmmsc_get_last_error(connection), TITLE_STR_SIZE);
		wejpscroller_set_text(WEJPSCROLLER(scroller), title_str, TRUE);
		xmms2ctrl_set_error(ERROR_CONNECTION);
	} else {
		xmmsc_mainloop_gmain_init(connection);
		XMMS_CALLBACK_SET(connection, xmmsc_playback_current_id,
	 	                  sig_handle_current_id, connection);
		XMMS_CALLBACK_SET(connection, xmmsc_broadcast_playback_current_id,
		                  bc_handle_current_id, connection);
		XMMS_CALLBACK_SET(connection, xmmsc_broadcast_medialib_entry_changed,
		                  bc_handle_medialib_entry_changed, connection);
		XMMS_CALLBACK_SET(connection, xmmsc_signal_playback_playtime, 
		                  sig_handle_playtime, title_str);
		XMMS_CALLBACK_SET(connection, xmmsc_broadcast_playback_status,
		                  bc_handle_playback_status_change, NULL);
		XMMS_CALLBACK_SET(connection, xmmsc_broadcast_playback_volume_changed,
		                  bc_handle_volume_change, NULL);

		xmmsc_disconnect_callback_set(connection, connection_lost, NULL);
		
		res = xmmsc_playback_status(connection);
		xmmsc_result_notifier_set(res, n_handle_playback_status_change, NULL);
		xmmsc_result_unref(res);
		
		res = xmmsc_playback_volume_get(connection);
		xmmsc_result_notifier_set(res, n_volume_init, NULL);
		xmmsc_result_unref(res);
	}
	return xmms2ctrl_get_error();
}

/* callback: close window (return FALSE = close window) */
static gint cb_delete_event(GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gint       main_x_pos = 0, main_y_pos = 0;
	gchar      buf[32] = "0";
	ConfigFile *cf = (ConfigFile *)data;

	gtk_window_get_position(GTK_WINDOW(widget), &main_x_pos, &main_y_pos);
	snprintf(buf, 30, "%d", main_x_pos);
	cfg_add_key(cf, "MainWindowPosX", buf);
	snprintf(buf, 30, "%d", main_y_pos);
	cfg_add_key(cf, "MainWindowPosY", buf);
	return FALSE;
}

static void cb_toggle_time(GtkWidget *widget, gpointer data)
{
	flag_time_remaining = !flag_time_remaining;
}

static void n_quit_gtk_main(xmmsc_result_t *res, void *userdata)
{
	if (xmms2ctrl_get_error() == ERROR_NONE)
		xmmsc_disconnect_callback_set(connection, NULL, NULL);
	gtk_main_quit();
}

static void cb_quit_function(GtkWidget *widget, gpointer data)
{
	if (strncasecmp(cfg_get_key_value(cf, "TerminateXmms2dOnExit"), "yes", 3) == 0 && 
	    xmms2ctrl_get_error() != ERROR_CONNECTION) { 
		xmmsc_result_t *res = xmmsc_quit(connection);
		xmmsc_result_notifier_set(res, n_quit_gtk_main, NULL);
		xmmsc_result_unref(res);
	} else {
		n_quit_gtk_main(NULL, NULL);
	}
}

static void cb_drag_data_received(GtkWidget          *widget,
                                  GdkDragContext     *context,
                                  gint                x,
                                  gint                y,
                                  GtkSelectionData   *data,
                                  guint               info,
                                  guint               time)
{
	xmmsc_result_t *res;

	if ((data->length >= 0) && (data->format == 8)) {
		gint n = 0, m = 0;
		gchar buffer[384], *uri_conv;
		/* seperate each file on the list: */
		while (n < strlen((gchar *)data->data)) {
			if (((guchar *)data->data)[n] == '\n') {
				buffer[m] = '\0';
				m = 0;
				/* decode %XX escape sequences to their corresponding chars: */
				uri_conv = decode_string(buffer);
				res = xmmsc_playlist_add(connection, uri_conv);
				xmmsc_result_unref(res);
				g_free(uri_conv);
			} else if (((guchar *)data->data)[n] > 31) {
				buffer[m] = ((guchar *)data->data)[n];
				if (m < 382) m++;
			}
			n++;
		}
		gtk_drag_finish(context, TRUE, FALSE, time);
	} else {
		gtk_drag_finish(context, FALSE, FALSE, time);
	}
}

static void load_config(gchar *path_to_config_file)
{
	gchar       *ipc_path_default;
	const gchar *username;
	gint         str_length = 0;

	cfg_init_config_file_struct(&cf);
	/* default config values */
	cfg_add_key(&cf, "ScrollerFont",                  "Sans Serif");
	cfg_add_key(&cf, "ScrollerFontSizePx",            "12");
	cfg_add_key(&cf, "ScrollerDelay",                 "27");
	cfg_add_key(&cf, "MainWindowPosX",                "10");
	cfg_add_key(&cf, "MainWindowPosY",                "10");
	cfg_add_key(&cf, "MediaLibWindowPosX",            "60");
	cfg_add_key(&cf, "MediaLibWindowPosY",            "50");
	cfg_add_key(&cf, "MediaLibWindowWidth",           "600");
	cfg_add_key(&cf, "MediaLibWindowHeight",          "360");
	cfg_add_key(&cf, "MediaLibFontSizePt",            "-1");
	cfg_add_key(&cf, "RestoreMediaLibSize",           "yes");
	cfg_add_key(&cf, "RestoreMainWindowPosition",     "no");
	cfg_add_key(&cf, "RestoreMediaLibPosition",       "no");
	cfg_add_key(&cf, "OpenMediaLibOnStartup",         "no");
	cfg_add_key(&cf, "MediaLibSearchColumnID",        "show");
	cfg_add_key(&cf, "MediaLibSearchColumnArtist",    "show");
	cfg_add_key(&cf, "MediaLibSearchColumnTrack",     "show");
	cfg_add_key(&cf, "MediaLibSearchColumnAlbum",     "show");
	cfg_add_key(&cf, "MediaLibSearchColumnRating",    "show");
	cfg_add_key(&cf, "MediaLibPlaylistColumnPos",     "show");
	cfg_add_key(&cf, "MediaLibPlaylistColumnID",      "show");
	cfg_add_key(&cf, "MediaLibPlaylistColumnArtist",  "show");
	cfg_add_key(&cf, "MediaLibPlaylistColumnTrack",   "show");
	cfg_add_key(&cf, "MediaLibPlaylistColumnAlbum",   "show");
	cfg_add_key(&cf, "MediaLibPlaylistColumnRating",  "show");
	cfg_add_key(&cf, "MediaLibAlbumTrackNrDigits",    "3");
	cfg_add_key(&cf, "TrayIcon",                      "yes");
	cfg_add_key(&cf, "AutoReconnect",                 "no");
	cfg_add_key(&cf, "ShowRemainingTime",             "no");
	cfg_add_key(&cf, "TerminateXmms2dOnExit",         "no");
	cfg_add_key(&cf, "StartLocalXmms2dOnStartup",     "no");
	username = g_get_user_name();
	str_length = strlen(username) + 128;
	ipc_path_default = g_malloc(str_length * sizeof(gchar));
	snprintf(ipc_path_default, str_length, "unix:///tmp/xmms-ipc-%s", username);
	cfg_add_key(&cf, "IPCPath", ipc_path_default);
	g_free(ipc_path_default);

	if (cfg_read_config_file(&cf, path_to_config_file) != 0) {
		printf("No config file found! Creating %s: ", path_to_config_file);
		if (cfg_write_config_file(&cf, path_to_config_file) != 0)
			printf("FAILED\n");
		else
			printf("OKAY\n");
	}
}

static void cb_docklet_menu()
{
	GtkWidget *menu, *entry, *image;
	
	menu = gtk_menu_new();
	
	entry = gtk_image_menu_item_new_from_stock(GTK_STOCK_MEDIA_PREVIOUS, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);
	g_signal_connect(G_OBJECT(entry), "activate",
	                 G_CALLBACK(cb_button_clicked), (gpointer)PREV_BUTTON);

	entry = gtk_image_menu_item_new_from_stock(GTK_STOCK_MEDIA_PLAY, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);
	g_signal_connect(G_OBJECT(entry), "activate",
	                 G_CALLBACK(cb_button_clicked), (gpointer)PLAY_BUTTON);

	entry = gtk_image_menu_item_new_from_stock(GTK_STOCK_MEDIA_PAUSE, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);
	g_signal_connect(G_OBJECT(entry), "activate",
	                 G_CALLBACK(cb_button_clicked), (gpointer)PAUSE_BUTTON);

	entry = gtk_image_menu_item_new_from_stock(GTK_STOCK_MEDIA_STOP, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);
	g_signal_connect(G_OBJECT(entry), "activate",
	                 G_CALLBACK(cb_button_clicked), (gpointer)STOP_BUTTON);

	entry = gtk_image_menu_item_new_from_stock(GTK_STOCK_MEDIA_NEXT, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);
	g_signal_connect(G_OBJECT(entry), "activate",
	                 G_CALLBACK(cb_button_clicked), (gpointer)NEXT_BUTTON);

	entry = gtk_image_menu_item_new_with_mnemonic("_Open Playlist Editor");
	image = gtk_image_new_from_pixbuf(gdk_pixbuf_new_from_xpm_data((const char **)playlist_xpm));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(entry), image);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);
	g_signal_connect(G_OBJECT(entry), "activate",
	                 G_CALLBACK(cb_button_clicked), (gpointer)PLAYLIST_BUTTON);

	entry = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);

	entry = gtk_image_menu_item_new_from_stock(GTK_STOCK_ABOUT, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);
	g_signal_connect(G_OBJECT(entry), "activate",
	                 G_CALLBACK(cb_button_clicked), (gpointer)ABOUT_BUTTON);

	entry = gtk_image_menu_item_new_from_stock(GTK_STOCK_QUIT, NULL);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), entry);
	g_signal_connect(G_OBJECT(entry), "activate",
	                 G_CALLBACK(cb_quit_function), NULL);

	gtk_widget_show_all(menu);
	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, 0, 
	               gtk_get_current_event_time());
}

static void cb_toggle_window(GtkWidget *window)
{
 	if (gtk_window_get_skip_taskbar_hint(GTK_WINDOW(window))) {
 		gtk_window_set_skip_taskbar_hint(GTK_WINDOW(window), FALSE);
		gtk_window_present(GTK_WINDOW(window));
 	} else {
		gtk_window_iconify(GTK_WINDOW(window));
		gtk_window_set_skip_taskbar_hint(GTK_WINDOW(window), TRUE);
	}
}

static void cb_docklet_clicked(GtkWidget      *widget,
                               GdkEventButton *event,
                               gpointer        data)
{
	if (event->type == GDK_BUTTON_PRESS) {
		switch (event->button) {
		case 1:
		case 2:
			cb_toggle_window(window);
			break;
		case 3:
			cb_docklet_menu();
			break;
		}
	}
}

static void create_trayicon(GdkPixbuf *icon);

static void cb_trayicon_destroy(GtkWidget *widget, gpointer data)
{
	printf("tray icon destroyed!\n");
	docklet = NULL;
	docklet_tooltip = NULL;
}

static void create_trayicon(GdkPixbuf *icon)
{
	GtkWidget   *vbox;
	GtkWidget   *image_docklet;

	docklet = egg_tray_icon_new("gxmms2");
	g_signal_connect(G_OBJECT(docklet), "destroy",
	                 G_CALLBACK(cb_trayicon_destroy), icon);
	vbox = gtk_event_box_new();
	icon = gdk_pixbuf_new_from_xpm_data((const char **)gxmms2_mini_xpm);
	image_docklet = gtk_image_new_from_pixbuf(icon);

	g_signal_connect(G_OBJECT(vbox), "button-press-event",
	                 G_CALLBACK(cb_docklet_clicked), NULL);

	gtk_container_add(GTK_CONTAINER(vbox), image_docklet);
	gtk_container_add(GTK_CONTAINER(docklet), vbox);
	gtk_widget_show_all(GTK_WIDGET(docklet));
		
	docklet_tooltip = gtk_tooltips_new();
	gtk_tooltips_set_tip(docklet_tooltip, GTK_WIDGET(docklet),
	                     "gxmms2", NULL);
	gtk_tooltips_enable(docklet_tooltip);
}

int main(int argc, char *argv[])
{
	GtkWidget   *table, *vbox, *hbox;
	GtkWidget   *button_prev, *button_next,     *button_play, *button_pause;
	GtkWidget   *button_stop, *button_playlist, *button_info, *button_about;
	GtkWidget   *image_prev,  *image_next,      *image_play,  *image_pause;
	GtkWidget   *image_stop,  *image_playlist,  *image_info,  *image_about;
	GtkTooltips *tooltips;
	GdkPixbuf   *icon;
	gchar       *path_to_config_file;
	gchar        userconf[PATH_MAX];
	gint         scroller_font_size;
	PangoFontDescription *font_desc = pango_font_description_new();

	enum {
		TARGET_STRING,
		TARGET_ROOTWIN
	};

	GtkTargetEntry target_table[] = {
		{ "text/uri-list",                 0, TARGET_STRING  },
		{ "application/x-rootwindow-drop", 0, TARGET_ROOTWIN }
	};
	guint n_targets = sizeof(target_table) / sizeof(target_table[0]);

	xmmsc_userconfdir_get(userconf, PATH_MAX);
	path_to_config_file = g_build_path(G_DIR_SEPARATOR_S,
	                                   userconf, "clients",
	                                   "gxmms2.conf", NULL);

	if (!g_file_test(path_to_config_file, G_FILE_TEST_EXISTS)) {
		gchar *dir = g_build_path(G_DIR_SEPARATOR_S,
		                          userconf, "clients", NULL);
		g_mkdir_with_parents(dir, 0755);
		g_free(dir);
	}

	load_config(path_to_config_file);

	gtk_init(&argc, &argv);

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_policy(GTK_WINDOW(window), FALSE, TRUE, FALSE);
	gtk_window_set_default_size(GTK_WINDOW(window), 200, 32);
	gtk_window_set_title(GTK_WINDOW(window), "gxmms2");
	gtk_container_set_border_width(GTK_CONTAINER(window), 5);
	gtk_window_set_resizable(GTK_WINDOW(window), FALSE);

	gtk_drag_dest_set(window, GTK_DEST_DEFAULT_ALL, 
	                  target_table, n_targets - 1, 
	                  GDK_ACTION_COPY | GDK_ACTION_MOVE);

	icon = gdk_pixbuf_new_from_xpm_data((const char **)gxmms2_mini_xpm);
	gtk_window_set_icon(GTK_WINDOW(window), icon);

	vbox = gtk_vbox_new(FALSE, 3);
	gtk_container_add(GTK_CONTAINER(window), vbox);	

	hbox = gtk_hbox_new(FALSE, 3);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	/* scale */
	scale = gtk_hscale_new_with_range(0, 200, 1);
	gtk_scale_set_draw_value(GTK_SCALE(scale), FALSE);
	gtk_container_add(GTK_CONTAINER(vbox), scale);


	table = gtk_table_new(1, 8, FALSE);
	gtk_container_add(GTK_CONTAINER(vbox), table);

	/* status icon */
	status_display_init(&sd);
	gtk_container_add(GTK_CONTAINER(hbox), status_display_get_image(&sd));

	/* (scroller) font size */
	scroller_font_size = atoi(cfg_get_key_value(cf, "ScrollerFontSizePx"));

	if (scroller_font_size > 0)	
		gtk_widget_set_size_request(GTK_WIDGET(hbox), -1, scroller_font_size + 3);

	/* scroller */
	scroller = wejpscroller_new(atoi(cfg_get_key_value(cf, "ScrollerDelay")));
	wejpscroller_set_dimensions(WEJPSCROLLER(scroller), 120, 
	                            scroller_font_size + 2);
	wejpscroller_set_font(WEJPSCROLLER(scroller),
	                      cfg_get_key_value(cf, "ScrollerFont"),
	                      scroller_font_size);
	gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(scroller), TRUE, TRUE, 1);

	/* eventbox time */
	eventbox_time = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(hbox), eventbox_time);

	/* time label */
	label_time = gtk_label_new("00:00");
	gtk_container_add(GTK_CONTAINER(eventbox_time), label_time);
	if (scroller_font_size > 0)
		pango_font_description_set_absolute_size(font_desc, 
		                                         scroller_font_size*PANGO_SCALE);
	pango_font_description_set_family_static(font_desc, 
	                                     cfg_get_key_value(cf, "ScrollerFont"));
	gtk_widget_modify_font(label_time, font_desc);
	pango_font_description_free(font_desc);

	/* volume icon */
	wv = wejpvolume_new(100, 0, 2);
	gtk_container_add(GTK_CONTAINER(hbox), wv);

	/* prev button */
	image_prev = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PREVIOUS, 
	                                      GTK_ICON_SIZE_MENU);
	button_prev = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button_prev), image_prev);

	gtk_table_attach(GTK_TABLE(table),
		             button_prev,
		             0, 1, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* play button */
	image_play = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PLAY, 
	                                      GTK_ICON_SIZE_MENU);
	button_play = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button_play), image_play);

	gtk_table_attach(GTK_TABLE(table),
		             button_play,
		             1, 2, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* pause button */
	image_pause = gtk_image_new_from_stock(GTK_STOCK_MEDIA_PAUSE, 
	                                      GTK_ICON_SIZE_MENU);
	button_pause = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button_pause), image_pause);

	gtk_table_attach(GTK_TABLE(table),
		             button_pause,
		             2, 3, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* stop button */
	image_stop = gtk_image_new_from_stock(GTK_STOCK_MEDIA_STOP, 
	                                      GTK_ICON_SIZE_MENU);
	button_stop = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button_stop), image_stop);

	gtk_table_attach(GTK_TABLE(table),
		             button_stop,
		             3, 4, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* next button */
	image_next = gtk_image_new_from_stock(GTK_STOCK_MEDIA_NEXT, 
	                                      GTK_ICON_SIZE_MENU);
	button_next = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button_next), image_next);

	gtk_table_attach(GTK_TABLE(table),
		             button_next,
		             4, 5, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* info button */
	image_info = gtk_image_new_from_stock(GTK_STOCK_DIALOG_INFO, 
	                                      GTK_ICON_SIZE_MENU);
	button_info = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button_info), image_info);

	gtk_table_attach(GTK_TABLE(table),
		             button_info,
		             5, 6, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* playlist button */
	icon = gdk_pixbuf_new_from_xpm_data((const char **)playlist_xpm);
	image_playlist = gtk_image_new_from_pixbuf(icon);
	button_playlist = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button_playlist), image_playlist);

	gtk_table_attach(GTK_TABLE(table),
		             button_playlist,
		             6, 7, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* about button */
	image_about = gtk_image_new_from_stock(GTK_STOCK_ABOUT, 
	                                       GTK_ICON_SIZE_MENU);

	button_about = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(button_about), image_about);

	gtk_table_attach(GTK_TABLE(table),
		             button_about,
		             7, 8, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* tooltips */
	tooltips = gtk_tooltips_new();
	gtk_tooltips_set_tip(tooltips, button_prev, 
	                     "Previous track", NULL);
	gtk_tooltips_set_tip(tooltips, button_next, 
	                     "Next track", NULL);
	gtk_tooltips_set_tip(tooltips, button_play, 
	                     "Play", NULL);
	gtk_tooltips_set_tip(tooltips, button_pause, 
	                     "Pause", NULL);
	gtk_tooltips_set_tip(tooltips, button_stop, 
	                     "Stop", NULL);
	gtk_tooltips_set_tip(tooltips, button_info, 
	                     "Track info", NULL);
	gtk_tooltips_set_tip(tooltips, button_playlist, 
	                     "Open playlist editor", NULL);
	gtk_tooltips_set_tip(tooltips, button_about, 
	                     "About gxmms2", NULL);

	g_signal_connect(G_OBJECT(window),          "delete_event",
	                 G_CALLBACK(cb_delete_event),  &cf);
	g_signal_connect(G_OBJECT(window),          "destroy",
	                 G_CALLBACK(cb_quit_function), NULL);
	g_signal_connect(G_OBJECT(button_play),     "clicked",
	                 G_CALLBACK(cb_button_clicked), (gpointer)PLAY_BUTTON);
	g_signal_connect(G_OBJECT(button_pause),    "clicked",
	                 G_CALLBACK(cb_button_clicked), (gpointer)PAUSE_BUTTON);
	g_signal_connect(G_OBJECT(button_stop),     "clicked",
	                 G_CALLBACK(cb_button_clicked), (gpointer)STOP_BUTTON);
	g_signal_connect(G_OBJECT(button_next),     "clicked",
	                 G_CALLBACK(cb_button_clicked), (gpointer)NEXT_BUTTON);
	g_signal_connect(G_OBJECT(button_prev),     "clicked",
	                 G_CALLBACK(cb_button_clicked), (gpointer)PREV_BUTTON);
	g_signal_connect(G_OBJECT(button_playlist), "clicked",
	                 G_CALLBACK(cb_button_clicked), (gpointer)PLAYLIST_BUTTON);
	g_signal_connect(G_OBJECT(button_info),     "clicked",
	                 G_CALLBACK(cb_button_clicked), (gpointer)INFO_BUTTON);
	g_signal_connect(G_OBJECT(button_about),    "clicked",
	                 G_CALLBACK(cb_button_clicked), (gpointer)ABOUT_BUTTON);
	g_signal_connect(G_OBJECT(scale),           "button-press-event",
	                 G_CALLBACK(cb_scale_press_event), NULL);
	g_signal_connect(G_OBJECT(scale),           "button-release-event",
	                 G_CALLBACK(cb_scale_release_event), NULL);
	g_signal_connect(G_OBJECT(eventbox_time),   "button-press-event",
	                 G_CALLBACK(cb_toggle_time), NULL);
	g_signal_connect(G_OBJECT(wv),              "volume-changed",
	                 G_CALLBACK(cb_volume_changed), NULL);
	g_signal_connect(G_OBJECT(window),          "key-press-event",
	                 G_CALLBACK(cb_key_press), NULL);
	g_signal_connect(G_OBJECT(window),          "drag_data_received",
	                 G_CALLBACK(cb_drag_data_received), NULL);

	if (strncasecmp(cfg_get_key_value(cf, "RestoreMainWindowPosition"), 
	                "yes", 3) == 0) {
		gtk_window_move(GTK_WINDOW(window),
		                atoi(cfg_get_key_value(cf, "MainWindowPosX")),
		                atoi(cfg_get_key_value(cf, "MainWindowPosY")));
	}

	/* docklet */
	if (strncasecmp(cfg_get_key_value(cf, "TrayIcon"), "yes", 3) == 0) {
		create_trayicon(icon);
	}

	if (strncasecmp(cfg_get_key_value(cf, "ShowRemainingTime"), "yes", 3) == 0)
		flag_time_remaining = TRUE;

	gtk_widget_show_all(window);

	connection = xmmsc_init("gxmms2");
	xmms2_connect();

	if (strncasecmp(cfg_get_key_value(cf, "OpenMediaLibOnStartup"), 
	                "yes", 3) == 0) {
		playlist = gmedialib_new(TRUE, TRUE, local_xmms2d, TRUE, TRUE);
		load_medialib_config(&cf, GMEDIALIB(playlist));
		gtk_window_present(GTK_WINDOW(playlist));
	}

	gtk_main();

	if (playlist) {
		gchar buf[8];

		snprintf(buf, 7, "%d", gml_window_get_width(GMEDIALIB(playlist)));
		cfg_add_key(&cf, "MediaLibWindowWidth",  buf);
		snprintf(buf, 7, "%d", gml_window_get_height(GMEDIALIB(playlist)));
		cfg_add_key(&cf, "MediaLibWindowHeight", buf);
		snprintf(buf, 7, "%d", gml_window_get_pos_x(GMEDIALIB(playlist)));
		cfg_add_key(&cf, "MediaLibWindowPosX", buf);
		snprintf(buf, 7, "%d", gml_window_get_pos_y(GMEDIALIB(playlist)));
		cfg_add_key(&cf, "MediaLibWindowPosY", buf);
		if (gml_notebook_is_mlib_search_visible(GMEDIALIB(playlist))) {
			snprintf(buf, 7, "%s",
			         gml_search_get_column_id_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibSearchColumnID",        buf);
			snprintf(buf, 7, "%s",
			         gml_search_get_column_artist_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibSearchColumnArtist",    buf);
			snprintf(buf, 7, "%s",
			         gml_search_get_column_track_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibSearchColumnTrack",     buf);
			snprintf(buf, 7, "%s",
			         gml_search_get_column_album_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibSearchColumnAlbum",     buf);
			snprintf(buf, 7, "%s",
			         gml_search_get_column_rating_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibSearchColumnRating",    buf);
			snprintf(buf, 7, "%s",
			         gml_pl_get_column_pos_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
		}
		if (gml_notebook_is_playlist_visible(GMEDIALIB(playlist))) {
			cfg_add_key(&cf, "MediaLibPlaylistColumnPos",     buf);
			snprintf(buf, 7, "%s",
			         gml_pl_get_column_id_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibPlaylistColumnID",      buf);
			snprintf(buf, 7, "%s",
			         gml_pl_get_column_artist_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibPlaylistColumnArtist",  buf);
			snprintf(buf, 7, "%s",
			         gml_pl_get_column_track_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibPlaylistColumnTrack",   buf);
			snprintf(buf, 7, "%s",
			         gml_pl_get_column_album_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibPlaylistColumnAlbum",   buf);
			snprintf(buf, 7, "%s",
			         gml_pl_get_column_rating_visible(GMEDIALIB(playlist)) ?
			         "show" : "hide");
			cfg_add_key(&cf, "MediaLibPlaylistColumnRating", buf);
		}
	}
	cfg_add_key(&cf, "ShowRemainingTime", (flag_time_remaining ? "yes" : "no"));

	cfg_write_config_file(&cf, path_to_config_file);

	cfg_free_config_file_struct(&cf);
	g_free(path_to_config_file);

	return 0;
}
