/*
 *     gtkatlantic - the gtk+ monopd client, enjoy network monopoly games
 *
 *
 *  Copyright © 2002-2014 Sylvain Rochet
 *
 *  gtkatlantic is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <gtk/gtk.h>
#include <libxml/parser.h>

#include "engine.h"

#include "xmlparse.h"
#include "game.h"
#include "client.h"
#include "interface.h"
#include "global.h"
#include "load.h"
#include "callback.h"
#include "display.h"
#include "trade.h"

/*
ok	server
ok	client
ok	msg
ok	display
dep	updateplayerlist
dep	updategamelist
ok	playerupdate
ok	deleteplayer
ok	estateupdate
ok	cardupdate
ok	estategroupupdate
ok	tradeupdate
ok	auctionupdate
ok	gameupdate
ok	deletegame
dep	commandlist
ok	configupdate
*/

/*
ok	.R

ok	.gn
no-need	.gl
ok	.gj
ok	.gx
todo	.gk
todo	.gu
ok	.gc
todo	.gS

ok	.Tn
ok	.Tc
ok	.Te
ok	.Tm
ok	.Ta
ok	.Tr

ok	.r
ok	.E
ok	.t

ok	.D
no-need	.p

ok	.eb
ok	.es
ok	.ea
ok	.hb
ok	.hs
ok	.em

ok	.ab

ok	.jc
ok	.jp
ok	.jr

ok	.T$
ok	.T%

no-need	.pi

todo	.gd
ok	.gs

no-need	.d
ok	.n
no-need	.f
*/


void xmlparse_getgamelist_plugger(guint32 connectid, gchar *buffer)  {

	xmlDocPtr doc;
	xmlNodePtr cur;

	doc = xmlParseMemory(buffer, strlen(buffer) );

	if(doc == NULL)  {

		printf("XML ERROR ON: %s %d\n", connection[connectid]->host, connection[connectid]->port);
		return;
	}

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		xmlFreeDoc(doc);
		return;
	}

	if( xmlStrcmp(cur->name, SERVER_XMLROOTELEMENT) )  {
		xmlFreeDoc(doc);
		return;
	}

	for( cur = cur->xmlChildrenNode ; cur != NULL ; cur = cur -> next)  {

		if(! xmlStrcmp(cur->name, (xmlChar*)"server") )  xmlparse_server(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"gameupdate") )  xmlparse_gamelist_gameupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"deletegame") )  xmlparse_gamelist_deletegame(connectid, doc, cur);
	}

	xmlFreeDoc(doc);
}




void xmlparse_game_plugger(guint32 connectid, gchar *buffer)  {

	xmlDocPtr doc;
	xmlNodePtr cur;

	doc = xmlParseMemory(buffer, strlen(buffer) );

	if(doc == NULL)   return;

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {
		xmlFreeDoc(doc);
		return;
	}

	if( xmlStrcmp(cur->name, SERVER_XMLROOTELEMENT) )  {
		xmlFreeDoc(doc);
		return;
	}

	for(cur = cur->xmlChildrenNode ; cur != NULL ; cur = cur -> next)  {

		if(! xmlStrcmp(cur->name, (xmlChar*)"gameupdate") )  xmlparse_gameupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"deletegame") )  xmlparse_deletegame(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"server") )  xmlparse_server(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"client") )  xmlparse_client(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"msg") )  xmlparse_message(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"estateupdate") )  xmlparse_estateupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"playerupdate") )  xmlparse_playerupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"auctionupdate") )  xmlparse_auctionupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"display") )  xmlparse_display(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"configupdate") )  xmlparse_configupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"tradeupdate") )  xmlparse_tradeupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"cardupdate") )  xmlparse_cardupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"estategroupupdate") )  xmlparse_estategroupupdate(connectid, doc, cur);
		if(! xmlStrcmp(cur->name, (xmlChar*)"deleteplayer") )  xmlparse_deleteplayer(connectid, doc, cur);
	}

	xmlFreeDoc(doc);
}




void xmlparse_server(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;
	(void)doc;

	tmp = xmlGetProp(cur, (xmlChar*)"version");
	if(tmp)  connection[connectid]->server_version = (gchar*)tmp;
}




void xmlparse_client(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;
	(void)connectid;
	(void)doc;

	tmp = xmlGetProp(cur, (xmlChar*)"playerid");
	if( tmp )  {
		global->my_playerid = atoi( (gchar*)tmp );
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"cookie");
	if( tmp )  {
		if(global->cookie) g_free(global->cookie);
		global->cookie = (gchar*)tmp;
	}
}




void xmlparse_gameupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur) {
	xmlChar *tmp;
	struct timeval tv;
	gint32 gameid;
	game *game;
	guint8 newstatus;
	(void)connectid;
	(void)doc;

	tmp = xmlGetProp(cur, (xmlChar*)"gameid");
	if (tmp) {
		gameid = atoi((gchar*)tmp);
		g_free(tmp);
	} else {
		return;
	}

	/* We do not need game templates */
	if (gameid < 0) {
		return;
	}

	game = game_find(gameid);
	if (!game) {
		game = game_new(gameid);
		if (!game) {
			return;
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"status");
	if(tmp) {
		if(!xmlStrcmp(tmp, (xmlChar*)"config") ) {
			newstatus = GAME_STATUS_CONFIG;
		}
		else if(!xmlStrcmp(tmp, (xmlChar*)"init") ) {
			newstatus = GAME_STATUS_INIT;
		}
		else if(!xmlStrcmp(tmp, (xmlChar*)"run") ) {
			newstatus = GAME_STATUS_RUN;
		}
		else if(!xmlStrcmp(tmp, (xmlChar*)"end") ) {
			newstatus = GAME_STATUS_END;
		}
		else {
			g_free(tmp);
			return;
		}

		if (game->status != newstatus) {
			game->status = newstatus;

			if (newstatus == GAME_STATUS_RUN) {
				gettimeofday(&tv, NULL);
				game->start_time = tv.tv_sec;
			}

			if (currentgame == game) {
				game_switch_status();
			}
		}
	}
	g_free(tmp);

	tmp = xmlGetProp(cur, (xmlChar*)"master");
	if (tmp)  {
		game->master = atoi((gchar*)tmp);
		g_free(tmp);

		/* set sensitive mode of start button */
		if (currentgame == game && game->status == GAME_STATUS_CONFIG) {
			GtkWidget *StartButton = g_object_get_data(G_OBJECT(global->MainVerticalBox), "start_button");
			gtk_widget_set_sensitive(StartButton, (global->my_playerid == game->master));
			
			GList *list = gtk_container_get_children(GTK_CONTAINER(currentgame->GameConfigBox));
			for (list = g_list_first(list); list; list = g_list_next(list)) {
				gtk_widget_set_sensitive(GTK_WIDGET(list->data), (global->my_playerid == game->master));
			}
			g_list_free(list);
		}
	}
}


void xmlparse_deletegame(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur) {
	xmlChar *tmp;
	gint32 gameid;
	(void)connectid;
	(void)doc;

	tmp = xmlGetProp(cur, (xmlChar*)"gameid");
	if (tmp) {
		gameid = atoi((gchar*)tmp);
		g_free(tmp);
	} else {
		return;
	}

	if (currentgame->gameid != gameid) {
		game_free(gameid);
	}
}



void xmlparse_message(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	gchar *type, *text, *author, *value;
	(void)connectid;
	(void)doc;

	type = (gchar*)xmlGetProp(cur, (xmlChar*)"type");

	if(! g_ascii_strncasecmp(type, "chat", 4) )  {

		author = (gchar*)xmlGetProp(cur, (xmlChar*)"author");
		value = (gchar*)xmlGetProp(cur, (xmlChar*)"value");
		if(value && author)  {

			if(! strncmp("[ACTION]", value, 8) )
				text = g_strdup_printf("* %s%s", author, value + 8);
			else if(! strncmp("/me", value, 3) )
				text = g_strdup_printf("* %s%s", author, value + 3);
			else
				text = g_strdup_printf("<%s> %s", author, value);

			text_insert_chat(text, strlen(text));
			g_free(text);
		}
		if(value[0] == '!')  parse_specific_chat_message(value);
		if(author)  g_free(author);
		if(value)  g_free(value);
	}
	else {

		gchar *typeup = g_utf8_strup( type, -1 );
		value = (gchar*)xmlGetProp(cur, (xmlChar*)"value");
		text = g_strdup_printf("%s: %s", typeup, value);
		text_insert_message(text, strlen(text));
		g_free(typeup);
		g_free(text);
		g_free(value);
	}

	g_free(type);
}




void xmlparse_estateupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	guint32 color[3], i;
	xmlChar *tmp;
	eng_obj *pic;
	gint32 star, id, t, u;
	gboolean refresh = FALSE;
	gboolean owner_mortgage_changed = FALSE;
	(void)connectid;
	(void)doc;

	if (!currentgame) {
		return;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"estateid");
	if(!tmp)  return;
	id = atoi( (gchar*)tmp );
	g_free(tmp);
	if(id < 0) return;

	tmp = xmlGetProp(cur, (xmlChar*)"name");
	if(tmp)  currentgame->estate[id].name = (gchar*)tmp;

	tmp = xmlGetProp(cur, (xmlChar*)"color");
	if(tmp) {

		sscanf((gchar*)tmp, "#%2X%2X%2X", &color[0], &color[1], &color[2]);
		currentgame->estate[id].color[0] = color[0];
		currentgame->estate[id].color[1] = color[1];
		currentgame->estate[id].color[2] = color[2];
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"bgcolor");
	if(tmp) {

		sscanf((gchar*)tmp, "#%2X%2X%2X", &color[0], &color[1], &color[2]);
		currentgame->estate[id].bgcolor[0] = color[0];
		currentgame->estate[id].bgcolor[1] = color[1];
		currentgame->estate[id].bgcolor[2] = color[2];
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"owner");
	if(tmp)  {

		t = currentgame->estate[id].owner = atoi((gchar*)tmp);
		g_free(tmp);
		owner_mortgage_changed = TRUE;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"mortgaged");
	if(tmp)  {

		u = currentgame->estate[id].mortgaged = atoi((gchar*)tmp);
		g_free(tmp);
		owner_mortgage_changed = TRUE;
	}

	if(owner_mortgage_changed)  {

		t = currentgame->estate[id].owner;
		u = currentgame->estate[id].mortgaged;

		/* reset unowned cards of this estate to all players */
		for(i = 0 ; i < MAX_PLAYERS ; i++)  {

			if(! global->player[i].playerid) continue;
			if (global->player[i].spectator) continue;
			if(global->player[i].game != currentgame->gameid)  continue;

			pic = global->player[i].playerlist_cards_pic[ get_playerlistcard_id_with_estate(id) ];

			eng_pic_set_alpha(pic, data->playerlist_cards_alphaunowned);
		}

		if(t > 0  &&  !u)  {

			/* star */
			star = global->player[ get_player_slot_with_playerid(t) ].buffer_star;

			pic = currentgame->estate[id].star_pic;
			eng_pic_set_width(pic, data->pngfile_star_width[star]);
			eng_pic_set_height(pic, data->pngfile_star_height[star]);
			eng_pic_set_buf(pic, data->pngfile_star_buf[star]);
			eng_pic_show(pic);

			/* playerlist card */
			pic = global->player[ get_player_slot_with_playerid(t) ].playerlist_cards_pic[ get_playerlistcard_id_with_estate(id) ];

			eng_pic_set_bgcolor(pic, data->playerlist_cards_cardbgcolor);
			eng_pic_set_alpha(pic, data->playerlist_cards_alphaowned);
		}
		else  if(t > 0  &&  u)  {

			/* star */
			star = global->player[ get_player_slot_with_playerid(t) ].buffer_star;

			pic = currentgame->estate[id].star_pic;
			eng_pic_set_width(pic, data->pngfile_star_m_width[star]);
			eng_pic_set_height(pic, data->pngfile_star_m_height[star]);
			eng_pic_set_buf(pic, data->pngfile_star_m_buf[star]);
			eng_pic_show(pic);

			/* playerlist card */
			pic = global->player[ get_player_slot_with_playerid(t) ].playerlist_cards_pic[ get_playerlistcard_id_with_estate(id) ];

			eng_pic_set_bgcolor(pic, data->playerlist_cards_cardbgcolormortgage);
			eng_pic_set_alpha(pic, data->playerlist_cards_alphamortgage);
		}
		else /* if( t <= 0 ) */ {

			/* star */
			eng_pic_unshow(currentgame->estate[id].star_pic);
		}

		/* update estatelist in trade panel */
		for(i = 0 ; i < MAX_TRADES ; i++)  {

			if(! currentgame->trade[i].open)  continue;
			if(currentgame->trade[i].current_component != TRADE_TYPE_ESTATE)  continue;

			trade_rebuild_subcomponent(i);
		}

		refresh = TRUE;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"houses");
	if(tmp)  {

		t = currentgame->estate[id].houses = atoi((gchar*)tmp);
 		g_free(tmp);

		if(t <= 0)   {

			/* houses */
			pic = currentgame->estate[id].house_pic;
			eng_pic_unshow(pic);
		}
		else  if(t > 0  &&  data->estate[id].type_house == TYPE_HOUSE_HORIZONTAL)  {

			/* houses */
			pic = currentgame->estate[id].house_pic;
			eng_pic_set_width(pic, data->pngfile_horiz_house_width[t]);
			eng_pic_set_height(pic, data->pngfile_horiz_house_height[t]);
			eng_pic_set_buf(pic, data->pngfile_horiz_house_buf[t]);
			eng_pic_show(pic);
		}
		else  if(t > 0  &&  data->estate[id].type_house == TYPE_HOUSE_VERTICAL)  {

			/* houses */
			pic = currentgame->estate[id].house_pic;
			eng_pic_set_width(pic, data->pngfile_vert_house_width[t]);
			eng_pic_set_height(pic, data->pngfile_vert_house_height[t]);
			eng_pic_set_buf(pic, data->pngfile_vert_house_buf[t]);
			eng_pic_show(pic);
		}

		refresh = TRUE;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"houseprice");
	if(tmp)  {
		currentgame->estate[id].houseprice = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"sellhouseprice");
	if(tmp)  {
		currentgame->estate[id].sellhouseprice = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"mortgageprice");
	if(tmp)  {
		currentgame->estate[id].mortgageprice = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"unmortgageprice");
	if(tmp)  {
		currentgame->estate[id].unmortgageprice = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"group");
	if(tmp)  {
		currentgame->estate[id].group = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_be_owned");
	if(tmp)  {
		currentgame->estate[id].can_be_owned = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_toggle_mortgage");
	if(tmp)  {
		currentgame->estate[id].can_toggle_mortgage = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_buy_houses");
	if(tmp)  {
		currentgame->estate[id].can_buy_houses = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_sell_houses");
	if(tmp)  {
		currentgame->estate[id].can_sell_houses = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"money");
	if(tmp)  {
		currentgame->estate[id].money = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"price");
	if(tmp)  {
		currentgame->estate[id].price = atoi((gchar*)tmp);
		g_free(tmp);
	}

	for(i = 0 ; i <= 5 ; i++) {

		xmlChar *key = (xmlChar*)g_strdup_printf("rent%d", i);
		tmp = xmlGetProp(cur, key);

		if(tmp)  {
			currentgame->estate[id].rent[i] = atoi((gchar*)tmp);
			g_free(tmp);
		}
		g_free(key);
	}

	if (refresh) {
		update_display();
	}
}




void xmlparse_playerupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	gint32 playerid, i;
	gint32 id;
	xmlChar *tmp;
	gboolean refresh = FALSE;
	(void)connectid;
	(void)doc;

	tmp = xmlGetProp(cur, (xmlChar*)"playerid");
	if(!tmp) return;
 	playerid = atoi((gchar*)tmp);
	g_free(tmp);
	if(!playerid) return;

	/* create new player */
	id = get_player_slot_with_playerid(playerid);
	if (id < 0)  {
		id = game_new_player_slot(playerid);
		if (id < 0) {
			return;
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"game");
	if(tmp)  {
		global->player[id].game = atoi((gchar*)tmp);
		g_free(tmp);

		if (playerid == global->my_playerid) {
			if (global->player[id].game < 0) {
				if (currentgame) {
					game_quit();
				}
			} else {
				if (!currentgame) {
					game *game = game_find(global->player[id].game);
					if (game) {
						currentgame = game;
						game_switch_status();
						printf("Current game: %d\n", game->gameid);
					}
				}
			}
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"host");
	if(tmp)  {

		if(global->player[id].host) g_free(global->player[id].host);
		global->player[id].host = (gchar*)tmp;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"image");
	if(tmp)  {

		if(global->player[id].image) g_free(global->player[id].image);
		global->player[id].image = (gchar*)tmp;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"name");
	if(tmp)  {

		if(global->player[id].name) g_free(global->player[id].name);
		global->player[id].name = g_strndup((gchar*)tmp, MAX_LENGTH_NICK);
		g_free(tmp);

		if(! strcmp(global->player[id].name, "_metaserver_"))  {
			game_free_player(id);
			return;
		}

		if (currentgame && global->player[id].game == currentgame->gameid) {
			/* playerlist name */
			/* FIXME: handle hasturn color */
			if(global->player[id].playerlist_LabelNamePlayer)
				gtk_label_set_text(GTK_LABEL(global->player[id].playerlist_LabelNamePlayer), global->player[id].name);

			/* playerlist in trade panel */
			for(i = 0 ; i < MAX_TRADES ; i++)  {

				if(! currentgame->trade[i].open)  continue;

				trade_rebuild_playerlist(i);
			}

			/* update sub component players in trade panel */
			for(i = 0 ; i < MAX_TRADES ; i++)  {

				if(! currentgame->trade[i].open)  continue;

				trade_rebuild_subcomponent(i);
			}
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"money");
	if(tmp)  {

		global->player[id].money = atoi((gchar*)tmp);

		if (currentgame && global->player[id].game == currentgame->gameid) {
			if(global->player[id].playerlist_LabelMoneyPlayer) {
				if(global->player[id].hasturn) {
					gchar *tmp = g_markup_printf_escaped("<span foreground=\"#ff0000\">%d</span>", global->player[id].money);
					gtk_label_set_markup(GTK_LABEL(global->player[id].playerlist_LabelMoneyPlayer), tmp);
					g_free(tmp);
				} else {
					gchar *tmp = g_markup_printf_escaped("<span foreground=\"#000000\">%d</span>", global->player[id].money);
					gtk_label_set_markup(GTK_LABEL(global->player[id].playerlist_LabelMoneyPlayer), tmp);
					g_free(tmp);
				}
			}
		}
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"location");
	if(tmp)  {
		global->player[id].location_to = atoi((gchar*)tmp);
		g_free(tmp);

		tmp = xmlGetProp(cur, (xmlChar*)"directmove");
		if (tmp) {
			global->player[id].directmove = atoi((gchar*)tmp);
			g_free(tmp);
		}

		if (currentgame && global->player[id].game == currentgame->gameid) {
			refresh = TRUE;
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"jailed");
	if(tmp)  {
		gboolean jailed = atoi((gchar*)tmp);;
		if (jailed != global->player[id].jailed) {
			global->player[id].jailed = jailed;
			global->player[id].directmove = TRUE; /* force directmove when player is going to or leaving jail */
		}
		g_free(tmp);

		if (currentgame && global->player[id].game == currentgame->gameid) {
			refresh = TRUE;
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"bankrupt");
	if(tmp)  {
		global->player[id].bankrupt = atoi((gchar*)tmp);
		g_free(tmp);

		if (currentgame && global->player[id].game == currentgame->gameid) {
			/* remove token */
			if(global->player[id].bankrupt) {
				eng_pic_unshow(global->player[id].token_pic);
				refresh = TRUE;
			}
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"hasturn");
	if(tmp)  {
		global->player[id].hasturn = atoi((gchar*)tmp);
		g_free(tmp);

		if (currentgame && global->player[id].game == currentgame->gameid) {
			/* set has turn attributes */
			if(global->player[id].hasturn)  {

				if (global->player[id].playerlist_LabelNamePlayer)  {
					gchar *tmp = g_markup_printf_escaped("<span foreground=\"#ff0000\">%s</span>", global->player[id].name);
					gtk_label_set_markup(GTK_LABEL(global->player[id].playerlist_LabelNamePlayer), tmp);
					g_free(tmp);

					tmp = g_markup_printf_escaped("<span foreground=\"#ff0000\">%d</span>", global->player[id].money);
					gtk_label_set_markup(GTK_LABEL(global->player[id].playerlist_LabelMoneyPlayer), tmp);
					g_free(tmp);
				}

				eng_pic_unset_alpha(global->player[id].token_pic);
			}
			else  {

				if (global->player[id].playerlist_LabelNamePlayer)  {
					gchar *tmp = g_markup_printf_escaped("<span foreground=\"#000000\">%s</span>", global->player[id].name);
					gtk_label_set_markup(GTK_LABEL(global->player[id].playerlist_LabelNamePlayer), tmp);
					g_free(tmp);

					tmp = g_markup_printf_escaped("<span foreground=\"#000000\">%d</span>", global->player[id].money);
					gtk_label_set_markup(GTK_LABEL(global->player[id].playerlist_LabelMoneyPlayer), tmp);
					g_free(tmp);
				}

				if(config->game_token_transparency)
					eng_pic_set_alpha(global->player[id].token_pic, 0x7f);
			}

			refresh = TRUE;
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_roll");
	if(tmp)  {
		global->player[id].can_roll = atoi((gchar*)tmp);
		g_free(tmp);

		if (currentgame && global->player[id].game == currentgame->gameid && currentgame->status == GAME_STATUS_RUN)  {
			GtkWidget *Button = g_object_get_data(G_OBJECT(currentgame->CommandsBox), "roll_button");

			if(global->player[id].can_roll)
				gtk_widget_show_all(Button);
			else
				gtk_widget_hide(Button);
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"can_buyestate");
	if(tmp)  {
		global->player[id].can_buyestate = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"spectator");
	if(tmp)  {
 		global->player[id].spectator = atoi((gchar*)tmp);
		g_free(tmp);
	}

	game_buildplayerlist();
	if (refresh) {
		game_initiate_token_movement();
		update_display();
	}
}




void xmlparse_auctionupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	GtkWidget *text, *entry, *box;
	GtkTextBuffer *textbuff;
	GtkTextIter textiter;
	GtkTextMark *textmark;

	guint32 actor = 0, auctionid, estateid, highbid = 0, highbidder, status;
	gchar *ptr, *message;

	xmlChar *tmp;

	(void)connectid;
	(void)doc;

	if (!currentgame || currentgame->status != GAME_STATUS_RUN) {
		return;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"auctionid");
	if(!tmp) return;
	auctionid = atoi((gchar*)tmp);
	g_free(tmp);

	/* initialise auction */
	tmp = xmlGetProp(cur, (xmlChar*)"actor");
	if(tmp)  {
		actor = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"estateid");
	if(tmp)  {

		estateid = atoi((gchar*)tmp);
		g_free(tmp);

		display_hide();
		interface_create_auctionbox();
		g_object_set_data(G_OBJECT(currentgame->BoardCenter), "auction_highbid", 0);

		text = g_object_get_data(G_OBJECT(currentgame->BoardCenter), "auction_text");
		entry = g_object_get_data(G_OBJECT(currentgame->BoardCenter), "auction_entry");

		ptr = g_strdup_printf("%d", auctionid);
		g_object_set_data_full(G_OBJECT(entry), "auctionid", ptr, g_free);

		message = g_strdup_printf("%s auction estate %s", global->player[ get_player_slot_with_playerid( actor ) ].name, currentgame->estate[ estateid ].name);

		textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
		gtk_text_buffer_get_end_iter(textbuff, &textiter);
		gtk_text_buffer_insert(textbuff, &textiter, message, strlen(message));
		gtk_text_buffer_insert(textbuff, &textiter, "\n", -1);


		g_free(message);
	}

	/* bid */
	tmp = xmlGetProp(cur, (xmlChar*)"highbid");
	if(tmp)  {
		highbid = atoi((gchar*)tmp);
		g_object_set_data(G_OBJECT(currentgame->BoardCenter), "auction_highbid", GINT_TO_POINTER(highbid));
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"highbidder");
	if(tmp)  {

		highbidder = atoi((gchar*)tmp);
		g_free(tmp);

		message = g_strdup_printf("%s bid %d", global->player[ get_player_slot_with_playerid(highbidder) ].name, highbid);

		text = g_object_get_data(G_OBJECT(currentgame->BoardCenter), "auction_text");

		textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
		gtk_text_buffer_get_end_iter(textbuff, &textiter);
		gtk_text_buffer_insert(textbuff, &textiter, message, strlen(message));
		gtk_text_buffer_insert(textbuff, &textiter, "\n", -1);

		/* Scroll the text */
		textmark = gtk_text_buffer_get_mark(textbuff, "endmark");
		gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(text), textmark, 0.0, FALSE, 0.0, 0.0);

		g_free(message);
	}

	/* status */
	tmp = xmlGetProp(cur, (xmlChar*)"status");
	if(tmp)  {

		status = atoi((gchar*)tmp);
		g_free(tmp);

		switch( status )  {

			case 1:
				text = g_object_get_data(G_OBJECT(currentgame->BoardCenter), "auction_text");

				message = g_strdup("<- going once ->");
				textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
				gtk_text_buffer_get_end_iter(textbuff, &textiter);
				gtk_text_buffer_insert(textbuff, &textiter, message, strlen(message));
				gtk_text_buffer_insert(textbuff, &textiter, "\n", -1);
				/* Scroll the text */
				textmark = gtk_text_buffer_get_mark(textbuff, "endmark");
				gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(text), textmark, 0.0, FALSE, 0.0, 0.0);

				g_free(message);
				break;

			case 2:
				text = g_object_get_data(G_OBJECT(currentgame->BoardCenter), "auction_text");

				message = g_strdup("\n<- going twice ->");
				textbuff = gtk_text_view_get_buffer(GTK_TEXT_VIEW(text));
				gtk_text_buffer_get_end_iter(textbuff, &textiter);
				gtk_text_buffer_insert(textbuff, &textiter, message, strlen(message));
				gtk_text_buffer_insert(textbuff, &textiter, "\n", -1);
				/* Scroll the text */
				textmark = gtk_text_buffer_get_mark(textbuff, "endmark");
				gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(text), textmark, 0.0, FALSE, 0.0, 0.0);

				g_free(message);
				break;

			case 3:
				box = g_object_get_data(G_OBJECT(currentgame->BoardCenter), "auctionbox");
				if(box) gtk_widget_destroy(box);
				g_object_set_data(G_OBJECT(currentgame->BoardCenter), "auctionbox", 0);
				display_show();
				break;
		}
	}
}




void xmlparse_display(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlNodePtr cur2;
	gint32 estateid;
	xmlChar *tmp;
	gchar *caption, *command;
	gboolean enable;
	(void)connectid;
	(void)doc;

	if (!currentgame || currentgame->status < GAME_STATUS_INIT) {
		return;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"cleartext");
	if(tmp)  {
		if( atoi((gchar*)tmp) )  display_clear_text();
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"clearbuttons");
	if(tmp)  {
		if( atoi((gchar*)tmp) )  display_clear_buttons();
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"estateid");
	if(tmp)   {
		estateid = atoi((gchar*)tmp);
		g_free(tmp);
		display_estate(estateid);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"text");
	if(tmp)  {

		if( strlen((gchar*)tmp) > 0)  display_text((gchar*)tmp);
		g_free(tmp);
	}

	/* buttons */
	for(cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next)  {

		if(! xmlStrcmp(cur2->name, (xmlChar*)"button") )  {

			caption = (gchar*)xmlGetProp(cur2, (xmlChar*)"caption");
			command = (gchar*)xmlGetProp(cur2, (xmlChar*)"command");

			tmp = xmlGetProp(cur2, (xmlChar*)"enabled");
			enable = atoi((gchar*)tmp);
			g_free(tmp);

			display_add_button((gchar*)caption, (gchar*)command, enable);
			g_free(caption);
			g_free(command);
		}
	}
}




void xmlparse_configupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	GtkWidget *CButton;
	xmlNodePtr cur2;
	gchar *title, *command, *description;
	guint32 signalid;
	xmlChar *tmp;
	game *game;
	gint32 id = 0;
	(void)connectid;
	(void)doc;

	tmp = xmlGetProp(cur, (xmlChar*)"gameid");
	if (tmp) {
		game = game_find(atoi((gchar*)tmp));
		g_free(tmp);

		if (!game) {
			return;
		}
	} else {
		return;
	}

	/* We are going to receive <configupdate/> events before we know in which game we are
	 * therefore we create the config box before the config panel is created
	 */
	if (!game->GameConfigBox) {
		game->GameConfigBox = gtk_box_new(GTK_ORIENTATION_VERTICAL, 0);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"configid");
	if (tmp) {
		id = atoi((gchar*)tmp);
		g_free(tmp);
	}

	/* commands using the new configupdate protocol */
	if (id > 0) {
		/* create it if not created */
		gchar *strid = g_strdup_printf("%d", id);
		if ( !(CButton = g_object_get_data(G_OBJECT(game->GameConfigBox), strid)) ) {

			/* We only handle boolean type for now */
			tmp = xmlGetProp(cur, (xmlChar*)"type");
			if (!tmp) {
				goto abort;
			}
			if(xmlStrcmp(tmp, (xmlChar*)"bool") )  {
				g_free(tmp);
				goto abort;
			}
			g_free(tmp);

			/* Disable spectators for now, server support is not finished yet */
			tmp = xmlGetProp(cur, (xmlChar*)"name");
			if (tmp) {
				if (!xmlStrcmp(tmp, (xmlChar*)"allowspectators") )  {
					g_free(tmp);
					goto abort;
				}
				g_free(tmp);
			}

			description = (gchar*)xmlGetProp(cur, (xmlChar*)"description");
			if (!description) {
				goto abort;
			}

			CButton = gtk_check_button_new_with_label(description);
			signalid = g_signal_connect(G_OBJECT(CButton), "toggled", G_CALLBACK(Callback_toggle_boolean_gameoption), NULL);
			gtk_widget_set_sensitive(CButton, (global->my_playerid == game->master));
			gtk_box_pack_start(GTK_BOX(game->GameConfigBox), CButton, FALSE, FALSE, 0);

			/* store signal identifier */
			g_object_set_data(G_OBJECT(CButton), "signal", GINT_TO_POINTER(signalid));

			/* store id under CButton */
			g_object_set_data(G_OBJECT(CButton), "id", GINT_TO_POINTER(id));

			/* save the pointer of button to use in the future */
			g_object_set_data(G_OBJECT(game->GameConfigBox), strid, CButton);

			g_free(description);
		}

		/* modify */
		tmp = xmlGetProp(cur, (xmlChar*)"value");
		if (tmp) {

			/* disconnect signal */
			signalid = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(CButton), "signal"));
			g_signal_handler_disconnect(G_OBJECT(CButton), signalid);

			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CButton), (atoi((gchar*)tmp) != 0));
			g_free(tmp);

			/* reconnect signal */
			signalid = g_signal_connect(G_OBJECT(CButton), "toggled", G_CALLBACK(Callback_toggle_boolean_gameoption), NULL);
			g_object_set_data(G_OBJECT(CButton), "signal", GINT_TO_POINTER(signalid));
		}

/*		tmp = xmlGetProp(cur, (xmlChar*)"edit");   USELESS */
abort:
		g_free(strid);

	}

	/* commands using the old configupdate protocol */
	else for( cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next)  {

		if(! xmlStrcmp(cur2->name, (xmlChar*)"option") )  {

			command = (gchar*)xmlGetProp(cur2, (xmlChar*)"command");
			if(!command)  continue;

			/* create it if not created */
			if ( !(CButton = g_object_get_data(G_OBJECT(game->GameConfigBox), command)) ) {

				tmp = xmlGetProp(cur2, (xmlChar*)"type");
				if (!tmp) {
					continue;
				}
				if(xmlStrcmp(tmp, (xmlChar*)"bool") )  {
					g_free(tmp);
					continue;
				}
				g_free(tmp);

				title = (gchar*)xmlGetProp(cur2, (xmlChar*)"title");
				if (!title) {
					continue;
				}

				CButton = gtk_check_button_new_with_label(title);
				signalid = g_signal_connect(G_OBJECT(CButton), "toggled", G_CALLBACK(Callback_toggle_boolean_gameoption), NULL);
				gtk_widget_set_sensitive(CButton, (global->my_playerid == game->master));
				gtk_box_pack_start(GTK_BOX(game->GameConfigBox), CButton, FALSE, FALSE, 0);

				/* store signal identifier */
				g_object_set_data(G_OBJECT(CButton), "signal", GINT_TO_POINTER(signalid));

				/* store command under CButton */
				g_object_set_data_full(G_OBJECT(CButton), "command", g_strdup(command), g_free);

				/* save the pointer of button to use in the future */
				g_object_set_data(G_OBJECT(game->GameConfigBox), command, CButton);

				g_free(title);
			}

			/* modify */
			tmp = xmlGetProp(cur2, (xmlChar*)"value");
			if (tmp) {

				/* disconnect signal */
				signalid = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(CButton), "signal"));
				g_signal_handler_disconnect(G_OBJECT(CButton), signalid);

				gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(CButton), (atoi((gchar*)tmp) != 0));
				g_free(tmp);

				/* reconnect signal */
				signalid = g_signal_connect(G_OBJECT(CButton), "toggled", G_CALLBACK(Callback_toggle_boolean_gameoption), NULL);
				g_object_set_data(G_OBJECT(CButton), "signal", GINT_TO_POINTER(signalid));
			}

/*			tmp = xmlGetProp(cur2, (xmlChar*)"edit");   USELESS */

			g_free(command);
		}
	}

	gtk_widget_show_all(game->GameConfigBox);
}




void xmlparse_tradeupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlNodePtr cur2;
	xmlChar *tmp;
	guint32 tradeid = 0, playerid = 0, cardid = 0, estateid = 0;
	guint32 targetplayer = 0, playerfrom = 0, playerto = 0, money = 0;
	gboolean accept = FALSE;
	(void)connectid;
	(void)doc;

	if (!currentgame || currentgame->status != GAME_STATUS_RUN) {
		return;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"tradeid");
	if(tmp)  {
		tradeid = atoi((gchar*)tmp);
		g_free(tmp);
	}
	else  return;

//	tmp = xmlGetProp(cur, (xmlChar*)"actor");
//	if(tmp)   {
//		actor = atoi((gchar*)tmp);
//		g_free(tmp);
//	}

	tmp = xmlGetProp(cur, (xmlChar*)"type");
	if(tmp)  {

		if(! xmlStrcmp(tmp, (xmlChar*)"rejected") )  {
			trade_destroy(tradeid);
			g_free(tmp);
			return;
		}

		if(! xmlStrcmp(tmp, (xmlChar*)"completed") )  {
			trade_destroy(tradeid);
			g_free(tmp);
			return;
		}

		if(! xmlStrcmp(tmp, (xmlChar*)"accepted") )  {
			g_free(tmp);
			return;
		}

		if(! xmlStrcmp(tmp, (xmlChar*)"new") )  {
			trade_initnew(tradeid);
			g_free(tmp);
		}
	}

	tmp = xmlGetProp(cur, (xmlChar*)"revision");
	if (tmp) {
		trade_update_revision(tradeid, atoi((gchar*)tmp));
		g_free(tmp);
	}

	/* create trade panel if wasn't created (server bug I suppose) */
	trade_initnew(tradeid);

	for(cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next)  {

		/* -- player -- */
		if(! xmlStrcmp(cur2->name, (xmlChar*)"tradeplayer") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"playerid");
			if(tmp)   {
				playerid = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"accept");
			if(tmp)   {
				accept = atoi((gchar*)tmp);
				g_free(tmp);
			}

			trade_update_player(tradeid, playerid, accept);
		}

		/* -- card -- */
		if(! xmlStrcmp(cur2->name, (xmlChar*)"tradecard") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"cardid");
			if(tmp)   {
				cardid = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"targetplayer");
			if(tmp)   {
				targetplayer = atoi((gchar*)tmp);
				g_free(tmp);
			}

			trade_update_card(tradeid, cardid, targetplayer);
		}

		/* -- estate -- */
		if(! xmlStrcmp(cur2->name, (xmlChar*)"tradeestate") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"estateid");
			if(tmp)   {
				estateid = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"targetplayer");
			if(tmp)   {
				targetplayer = atoi((gchar*)tmp);
				g_free(tmp);
			}

			trade_update_estate(tradeid, estateid, targetplayer);
		}

		/* -- money -- */
		if(! xmlStrcmp(cur2->name, (xmlChar*)"trademoney") )  {

			tmp = xmlGetProp(cur2, (xmlChar*)"playerfrom");
			if(tmp)   {
				playerfrom = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"playerto");
			if(tmp)   {
				playerto = atoi((gchar*)tmp);
				g_free(tmp);
			}

			tmp = xmlGetProp(cur2, (xmlChar*)"money");
			if(tmp)   {
				money = atoi((gchar*)tmp);
				g_free(tmp);
			}

			trade_update_money(tradeid, playerfrom, playerto, money);
		}
	}

}




void xmlparse_cardupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;
	gint32 cardid = 0, owner = 0, i, cardslot = 0;
	(void)connectid;
	(void)doc;

	/* 
	 * FIXME: this is broken when reconnecting, we discard <cardupdate/> because we don't know yet in which game we are participating
	 * fixing that first require a server change to send playerupdate game attribute before cardupdate. Cardupdate doesn't have a gameid
	 * attribute so we don't know for which game it applies.
	 */
	if (!currentgame || currentgame->status != GAME_STATUS_RUN) {
		return;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"cardid");
	if(tmp)  {
		cardid = atoi((gchar*)tmp);
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"owner");
	if(tmp)  {
		owner = atoi((gchar*)tmp);
		g_free(tmp);
	}

	/* create new card slot */
	if(owner >= 0)  {

		cardslot = get_card_slot_with_cardid(cardid);
		if(cardslot < 0)  if(! game_get_valid_card_slot(&cardslot) )  return;

		currentgame->card[cardslot].cardid = cardid;
		currentgame->card[cardslot].owner = owner;
	}

	/* destroy a card slot */
	else  {

		cardslot = get_card_slot_with_cardid(cardid);
		if(cardslot < 0)  return;

		currentgame->card[cardslot].cardid = 0;
		currentgame->card[cardslot].owner = 0;
	}

	/* update component list in trade panel */
	for(i = 0 ; i < MAX_TRADES ; i++)  {

		if(! currentgame->trade[i].open)  continue;

		trade_rebuild_component(i);
		trade_rebuild_subcomponent(i);
	}
}




void xmlparse_estategroupupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;
	gint32 groupid = -1;
	(void)connectid;
	(void)doc;

	if (!currentgame) {
		return;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"groupid");
	if(tmp)  {
		groupid = atoi((gchar*)tmp);
		g_free(tmp);
	}

	if(groupid < 0)  return;

	if(currentgame->group[groupid].name)  g_free(currentgame->group[groupid].name);

	currentgame->group[groupid].name = (gchar*)xmlGetProp(cur, (xmlChar*)"name");
}




void xmlparse_deleteplayer(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur)  {

	xmlChar *tmp;
	gint32 playerid = -1, id;
	(void)connectid;
	(void)doc;

	tmp = xmlGetProp(cur, (xmlChar*)"playerid");
	if(tmp)  {
		playerid = atoi((gchar*)tmp);
		g_free(tmp);
	}

	if(playerid < 0) return;

	id = get_player_slot_with_playerid(playerid);
	if(id < 0)  return;

	/* delete a player */
	if (global->player[id].playerlist_token_frame == NULL) {  /* never delete a player on board */ /* FIXME: this is kind of a ugly hack, a FSM should be better */
		game_free_player(id);
	}
	game_buildplayerlist();
}




void xmlparse_gamelist_gameupdate(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur) {
	xmlChar *tmp;
	gchar *game = NULL, *players = NULL, *description, *gametype;
	gint32 id;
	gchar *host, *version;
	gint32 port;
	GtkListStore *store;
	GtkTreeIter iter;
	gboolean valid;
	gboolean update = FALSE;
	(void)doc;

	if (connectid != global->customserver_connectid) {
		return;
	}

	if (global->phase != PHASE_GETGAMES)  {
		client_disconnect(connectid);
		global->customserver_connectid = 0;
		return;
	}

	store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(global->GameList)));
	if (store == NULL) {
		return;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"gameid");
	id = atoi((gchar*)tmp);
	g_free(tmp);

	host = connection[connectid]->host;
	port = connection[connectid]->port;

	/* lookup if gameid on same server already exist */
	if (id > 0) {
		valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
		while (valid) {
			gchar *ehost;
			gint32 eport;
			gint32 source;
			gint32 eid;

			gtk_tree_model_get(GTK_TREE_MODEL(store), &iter, GAMELIST_COLUMN_ID, &eid, GAMELIST_COLUMN_HOST, &ehost, GAMELIST_COLUMN_PORT, &eport, GAMELIST_COLUMN_SOURCE, &source, -1);
			if ( eid == id  &&  source == GAMELIST_SOURCE_CUSTOM  &&  !strcmp(host, ehost)  &&  eport == port )  {
				update = TRUE;
				g_free(ehost);
				break;
			}
			g_free(ehost);

			valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter); /* get next iter so we can remove the current entry safely */
		}
	}

	gtk_widget_hide(global->ServerList);
	gtk_widget_show(global->GameList);

	tmp = xmlGetProp(cur, (xmlChar*)"canbejoined");
	if (tmp) {
		if (atoi((gchar*)tmp) == 0)  {
			if (update == TRUE) {
				gtk_list_store_remove(store, &iter);
			}
			g_free(tmp);
			return;
		}
		g_free(tmp);
	}

	tmp = xmlGetProp(cur, (xmlChar*)"name");
	if (tmp) {
		/* ONLY TEMPORARY ! */
		if(xmlStrcmp(tmp, (xmlChar*)"Monopoly"))  {
			g_free(tmp);
			return;
		}

		if (id < 0) {
			game = g_strdup_printf("Create a new %s game", (gchar*)tmp);
		} else {
			game = g_strdup_printf("Join %s game #%d", (gchar*)tmp, id);
		}
		g_free(tmp);
	}

	description = (gchar*)xmlGetProp(cur, (xmlChar*)"description");
	if (update == TRUE && description) {
		gtk_list_store_set(store, &iter,
		GAMELIST_COLUMN_DESCRIPTION, description,
		-1);
	}

	players = (gchar*)xmlGetProp(cur, (xmlChar*)"players");
	if (update == TRUE && players) {
		gtk_list_store_set(store, &iter,
		GAMELIST_COLUMN_PLAYERS, players,
		-1);
	}

	gametype = (gchar*)xmlGetProp(cur, (xmlChar*)"gametype");

	if(update == FALSE) {
		version = connection[connectid]->server_version;

		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
		GAMELIST_COLUMN_HOST, host,
		GAMELIST_COLUMN_VERSION, version,
		GAMELIST_COLUMN_GAME, game,
		GAMELIST_COLUMN_DESCRIPTION, description,
		GAMELIST_COLUMN_PLAYERS, players,
		GAMELIST_COLUMN_PORT, port,
		GAMELIST_COLUMN_GAMETYPE, gametype,
		GAMELIST_COLUMN_ID, id,
		GAMELIST_COLUMN_SOURCE, GAMELIST_SOURCE_CUSTOM,
		-1);
	}

	g_free(game);
	g_free(description);
	if(players) g_free(players);
	g_free(gametype);
}


void xmlparse_gamelist_deletegame(guint32 connectid, xmlDocPtr doc, xmlNodePtr cur) {
	xmlChar *tmp;
	gint32 id;
	gchar *host;
	gint32 port;
	GtkListStore *store;
	GtkTreeIter iter;
	gboolean valid;
	(void)doc;

	if (connectid != global->customserver_connectid) {
		return;
	}

	if (global->phase != PHASE_GETGAMES)  {
		client_disconnect(connectid);
		global->customserver_connectid = 0;
		return;
	}

	tmp = xmlGetProp(cur, (xmlChar*)"gameid");
	id = atoi((gchar*)tmp);
	g_free(tmp);

	host = connection[connectid]->host;
	port = connection[connectid]->port;

	store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(global->GameList)));
	if (store == NULL) {
		return;
	}

	valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
	while (valid) {
		gchar *ehost;
		gint32 eport;
		gint32 source;
		gint32 eid;
		GtkTreeIter curiter = iter;

		valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter); /* get next iter so we can remove the current entry safely */

		gtk_tree_model_get(GTK_TREE_MODEL(store), &curiter, GAMELIST_COLUMN_ID, &eid, GAMELIST_COLUMN_HOST, &ehost, GAMELIST_COLUMN_PORT, &eport, GAMELIST_COLUMN_SOURCE, &source, -1);
		if ( eid == id  &&  source == GAMELIST_SOURCE_CUSTOM  &&  !strcmp(host, ehost)  &&  eport == port )  {
			gtk_list_store_remove(store, &curiter);
		}
		g_free(ehost);
	}
}


void xmlparse_metaserver(guint32 connectid, gchar *buffer) {

	xmlChar *tmp;
	xmlDocPtr doc;
	xmlNodePtr cur, cur2;
	guint32 do_close = TRUE;
	GtkListStore *store;

	if (global->phase != PHASE_GETGAMES)  {
		client_disconnect(connectid);
		global->metaserver_connectid = 0;
		return;
	}

	doc = xmlParseMemory(buffer,  strlen(buffer) );
	if(doc == NULL)  return;

	cur = xmlDocGetRootElement(doc);
	if (cur == NULL) {

		xmlFreeDoc(doc);
		return;
	}

	if( xmlStrcmp(cur->name, METASERVER_XMLROOTELEMENT) )  {

		xmlFreeDoc(doc);
		return;
	}

	if(connection[connectid]->type == CONNECT_TYPE_META_GETLIST)  {
		store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(global->ServerList)));
		if (store == NULL) {
			return;
		}
		gtk_list_store_clear(store);
		gtk_widget_hide(global->GameList);
		gtk_widget_show(global->ServerList);
	}
	else /* if(connection[connectid]->type == CONNECT_TYPE_META_GETGAME) */ {
		GtkTreeIter iter;
		gboolean valid;

		store = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(global->GameList)));
		if (store == NULL) {
			return;
		}

		/* remove all games previously fetched from metaserver */
		valid = gtk_tree_model_get_iter_first(GTK_TREE_MODEL(store), &iter);
		while (valid) {
			gint32 source;
			GtkTreeIter curiter = iter;

			valid = gtk_tree_model_iter_next(GTK_TREE_MODEL(store), &iter); /* get next iter so we can remove the current entry safely */

			gtk_tree_model_get(GTK_TREE_MODEL(store), &curiter, GAMELIST_COLUMN_SOURCE, &source, -1);
			if (source == GAMELIST_SOURCE_METASERVER)  {
				gtk_list_store_remove(store, &curiter);
			}
		}
		gtk_widget_hide(global->ServerList);
		gtk_widget_show(global->GameList);
	}

	for(cur = cur->xmlChildrenNode ; cur != NULL ; cur = cur -> next)  {

		if(! xmlStrcmp(cur->name, (xmlChar*)"metaserver") )  {
			gchar *text;

			tmp = xmlGetProp(cur, (xmlChar*)"version");
			text = g_strdup_printf("Atlantic metaserver version %s", tmp);
			g_free(tmp);
			text_insert_message(text, strlen(text));
			g_free(text);

			do_close = FALSE;
		}

		if(! xmlStrcmp(cur->name, (xmlChar*)"msg") )  {
			gchar *text;

			tmp = xmlGetProp(cur, (xmlChar*)"value");
			text = g_strdup_printf("%s", tmp);
			g_free(tmp);
			interface_create_messagewin(text);
			g_free(text);

			do_close = FALSE;
		}


		if(connection[connectid]->type == CONNECT_TYPE_META_GETLIST
		  && !xmlStrcmp(cur->name, (xmlChar*)"server") )  {
			gchar *host, *version;
			gint32 port, users;
			GtkTreeIter iter;

			host = (gchar*)xmlGetProp(cur, (xmlChar*)"host");
			tmp = xmlGetProp(cur, (xmlChar*)"port");
			port = atoi((gchar*)tmp);
			g_free(tmp);
			version = (gchar*)xmlGetProp(cur, (xmlChar*)"version");
			tmp = xmlGetProp(cur, (xmlChar*)"users");
			users = atoi((gchar*)tmp);
			g_free(tmp);

			gtk_list_store_append(store, &iter);
			gtk_list_store_set(store, &iter,
			  SERVERLIST_COLUMN_HOST, host,
			  SERVERLIST_COLUMN_PORT, port,
			  SERVERLIST_COLUMN_VERSION, version,
			  SERVERLIST_COLUMN_USERS, users,
			  -1);

			g_free(host);
			g_free(version);
		}

		if(connection[connectid]->type == CONNECT_TYPE_META_GETGAME
		   && !xmlStrcmp(cur->name, (xmlChar*)"servergamelist") )  {
			gchar *host, *version;
			gint32 port;

			host = (gchar*)xmlGetProp(cur, (xmlChar*)"host");
			tmp = xmlGetProp(cur, (xmlChar*)"port");
			port = atoi((gchar*)tmp);
			g_free(tmp);
			version = (gchar*)xmlGetProp(cur, (xmlChar*)"version");

			for(cur2 = cur->xmlChildrenNode ; cur2 != NULL ; cur2 = cur2 -> next )  {

				if(! xmlStrcmp(cur2->name, (xmlChar*)"game") )  {
					GtkTreeIter iter;
					gchar *game, *players = NULL, *description, *gametype;
					gint32 id;

					tmp = xmlGetProp(cur2, (xmlChar*)"canbejoined");
					if(tmp  &&  atoi((gchar*)tmp) == 0 )  {
						g_free(tmp);
						continue;
					}
					if(tmp) g_free(tmp);

					/* ONLY TEMPORARY ! */
					tmp = xmlGetProp(cur2, (xmlChar*)"gametype");
					if(xmlStrcmp(tmp, (xmlChar*)"city")  &&  xmlStrcmp(tmp, (xmlChar*)"french_city") )  {
						g_free(tmp);
						continue;
					}
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"id");
					id = atoi((gchar*)tmp);
					g_free(tmp);

					tmp = xmlGetProp(cur2, (xmlChar*)"name");
					if(id < 0) {
						game = g_strdup_printf("Create a new %s game", (gchar*)tmp);
					} else {
						game = g_strdup_printf("Join %s game #%d", (gchar*)tmp, id);
					}
					g_free(tmp);

					description = (gchar*)xmlGetProp(cur2, (xmlChar*)"description");
					players = (gchar*)xmlGetProp(cur2, (xmlChar*)"players");
					gametype = (gchar*)xmlGetProp(cur2, (xmlChar*)"gametype");

					gtk_list_store_append(store, &iter);
					gtk_list_store_set(store, &iter,
					  GAMELIST_COLUMN_HOST, host,
					  GAMELIST_COLUMN_VERSION, version,
					  GAMELIST_COLUMN_GAME, game,
					  GAMELIST_COLUMN_DESCRIPTION, description,
					  GAMELIST_COLUMN_PLAYERS, players,
					  GAMELIST_COLUMN_PORT, port,
					  GAMELIST_COLUMN_GAMETYPE, gametype,
					  GAMELIST_COLUMN_ID, id,
					  GAMELIST_COLUMN_SOURCE, GAMELIST_SOURCE_METASERVER,
					  -1);

					g_free(game);
					g_free(description);
					if(players) g_free(players);
					g_free(gametype);
				}
			}

			g_free(host);
			g_free(version);
		}

	}

	xmlFreeDoc(doc);
	if (do_close) {
		client_disconnect(connectid);
		global->metaserver_connectid = 0;
	}
}
