# GPixPod - organize photos on your iPod, freely!
# Copyright (C) 2006 Flavio Gargiulo (FLAGAR.com)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

from struct import *
from glob import glob
import time, os, cPickle

def readChunk(file, offset, size=4):
    """ Read a chunk of data from a file """
    file.seek(offset)
    return file.read(size)

def unpackChunk(file, offset, format='<l'):
    """ Unpack a chunk of data using the given format """
    size = calcsize(format)
    chunk = readChunk(file, offset, size)
    return unpack(format, chunk)

def updateChunk(chunk, pos, format, value):
    """ Update part of a chunk with new value with the given format """
    size = calcsize(format)
    return chunk[0:pos]+pack(format, value)+chunk[pos+size:]

def fileWrite(filename, data, mode='w+b'):
    """ Write contents to a file in one-pass """
    f = open(filename, mode)
    f.write(data)
    f.close()

def fileRead(filename, mode='rb'):
    """ Read and return the contents of a file in one-pass """
    f = open(filename, mode)
    x = f.read()
    f.close()
    return x

def getPadding(length):
    """ Calculate the padding on a 4-bytes boundary basis """
    # Maybe it could be more elegant with division and rounding?
    padding = 4 - (length % 4)
    if padding < 4:
        return padding
    else:
        return 0

def appleTime():
    """ Get current time in Apple format (+2082844800", Apple time is from 1904-01-01) """
    return int(time.time())+2082844800

def getRatioSize(width, height, max_width, max_height):
    """ Recalculate width and height keeping ratio aspect scaling within the size limits """
    # Calculating proper width/height with ratio within limits
    new_width = width
    new_height = height
    if new_width > max_width or new_height > max_height:
        new_width = max_width
        new_height = max_width*height/width
    if new_height > max_height:
        new_height = max_height
        new_width = max_height*width/height
    return (new_width, new_height)

def detectIpodModel(ipod_mountpoint):
    """ Get iPod model """
    sysinfo_filename = os.path.join(ipod_mountpoint, 'iPod_Control', 'Device', 'SysInfo')
    if os.path.isfile(sysinfo_filename):
        sysinfo_file = open(sysinfo_filename)
        for sysinfo_line in sysinfo_file:
            if sysinfo_line.split(':')[0] == 'boardHwSwInterfaceRev':
	        ipod_mcode = sysinfo_line.split(' ')[1]
    	        # Infos got from http://ipodlinux.org/Generations
	        if ipod_mcode == '0x00010000':
	            return '1G'
	        elif ipod_mcode == '0x00020000':
	            return '2G'
	        elif ipod_mcode == '0x00030001':
	            return '3G'
	        elif ipod_mcode == '0x00040013':
	            return '1G Mini'
	        elif ipod_mcode == '0x00050013':
	            return '4G'
	        elif ipod_mcode == '0x00060000':
	            return 'Photo'
	        elif ipod_mcode == '0x00060004':
	            return 'Color'
	        elif ipod_mcode == '0x00070002':
	            return '2G Mini'
	        elif ipod_mcode == '0x000B0005':
	            return '5G'
	        elif ipod_mcode == '0x000C0005':
	            return 'Nano'
                elif ipod_mcode == '0x000C0006':
                    return 'Nano'  # Latest, 1Gb model (Nano 2nd gen.?)
                else:
	            return 'Unrecognized'
    else:
        print "Problems opening %s. Does this file exist? If it doesn't exist, then it should be an iPod shuffle?" % sysinfo_filename

def retrieveIpodModel():
    """ Get manually configured iPod model """
    configfile = os.path.join(os.path.expanduser('~'), '.gpixpod', 'config')
    try:
        configf = open(configfile)
        configsettings = cPickle.load(configf)
        ipodmodel = configsettings['ipod_model']
        configf.close()
        return ipodmodel
    except:
        return None

def getIpodModel(ipod_mountpoint):
    """ Get iPod model trying detecting and if unsuccessful, trying reading manually configured model """
    ipodmodel = retrieveIpodModel()
    if ipodmodel == None:
        ipodmodel = detectIpodModel(ipod_mountpoint)
    return ipodmodel

def expand_paths(mixed, recursive):
    """ Expand paths for filenames with wildcards (*) and directories with their contents """
    # Processing wildcards with glob
    filenames = filenames2 = []
    for x in range(len(mixed)):
        filenames.extend(glob(mixed[x]))
    # Extracting contents of the directories, instead of them if recursive
    for x in range(len(filenames)):
        if recursive and os.path.isdir(filenames[x]):
            for topdir, dirnames, files in os.walk(filenames[x]):
                for y in files:
                    filenames2.append(os.path.join(topdir, y))
        elif os.path.isfile(filenames[x]):
            filenames2.append(filenames[x])
    return list(set(filenames2))
