/***************************************************************************
 *            qof-expenses.c
 *
 *  Thu Oct 21 07:59:13 2004-2005
 *  Copyright  2004-2005  Neil Williams  <linux@codehelp.co.uk>
 ****************************************************************************/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
  */

#include "config.h"
#include <stdlib.h>
#include <glib.h>
#include <libintl.h>
#include <glib/gprintf.h>
#include <qof.h>
#include <locale.h>
#include "qof-expenses.h"
#define _(String) dgettext (LIBRARY_GETTEXT_PACKAGE, String)

static QofLogModule log_module = GPE_MOD_EXP;

AS_STRING_FUNC  (ExpenseDistance, DISTANCE_LIST)
FROM_STRING_FUNC(ExpenseDistance, DISTANCE_LIST)
AS_STRING_FUNC  (ExpensePayment, PAYMENT_TYPE_LIST)
FROM_STRING_FUNC(ExpensePayment, PAYMENT_TYPE_LIST)
/* leave the macro in place to support types.
 allow translation by mapping the gcc -E output.
*/
AS_STRING_FUNC  (ExpenseType, EXPENSE_TYPE_LIST)
FROM_STRING_FUNC(ExpenseType, EXPENSE_TYPE_LIST)

/** \brief Copy of qof-main function

Copied into qof-expenses library to prevent a spurious
dependency.
*/
static gchar *
qof_main_make_utf8 (gchar * string)
{
	gchar *value;

	if (!string)
		return NULL;
	if (g_utf8_validate (string, -1, NULL))
		return string;
	value = g_locale_to_utf8 (string, -1, NULL, NULL, NULL);
	if (!value)
	{
		PWARN (" unable to convert from locale %s", string);
		PINFO ("trying to convert from ISO-8859-15.");
		value = g_convert (string, -1, "UTF-8", "ISO-8859-15",
			NULL, NULL, NULL);
		if (!value)
		{
			PERR (" conversion failed");
			return string;
		}
		return value;
	}
	return value;
}

/** \brief Currency Table data
	
Table relating the currencies[5] to the actual currency names, mnemonics
 and symbols. In pilot-qof it is indexed by GINT_TO_POINTER(pq_code). */
static GHashTable *gpe_currency_table = NULL;

void
gpe_currency_foreach(GpeCurrencyCB cb, gpointer user_data)
{
	g_hash_table_foreach(gpe_currency_table, cb, user_data);
}

/** \brief Populate the currency table with the known currencies.

 Custom currencies are not included.

 All mnemonics are from gnucash and use the ISO4217 namespace

 custom currencies are the same and may end up with the same
 struct: gchar name[16], gchar symbol[4], gchar rate[8].

 Ignore currencies[5] (the five available currencies
 from the main table above) but:

 \todo store currencies[4] from AppInfo which
 are the four custom currencies.
*/

static void
populate_currencies (void)
{
	gpe_currency_table = g_hash_table_new(g_direct_hash, g_direct_equal);
	{	/* Australia 	0	100		AU$		AUD */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 0;
		c->fraction = 100;
		c->symbol = "AU$";
		c->mnemonic = "AUD";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Austria 	1	100		€		ATS */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 1;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "ATS";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Belgium  	2	100		€		BEF */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 2;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "BEF";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Brazil  	3	100		R$		BRL */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 3;
		c->fraction = 100;
		c->symbol = "R$";
		c->mnemonic = "BRL";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Canada	4	100		$CN		CAD */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 4;
		c->fraction = 100;
		c->symbol = "$CN";
		c->mnemonic = "CAD";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Denmark   	5	100		DKK		DKK */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 5;
		c->fraction = 100;
		c->symbol = "DKK";
		c->mnemonic = "DKK";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Finland   	6	100		€		FIM */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 6;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "FIM";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* France    	7	100		€		FRF */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 7;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "FRF";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Germany   	8	100		€		DEM */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 8;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "DEM";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Hong Kong	9	100		HK$		HKD */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 9;
		c->fraction = 100;
		c->symbol = "HK$";
		c->mnemonic = "HKD";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Iceland 	10	100		ISK		ISK */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 10;
		c->fraction = 100;
		c->symbol = "ISK";
		c->mnemonic = "ISK";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Ireland   	11	100		€		IEP */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 11;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "IEP";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Italy   	12	1		EUR		ITL */
		/* The Italian Lira had a fraction == 1*/
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 12;
		c->fraction = 100;
		c->symbol = "EUR";
		c->mnemonic = "ITL";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Japan     	13	1	&#165;	¥		JPY */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 13;
		c->fraction = 1;
		c->symbol = "¥";
		c->mnemonic = "JPY";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Luxembourg 	14	100	&#136;	€		LUF */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 14;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "LUF";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Mexico   	15	100		MXP		MXP */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 15;
		c->fraction = 100;
		c->symbol = "MXP";
		c->mnemonic = "MXP";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Netherlands 	16	100		€		ANG */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 16;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "ANG";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* New Zealand 	17	100		$NZ		NZD */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 17;
		c->fraction = 100;
		c->symbol = "$NZ";
		c->mnemonic = "NZD";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Norway    	18	100		NOK		NOK */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 18;
		c->fraction = 100;
		c->symbol = "NOK";
		c->mnemonic = "NOK";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Spain   	19	100		€		ESP */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 19;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "ESP";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Sweden 	20	100		SEK		SEK */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 20;
		c->fraction = 100;
		c->symbol = "SEK";
		c->mnemonic = "SEK";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Switzerland 	21	100		CHF		CHF */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 21;
		c->fraction = 100;
		c->symbol = "CHF";
		c->mnemonic = "CHF";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* United Kingdom 22	100		£		GBP */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 22;
		c->fraction = 100;
		c->symbol = "£";
		c->mnemonic = "GBP";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* United States  23	100		$US		USD */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 23;
		c->fraction = 100;
		c->symbol = "$US";
		c->mnemonic = "USD";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* India     	24	100		Rs		INR */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 24;
		c->fraction = 100;
		c->symbol = "Rs";
		c->mnemonic = "INR";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Indonesia 	25	1		Rp		IDR */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 25;
		c->fraction = 1;
		c->symbol = "Rp";
		c->mnemonic = "IDR";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Korea   	26	100		KRW		KRW  (South) */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 26;
		c->fraction = 100;
		c->symbol = "KRW";
		c->mnemonic = "KRW";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Malaysia  	27	100		RM		MYR */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 27;
		c->fraction = 100;
		c->symbol = "RM";
		c->mnemonic = "MYR";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* P.R.C. // People's Rep. China 28	100	RMB		CNY */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 28;
		c->fraction = 100;
		c->symbol = "RMB";
		c->mnemonic = "CNY";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Phillipines	29	100		P		PHP */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 29;
		c->fraction = 100;
		c->symbol = "P";
		c->mnemonic = "PHP";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Singapore 	30	100		$		SGD */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 30;
		c->fraction = 100;
		c->symbol = "$";
		c->mnemonic = "SGD";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Thailand  	31	100		BHT		THB */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 31;
		c->fraction = 100;
		c->symbol = "BHT";
		c->mnemonic = "THB";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* Taiwan	32	100		NT$		TWD */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 32;
		c->fraction = 100;
		c->symbol = "NT$";
		c->mnemonic = "TWD";
		c->non_utf8 = FALSE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
	{	/* EU (Euro) 	133	100		€		EUR */
		GpeCurrency *c = g_new0(GpeCurrency, 1);
		c->pq_code = 133;
		c->fraction = 100;
		c->symbol = "€";
		c->mnemonic = "EUR";
		c->non_utf8 = TRUE;
		g_hash_table_insert(gpe_currency_table, 
			GINT_TO_POINTER(c->pq_code), (gpointer)c);
	}
}

static gboolean
check_name (gpointer G_GNUC_UNUSED key, gpointer value, gpointer data)
{
	gchar * mnemonic = (gchar*) data;
	GpeCurrency * currency = (GpeCurrency*) value;
	if (0 == safe_strcmp (mnemonic, currency->mnemonic))
		return TRUE;
	return FALSE;
}

GpeCurrency*
gpe_currency_lookup_name (QofInstance * inst, gchar * mnemonic)
{
	GpeCurrency *currency;

	currency = NULL;
	if(!gpe_currency_table) populate_currencies();
	currency = (GpeCurrency*) g_hash_table_find 
		(gpe_currency_table, check_name, mnemonic);
	gpe_currency_lookup(inst, currency->pq_code);
	return currency;
}

GpeCurrency*
gpe_currency_lookup (QofInstance* inst, gint currency_code)
{
	GpeCurrency *currency;

	currency = NULL;
	if(!gpe_currency_table) populate_currencies();
	currency = (GpeCurrency*)g_hash_table_lookup(
		gpe_currency_table, GINT_TO_POINTER(currency_code));
	if(!currency)
	{
		PERR (" unsupported currency! %d", currency_code);
		return NULL;
	}
	kvp_frame_set_string(qof_instance_get_slots(inst),
		PQ_CURRENCY_MNEMONIC, currency->mnemonic);
	kvp_frame_set_gint64(qof_instance_get_slots(inst),
		PQ_CURRENCY_FRACTION, currency->fraction);
	if(currency->non_utf8)
	{
		gchar * k_symbol = g_strdup (currency->symbol);
		kvp_frame_set_string(qof_instance_get_slots(inst),
			PQ_CURRENCY_SYMBOL, qof_main_make_utf8(k_symbol));
	}
	else
	{
		kvp_frame_set_string(qof_instance_get_slots(inst),
			PQ_CURRENCY_SYMBOL, currency->symbol);
	}
	return currency;
}	

typedef struct ExpenseCustomCurrency {
		gchar name[16];
		gchar symbol[4];
		gchar rate[8];
} ExpenseCustomCurrency_t;

struct Expense {
		struct tm date;
		ExpenseType type;
		ExpensePayment payment;
		gint currency;
		gchar *amount;
		gchar *vendor;
		gchar *city;
		gchar *attendees;
		gchar *note;
};

struct ExpenseAppInfo {
//	struct CategoryAppInfo category;
	struct ExpenseCustomCurrency currencies[4];
};

struct QofExp_s
{
	QofInstance inst;
	Expense_t wrap;
	ExpenseDistance distance_unit;
	gchar *category;
	const gchar* print_string;
	GpeCurrency *currency;
	gdouble temp_amount;     /**<  The amount is set before the
		currency code is available so this acts as a buffer until both
		pieces of data are available. */
	gboolean reset_amount;   /**< Denote whether the buffer should be used. */
};

static QofExp *
expense_create (QofBook *book)
{
	Expense_t *qe;
	QofExp *obj;
	QofCollection *coll;
	GList *all;

	obj = g_new0(QofExp, 1);
	qof_instance_init (&obj->inst, GPE_QOF_EXPENSES, book);
	coll = qof_book_get_collection (book, GPE_QOF_EXPENSES);
	all = qof_collection_get_data (coll);
	all = g_list_prepend (all, obj);
	qof_collection_set_data (coll, all);
	qe = &obj->wrap;
	{
		glong nanosecs;
		QofTime *qt;
		QofDate *qd;

		qt = qof_time_get_current ();
		nanosecs = qof_time_get_nanosecs (qt);
		qd = qof_date_from_qtime (qt);
		if (!qof_date_to_struct_tm (qd, &qe->date, 
			&nanosecs))
			DEBUG (" failed to set initial date");
		qof_date_free (qd);
		qof_time_free (qt);
	}
	qe->amount = "0";
	/* 0 == AU$ so use an init value */
	qe->currency = -1;
	if(!gpe_currency_table) populate_currencies();
	qof_event_gen ((QofEntity*)obj, QOF_EVENT_CREATE, NULL);
	return obj;
}

static const gchar*
qof_exp_paymentAsString(ExpensePayment payment)
{
	const gchar *string;

	string = "Cash"; // following install-expenses convention.
	string = ExpensePaymentasString(payment);
	return string;
}

static const gchar*
qof_exp_typeAsString(ExpenseType type)
{
	const gchar* string;

	string = "Bus"; // following install-expenses convention.
	string = ExpenseTypeasString(type);
	return string;
}

static ExpensePayment
qof_exp_paymentFromString(const gchar* payment_string)
{
	return ExpensePaymentfromString(payment_string);
}

static ExpenseType
qof_exp_typeFromString(const gchar* type_string)
{
	return ExpenseTypefromString(type_string);
}

static QofTime *
exp_getTime (QofExp * e)
{
	Expense_t *qe;
	QofDate *qd;
	QofTime *qt;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	qd = qof_date_from_struct_tm (&qe->date);
	qt = qof_date_to_qtime (qd);
	qof_date_free (qd);
	return qt;
}

static gchar*
exp_getType (QofExp * e)
{
	Expense_t *qe;
	gchar* string;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	string = g_strdup(qof_exp_typeAsString(qe->type));
	return string;
}

static gchar*
exp_getPayment (QofExp * e)
{
	Expense_t *qe;
	gchar* string;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	string = g_strdup(qof_exp_paymentAsString(qe->payment));
	return string;
}

static gint
exp_getCurrency (QofExp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, -1);
	qe = &e->wrap;
	return qe->currency;
}

static QofNumeric
exp_getAmount (QofExp * e)
{
	Expense_t *qe;
	QofNumeric amount;
	gdouble pi_amount;
	gchar *numeric_char;

	amount = qof_numeric_zero ();
	g_return_val_if_fail (e != NULL, amount);
	qe = &e->wrap;
	if(qe->amount == 0) { return amount; }
	/* floating point as a string converts to gnc_numeric */
	pi_amount = strtod (qe->amount, NULL);
	if(e->currency)
	{
		amount = qof_numeric_from_double (pi_amount, e->currency->fraction,
			QOF_HOW_DENOM_EXACT | QOF_HOW_RND_ROUND);
	}
	else /* default: use the most common fraction in the Palm currency list. */
	{
	amount = qof_numeric_from_double (pi_amount, 100, 
		QOF_HOW_DENOM_EXACT | QOF_HOW_RND_ROUND);
	}
	numeric_char = qof_numeric_to_string(amount);
	g_free(numeric_char);

	if (qof_numeric_check (amount) == QOF_ERROR_OK)
	{
		return amount;
	}
	return qof_numeric_zero ();
}

static const gchar *
exp_getVendor (QofExp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->vendor;
}

static const gchar *
exp_getCity (QofExp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->city;
}

static const gchar *
exp_getAttendees (QofExp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->attendees;
}

static const gchar *
exp_getNote (QofExp * e)
{
	Expense_t *qe;

	g_return_val_if_fail (e != NULL, NULL);
	qe = &e->wrap;
	return qe->note;
}

static const gchar*
exp_getDistance(QofExp *e)
{
	g_return_val_if_fail(e != NULL, NULL);
	return ExpenseDistanceasString(e->distance_unit);
}

static const gchar* 
exp_getCategory(QofExp *e)
{
	g_return_val_if_fail(e != NULL, NULL);
	return e->category;
}

static void
exp_setTime (QofExp * e, QofTime *h)
{
	Expense_t *qe;
	glong nanosecs;
	QofDate *qd;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	if (!h)
		return;
	nanosecs = qof_time_get_nanosecs (h);
	qd = qof_date_from_qtime (h);
	qof_date_to_struct_tm (qd, &qe->date, 
		&nanosecs);
	qof_date_free (qd);
}

static void
exp_setType (QofExp * e, const gchar *type_string)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->type = qof_exp_typeFromString(type_string);
}

static void
exp_setPayment (QofExp * e, const gchar *payment_string)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->payment = qof_exp_paymentFromString(payment_string);
}

static void
exp_combine_currency_with_amount(QofExp *e)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	if(!e->currency || qe->currency < 0)
	{
		/* the gint32 currency is actually to be set BEFORE the numeric amount. :-( */
		e->currency = gpe_currency_lookup((QofInstance*)e, qe->currency);
	}
	if(!e->currency)
	{
		PERR (" Unable to identify currency fraction."
			" Using two decimal places.");
		/* Amount is stored in the Palm as a string version
		of a floating point number. */
		qe->amount = g_strdup_printf ("%.2f", e->temp_amount);
		return;
	}
	switch (e->currency->fraction)
	{
		case 1 : 
		{
			qe->amount = g_strdup_printf ("%.0f", e->temp_amount);
			break;
		}
		case 10 : 
		{
			qe->amount = g_strdup_printf ("%.1f", e->temp_amount);
			break;
		}
		case 100 : 
		{
			qe->amount = g_strdup_printf ("%.2f", e->temp_amount);
			break;
		}
		case 1000 : 
		{
			qe->amount = g_strdup_printf ("%.3f", e->temp_amount);
			break;
		}
		default :
		{
			PERR (" Invalid currency fraction."
				" Using two decimal places as default.");
			qe->amount = g_strdup_printf ("%.2f", e->temp_amount);
		}
	}
}

static void
exp_setCurrency (QofExp * e, gint data)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->currency = data;
	e->currency = gpe_currency_lookup((QofInstance*)e, data);
	if(e->reset_amount)
	{
		exp_combine_currency_with_amount(e);
	}
	e->reset_amount = FALSE;
}

static void
exp_setAmount (QofExp * e, QofNumeric h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	e->temp_amount = qof_numeric_to_double (h);
	e->reset_amount = TRUE;
	/* if an amount can ever be set without a currency_code,
		   this needs to be reviewed. */
	/** \todo FIXME: INSERT handler can set one without the
	other. Need to use the pref.default_currency? */
	if(e->currency) exp_combine_currency_with_amount(e);
}

static void
exp_setVendor (QofExp * e, gchar *h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->vendor = g_strdup (qof_main_make_utf8(h));
}

static void
exp_setCity (QofExp * e, gchar *h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->city = g_strdup (qof_main_make_utf8(h));
}

static void
exp_setAttendees (QofExp * e, gchar *h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->attendees = g_strdup (qof_main_make_utf8(h));
}

static void
exp_setNote (QofExp * e, gchar *h)
{
	Expense_t *qe;

	g_return_if_fail (e != NULL);
	qe = &e->wrap;
	qe->note = g_strdup (qof_main_make_utf8(h));
}

static void
exp_setDistance(QofExp *e, const gchar *distance_name)
{

	g_return_if_fail(e);
	e->distance_unit = ExpenseDistancefromString(distance_name);
}

static void
exp_setCategory(QofExp *e, gchar *n)
{
	g_return_if_fail(e != NULL);
	e->category = g_strdup(qof_main_make_utf8(n));
}

static const gchar*
expensePrintable (gpointer instance)
{
	QofExp *obj;
	
	obj = (QofExp*)instance;
	if(!obj) return NULL;
	if(exp_getType(obj))
	{
	return g_strconcat(exp_getType(obj), " ",
		exp_getVendor(obj), " ", exp_getCity(obj), NULL);
	}
	return NULL;
}

static QofObject expenses_object_def = {
      .interface_version = QOF_OBJECT_VERSION,
      .e_type =            GPE_QOF_EXPENSES,
      .type_label =        QOF_EXPENSES_DESC,
      .create =            ((gpointer)expense_create),
      .book_begin =        NULL,
      .book_end =          NULL,
      .is_dirty =          qof_collection_is_dirty,
      .mark_clean =        qof_collection_mark_clean,
      .foreach =           qof_collection_foreach,
	  .printable =         expensePrintable,
      .version_cmp =       (gint (*)(gpointer, gpointer)) qof_instance_version_cmp,
};

gboolean
ExpensesRegister (void)
{
	static QofParam params[] = {
	 { EXP_DATE,       QOF_TYPE_TIME,    (QofAccessFunc) exp_getTime,      (QofSetterFunc) exp_setTime, NULL },
	 { EXP_TYPE,       QOF_TYPE_STRING,  (QofAccessFunc) exp_getType,      (QofSetterFunc) exp_setType, NULL },
	 { EXP_PAYMENT,    QOF_TYPE_STRING,  (QofAccessFunc) exp_getPayment,   (QofSetterFunc) exp_setPayment, NULL },
	 { EXP_CURRENCY,   QOF_TYPE_INT32,   (QofAccessFunc) exp_getCurrency,  (QofSetterFunc) exp_setCurrency, NULL },
	 { EXP_AMOUNT,     QOF_TYPE_NUMERIC, (QofAccessFunc) exp_getAmount,    (QofSetterFunc) exp_setAmount, NULL },
	 { EXP_VENDOR,     QOF_TYPE_STRING,  (QofAccessFunc) exp_getVendor,    (QofSetterFunc) exp_setVendor, NULL },
	 { EXP_CITY,       QOF_TYPE_STRING,  (QofAccessFunc) exp_getCity,      (QofSetterFunc) exp_setCity, NULL },
	 { EXP_ATTENDEES,  QOF_TYPE_STRING,  (QofAccessFunc) exp_getAttendees, (QofSetterFunc) exp_setAttendees, NULL },
	 { EXP_NOTE,       QOF_TYPE_STRING,  (QofAccessFunc) exp_getNote,      (QofSetterFunc) exp_setNote, NULL },
	 { EXP_DISTANCE,   QOF_TYPE_STRING,  (QofAccessFunc) exp_getDistance,  (QofSetterFunc) exp_setDistance, NULL },
	 { EXP_CATEGORY,   QOF_TYPE_STRING,  (QofAccessFunc) exp_getCategory,  (QofSetterFunc) exp_setCategory, NULL },
	 { EXP_KVP,        QOF_TYPE_KVP,     (QofAccessFunc) qof_instance_get_slots, NULL, NULL },
	 { QOF_PARAM_BOOK, QOF_ID_BOOK,      (QofAccessFunc) qof_instance_get_book, NULL, NULL },
	 { QOF_PARAM_GUID, QOF_TYPE_GUID,    (QofAccessFunc) qof_instance_get_guid, NULL, NULL },
	 { NULL, NULL, NULL, NULL, NULL },
	};

	bindtextdomain (LIBRARY_GETTEXT_PACKAGE, LOCALE_DIR);

	qof_class_register (GPE_QOF_EXPENSES, NULL, params);
	if(!gpe_currency_table) populate_currencies();

	return qof_object_register (&expenses_object_def);
}
