/*
     This file is part of gnunet-qt.
     (C) 2006, 2007 Nils Durner (and other contributing authors)

     gnunet-qt is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     gnunet-qt is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/core/wndMain.cc
 * @brief Main window of gnunet-qt
 * @author Nils Durner
 */

#include <QWhatsThis>
#include <QSizeGrip>

#include "wndMain.h"

class GStatusTextEvent : public QEvent
{
public:
  GStatusTextEvent(QEvent::Type t) : QEvent(t){};

  QPixmap icon;
  QString text;
};

class GNetworkStatusEvent : public GStatusTextEvent
{
public:
  GNetworkStatusEvent(QEvent::Type t) : GStatusTextEvent(t){};
};

GMainWindow::GMainWindow(QWidget *parent) : QMainWindow(parent)
{
  QHBoxLayout *lay;
  
  setupUi(this);
  delete tabRoot;
  
  QStatusBar *bar = statusBar();
  if (bar)
  {
    lay = new QHBoxLayout();
    
    statusText.setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
    
    bar->setSizeGripEnabled(false);

    lay->addWidget(&statusIcon);
    lay->addWidget(&statusText);
    lay->addWidget(&networkIcon);
    lay->addWidget(&networkText);
    delete bar->layout();
    bar->setLayout(lay);    

    networkIcon.setToolTip(tr("Number of connected peers"));
    networkText.setToolTip(tr("Number of connected peers"));
  }
  
  connect(action_Context_help, SIGNAL(triggered()), this, SLOT(contextHelp()));
  connect(actionAbout, SIGNAL(triggered()), this, SLOT(about()));
  connect(actionExit, SIGNAL(triggered()), qApp, SLOT(quit()));
}

void GMainWindow::contextHelp()
{
  if (action_Context_help->isChecked())
    QWhatsThis::enterWhatsThisMode();
  else
    QWhatsThis::leaveWhatsThisMode();
}

void GMainWindow::about()
{
  GAboutDlg dlg(this);
  
  dlg.exec();
}

void GMainWindow::addApplication(QWidget *wnd, const QIcon &icon, const QString &label)
{
  tabWidget->addTab(wnd, icon, label);
  connect(wnd, SIGNAL(setStatusText(const QPixmap &, const QString)),
    this, SLOT(setStatusText(const QPixmap &, const QString)));
  connect(wnd, SIGNAL(setNetworkStatus(const QPixmap &, const QString)),
    this, SLOT(setNetworkStatus(const QPixmap &, const QString)));
}

void GMainWindow::setStatusText(const QPixmap &icon, const QString strText)
{
  GStatusTextEvent *e;
  
  e = new GStatusTextEvent(QEvent::User);
  e->icon = icon;
  e->text = strText;

  qApp->postEvent(this, e);
}

void GMainWindow::setNetworkStatus(const QPixmap &icon, const QString strText)
{
  GNetworkStatusEvent *e;
  
  e = new GNetworkStatusEvent(QEvent::User);
  e->icon = icon;
  e->text = strText;
  
  qApp->postEvent(this, e);
}

bool GMainWindow::event(QEvent *event)
{
  GStatusTextEvent *e;
  
  e = dynamic_cast<class GNetworkStatusEvent *> (event);
  if (e)
  {
    networkIcon.setPixmap(e->icon);
    networkText.setText(e->text);
    return true;
  }
  else
  {
    e = dynamic_cast<class GStatusTextEvent *> (event);
    if (e)
    {
      statusIcon.setPixmap(e->icon);
      statusText.setText(e->text);
      return true;
    }
  }
  
  return QMainWindow::event(event);  
}

/* end of wndMain.cc */
