/*
     This file is part of gnunet-qt.
     (C) 2007 Nils Durner (and other contributing authors)

     gnunet-qt is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     gnunet-qt is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with gnunet-qt; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/stats/statsPlugin.cc
 * @brief gnunet-qt's statistics tab
 * @author Nils Durner
 */
#include "config.h"
#include "statsPlugin.h"

#include <QTreeWidgetItem>

#include <GNUnet/gnunet_directories.h>
#include <GNUnet/gnunet_util.h>
#include <GNUnet/gnunet_util_boot.h>
#include <GNUnet/gnunet_stats_lib.h>


GStatsPlugin::GStatsPlugin(struct GC_Configuration *config,
  struct GE_Context *errorContext) : GPlugin()
{
  setupUi(this);
  
  this->config = config;
  this->errorContext = errorContext;
  
  connect(pbRefresh, SIGNAL(clicked(bool)), this, SLOT(refresh()));
  
  treeStats->setColumnWidth(0, 550);
}

GStatsPlugin::~GStatsPlugin()
{
}

static int printStatistics(const char *name, unsigned long long value, void *param)
{  
  QTreeWidgetItem *item = new QTreeWidgetItem(0);
  
  item->setText(0, QString::fromUtf8(name));
  item->setText(1, QString::number(value));

  ((QTreeWidget *) param)->addTopLevelItem(item);
  
  return OK;
}

void GStatsPlugin::refresh()
{
  int res;
  struct ClientServerConnection *sock;

  treeStats->clear();

  sock = client_connection_create(errorContext, config);
  if (!sock)
  {
    GE_LOG(errorContext, (GE_KIND) (GE_ERROR | GE_USER | GE_IMMEDIATE),
      qPrintable(tr("Error establishing connection with background process gnunetd.")));
    return;
  }

#if HAVE_STATS
  res = STATS_getStatistics(errorContext,
			    sock,
			    &printStatistics, 
			    treeStats);
#else
  res = requestStatistics(errorContext, sock,
    (StatisticsProcessor) &printStatistics, treeStats);
#endif
  if (res != OK)
    GE_LOG(errorContext, (GE_KIND) (GE_ERROR | GE_USER | GE_IMMEDIATE),
      qPrintable(tr("Error reading information from background process gnunetd.")));
  
  connection_destroy(sock);
}

extern "C"
{
  
  GNUNETQT_API GPlugin *init_plugin(struct GC_Configuration *config,
    struct GE_Context *errorContext)
  {
    return new GStatsPlugin(config, errorContext);
  }
  
  GNUNETQT_API void shutdown_plugin(GPlugin *plugin)
  {
    delete (GStatsPlugin *) plugin;
  }
  
} // extern "C"

/* enf of statsPlugin.cc */
