# GNU Enterprise Application Server - Generators - Class definition
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: classdef.py 7872 2005-08-20 11:41:58Z reinhard $

import datetime

from gnue.appserver import repository
from gnue.common.apps import errors

# =============================================================================
# Exceptions
# =============================================================================

class EmptyClassError (errors.UserError):
  def __init__ (self, classname):
    msg = u_("The class '%s' has no properties to be displayed") % classname
    errors.UserError.__init__ (self, msg)


# =============================================================================
# This class implements a collection of properties for a business class
# =============================================================================

class ClassDef:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, session, klass, language, asReference = False):
    """
    @param session: Language interface session instance
    @param klass: Language interface object with the class to be processed
    @param language: language to be preferred for labels
    @param asReference: if True, only references should be held
    """

    self.__session   = session
    self.__class     = klass
    self.__language  = language
    self.classname   = "%s_%s" % (klass.module.name, klass.name)
    self.classLabel  = "%s %s" % (klass.module.name.title (),
                                  klass.name.title ())
    self.isReference = asReference
    self.isLookup    = False

    (self.properties, self.specials) = self.__loadProperties ()
    if not len (self.properties):
      raise EmptyClassError, self.classname

    self.__loadLabels (klass.gnue_id, self.properties)

    self.__updateReferences ()

    self.virtualPages = self.__buildVirtualPages ()


  # ---------------------------------------------------------------------------
  # Get the name of the sort column for a class definition
  # ---------------------------------------------------------------------------

  def sortColumn (self):
    """
    This function iterates over all search-properties and returns the property
    with the lowest search-value or None if no search properties are available

    @return: property instance with the lowest search-order or None
    """

    search = []
    for item in self.properties:
      if item.search is not None:
        search.append ((item.search, item.dbField))

    search.sort ()

    return len (search) and search [0][1] or None


  # ---------------------------------------------------------------------------
  # Release all resources and pointers held by a classdef instance
  # ---------------------------------------------------------------------------

  def release (self):
    """
    Release all references and pointers of a class definition instance. This
    allows the garbage collection to work fine.
    """

    for item in self.properties + self.specials:
      item.parent = None
      if item.reference is not None:
        item.reference.release ()

      item.reference = None


  # ---------------------------------------------------------------------------
  # Create a sequence of languages to be processed for labels
  # ---------------------------------------------------------------------------

  def __buildLanguageList (self, language):
    """
    This function creates a sequence of languages to be processed for labels.
    The order is from most general (starting with 'C') to most specific.

    @param language: a language like 'de_AT'
    @return: sequence of languages like ['C', 'de', 'de_AT']
    """

    result = ['C']

    if '_' in language:
      add = language.split ('_') [0]
      if not add in result:
        result.append (add)

    if not language in result:
      result.append (language)

    return result


  # ---------------------------------------------------------------------------
  # Load all properties and calculated fields of the requested class
  # ---------------------------------------------------------------------------

  def __loadProperties (self):
    """
    This function populates two squences of properties of the given class.
    The sequence 'properties' contains all 'normal' properties and calculated
    fields, and the sequence 'specials' holds all allowed special properties.
    Both sequences are unordered. If this instance had 'isReference' set, only
    reference properties will be added.
  
    @return: tuple with both sequences (properties, specials)
    """

    properties = []
    specials   = []
    classId    = self.__class.gnue_id

    cond  = {'gnue_class': classId}
    pList = ['gnue_name', 'gnue_length', 'gnue_scale', 'gnue_type']

    for prop in self.__session.find ('gnue_property', cond, [], pList):
      if prop.gnue_module.gnue_name == 'gnue' or prop.gnue_type == 'id':
        if prop.gnue_name in ['createdate', 'createuser', 'modifydate',
                              'modifyuser']:
          if not self.isReference:
            specials.append (Property (self, prop, isSpecial = True))

        continue

      # Do not include filter properties
      filterClass = prop.gnue_class.gnue_filter
      if filterClass:
        if prop.gnue_name == filterClass.gnue_name and \
           prop.gnue_module.gnue_name == filterClass.gnue_module.gnue_name:
          continue

      properties.append (Property (self, prop))

    # Add all calculated fields from the procedure definitions
    cond = ['and', ['eq', ['field', 'gnue_class'], ['const', classId]],
                   ['notnull', ['field', 'gnue_type']],
                   ['like', ['field', 'gnue_name'], ['const', 'get%']]]

    for proc in self.__session.find ('gnue_procedure', cond, [], pList):
      # calculated fields must not have parameters
      if self.__session.find ('gnue_parameter',
          {'gnue_procedure': proc.gnue_id}, [], []):
        continue

      properties.append (Property (self, proc, isCalculated = True))

    return (properties, specials)


  # ---------------------------------------------------------------------------
  # Load labels for the given class and update the property sequence
  # ---------------------------------------------------------------------------

  def __loadLabels (self, classId, properties):
    """
    This function loads all label information for a given class and updates all
    items given by a property sequence. All hidden properties (position = 0)
    are removed from the property sequence.

    @param classId: gnue_id of the class to fetch labels for
    @param properties: unordered sequence of properties to be updated
    """

    pList = ['gnue_label', 'gnue_position', 'gnue_search', 'gnue_info',
             'gnue_page']

    for locale in self.__buildLanguageList (self.__language):
      cond = ['and', ['eq', ['field', 'gnue_language'], ['const', locale]],
                     ['eq', ['field', 'gnue_property.gnue_class'],
                            ['const', classId]]]

      if self.isReference:
        cond.append (['or', ['notnull', ['field', 'gnue_search']],
                            ['notnull', ['field', 'gnue_info']]])

      for label in self.__session.find ('gnue_label', cond, [], pList):

        full = repository.createName (label.gnue_property.module.name,
                                     label.gnue_property.name)
        if full == 'gnue_id' and len (label.label):
          self.classLabel = label.label
          continue

        for item in properties:
          if item.propDef.gnue_id == label.gnue_property.gnue_id:
            item.updateLabel (label)

      # Add labels for calculated properties
      cond = ['and', ['eq', ['field', 'gnue_language'], ['const', locale]],
                     ['eq', ['field', 'gnue_procedure.gnue_class'],
                     ['const', classId]]]

      if self.isReference:
        cond.append (['or', ['notnull', ['field', 'gnue_search']],
                            ['notnull', ['field', 'gnue_info']]])

      for label in self.__session.find ('gnue_label', cond, [], pList):
        for item in properties:
          if item.propDef.gnue_id == label.gnue_procedure.gnue_id:
            item.updateLabel (label)

    if self.isReference:
      for item in properties [:]:
        if item.search is None and item.info is None:
          properties.remove (item)

    # Remove all properties with a explicit position of 0. These are 'hidden'
    # properties.
    for item in properties [:]:
      if item.position == 0:
        properties.remove (item)


  # ---------------------------------------------------------------------------
  # Update all reference properties
  # ---------------------------------------------------------------------------

  def __updateReferences (self):
    """
    This function updates reference information for the class definition. If a
    class definition is a reference, all properties will be sorted according to
    their search- and info-attributes. If the class definition is not a
    reference, this function iterates over all properties, and creates such
    reference class definitions for reference properties.
    """

    if self.isReference:
      order = []

      for item in self.properties:
        order.append ((item.search or item.info, item))

      order.sort ()
      self.properties = [item [1] for item in order]

      # a reference is a 'lookup reference' if the first search field has a
      # search-order other than 0. Otherwise it would be a 'search reference'
      if len (self.properties):
        first = self.properties [0]
        self.isLookup = first.search is not None and first.search != 0
      else:
        self.isLookup = False

      for item in self.properties:
        item.updateStyle ()

    else:
      for item in self.properties:
        if '_' in item.propDef.gnue_type:
          (refMod, refName) = repository.splitName (item.propDef.gnue_type)
          refClass = self.__session.find ('gnue_class',
              {'gnue_name': refName, 'gnue_module.gnue_name': refMod}, [], [])
          cDef = ClassDef (self.__session, refClass [0], self.__language, True)

          if len (cDef.properties):
            item.reference = cDef
            for refItem in cDef.properties:
              refItem.fieldName = "%s%s" % (item.dbField, refItem.fieldName)


  # ---------------------------------------------------------------------------
  # Create a sequence of all virtual pages
  # ---------------------------------------------------------------------------

  def __buildVirtualPages (self):
    """
    This function creates a sequence of all virtual pages. A virtual page is
    tuple with a pagename and a sequence of all properties of the page, already
    ordered according to their position attribute.

    @return: sequence of page-tuples: (name, [property, property, ...])
    """

    # First create a dictionary with all properties per page
    pageDict = {}
    for item in self.properties:
      if not pageDict.has_key (item.page):
        pageDict [item.page] = []
      pageDict [item.page].append (item)

    # Order all properties per page according to position, where all properties
    # with a 'None'-position are at the end, ordered by their label
    for (pageName, items) in pageDict.items ():
      nullOrder = []
      posOrder  = []

      for item in items:
        if item.position is None:
          nullOrder.append ((item.label, item))
        else:
          posOrder.append ((item.position, item.label, item))

      nullOrder.sort ()
      posOrder.sort ()

      pageDict [pageName] = [item [-1] for item in posOrder + nullOrder]

    # now create a sequence of tuples, ordered by the pagename
    pageNames = pageDict.keys ()
    pageNames.sort ()

    return [(p, pageDict [p]) for p in pageNames]



    

# =============================================================================
# This class implements a single property of a class
# =============================================================================

class Property:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, parent, propDef, isCalculated = False, isSpecial = False):
    """
    @param parent: class definition instance which holds this property
    @param propDef: language interface class of the property to encapsulate
    @param isCalculated: set to True if the property is a calculated field
    @param isSpecial: set to True if the property is a special property
    """

    self.parent   = parent
    self.propDef  = propDef
    self.label    = propDef.gnue_name
    self.page     = propDef.module.name.title ()
    self.position = None
    self.search   = None
    self.info     = None
    self.labelPos = None

    self.fullName = repository.createName (propDef.gnue_module.gnue_name,
                                          propDef.gnue_name)
    if isCalculated:
      self.dbField = repository.createName (propDef.gnue_module.gnue_name,
                                           propDef.gnue_name [3:])
    else:
      self.dbField = self.fullName

    self.fieldName    = "fld%s" % self.dbField.title ().replace ('_', '')
    self.typecast     = None
    self.inputmask    = None
    self.displaymask  = None
    self.isCalculated = isCalculated
    self.style        = (isSpecial or isCalculated) and 'label' or None

    masks  = ['%x', '%X', '%x %X']

    if propDef.gnue_type in ['date', 'time', 'datetime']:
      typeId = ['date','time', 'datetime'].index (propDef.gnue_type)

      self.typecast    = 'date'
      self.inputmask   = masks [typeId]
      self.displaymask = masks [typeId]
      self.fieldLength = \
          len ((datetime.datetime.now ()).strftime (self.displaymask))

    elif propDef.gnue_type == 'number':
      self.typecast = 'number'
      self.fieldLength = (propDef.gnue_length or 0) + \
                         (propDef.gnue_scale  or 0) + 1

    elif propDef.gnue_type == 'boolean':
      self.fieldLength = 1
      self.style       = 'checkbox'

    else:
      self.fieldLength = propDef.gnue_length or 32

    self.stretchable = propDef.gnue_type == 'string' and \
                       propDef.gnue_length is None

    if self.stretchable:
      self.minHeight = None
    else:
      self.minHeight = 1
    self.isSpecial   = isSpecial
    self.reference   = None



  # ---------------------------------------------------------------------------
  # Update the label information for the property
  # ---------------------------------------------------------------------------

  def updateLabel (self, labelDef):
    """
    This function updates the label information for the encapsulated property

    @param labelDef: language interface object with the lable information
    """

    self.label      = labelDef.label
    self.page       = labelDef.page and labelDef.page or self.page
    self.position   = labelDef.position
    self.search     = labelDef.search
    self.info       = labelDef.info


  # ---------------------------------------------------------------------------
  # Update the property style depending on reference information
  # ---------------------------------------------------------------------------

  def updateStyle (self):
    """
    This function set's the style for reference properties.
    """

    if self.parent.isReference:
      # if we're part of a lookup use either a dropdown or a label depending
      # wether we're a search- or info-property
      if self.parent.isLookup:
        self.style = self.search and 'dropdown' or 'label'

      # in a search-reference all properties except the first one are labels
      else:
        if self.parent.properties [0] != self:
          self.style = 'label'



  # ---------------------------------------------------------------------------
  # Get the width for all widgets of a property
  # ---------------------------------------------------------------------------

  def widgetWidth (self, maxSpace = None):
    """
    This function calculates the width (in character) for all widgets needed by
    a property. This includes any reference properties if available.

    @param maxSpace: maximum available space for the property
    @return: horizontal space needed for the property
    """

    # make sure we have a minimum length of 7 for dropdown properties
    result = max (self.fieldLength, self.style == 'dropdown' and 7 or None)

    if self.reference is None:
      # stretchable properties are allowed to consume all space
      if self.stretchable:
        result = maxSpace

      else:
        # if we have a space limit given use the minimum value
        if maxSpace is not None:
          result = min (maxSpace, result)

    else:
      # the widgetWidth for lookup properties is the sum of all reference
      # elements plus a space between every reference element
      if self.reference.isLookup:
        result = len (self.reference.properties) - 1

        for item in self.reference.properties:
          result += (item.widgetWidth (maxSpace) or 0)


      # the width for a search property would be the length of the first
      # property and the length of the lookup-button
      else:
        result = self.reference.properties [0].widgetWidth (maxSpace) + 6

    return result

