# GNU Enterprise Application Server - Session Manager Object
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: geasSessionManager.py 7939 2005-09-21 11:17:08Z johannes $

import geasSession
import geasAuthentication
import os.path
import fnmatch
import time

from gnue import paths
from gnue.common.datasources import GConnections
from gnue.common.apps import i18n, GConfig, errors
from gnue.appserver import repository
from gnue.appserver.gcd import readgcd
from gnue.appserver.gld import readgld

# =============================================================================
# Exceptions
# =============================================================================

class SessionNotFoundError (errors.SystemError):
  def __init__ (self, sessionId):
    msg = u_("Cannot find a session with ID '%s'") % sessionId
    errors.SystemError.__init__ (self, msg)

# =============================================================================
# Session Manager class
# =============================================================================

class geasSessionManager:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, modulepath = None, scanModules = False):
    self._connections = connections
    self._sessions    = {}
    self._modulepath  = modulepath or \
                        os.path.join (paths.data, "share", "gnue", "appserver")

    self.repository = repository.Repository (connections)
    self.updateRepository (scanModules, haltOnError = True)

    cfg = gConfigDict (section = 'appserver')
    dbauth = cfg.get ('authentication', 'False')
    if dbauth.lower () in ['true', 'yes', 'y']:
      assert gDebug (1, "Using DB Auth Agent")
      self._authAdapter = geasAuthentication.geasDBAuthAgent (connections,
                                                        gConfig ('connection'))
    else:
      assert gDebug (1, "Using dummy Auth Agent")
      self._authAdapter = geasAuthentication.geasAuthAgent ()

    self._langRuntimes = {}


  # ---------------------------------------------------------------------------
  # Find a session from session ID
  # ---------------------------------------------------------------------------

  def _getSession (self, sess_id):

    if self._sessions.has_key (sess_id):
      return self._sessions [sess_id]
    else:
      raise SessionNotFoundError, (sess_id)


  # ---------------------------------------------------------------------------
  # Update the class repository
  # ---------------------------------------------------------------------------

  def updateRepository (self, scanModules, haltOnError = False):
    """
    This function updates the class repository used by the session manager.
    First the modulepath will be scanned for gcd- and gld-files. All files
    found will be integrated and finally the repository get's updated.
    """

    if not scanModules:
      self._loadRepository ()
      return

    try:
      (gcd, gld) = self._scanModulePath ()

      # Lookup dictionaries for reuse by readgld
      lpModule    = {}
      lpClass     = {}
      lpProperty  = {}
      lpProcedure = {}

      if len (gcd):
        gcdreader = readgcd.gcdReader (self._connections,
                                    gConfig ('connection'), gcd)
        gcdreader.run ()

        lpModule     = gcdreader.lpModule
        lpClass      = gcdreader.lpClass
        lpProperty   = gcdreader.lpProperty
        lpProcedure  = gcdreader.lpProcedure

      if len (gld):
        reader = readgld.gldReader (self._connections, gConfig ('connection'),
            gld, lpModule, lpClass, lpProperty, lpProcedure)
        reader.run ()

    except:
      if haltOnError:
        raise

      msg = u_("Failed reloading repository: %s") % errors.getException () [3]
      assert gDebug (1, msg)
      print o(msg)

    else:
      self._loadRepository ()


  # ---------------------------------------------------------------------------
  # Load the class repository
  # ---------------------------------------------------------------------------

  def _loadRepository (self):
    """
    This function loads the class repository
    """

    print o(u_("Reloading class repository ..."))

    self.repository.load ()

    print o(u_("Class repository loaded"))


  # ---------------------------------------------------------------------------
  # Scan the module path for all GCD- and GLD-files
  # ---------------------------------------------------------------------------

  def _scanModulePath (self):
    """
    This function splits the modulepath by semicolons and scans all parts for
    gcd- and gld-files.

    @return: tuple of sequences (gcd, gld)
    """

    parts = [os.path.normcase (p) for p in self._modulepath.split (';')]
    gcds, glds = [], []

    for root in parts:
      assert gDebug (1, "Scanning path '%s'" % root)

      arg = {'gcd': [], 'gld': []}
      os.path.walk (root, self.__visitPath, arg)
      gcds.extend (arg ['gcd'])
      glds.extend (arg ['gld'])

    return (gcds, glds)


  # ---------------------------------------------------------------------------
  # Search a given path for gcd- and gld-files
  # ---------------------------------------------------------------------------

  def __visitPath (self, arg, dirname, files):
    """
    This function get's called by os.path.walk () and processes all files in a
    given directory.
    """

    for name in files:
      fullname = os.path.normpath (os.path.join (dirname, name))

      # Since os.path.walk () does *not* walk symbolic links we start another
      # run for every linked directory. 
      if os.path.islink (fullname) and os.path.isdir (fullname):
        os.path.walk (fullname, self.__visitPath, arg)
      else:
        if fnmatch.fnmatch (name, '*.gcd'):
          arg ['gcd'].append (fullname)
        elif fnmatch.fnmatch (name, '*.gld'):
          arg ['gld'].append (fullname)


  # ===========================================================================
  # official API functions
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Open the connection
  # ---------------------------------------------------------------------------

  def open (self, authentication):

    assert gEnter (4)

    loginHandler = self._connections._loginHandler
    loginOptions = self._connections._loginOptions
    location     = self._connections._location
    
    conn = GConnections.GConnections(location, loginHandler, loginOptions)
   
    sess = geasSession.geasSession (conn, self._authAdapter, self,
                                    authentication)

    sess.login (authentication.get ('_username'),
                authentication.get ('_password'))

    # We use the memory address of the geasSession object as session id.  This
    # is always unique and even thread safe.
    session_id = id (sess)
    self._sessions [session_id] = sess

    result = session_id
    assert gLeave (4, result)

    return result

  # ---------------------------------------------------------------------------
  # Close the connection
  # ---------------------------------------------------------------------------

  def close (self, session_id, commit):

    assert gEnter (4)

    s = self._getSession (session_id)
    if commit:
      s.commit ()
    s.logout ()
    del self._sessions [session_id]

    assert gLeave (4)

  # ---------------------------------------------------------------------------
  # Commit current transaction
  # ---------------------------------------------------------------------------

  def commit (self, session_id):

    assert gEnter (4)

    s = self._getSession (session_id)
    s.commit ()

    assert gLeave (4)

  # ---------------------------------------------------------------------------
  # Rollback current transaction
  # ---------------------------------------------------------------------------

  def rollback (self, session_id):

    assert gEnter (4)

    s = self._getSession (session_id)
    s.rollback ()

    assert gLeave (4)


  # ---------------------------------------------------------------------------
  # Build list from query
  # ---------------------------------------------------------------------------

  def request (self, session_id, classname, conditions, sortorder,
               propertylist):

    assert gEnter (4)

    s = self._getSession (session_id)
    result = s.request (classname, conditions, sortorder, propertylist)

    assert gLeave (4, result)
    return result


  # ---------------------------------------------------------------------------
  # Count number of objects in list
  # ---------------------------------------------------------------------------

  def count (self, session_id, list_id):

    assert gEnter (4)

    s = self._getSession (session_id)
    result = s.count (list_id)

    assert gLeave (4, result)
    return result

  # ---------------------------------------------------------------------------
  # Fetch objects from list
  # ---------------------------------------------------------------------------

  def fetch (self, session_id, list_id, start, count, close = False):

    assert gEnter (4)
    s = self._getSession (session_id)
    result = s.fetch (list_id, start, count)

    assert gLeave (4, result)
    return result


  # ---------------------------------------------------------------------------
  # Load objects from ids
  # ---------------------------------------------------------------------------

  def load (self, session_id, classname, obj_id_list, propertylist):

    assert gEnter (4)

    s = self._getSession (session_id)
    result = s.load (classname, obj_id_list, propertylist)

    assert gLeave (4, result)
    return result

  # ---------------------------------------------------------------------------
  # Store objects
  # ---------------------------------------------------------------------------

  def store (self, session_id, classname, obj_id_list, propertylist, data):

    assert gEnter (4)

    s = self._getSession (session_id)
    result = s.store (classname, obj_id_list, propertylist, data)

    assert gLeave (4, result)
    return result

  # ---------------------------------------------------------------------------
  # Delete objects
  # ---------------------------------------------------------------------------

  def delete (self, session_id, classname, obj_id_list):

    assert gEnter (4)

    s = self._getSession (session_id)
    s.delete (classname, obj_id_list)

    assert gLeave (4)


  # ---------------------------------------------------------------------------
  # Call procedure
  # ---------------------------------------------------------------------------

  def call (self, session_id, classname, obj_id_list, procedurename,
            parameters):

    assert gEnter (4)

    s = self._getSession (session_id)
    result = s.call (classname, obj_id_list, procedurename, parameters)

    assert gLeave (4, result)
    return result


  # ---------------------------------------------------------------------------
  # Get a structure describing all available filters
  # ---------------------------------------------------------------------------

  def getFilters (self, language):

    assert gEnter (4)

    return self.repository.getFilterDefinition (language)

    assert gLeave (4, result)
    return result
