# GNU Enterprise Application Server - RPC Server Application
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: geasRpcServer.py 7989 2005-09-24 17:28:38Z btami $

import time
import os
import sys
import signal
import thread

import gnue.paths

from gnue.common.apps import GConfig, i18n
from gnue.common.apps.i18n import translate as _        # for epydoc
from gnue.common.apps.GServerApp import GServerApp
from gnue.common.datasources.GLoginHandler import SilentLoginHandler
from gnue.common.rpc import server
from gnue.common.utils.FileUtils import openResource

from gnue.appserver import VERSION
from gnue.appserver import geasSessionManager
from gnue.appserver import geasConfiguration


# =============================================================================
# RPC application class
# =============================================================================

class geasRpcServerApp (GServerApp):

  NAME = "GNUe Application Server"
  VERSION = VERSION
  USAGE = GServerApp.USAGE
  COMMAND = 'gnue-appserver'
  SUMMARY = _(
"""The GNUe Application Server is the core of the n-tier variant of the
GNU Enterprise system. To the front end (be it GNUe Forms, GNUe Reports or
any other tool), it provides user-defineable business objects with arbitary
fields and methods. While transforming access to those fields and methods
into database communication and calling of scripts, it cares about stability,
security, speed, and consistency.""")

  COMMAND_OPTIONS = [

    ['rpctype', None, 'rpc-type', 1, None, 'type', _(
"""Set the GNURPC connection type.  The currently supported values for <type>
are 'xmlrpc', 'xmlrpc.py_xmlrpc', 'xmlrpc.pw_xmlrpc' and 'pyro'.  For more
information on GNURPC connection types have a look at
common/doc/RPC-abstraction.""")],

    ['rpcport', None, 'rpc-port', 1, None, 'port', _(
"""Set the GNURPC port.  For more information on GNURPC have a look at
common/doc/RPC-abstraction.""")],

    ['database', None, 'database', 1, None, 'name', _(
"""Depreciated: Set the Database to use for loading and storing data to <name>.
gnue-appserver will search for it in connections.conf.""")],

    ['connection', 'c', 'connection', 1, None, 'name', _(
"""Set the Database to use for loading and storing data to <name>.
gnue-appserver will search for it in connections.conf.""")],

    ['username', 'u', 'username', 1, None, 'user', _(
"""Set the username for the used database.""")],

    ['password', 'p', 'password', 1, None, 'password', _(
"""Set the password for the used database.""")],

    ['selftest', None, 'selftest', 0, None, None, _(
"""Test appservers connection to the backend database, check correctness of
global settings, etc.""")],

    ['web-frontend', None, 'web-frontend', 0, None, None, _(
"""Enable appservers web frontend. Just works for the rpc-type XMLRPC.  The
webfrontend is at the same port as XMLRPC.""")],

    ['loglevel', None, 'loglevel', 1, None, 'loglevel', _(
"""If set to 1, provides information on data dispatched to the RPC
interface.""")],
    ['modulepath', 'm', 'modulepath', True, None, 'pathlist',
    _("Semicolon-separated list of paths to load modules from")]
    ]

#  USE_DATABASE_OPTIONS = 1 # Conflicts with the existing username and password

  # ---------------------------------------------------------------------------
  # Initialize the object
  # ---------------------------------------------------------------------------

  def __init__ (self, connections=None):

    GServerApp.__init__ (self, connections, 'appserver',
                         geasConfiguration.ConfigOptions)
    self.configurationManager.registerAlias ('gConfig', 'appserver')

    # overwrite configuration settings by command line parameters
    cparser = self.configurationManager._loadedConfigs ['gnue.conf']

    if self.OPTIONS ["rpctype"] is not None:
      cparser.set ('appserver', 'rpctype', self.OPTIONS ["rpctype"])

    if self.OPTIONS ["rpcport"] is not None:
      cparser.set ('appserver', 'rpcport', self.OPTIONS ["rpcport"])

    if self.OPTIONS ["database"] is not None:
      assert gDebug (1, "DEPRECIATION WARNING: use of 'database' option is "
                 "depreciated. Please use 'connection' instead.")
      cparser.set ('appserver', 'connection', self.OPTIONS ["database"])
    else:
      # do we have a 'database' entry in gnue.conf ?
      dbName  = cparser.get ('appserver', 'database')
      coName = cparser.get ('appserver', 'connection')

      # if there is a 'database' entry set and the 'connection' entry is either
      # default or not set (which is the same) we use the 'database' specified
      if dbName and coName == 'gnue':
        cparser.set ('appserver', 'connection', dbName)

    if self.OPTIONS ['connection'] is not None:
      cparser.set ('appserver', 'connection', self.OPTIONS ['connection'])

    if self.OPTIONS ["modulepath"] is not None:
      cparser.set ('appserver', 'modulepath', self.OPTIONS ['modulepath'])


  # ---------------------------------------------------------------------------
  # Set a list of transports
  # ---------------------------------------------------------------------------

  def setTransports (self, transports):
    self._transports = transports

  # ---------------------------------------------------------------------------
  # Initialize the server
  # ---------------------------------------------------------------------------

  def phaseInit (self):
    rpctype = gConfig ("rpctype")

    if rpctype in ('xmlrpc','xmlrpc.pw_xmlrpc','xmlrpc.py_xmlrpc'):
      port = gConfig ("rpcport")
      print u_("Exporting our services via %(rpctype)s (port %(port)s) ...") % \
               {"rpctype": rpctype, "port": port}
      params = {'port': int (port),
                'allowed_hosts': gConfig ('allowed_hosts'),
                'bindto': gConfig ('bindto'),
                'servertype': gConfig ('servertype')}

      if self.OPTIONS ["web-frontend"]:
        httpbind = {'/': gConfig ('httpdir'),
                    '/status': self.htmlStatus}
        params.update ({'httpbind': httpbind})

      if self.OPTIONS.has_key('loglevel'):
        loglevel = self.OPTIONS ["loglevel"]
        params.update ({'loglevel': loglevel})

      self.setTransports({rpctype: params})

    elif rpctype == "pyro":
      port = gConfig ("rpcport")
      print u_("Exporting our services via %(rpctype)s (port %(port)s) ...") % \
               {"rpctype": rpctype, "port": port}
      params = {'port': int (port),
                'bindto': gConfig ('bindto'),
                'allowed_hosts': gConfig ('allowed_hosts')}
      self.setTransports ({'pyro': params})

    elif rpctype == "sockets":
      # Sockets not working yet
      print _("Exporting our services via sockets (EXPERIMENTAL!) ...")
      self.setTransports ({'sockets':{}})

    else:
      # wrong transport protocol set. exiting
      print _("The protocol you've set is currently not supported.")
      sys.exit (-1)

  # ---------------------------------------------------------------------------
  # Show current status via html
  # ---------------------------------------------------------------------------

  def htmlStatus (self):
    out = "<HTML><HEAD></HEAD><BODY>"
    out += u_("Status: %s Sessions opened") % self.sm._sessNo
    out += "</BODY></HTML>"
    return out

  # ---------------------------------------------------------------------------
  # Run the server
  # ---------------------------------------------------------------------------

  def run (self):
    if self.OPTIONS ["selftest"]:
      self.selftest ()
      return

    # Create a new SessionManager instance which will be served by the various
    # transport adapters
    service = self.requestSessionManager ()
    servers = server.bind (self._transports, service)

    # be verbose
    print _("\n... GNUe Application Server up and running ...\n")

    # Daemonize (if appropriate)
    GServerApp.run (self)

    try:
      signal.signal (signal.SIGHUP, self._hangupServer)
      
    except AttributeError:
      pass

    # Start the server for the different protocolls
    for adapter in servers.values ():
      thread.start_new_thread (adapter.serve, ())

    # wait for the servers shut down
    try:
      while 1:
        # sys.maxint = Owerflow error on win32
        time.sleep (1999999)

    except KeyboardInterrupt:
      adapter.shutdown ()
      print _("Appserver is shutting down....ok")


  # ---------------------------------------------------------------------------
  # Request a session manager (called once for every connection)
  # ---------------------------------------------------------------------------

  def requestSessionManager (self):
    if hasattr (self, "sm"):
      return self.sm

    user  = self.OPTIONS ['username'] or 'gnue'
    paswd = self.OPTIONS ['password'] or ''

    loginHandler = SilentLoginHandler (_username = user, _password = paswd)
    self.connections.setLoginHandler (loginHandler)

    self.sm = geasSessionManager.geasSessionManager (self.connections,
                                       gConfig ('modulepath'), True)
    return self.sm


  # ---------------------------------------------------------------------------
  # Handle a 'Hangup' signal
  # ---------------------------------------------------------------------------

  def _hangupServer (self, signal, frame):
    """
    This function handles a SIGHUP signal and reloads the class repository.
    """

    if self.sm:
      self.sm.updateRepository (scanModules = True)

  # ---------------------------------------------------------------------------
  # Self test
  # ---------------------------------------------------------------------------

  def selftest (self):
    ## Starting Appserver selftest
    print _(
"""
GNUe Application Server is running a simple self test
=====================================================

PREREQUISITE: You have to populate the backend db with
'the "address_person" example.
""")

    print _("Step 1: Starting Session Manager ...")
    sm = self.requestSessionManager ()

    print _("Step 2: Opening session (user 'hacker', password 'secret') ...")
    session=sm.open ({'user'     : "hacker",
                      'password' : "secret",
                      'language' : i18n.language})

    print _("Step 3: Creating object list ...")
    list = sm.request (session, "address_person", [], ["address_zip"],
                       ["address_name", "address_street", "address_city"])

    print _("Step 4: Retrieving first instance ...")
    rset = sm.fetch (session,list,0,1)

    if len (rset):
      print o(u_("""
These are the values of the first instance:
  Name  : %(name)s
  Street: %(street)s
  City  : %(city)s
""") % { "name": rset[0][1], "street": rset[0][2], "city": rset[0][3] })


    print o(u_("Step 5: Retrieving defined filters ..."))
    res = sm.getFilters (i18n.language)
    print "Filters:", res

    print _('Selftest passed!')


# =============================================================================
# Main program
# =============================================================================

if __name__ == '__main__':
  app = geasRpcServerApp ()
  app.phaseInit ()
  app.run ()
