# GNU Enterprise Application Server - Class Repository
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: repository.py 7360 2005-04-13 14:00:02Z johannes $

from gnue.common.apps import errors
from gnue.common.utils.CaselessDict import CaselessDict

from gnue.appserver import data
from gnue.appserver.language import Session

# =============================================================================
# Exceptions
# =============================================================================

class ModuleNotFoundError (errors.ApplicationError):
  def __init__ (self, module):
    msg = u_("Module '%s' not found in class repository") % module
    errors.ApplicationError.__init__ (self, msg)

class ClassNotFoundError (errors.ApplicationError):
  def __init__ (self, classname):
    msg = u_("Class '%s' not found in class repository") % classname
    errors.ApplicationError.__init__ (self, msg)

class FilterNotFoundError (errors.ApplicationError):
  def __init__ (self, classname):
    msg = u_("Filter '%s' not found in class repository") % classname
    errors.ApplicationError.__init__ (self, msg)

class PropertyNotFoundError (errors.ApplicationError):
  def __init__ (self, name, classname):
    msg = u_("Class '%(classname)s' has no property '%(property)s'") \
          % {"classname": classname, "property": name}
    errors.ApplicationError.__init__ (self, msg)

class ProcedureNotFoundError (errors.ApplicationError):
  def __init__ (self, name, classname):
    msg = u_("Class '%(classname)s' has no procedure '%(procedure)s'") \
          % {"classname": classname, "procedure": name}
    errors.ApplicationError.__init__ (self, msg)

class ParameterNotFoundError (errors.ApplicationError):
  def __init__ (self, parameter, procedure):
    msg = u_("Procedure '%(procedure)s' has no parameter '%(parameter)s'") \
          % {"procedure": procedure, "parameter": parameter}
    errors.ApplicationError.__init__ (self, msg)

class MissingFilterClassError (errors.ApplicationError):
  def __init__ (self, classname, filterid):
    msg = u_("Filter '%(filter)s' defined in class '%(classname)s' not found "
             "in class repository") \
          % {'classname': classname, 'filter': filterid}
    errors.ApplicationError.__init__ (self, msg)

class ValidationError (errors.ApplicationError):
  pass

class TypeNameError (ValidationError):
  def __init__ (self, typename):
    msg = u_("'%s' is not a valid type") % typename
    ValidationError.__init__ (self, msg)

class TypeFormatError (ValidationError):
  pass

class ParameterValidationError (ValidationError):
  def __init__ (self, procedure, parameter, error):
    msg = u_("Error validating parameter '%(parameter)s' of procedure "
             "'%(procedure)s': %(errors)s") \
          % {'procedure': procedure, 'parameter': parameter, 'error': error}
    ValidationError.__init__ (self, msg)

class InvalidNameError (errors.ApplicationError):
  def __init__ (self, name):
    msg = u_("'%s' is not a valid, fully qualified identifier") % name
    errors.ApplicationError.__init__ (self, name)

# =============================================================================
# This class implements the class repository
# =============================================================================

class Repository:
  """
  This class provides access to the class repository of AppServer. To load the
  repository from the backend database call the method 'load'. After this the
  following instance variables are available:

  @ivar modules: dictionary with all modules available. The dictionary has both,
      the modulenames and their gnue_id's as keys.
  @ivar classes: dictionary with all classes available. The dictionary has
      both, the classnames and their gnue_id's as keys.
  @ivar filters: dictionary with all filter classes available. The dictionary
      uses the 'gnue_id' as keys.
  """

  # ---------------------------------------------------------------------------
  # Create a new repository instance
  # ---------------------------------------------------------------------------

  def __init__ (self, session):
    """
    @param session: geasSession instance to use for data retrieval
    """

    self._session     = session
    self._langSession = Session.InternalSession (session.sm)
    self._connection  = data.connection (session.connections, session.database)

    self.modules = None
    self.classes = None
    self.filters = None


  # ---------------------------------------------------------------------------
  # Load the repository into modules and classes
  # ---------------------------------------------------------------------------

  def load (self):
    """
    This function reloads the whole class repository, which is available
    through the properties 'modules' and 'classes'.
    """

    modules    = CaselessDict (ModuleNotFoundError)
    classes    = CaselessDict (ClassNotFoundError)
    filters    = CaselessDict (FilterNotFoundError)

    properties = {}
    procedures = {}

    try:
      # First load all raw data from the gnue_* tables
      self.__loadRawData ()

      # Wrap all modules
      for (key, value) in self.__rawModules.items ():
        module = Module (self._langSession, value)
        modules [key] = module
        modules [module.fullName] = module

      # Wrap all classes and install them into the apropriate module
      for (key, value) in self.__rawClasses.items ():
        aClass = Class (self._langSession, value, self.__rawModules)
        module = aClass.gnue_module
        modules [module]._installClass (aClass)

        classes [key]             = aClass
        classes [aClass.fullName] = aClass

        # If a class has a filter defined, we add it to the filter dictionary,
        # which will be completed right after loading all classes
        if aClass.gnue_filter is not None:
          if not self.__rawClasses.has_key (aClass.gnue_filter):
            raise MissingFilterClassError, (aClass.fullName, aClass.gnue_filter)

          filters [aClass.gnue_filter] = None

      # After all classes are loaded, complete the filter dictionary
      for key in filters.keys ():
        filters [key] = classes [key]

      # Wrap all properties and install them into the apropriate class
      for (key, value) in self.__rawProperties.items ():
        aProperty = Property (self._langSession, value, self.__rawModules)
        classes [aProperty.gnue_class]._installProperty (aProperty)
        properties [key] = aProperty

      # Wrap all procedures and link them into a lookup dictionary first
      for (key, value) in self.__rawProcedures.items ():
        aProcedure = Procedure (self._langSession, value, self.__rawModules)
        procedures [key] = aProcedure

      # Now create all parameters and install them to their procedures.
      for (key, value) in self.__rawParameters.items ():
        aParameter = Parameter (self._langSession, value)
        procedures [aParameter.gnue_procedure]._installParameter (aParameter)

      # After having all parameters installed to their procedures, we can then
      # run their second level initialization. Here it'll be determined wether
      # a procedure is a calculated field or not. Finally the procedures will
      # be installed to their corresponding class.
      for item in procedures.values ():
        item.secondaryInit (self.__rawModules)
        classes [item.gnue_class]._installProcedure (item)

      # Now finalize and verify all properties and procedures
      for item in properties.values ():
        item.finalize (classes)

      # The verification of procedures implicitly runs a type-check for all
      # calculated properties.
      map (verifyProcedure, procedures.values ())


      # To finish the build process, all references (unicode-strings) are
      # replaced by the corresponding wrapper instance. This way a
      # 'someClass.gnue_module.fullName' will work.
      for item in modules.values ():
        item.updateLinks (modules, classes)

    except:
      raise

    else:
      # We keep referernces to the old dictionaries before replacing them, so
      # we can release them later. This eliminates a time gap between having
      # and old and a new repository available.
      oldmodules = self.modules
      oldclasses = self.classes
      oldfilters = self.filters

      self.modules = modules
      self.classes = classes
      self.filters = filters

      # finally, if we had an old repository, release it
      if oldmodules is not None:
        self.release (oldmodules, oldclasses, oldfilters)


  # ---------------------------------------------------------------------------
  # Release resources of the repository
  # ---------------------------------------------------------------------------

  def release (self, modules = None, classes = None, filters = None):
    """
    Release the current repository dictionaries so garbage collection still
    works fine.

    @param modules: module dictionary to release
    @param classes: class dictionary to release
    @param filters: filter dictionary to release
    """

    if modules is None: modules = self.modules
    if classes is None: classes = self.classes
    if filters is None: filters = self.filters

    for item in modules.values ():
      item.updateLinks (None, None, True)

    filters.clear ()
    classes.clear ()
    modules.clear ()


  # ---------------------------------------------------------------------------
  # Load the raw data from all repository tables
  # ---------------------------------------------------------------------------

  def __loadRawData (self):
    """
    This function reads all repository tables into dictionaries.
    """

    self.__rawModules = self.__loadTable (u'gnue_module', [u'gnue_name'])
    self.__rawClasses = self.__loadTable (u'gnue_class', \
                                [u'gnue_module', u'gnue_name', u'gnue_filter'])
    self.__rawProperties = self.__loadTable (u'gnue_property', \
                  [u'gnue_module', u'gnue_class', u'gnue_name', u'gnue_length',
                   u'gnue_scale', u'gnue_nullable', u'gnue_type'])
    self.__rawProcedures = self.__loadTable (u'gnue_procedure', \
                [u'gnue_module', u'gnue_class', u'gnue_name', u'gnue_language',
                 u'gnue_length', u'gnue_nullable', u'gnue_scale', u'gnue_type',
                 u'gnue_comment'])

    self.__rawParameters = self.__loadTable (u'gnue_parameter', \
                [u'gnue_procedure', u'gnue_name', u'gnue_type', u'gnue_length',
                 u'gnue_scale'])


  # ---------------------------------------------------------------------------
  # Load fields from a given table
  # ---------------------------------------------------------------------------

  def __loadTable (self, table, fields):
    """
    This function reads all rows from a table into a dictionary.

    @param table: name of the table to load data from.
    @param fields: sequence of fields to load per record.

    @return: dictionary with the primary key as key and another dictionary as
        value, using the fieldnames as keys.
    """

    result    = {}
    contents  = {None: (table, None, None, fields)}
    resultSet = self._connection.query (contents, None, [{'name': u'gnue_id'}])

    try:
      rec = resultSet.nextRecord ()
      while rec is not None:
        row = result.setdefault (rec.getField (u'gnue_id'), {})

        for field in fields:
          row [field] = rec.getField (field)

        rec = resultSet.nextRecord ()

    finally:
      resultSet.close ()

    return result


# =============================================================================
# Base class implementing a repository element
# =============================================================================

class BaseItem:

  # ---------------------------------------------------------------------------
  # Create a new instance of a repository item
  # ---------------------------------------------------------------------------

  def __init__ (self, session, classname, predefined = None):
    """
    This function creates a new instance of a repository item. If a predefined
    dictionary is given, all it's elements are copied into the instances'
    __dict__. Access to other attributes will be delegated to the corresponding
    language interface Object.

    @param session: language interface session
    @param classname: name of the class wrapped by this item
    @param predefined: dictionary with predefined values
    """

    self._session  = session
    self.__object  = None
    self.classname = classname

    if predefined is not None:
      self.__dict__.update (predefined)


  # ---------------------------------------------------------------------------
  # Access to a non-prefdefined attribute
  # ---------------------------------------------------------------------------

  def __getattr__ (self, attr):
    """
    This function delegates property access to the bound language interface
    object and caches the result for following access.

    @param attr: name of the attribute to retrieve
    @return: value of the attribute
    """

    if self.__object is None:
      self.__object = self._session._get (self.classname, self.gnue_id)

    # Fetch the result of a given attribute and keep it cached for later reuse
    result = getattr (self.__object, attr)
    self.__dict__ [attr] = result

    return result


  # ---------------------------------------------------------------------------
  # Dictionary emulation for wrapper instances
  # ---------------------------------------------------------------------------

  def __getitem__ (self, attr):
    """
    This function emulates a dictionary access to the wrapper instance.

    @param attr: attribute to return
    @return: value of the attribute 'attr'
    """

    return getattr (self, attr)


# =============================================================================
# This class implements a wrapper for module items of the repository
# =============================================================================

class Module (BaseItem):

  # ---------------------------------------------------------------------------
  # Create a new module item of the repository
  # ---------------------------------------------------------------------------

  def __init__ (self, session, predefined = None):
    """
    @param session: language interface session
    @param predefined: dictionary with predefined ('builtin') values
    """

    BaseItem.__init__ (self, session, u'gnue_module', predefined)

    self.classes  = CaselessDict (ClassNotFoundError)
    self.fullName = self.gnue_name


  # ---------------------------------------------------------------------------
  # Install a class into the module's class list
  # ---------------------------------------------------------------------------

  def _installClass (self, aClass):
    """
    This function adds the given class to the modules' list of classes.
    @param aClass: Class instance to be added
    """

    self.classes [aClass.gnue_id]  = aClass
    self.classes [aClass.fullName] = aClass


  # ---------------------------------------------------------------------------
  # Update links of a module item
  # ---------------------------------------------------------------------------

  def updateLinks (self, modules, classes, unlink = False):
    """
    This class updates links of a module item by updating the links of all it's
    classes.

    @param modules: dictionary with all modules available
    @param classes: dictionary with all classes available
    @param unlink: if set to True, references will be cleared
    """

    for (key, item) in self.classes.items ():
      item.updateLinks (modules, classes, unlink)



# =============================================================================
# This class implements a class item of the repository
# =============================================================================

class Class (BaseItem):

  # ---------------------------------------------------------------------------
  # Create a new class item of the repository
  # ---------------------------------------------------------------------------

  def __init__ (self, session, predefined, moduleLookup):
    """
    @param session: language interface session
    @param predefined: dictionary with predefined ('builtin') values
    @param moduleLookup: (raw) lookup-dictionary with all modules available
    """

    BaseItem.__init__ (self, session, u'gnue_class', predefined)

    moduleName    = moduleLookup [self.gnue_module] ['gnue_name']
    self.fullName = createName (moduleName, self.gnue_name)
    self.table    = self.fullName

    self.properties = CaselessDict (PropertyNotFoundError, self.fullName)
    self.procedures = CaselessDict (ProcedureNotFoundError, self.fullName)
    self.masters    = CaselessDict (ClassNotFoundError)

    self.OnInit     = []
    self.OnChange   = []
    self.OnValidate = []
    self.OnDelete   = []


  # ---------------------------------------------------------------------------
  # Install a new property to the property dictionary of the class
  # ---------------------------------------------------------------------------

  def _installProperty (self, aProperty):
    """
    This function installs a given property to the property dictionary

    @param aProperty: Property instance to be installed
    """

    self.properties [aProperty.gnue_id]  = aProperty
    self.properties [aProperty.fullName] = aProperty


  # ---------------------------------------------------------------------------
  # Install a new procedure to the procedure dictionary of the class
  # ---------------------------------------------------------------------------

  def _installProcedure (self, aProcedure):
    """
    This function adds a procedure to the procedure dictionary. If the
    procedure is a 'calculated property' it also adds it to the property
    dictionary. If a procedure is a trigger it will be added to the apropriate
    trigger sequence.

    @param aProcedure: procedure to be installed
    """

    self.procedures [aProcedure.gnue_id]  = aProcedure
    self.procedures [aProcedure.fullName] = aProcedure

    if aProcedure.isCalculated:
      aProperty = CalculatedProperty (aProcedure)
      self._installProperty (aProperty)

    # If the procedure is an OnInit trigger add it to the OnInit sequence. If 
    # the trigger is defined by the classes' module, it will be the first
    # trigger to fire.
    if aProcedure.gnue_name.upper () == 'ONINIT':
      if aProcedure.gnue_module == self.gnue_module:
        self.OnInit.insert (0, aProcedure)
      else:
        self.OnInit.append (aProcedure)

    # If the procedure is an OnValidate trigger add it to the trigger sequence.
    # The OnValidate trigger defined by the classes' module will be the last
    # one fired.
    if aProcedure.gnue_name.upper () == 'ONVALIDATE':
      if aProcedure.gnue_module == self.gnue_module:
        self.OnValidate.append (aProcedure)
      else:
        self.OnValidate.insert (0, aProcedure)

    # If the procedure is an OnChange trigger add it to the trigger sequence.
    if aProcedure.gnue_name.upper () == 'ONCHANGE':
      self.OnChange.append (aProcedure)

    # If the procedure is an OnDelete trigger add it to the trigger sequence.
    if aProcedure.gnue_name.upper () == 'ONDELETE':
      self.OnDelete.append (aProcedure)


  # ---------------------------------------------------------------------------
  # Add a given class as master of the class
  # ---------------------------------------------------------------------------

  def addMasterClass (self, aProperty, aMaster):
    """
    Add a given class to the dictionary of master-classes.

    @param aProperty: name of the property holding the pointer to master-class
    @param aMaster: class wrapper instance to be added
    """

    self.masters.setdefault (aMaster.fullName, []).append (aProperty)


  # ---------------------------------------------------------------------------
  # Update all links within a class wrapper
  # ---------------------------------------------------------------------------

  def updateLinks (self, modules, classes, unlink = False):
    """
    This class updates links of a class item by updating the links of all it's
    properties and procedures.

    @param modules: dictionary with all modules available
    @param unlink: if set to True, references will be cleared
    """

    if unlink:
      self.gnue_module = None
      self.gnue_filter = None

      self.masters.clear ()

      del self.OnInit [:]
      del self.OnChange [:]
      del self.OnValidate [:]
      del self.OnDelete [:]

    else:
      if not isinstance (self.gnue_module, Module):
        self.gnue_module = modules [self.gnue_module]

      if self.gnue_filter and not isinstance (self.gnue_filter, Class):
        self.gnue_filter = classes [self.gnue_filter]

    for item in self.properties.values () + self.procedures.values ():
      item.updateLinks (self, modules, unlink)



# =============================================================================
# This class wraps a property item of the repository
# =============================================================================

class Property (BaseItem):
  
  # ---------------------------------------------------------------------------
  # Create a new property item
  # ---------------------------------------------------------------------------

  def __init__ (self, session, predefined, moduleLookup):
    """
    @param session: language interface session
    @param predefined: dictionary with predefined ('builtin') values
    @param moduleLookup: (raw) lookup-dictionary with all modules available
    """

    updateTypeInfo (predefined)
    BaseItem.__init__ (self, session, u'gnue_property', predefined)

    moduleName        = moduleLookup [self.gnue_module]['gnue_name']
    self.fullName     = createName (moduleName, self.gnue_name)
    self.column       = self.fullName

    self.isCalculated    = False
    self.isReference     = False
    self.referencedClass = None


  # ---------------------------------------------------------------------------
  # Update the links of a property item
  # ---------------------------------------------------------------------------

  def updateLinks (self, aClass, modules, unlink = False):
    """
    This function updates the links of a property item to it's module and
    class. If unlink is set the references are cleared.

    @param aClass: Class instance the property belongs to
    @param modules: Module dictionary with all modules available
    @param unlink: this boolean Flag determines wether to establish links or to
        break them
    """

    if unlink:
      self.gnue_module = None
      self.gnue_class  = None
      self.referencedClass = None

    else:
      self.gnue_class = aClass
      if not isinstance (self.gnue_module, Module):
        self.gnue_module = modules [self.gnue_module]


  # ---------------------------------------------------------------------------
  # Finalize the definition of a property
  # ---------------------------------------------------------------------------

  def finalize (self, classes):
    """
    This function verifies the type of the property and set's the reference
    flags for rerference properties.

    @param classes: dictionary with all classes available
    """

    r = verifyType (self.gnue_type, self.gnue_length, self.gnue_scale, classes)

    self.isReference     = r is not None
    self.referencedClass = r

    # If the property is a reference to another class, the referenced class
    # is a master of this class
    if r is not None:
      classes [self.gnue_class].addMasterClass (self.fullName, r)



# =============================================================================
# This class wraps a calculated property (= special procedures)
# =============================================================================

class CalculatedProperty (BaseItem):

  # ---------------------------------------------------------------------------
  # Create a new calculated property
  # ---------------------------------------------------------------------------

  def __init__ (self, aProcedure):
    """
    @param aProcedure: procedure to bind as calculated property
    """

    predefined = {
      'gnue_id'      : aProcedure.gnue_id,
      'gnue_name'    : aProcedure.calcName,
      'gnue_type'    : aProcedure.gnue_type,
      'gnue_length'  : aProcedure.gnue_length,
      'gnue_scale'   : aProcedure.gnue_scale,
      'gnue_module'  : aProcedure.gnue_module,
      'gnue_class'   : aProcedure.gnue_class,
      'gnue_nullable': aProcedure.gnue_nullable}
    updateTypeInfo (predefined)

    BaseItem.__init__ (self, aProcedure._session, u'gnue_procedure', predefined)

    self.fullName  = aProcedure.calcFullName
    self.column    = None

    self.isReference     = False
    self.referencedClass = None
    self.isCalculated    = True
    self.procedure       = aProcedure


  # ---------------------------------------------------------------------------
  # Update the links of a calculated property item
  # ---------------------------------------------------------------------------

  def updateLinks (self, aClass, modules, unlink = False):
    """
    If in unlink mode this function clears the pointer to the bound procedure.

    @param aClass: Class instance the property belongs to
    @param modules: Module dictionary with all modules available
    @param unlink: this boolean Flag determines wether to establish links or to
        break them
    """

    if unlink:
      self.gnue_module = None
      self.gnue_class  = None
      self.procedure   = None

    else:
      self.gnue_class = aClass
      if not isinstance (self.gnue_module, Module):
        self.gnue_module = modules [self.gnue_module]




# =============================================================================
# This class wraps a procedure item of the repository
# =============================================================================

class Procedure (BaseItem):

  # ---------------------------------------------------------------------------
  # Create a new procedure wrapper
  # ---------------------------------------------------------------------------

  def __init__ (self, session, predefined, moduleLookup):
    """
    @param session: language interface session
    @param predefined: dictionary with predefined ('builtin') values
    @param moduleLookup: (raw) lookup-dictionary with all modules available
    """

    BaseItem.__init__ (self, session, u'gnue_procedure', predefined)

    moduleName    = moduleLookup [self.gnue_module]['gnue_name']
    self.fullName = createName (moduleName, self.gnue_name)

    self.isCalculated = False
    self.calcFullName = None
    self.calcName     = None

    self.parameters = CaselessDict (ParameterNotFoundError)


  # ---------------------------------------------------------------------------
  # Second level initialization for a procedure
  # ---------------------------------------------------------------------------

  def secondaryInit (self, moduleLookup):
    """
    This function determines wether a procedure can be used as a calculated
    field or not. If so the flag 'isCalcualted' will be set.

    @param moduleLookup: lookup dictionary for modules
    """

    self.isCalculated = self.gnue_type is not None and \
        not len (self.parameters) and self.gnue_name [:3].lower () == 'get'

    if self.isCalculated:
      self.calcName     = self.gnue_name [3:]
      moduleName        = moduleLookup [self.gnue_module]['gnue_name']
      self.calcFullName = createName (moduleName, self.calcName)


  # ---------------------------------------------------------------------------
  # Install a parameter to the procedures parameter dictionary
  # ---------------------------------------------------------------------------

  def _installParameter (self, aParameter):
    """
    This function installs the given parameter to the procedure's parameter
    dictionary.

    @param aParameter: the parameter to be installed.
    """

    self.parameters [aParameter.gnue_id]  = aParameter
    self.parameters [aParameter.fullName] = aParameter


  # ---------------------------------------------------------------------------
  # Update the links of a procedure item
  # ---------------------------------------------------------------------------

  def updateLinks (self, aClass, modules, unlink = False):
    """
    This function updates the links of a procedure item to it's module and
    class. If unlink is set the references are cleared.

    @param aClass: Class instance the procedure belongs to
    @param modules: Module dictionary with all modules available
    @param unlink: this boolean Flag determines wether to establish links or to
        break them
    """

    if unlink:
      self.gnue_module = None
      self.gnue_class  = None

    else:
      self.gnue_class = aClass
      if not isinstance (self.gnue_module, Module):
        self.gnue_module = modules [self.gnue_module]

    for param in self.parameters.values ():
      param.updateLinks (self, unlink)



# =============================================================================
# This class implements a parameter item of the repository
# =============================================================================

class Parameter (BaseItem):

  # ---------------------------------------------------------------------------
  # Create a new parameter wrapper instance
  # ---------------------------------------------------------------------------

  def __init__ (self, session, predefined):
    """
    @param session: language interface session
    @param predefined: dictionary with predefined ('builtin') values
    """

    BaseItem.__init__ (self, session, u'gnue_parameter', predefined)
    self.fullName = self.gnue_name

    if NONREF_TYPES.has_key (self.gnue_type):
      self.dbType = self.gnue_type
    else:
      self.dbType = REF_TYPE


  # ---------------------------------------------------------------------------
  # Update links
  # ---------------------------------------------------------------------------

  def updateLinks (self, aProcedure, unlink = False):
    """
    This functions updates the link to the owning procedure.

    @param aProcedure: the owning procedure
    @param unlink: if set to True the reference will be cleared
    """

    if unlink:
      self.gnue_procedure = None

    else:
      self.gnue_procedure = aProcedure



# =============================================================================
# Type checking support
# =============================================================================

NONREF_TYPES = {'boolean': True, 'date'  : True, 'datetime': True,
                'number' : True, 'string': True, 'time'    : True}
BASE_TYPES   = {'id': True}
BASE_TYPES.update (NONREF_TYPES)

NOLS_TYPES   = {'id': 1, 'date': 1, 'time': 1, 'datetime': 1, 'boolean': 1}

REF_TYPE     = "string"
REF_LENGTH   = 32
REF_SCALE    = 0


# -----------------------------------------------------------------------------
# Check if a combination of typename, length and scale is valid
# -----------------------------------------------------------------------------

def verifyBasetype (typename, length, scale):
  """
  This function verifies a given typename with length and scale. If this
  combination makes no sense a TypeFormatError will be raised. If typename
  is no valid base type a TypeNameError will be raised.

  @param typename: name of the datatype
  @param length: length of the datatype
  @param scale: scale of the datatype
  """

  if not BASE_TYPES.has_key (typename):
    raise TypeNameError, (typename)

  # A string type must not have a scale
  if typename == 'string':
    if scale:
      raise TypeFormatError, u_("string does not support 'scale'")

  # All of the following types must not have length nor scale
  if NOLS_TYPES.has_key (typename):
    if length:
      raise TypeFormatError, u_("%s does not support 'length'") % typename
    if scale:
      raise TypeFormatError, u_("%s does not support 'scale'") % typename

  # A number must have at least a 'length'
  if typename == 'number':
    if not length:
      raise TypeFormatError, u_("number without 'length'")


# -----------------------------------------------------------------------------
# Verify a given type
# -----------------------------------------------------------------------------

def verifyType (typename, length, scale, classes):
  """
  This function verifies a given type, length and scale combination, optionally
  using the given class dictionary for lookups of references.

  @param typename: name of the datatype
  @param length: length of the datatype
  @param scale: scale of the datatype
  @param classes: class dictionary to check for reference types

  @return: If 'typename' is a reference type this function returns the class
      definition of this reference type, otherwise None
  """

  if classes.has_key (typename):
    if length:
      raise TypeFormatError, u_("Reference types must not have a 'length'")

    if scale:
      raise TypeFormatError, u_("Reference types must not have a 'scale'")

    return classes [typename]

  else:
    verifyBasetype (typename, length, scale)
    return None


# -----------------------------------------------------------------------------
# Verify a procedure definition
# -----------------------------------------------------------------------------

def verifyProcedure (aProc):
  """
  This function checks the resulttype of a procedure definition, and all
  parameter types (if available).

  @param aProc: procedure wrapper item to be checked
  """

  # If a result type is specified, check it
  if aProc.gnue_type is not None:
    verifyBasetype (aProc.gnue_type, aProc.gnue_length, aProc.gnue_scale)

  else:
    # otherwise there must not be anything concerning a result type
    if aProc.gnue_length:
      raise TypeFormatError, u_("%s: Procedure has no result, but a 'length' "
                                "is specified.") % aProc.fullName
    if aProc.gnue_scale:
      raise TypeFormatError, u_("%s: Procedure has no result, but a 'scale' "
                                "is specified.") % aProc.fullName

  # verify all given parameter types
  for pa in aProc.parameters.values ():
    try:
      verifyBasetype (pa.gnue_type, pa.gnue_length, pa.gnue_scale)

    except ValidationError, vErr:
      raise ParameterValidationError, \
          (aProc.fullName, pa.fullName, vErr.message)


# -----------------------------------------------------------------------------
# Create a type information dictionary for a given wrapper item
# -----------------------------------------------------------------------------

def updateTypeInfo (item):
  """
  This function updates all type information in a given dictionary. It assumes
  to find the keys 'gnue_type', 'gnue_length' and 'gnue_scale' in the
  dictionary. After updating the dictionary will contain the following
  additional keys: fullType, dbFullType, dbType, dbLength and dbScale

  @param item: dictionary to update type information
  """

  gType  = dbType   = fullType = item ['gnue_type']
  length = dbLength = item ['gnue_length']
  scale  = dbScale  = item ['gnue_scale']

  if gType in ["string", "number"]:
    if gType == "number" and length and scale:
      fullType = "%s(%d,%d)" % (gType, length, scale)

    elif length:
      fullType = "%s(%d)" % (gType, length)

  # build database specific type information
  if not NONREF_TYPES.has_key (gType):
    (dbType, dbLength, dbScale) = (REF_TYPE, REF_LENGTH, REF_SCALE)

  dbFullType = dbType
  if dbType in ["string", "number"]:
    if dbType == "number" and dbLength and dbScale:
      dbFullType = "%s(%d,%d)" % (dbType, dbLength, dbScale)

    elif dbLength:
      dbFullType = "%s(%d)" % (dbType, dbLength)

  item.update ({'fullType'  : fullType,
                'dbFullType': dbFullType,
                'dbType'    : dbType,
                'dbLength'  : dbLength,
                'dbScale'   : dbScale})


# -----------------------------------------------------------------------------
# Create a fully qualified name from namespace and identifier
# -----------------------------------------------------------------------------

def createName (namespace, identifier):
  """
  This function creates a fully qualified name from namespace and identifier.
  If the result is not a valid name, an InvalidNameError will be raised.

  @param namespace: the namespace to use
  @param identifier: the identifier to use

  @return: fully qualified name: 'namespace_idendifier'
  """

  result = "%s_%s" % (namespace, identifier)

  if len (result.split ('_')) != 2:
    raise InvalidNameError, result

  return result


# -----------------------------------------------------------------------------
# Split a fully qualified name into namespace and identifier
# -----------------------------------------------------------------------------

def splitName (name):
  """
  This function splits the given name into namespace and identifier parts. If
  name does not contain a namespace at all, it will be left empty. If the
  resulting tuple has more than two parts an InvalidNameError will be raised.

  @param name: the name to be split into namespace and identifier

  @return: tuple (namespace, identifier) where namespace could be an empty
      string if name had no namespace information at all
  """

  parts = name.split ('_')
  items = len (parts)

  if items == 1:
    result = ('', name)

  elif items == 2:
    result = tuple (parts)

  else:
    raise InvalidNameError, name

  return result
