# GNU Enterprise Application Server - Language interface session
#
# Copyright 2003-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Session.py 7104 2005-03-07 16:49:25Z johannes $

import types
import string

from ObjectList import ObjectList
from Object import Object
from gnue.common.apps import i18n, errors


# ===========================================================================
# Exceptions
# ===========================================================================

class NoContextError (errors.ApplicationError):
  def __init__ (self, name):
    msg = u_("Cannot qualifiy name '%s', no context specified") % name
    errors.ApplicationError.__init__ (self, msg)

class MessageNotFoundError (errors.ApplicationError):
  def __init__ (self, message):
    msg = u_("Message '%s' not found") % message
    errors.ApplicationError.__init__ (self, msg)


# ===========================================================================
# CLASS session: implement a session of the language interface
# ===========================================================================

class Session:

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------

  def __init__ (self, sessionManager, sessionId, params = {}):
    """
    Create an instance of a language interface session

    @param sessionManager: the session manager to be used
    @param sessionId: id of the session encapsulated by the instance
    @param params: dictionary of session parameters
    """

    self.__sm         = sessionManager
    self.__session_id = sessionId
    self.__context    = None
    self.parameters   = params
    self.classdefs    = {}


  # -------------------------------------------------------------------------
  # Get the session's sessionManager
  # -------------------------------------------------------------------------

  def getSessionManager (self):
    """
    This function returns the session manager this session is bound to
    @return: session manager instance
    """

    return self.__sm


  # -------------------------------------------------------------------------
  # Get the session's id
  # -------------------------------------------------------------------------

  def getSessionId (self):
    """
    This function returns the id of this session in the session manager.
    @return: id of the session
    """

    return self.__session_id


  # -------------------------------------------------------------------------
  # Get or created the class definition dictionary for a given class
  # -------------------------------------------------------------------------

  def getClassDefinition (self, classname):
    """
    This function returns the class definition dictionary for a given
    classname. If it does not exist it will be created. Such a dictionary
    contains the datatype per class-element (property/procedure).

    @param classname: name of the class to return a definition dictionary for
    @return: class definition dictionary for the requested class
    """

    return {}

    key = classname.lower ()
    if not self.classdefs.has_key (key):
      self.classdefs [key] = {}

    return self.classdefs [key]


  # -------------------------------------------------------------------------
  # Set the default context
  # -------------------------------------------------------------------------

  def setcontext (self, context):
    """
    This function specifies the current context of the session. All subsequent
    calls to qualify using an unqualified name will be extended with this
    context.

    @param context: context to set
    """

    self.__context = context


  # -------------------------------------------------------------------------
  # Ensure the given name is fully qualified using the current context
  # -------------------------------------------------------------------------

  def qualify (self, name):
    """
    This function returns a fully qualified named. If the given name isn't
    already qualified the current context will be added. If no such context is
    available a NoContextError exception will be raised.

    @param name: name to be returned as fully qulified name
    @return: fully qualified name
    """

    # Although the 'in'-operation is an expensive operation, it's still faster
    # than using string.find () here
    if '_' in name:
      return name

    else:
      if self.__context is None:
        raise NoContextError (name)
      else:
        # This looks a bit ugly but it's faster than using "%s_%s" % (...)
        return self.__context + "_" + name


  # ---------------------------------------------------------------------------
  # Qualify an identifier consisting of multiple parts
  # ---------------------------------------------------------------------------

  def qualifyMultiple (self, name):
    """
    This function returns a fully qualified identifier for a name with multiple
    parts, i.e. 'foo.bar.baz' will be qualified in a context of 'ctx' as
    'ctx_foo.ctx_bar.ctx_baz'.

    @param name: name to be qualified
    @return: fully qualified version of name
    """

    return string.join ([self.qualify (p) for p in name.split ('.')], ".")


  # -------------------------------------------------------------------------
  # Close the session 
  # -------------------------------------------------------------------------

  def close (self):
    """
    This function closes the encapsulated session on the session manager.
    """

    gEnter (6)

    if self.__sm is not None:
      self.__sm.close (self.__session_id, None)
      self.__sm = self.__session_id = None

    gLeave (6)


  # -------------------------------------------------------------------------
  # Commit the current transaction
  # -------------------------------------------------------------------------

  def commit (self):
    """
    This function commits all pending changes of the session.
    """

    gEnter (6)

    if self.__session_id is not None:
      self.__sm.commit (self.__session_id)

    gLeave (6)


  # -------------------------------------------------------------------------
  # Revoke the current transaction
  # -------------------------------------------------------------------------

  def rollback (self):
    """
    This function revokes all pending changes of the current session.
    """

    gEnter (6)

    if self.__session_id is not None:
      self.__sm.rollback (self.__session_id)

    gLeave (6)


  # -------------------------------------------------------------------------
  # Return a collection of 'classname' matching the given arguments
  # -------------------------------------------------------------------------

  def find (self, classname, conditions = None,
            sortorder = [('gnue_id', False)], properties = []):
    """
    This function returns a new collection of classes matching the given
    conditions, having a given order and using a given set of 'preloaded'
    properties.

    @param classname: name of the class to be fetched
    @param conditions: condition which must be matched. This could be a
        dictionary, a sequence in prefix notation or a GCondition tree
    @param sortorder: order to sort the instances. This is a sequence of
        sort-elements where such an element could be a name, a tuple/triple or a
        dictionary with the keys 'name, descending, ignorecase'
    @param properties: sequence of propertynames to be loaded for each instance
        in the collection

    @return: ObjectList instance with all matching instances of the given class
    """

    gEnter (6)

    name = self.qualify (classname)
    cdef = self.getClassDefinition (name)

    result = ObjectList (self, name, conditions, sortorder, properties, cdef)

    return gLeave (6, result)


  # -------------------------------------------------------------------------
  # Create a new instance of classname
  # -------------------------------------------------------------------------

  def new (self, classname):
    """
    This function creates a new instance of a given class.
    @param classname: name of the class to create an instance of

    @return: Object instance of the requested class
    """

    gEnter (6)

    result = self._get (classname)

    return gLeave (6, result)


  # ---------------------------------------------------------------------------
  # Get an instance of a class with a given object Id
  # ---------------------------------------------------------------------------

  def _get (self, classname, objectId = None):
    """
    This function returns an instance of a given class having the specified
    objectId. If the objectId is None, a new one will be generated.
    If the objectId does not exist nothing happens.

    @param classname: name of the class to create an instance for
    @param objectId: gnue_id of the instance or None to create a new one

    @return: Object instance of the requested class
    """

    gEnter (6)

    name   = self.qualify (classname)
    cDef   = self.getClassDefinition (name)
    result = Object (self, name, objectId, cDef)

    return gLeave (6, result)


  # -------------------------------------------------------------------------
  # Get an instance of a class having a given objectId
  # -------------------------------------------------------------------------

  def get (self, classname, objectId = None):
    """
    This function returns an instance of a given class having the specified
    objectId. If the objectId is None, a new one will be generated.
    If the objectId does not exist in the backend, an InstanceNotFoundError
    will be raised.

    @param classname: name of the class to create an instance for
    @param objectId: gnue_id of the instance or None to create a new one

    @return: Object instance of the requested class
    """

    gEnter (6)

    name = self.qualify (classname)
    self.__sm.load (self.__session_id, name, [objectId], [u'gnue_id'])

    cDef   = self.getClassDefinition (name)
    result = Object (self, name, objectId, cDef)

    return gLeave (6, result)


  # -------------------------------------------------------------------------
  # Get a message from the message catalogue
  # -------------------------------------------------------------------------

  def message (self, messageName, *args):
    """
    This function returns a message from the message catalogue using the given
    arguments. If the session has a '_language' parameter this language(s) are
    treated with higher priority.

    @param messageName: name of the message to be returned
    @param args: variable arguments applied to the messaged

    @return: the message as unicode string
    """

    gEnter (6)

    languages = ['C']
    current = self.parameters.get ('_language', 'C')

    if '_' in current:
      add = current.split ('_') [0]
      if not add in languages:
        languages.insert (0, add)

    if not current in languages:
      languages.insert (0, current)

    (module, message) = self.qualify (messageName).lower ().split ('_')

    for lang in languages:
      cond = ['and', ['eq', ['lower', ['field', 'gnue_module.gnue_name']],
                            ['const', module]],
                     ['eq', ['lower', ['field', 'gnue_name']],
                            ['const', message]],
                     ['eq', ['field', 'gnue_language'], ['const', lang]]]

      result = self.find ('gnue_message', cond, [], ['gnue_text'])
      if len (result):
        text = result [0].gnue_text

        if len (args):
          if isinstance (args [0], types.DictType):
            text = text % args [0]
          else:
            text = text % args

        return gLeave (6, text)

    raise MessageNotFoundError, self.qualify (messageName)



# =============================================================================
# Login-Session creates a new Session-Id using 'user' and 'password'
# =============================================================================
class LoginSession (Session):

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------
  def __init__ (self, sessionManager, user, password, params = {}):
    Session.__init__ (self, sessionManager,
                  sessionManager.open ({'user': user, 'password': password}),
                  params)


# =============================================================================
# InternalSession uses 0 as Session-Id
# =============================================================================
class InternalSession(Session):

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------
  def __init__ (self, sessionManager, params = {}):
    Session.__init__ (self, sessionManager, 0, params)

