# GNU Enterprise Appserver - GNUe Language Definitions - XML-Parser
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GLParser.py 6938 2005-02-02 14:48:02Z johannes $

from gnue.common.definitions import GParser, GObjects, GRootObj
from gnue.common.definitions.GParserHelpers import GContent
from gnue.common.formatting import GTypecast
from gnue.appserver.classrep import helpers, Namespace

xmlElements   = None

# =============================================================================
# Exceptions
# =============================================================================

class Error (gException):
  pass

class NoModuleError (Error):
  def __init__ (self, classname):
    msg = u_("Class '%s' has no module") % classname
    Error.__init__ (self, msg)

class ModuleMismatchError (Error):
  def __init__ (self, classname, module):
    msg = u_("The class '%(class)s' mismatches the given module attribute "
             "'%(module)s'") % {'class': classname, 'module': module}
    Error.__init__ (self, msg)

class RefStyleError (Error):
  def __init__ (self, itemname):
    msg = u_("Element '%s' can either have a search- or info-attribute.") \
          % itemname
    Error.__init__ (self, msg)


# =============================================================================
# load an XML object tree from a given stream and return it's root object
# =============================================================================

def loadFile (stream, initialize = True):
  return GParser.loadXMLObject (stream, xmlSchemaHandler, 'GLModule',
      'module', initialize)


# =============================================================================
# Build a dictionary tree with all available XML elements
# =============================================================================

def getXMLelements ():
  global xmlElements

  if xmlElements is None:
    xmlElements = {
      'module': {
        'BaseClass' : GLModule,
        'Required'  : True,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'language': {
            'Required': True,
            'Typecast': GTypecast.text}
        },
        'ParentTags': (None)
      },

      'class': {
        'BaseClass': GLClass,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'page': {
            'Typecast': GTypecast.text},
          'label': {
            'Typecast': GTypecast.text},
        },
        'ParentTags': ('module',)
      },

      'property': {
        'BaseClass': GLProperty,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'pos': {
            'Default' : 0,
            'Typecast': GTypecast.integer},
          'label': {
            'Typecast': GTypecast.text},
          'page': {
            'Typecast': GTypecast.text},
          'info': {
            'Typecast': GTypecast.integer},
          'search': {
            'Typecast': GTypecast.integer}
        },
        'MixedContent'  : True,
        'KeepWhitespace': True,
        'ParentTags'    : ('class',)
      },

      'procedure': {
        'BaseClass': GLProcedure,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
          'pos': {
            'Default' : 0,
            'Typecast': GTypecast.integer},
          'label': {
            'Typecast': GTypecast.text},
          'page': {
            'Typecast': GTypecast.text},
          'info': {
            'Typecast': GTypecast.integer},
          'search': {
            'Typecast': GTypecast.integer}
        },
        'MixedContent'  : True,
        'KeepWhitespace': True,
        'ParentTags'    : ('class',)
      },

      'message': {
        'BaseClass': GLMessage,
        'Attributes': {
          'name': {
            'Required': True,
            'Unique'  : True,
            'Typecast': GTypecast.name},
        },
        'MixedContent'  : True,
        'KeepWhitespace': True,
        'ParentTags'    : ('module',)
      },
    }

  return GParser.buildImportableTags ('module', xmlElements)


# =============================================================================
# Class called by the XML parser to process the XML file
# =============================================================================

class xmlSchemaHandler (GParser.xmlHandler):
  def __init__ (self):
    GParser.xmlHandler.__init__ (self)
    self.xmlElements = getXMLelements ()


# =============================================================================
# Base class for all GNUe Layout Definition classes
# =============================================================================

class GLObject (GObjects.GObj):
  pass


# =============================================================================
# Base class for all objects with a type-attribute
# =============================================================================

class GLClassElement (GLObject):
  def __init__ (self, parent = None, objType = None):
    GLObject.__init__ (self, parent, type = objType)

    self.module   = None
    self.fullName = None

    self._inits.extend ([None, self._validate])


  # ---------------------------------------------------------------------------
  # If an instance has a type attribute, verify it
  # ---------------------------------------------------------------------------

  def _validate (self):
    self.module   = self.findParentOfType ('GLModule').name

    nameParts = self.name.split ('_')
    if len (nameParts) > 1:
      (self.module, self.name) = nameParts [:2]

    self.fullName = Namespace.createName (self.module, self.name)

    if not hasattr (self, 'page'):
      self.page = self._parent.page

    if not hasattr (self, 'search'): self.search = None
    if not hasattr (self, 'info'):   self.info   = None

    if hasattr (self, 'pos'): self.position = self.pos

    if self.search is not None and self.info is not None:
      raise RefStyleError, (self.fullName)

    if self.search is not None:
      self.info = None

    if self.info is not None:
      self.search = None



# =============================================================================
# The module object, root object of a GLD tree
# =============================================================================

class GLModule (GRootObj.GRootObj, GLObject):
  def __init__ (self, parent = None):
    GRootObj.GRootObj.__init__ (self, 'module', getXMLelements, self.__module__)
    GLObject.__init__ (self, parent, type = 'GLModule')


# =============================================================================
# The class object
# =============================================================================

class GLClass (GLObject):
  def __init__ (self, parent):
    GLObject.__init__ (self, parent, type = 'GLClass')
    self.fullName = None
    self.page     = None
    self._inits.extend ([None, self._validate, self._complete])

  # ---------------------------------------------------------------------------
  # Validate a class definition
  # ---------------------------------------------------------------------------

  def _validate (self):
    if not isinstance (self._parent, GLModule):
      raise NoModuleError, self.name

    nameParts = self.name.split ('_')
    if len (nameParts) > 1:
      if hasattr (self, 'module') and self.module != nameParts [0]:
        raise ModuleMismatchError, (self.name, self.module)

      (self.module, self.name) = nameParts [:2]

    if not hasattr (self, 'module'):
      self.module = self._parent.name

    self.fullName = Namespace.createName (self.module, self.name)


  # ---------------------------------------------------------------------------
  # Complete a class definition
  # ---------------------------------------------------------------------------

  def _complete (self):
    """
    """

    if hasattr (self, 'label') and self.label is not None:
      add = GLProperty (self)

      add.module   = 'gnue'
      add.name     = 'id'
      add.fullName = 'gnue_id'
      add.label    = self.label
      add.page     = self.page
      add.search   = None
      add.info     = None
      add.pos      = None


# =============================================================================
# The property object
# =============================================================================

class GLProperty (GLClassElement):
  def __init__ (self, parent):
    GLClassElement.__init__ (self, parent, objType = 'GLProperty')

# =============================================================================
# The procedure object
# =============================================================================

class GLProcedure (GLClassElement):
  def __init__ (self, parent):
    GLClassElement.__init__ (self, parent, objType = 'GLProcedure')

# =============================================================================
# The message object
# =============================================================================

class GLMessage (GLObject):
  def __init__ (self, parent):
    GLObject.__init__ (self, parent, type = 'GLMessage')
    self.fullName = None
    self._inits.extend ([None, self._validate])


  # ---------------------------------------------------------------------------
  # Validate a message
  # ---------------------------------------------------------------------------

  def _validate (self):
    parent        = self.findParentOfType ('GLModule')
    self.module   = parent.name
    self.language = parent.language
    self.fullName = Namespace.createName (self.module, self.name)
