# GNU Enterprise Application Server - Simple Front End
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: frontend.py 6851 2005-01-03 20:59:28Z jcater $

import os, getpass
from gnue.common.apps import GClientApp
from gnue.common.datasources import GLoginHandler
import geasSessionManager

# =============================================================================
# Helper functions
# =============================================================================

# -----------------------------------------------------------------------------
# Input a string, giving a prompt and a default value
# -----------------------------------------------------------------------------

def ask (prompt, default):
  result = raw_input ("? " + prompt + " [" + default + "]: ")
  if result == "":
    result = default
  return result

# -----------------------------------------------------------------------------
# Input an integer, giving a prompt and a default value
# -----------------------------------------------------------------------------

def askint (prompt, default):
  try:
    return int (ask (prompt, default))
  except ValueError:
    print "! Please enter a numeric value"

# =============================================================================
# Frontend application
# =============================================================================

class application (GClientApp.GClientApp):

  def __init__ (self):
    GClientApp.GClientApp.__init__ (self)

    # this should (IMHO) happen in geasSessionManager -- reinhard
    self.connections.setLoginHandler (geasLoginHandler ())

    # create session manager object representing the server
    self.server = geasSessionManager.geasSessionManager (self.connections)

    # the current session_id
    self.session = None

    # the current list_id
    self.list = None

    # the parameters that were used to build the list
    self.classname = None
    self.propertylist = None

  # ---------------------------------------------------------------------------
  # Main menu
  # ---------------------------------------------------------------------------

  def menu (self):
    print
    print "* The current list is of class %s and contains %d objects" \
          % (self.classname, self.server.count (self.session, self.list))
    print
    print "  1 - Display objects"
    print "  2 - Display more data"
    print "  3 - Modify objects"
    print "  4 - Add object"
    print "  5 - Delete objects"
    print "  8 - Request another list"
    print "  9 - Exit"
    print

    while 1:
      choice = askint ("Your choice", "1")

      if choice == 1:
        return self.display
      elif choice == 2:
        return self.more
      elif choice == 3:
        return self.modify
      elif choice == 4:
        return self.add
      elif choice == 5:
        return self.delete
      elif choice == 8:
        return self.request
      elif choice == 9:
        return None

  # ---------------------------------------------------------------------------
  # Display objects
  # ---------------------------------------------------------------------------

  def display (self):
    start = askint ("Start with object number", "0")

    # get maximum possible object count from given start position
    if start < 0:                       # from end of list
      maxcount = -start
    else:
      maxcount = self.server.count (self.session, self.list) - start

    count = askint ("Number of objects to display", "%d" % maxcount)

    # now, fetch the data from the server
    rset = self.server.fetch (self.session, self.list, start, count)

    for index in range (0, count):
      object = rset [index]
      # "object" now holds a list of strings, where item number 0 is always
      # the gnue_id, and items 1-n are the values of the properties.
      print "* Object number %d with id %s:" % (start + index, object [0])
      for i in range (0, len (self.propertylist)):
        print "*   " + self.propertylist [i] + ":", object [i + 1]

  # ---------------------------------------------------------------------------
  # Display more data
  # ---------------------------------------------------------------------------

  def more (self):
    # select the properties to display
    propertylist = []
    while 1:
      property = raw_input ("? Property to read: ")
      if property:
        break
      print "! Please enter at least one property"

    while property:
      propertylist.append (property)
      property = raw_input ("? Property to read (empty to finish): ")

    # select objects to display
    start = askint ("Start with object number", "0")

    # get maximum possible object count from given start position
    if start < 0:                       # from end of list
      maxcount = -start
    else:
      maxcount = self.server.count (self.session, self.list) - start

    count = askint ("Number of objects to display", "%d" % maxcount)

    # first, get all object_ids
    rset = self.server.fetch (self.session, self.list, start, count)
    object_ids = []
    for index in range (0, count):
      object = rset [index]
      object_ids.append (object [0])

    # now, we have all object_ids and we can use "load" to get the data
    rset = self.server.load (self.session, self.classname, object_ids,
                             propertylist)
    print rset
    for index in range (0, count):
      object = rset [index]
      print "* Object number %d with id %s:" % (start + index,
                                                object_ids [index])
      for i in range (0, len (propertylist)):
        print "*   " + propertylist [i] + ":", object [i]

  # ---------------------------------------------------------------------------
  # Modify objects
  # ---------------------------------------------------------------------------

  def modify (self):
    # select the properties to modify
    propertylist = []
    while 1:
      property = raw_input ("? Property to modify: ")
      if property:
        break
      print "! Please enter at least one property"

    while property:
      propertylist.append (property)
      property = raw_input ("? Property to modify (empty to finish): ")

    # select objects to modify
    start = askint ("Start with object number", "0")

    # get maximum possible object count from given start position
    if start < 0:                       # from end of list
      maxcount = -start
    else:
      maxcount = self.server.count (self.session, self.list) - start

    count = askint ("Number of objects to modify", "%d" % maxcount)

    # First, get all object_ids
    rset = self.server.fetch (self.session, self.list, start, count)
    object_ids = []
    for index in range (0, count):
      object = rset [index]
      object_ids.append (object [0])

    # Now, we have all object_ids and we can use "load" to get the data.
    # This is not necessary for the subsequent call of "store", it's just
    # convenient for the user.
    rset = self.server.load (self.session, self.classname, object_ids,
                             propertylist)
    data = []
    for index in range (0, count):
      object = rset [index]
      print "* Object number %d with id %s:" % (start + index,
                                                object_ids [index])
      row = []
      for i in range (0, len (propertylist)):
        row.append (ask (propertylist [i], object [i]))
      data.append (row)

    # Now, we have the new data in our 2-dimensional array and we can call
    # the "store" function.
    self.server.store (self.session, self.classname, object_ids, propertylist,
                       data)
    self.server.commit (self.session)

  # ---------------------------------------------------------------------------
  # Add new object
  # ---------------------------------------------------------------------------

  def add (self):
    # select the properties to add
    propertylist = []
    while 1:
      property = raw_input ("? Property to add: ")
      if property:
        break
      print "! Please enter at least one property"

    while property:
      propertylist.append (property)
      property = raw_input ("? Property to add (empty to finish): ")

    row = []
    for property in propertylist:
      row.append (ask (property, ""))

    # Now, we have the new data in "row" and we can call the "store" function.
    self.server.store (self.session, self.classname, [None], propertylist,
                       [row])
    self.server.commit (self.session)

  # ---------------------------------------------------------------------------
  # Delete objects
  # ---------------------------------------------------------------------------

  def delete (self):
    # select objects to modify
    start = askint ("Start with object number", "0")

    # get maximum possible object count from given start position
    if start < 0:                       # from end of list
      maxcount = -start
    else:
      maxcount = self.server.count (self.session, self.list) - start

    count = askint ("Number of objects to delete", "%d" % maxcount)

    # First, get all object_ids
    rset = self.server.fetch (self.session, self.list, start, count)
    object_ids = []
    for index in range (0, count):
      object = rset [index]
      object_ids.append (object [0])

    # Now, delete the objects
    self.server.delete (self.session, self.classname, object_ids)
    self.server.commit (self.session)

  # ---------------------------------------------------------------------------
  # Request a list of objects
  # ---------------------------------------------------------------------------

  def request (self):
    # select the class to operate upon
    if self.classname:
      self.classname = ask ("Classname", self.classname)
    else:
      while 1:
        self.classname = raw_input ("? Classname: ")
        if self.classname:
          break
        print "! Please enter a classname"

    # select the properties to fetch
    self.propertylist = []
    while 1:
      property = raw_input ("? Property to read: ")
      if property:
        break
      print "! Please enter at least one property"

    while property:
      self.propertylist.append (property)
      property = raw_input ("? Property to read (empty to finish): ")

    # TODO: define filter

    # select the properties to sort by
    sortlist = []
    sort = ask ("Sort key", self.propertylist [0])
    while sort:
      sortlist.append (sort)
      sort = raw_input ("? Further sort key (empty to finish): ")

    # request the list
    self.list = self.server.request (self.session, self.classname, [],
                                     sortlist, self.propertylist)

  # ---------------------------------------------------------------------------
  # Main loop
  # ---------------------------------------------------------------------------

  def run (self):
    # open a session
    # user = ask ("Username", getpass.getuser ())
    # password = getpass.getpass ("? Password: ")
    user = ask ("Username", "hacker")
    password = ask ("Password", "secret")
    self.session = self.server.open ({"user": user, "password": password})

    action = self.request

    while action:
      action ()
      action = self.menu ()

    self.server.close (self.session, None)

    print "Thank you for playing!"

# =============================================================================
# Login Handler
# =============================================================================

class geasLoginHandler (GLoginHandler.LoginHandler):
  def getLogin (self, *arguments):
    # log in as the user "gnue"
    return {"_username": "gnue", "_password": None}

# =============================================================================
# Let it rock
# =============================================================================

if __name__ == "__main__":
  app = application ()
  app.run ()
