# GNU Enterprise Application Server - Per-Session Cache
#
# Copyright 2004-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: data.py 7092 2005-03-04 23:04:27Z reinhard $

from types import *

import string
import copy

from gnue.common.apps import errors
from gnue.common.datasources import GDataSource, GConditions, GConnections
from gnue.common.utils.uuid import UUID

class StateChangeError (errors.SystemError):
  def __init__ (self, table, row):
    msg = u_("Changing state from 'commitable' to 'initialized' not allowed "
             "in table '%(table)s' row %(row)s") \
          % {'table': table, 'row': row}
    errors.SystemError.__init__ (self, msg)

class InvalidCacheError (errors.SystemError):
  def __init__ (self, table, row, state):
    msg = u_("Row '%(row)s' of table '%(table)s' has an invalid state "
             "'%(state)s'") \
          % {'table': table, 'row': row, 'state': state}
    errors.SystemError.__init__ (self, msg)

class OrderBySequenceError (errors.ApplicationError):
  pass

# =============================================================================
# Cache class
# =============================================================================

class _cache:
  """
  This class is acutally not more than a 3-dimensional array with the
  dimensions called "table", "row", and "field". Any combination of these can
  either have a string value, a value of None, or not be available.

  For any data item, the cache remembers the current value as well as the
  original value.

  This class doesn't do database access. It gets the values to store via the
  "write" method.

  This class is only used internally.
  """

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self):
    self.__old   = {}                     # Original data
    self.__new   = {}                     # Changed (dirty) data
    self.__state = {}                     # State of the data

  # ---------------------------------------------------------------------------
  # Store data in the cache
  # ---------------------------------------------------------------------------

  def write (self, table, row, field, value, dirty):
    """
    Write data to the cache. If "dirty" is false (0), the cache takes the given
    value as the original value for the field. If "dirty" is true (1), the
    value is taken as the modified value for the field, and the original value
    is remembered (if it was set before).

    It is possible to set a dirty value without having set an original value
    before.
    """
    checktype (table, UnicodeType)
    checktype (row, UnicodeType)
    checktype (field, UnicodeType)

    if dirty:
      tables = self.__new
    else:
      tables = self.__old

    if not tables.has_key (table):
      tables [table] = {}
    rows = tables [table]

    if not rows.has_key (row):
      rows [row] = {}
    fields = rows [row]

    fields [field] = value

    # Update state information of the row
    self.touch (table, row, dirty)



  # ---------------------------------------------------------------------------
  # Return whether a certain value is stored in the clean/dirty cache
  # ---------------------------------------------------------------------------

  def __has (self, table, row, field, dirty):

    if dirty:
      tables = self.__new
    else:
      tables = self.__old

    # We concatenate all has_key () checks so it can return as early as
    # possible. The function is about 3 times faster than using normal if's
    return tables.has_key (table) and tables [table].has_key (row) and \
        tables [table][row].has_key (field)


  # ---------------------------------------------------------------------------
  # Return whether a certain value is stored in the cache or not
  # ---------------------------------------------------------------------------

  def has (self, table, row, field, dirty = None):
    """
    Return true (1) if the given item is stored in the cache (either in a clean
    or in a dirty version). Return false (0) if it isn't.
    """
    checktype (table, UnicodeType)
    checktype (row, UnicodeType)
    checktype (field, UnicodeType)

    if dirty is None:
      # We could use self.__has (), here but avoiding an additional function
      # call here is a big deal regarding performance, as this function is
      # called about 100000 times for a (quite small) classrepository
      return (self.__old.has_key (table) and self.__old [table].has_key (row) \
              and self.__old [table][row].has_key (field)) or \
             (self.__new.has_key (table) and self.__new [table].has_key (row) \
              and self.__new [table][row].has_key (field))
    else:
      if dirty:
        tables = self.__new
      else:
        tables = self.__old
      return tables.has_key (table) and tables [table].has_key (row) and \
        tables [table][row].has_key (field)


  # ---------------------------------------------------------------------------
  # Read data from the cache
  # ---------------------------------------------------------------------------

  def read (self, table, row, field, dirty = True):
    """
    Read data from the cache. Depending on the dirty-flag set this function
    returns the current version, no matter if it's dirty or not. If the
    dirty-flag is False only the field's old value will be returned.

    If the given item isn't available, an exception is raised.
    """
    checktype (table, UnicodeType)
    checktype (row, UnicodeType)
    checktype (field, UnicodeType)

    if dirty and self.__has (table, row, field, 1):
      tables = self.__new               # Data is in dirty cache
    else:
      tables = self.__old               # Data isn't in dirty cache, so search
                                        # in clean cache
    rows = tables [table]
    fields = rows [row]
    return fields [field]


  # ---------------------------------------------------------------------------
  # Get the status of a record
  # ---------------------------------------------------------------------------

  def status (self, table, row):
    """
    Returns the status of the given row. Returns one of the following results:

    'initializing': newly created record with initialization not yet finished

    'initialized': newly created and initialized records with no modifications

    'inserted': newly created record with modifications

    'changed': existing record with modifications

    'deleted': deleted record

    For this function to work, an original value for the 'gnue_id' field must
    be available for any record except for newly created ones, and setting
    'gnue_id' to None means deleting the record.
    """
    checktype (table, UnicodeType)
    checktype (row, UnicodeType)

    if not self.__has (table, row, u'gnue_id', 0):
      return ''                         # row is not in cache at all

    old_id = self.__old [table] [row] [u'gnue_id']

    if self.__has (table, row, u'gnue_id', 1):
      new_id = self.__new [table] [row] [u'gnue_id']
    else:
      new_id = old_id

    if old_id is None:
      if new_id is None:
        return ''                       # row was inserted and deleted
      else:
        rowState = self.__getState (table, row)
        if rowState == 'commitable':
          return 'inserted'
        else:
          return rowState
    else:
      if new_id is None:
        return 'deleted'
      else:
        if self.__new.has_key (table):
          rows = self.__new [table]
          if rows.has_key (row):
            return 'changed'
        return ''                       # row has no dirty fields


  # ---------------------------------------------------------------------------
  # List all tables with dirty records
  # ---------------------------------------------------------------------------

  def dirtyTables (self):
    """
    Returns a dictionary of tables with dirty data (inserted, changed or
    deleted rows), where the key is the table name and the value is a
    dictionary of all dirty rows in the table, where the key is the row id and
    the value is a dictionary of all dirty fields in that row, where the key is
    the field name and the value is the current value of the field. Got it?
    """

    return self.__new

  # ---------------------------------------------------------------------------
  # Clear the whole cache
  # ---------------------------------------------------------------------------

  def clear (self, oldOnly = False):
    """
    Forget all data in the cache, original values as well as dirty values.
    """

    if oldOnly:
      # on a commit we need to remove all 'commited' stuff from cache, in order
      # to get new data from other transactions in.
      for table in self.__old.keys ():
        for row in self.__old [table].keys ():
          # state of an 'old' row is empty if it has been deleted or if it's
          # really clean.
          if self.status (table, row) == '':
            del self.__old [table][row]

        # if a table has no more rows, remove it too
        if not self.__old [table].keys ():
          del self.__old [table]

    else:
      self.__old   = {}
      self.__new   = {}
      self.__state = {}


  # ---------------------------------------------------------------------------
  # Update the state information of a given row
  # ---------------------------------------------------------------------------

  def initialized (self, table, row):
    """
    This function sets a row of a table to 'initialized'. This means the next
    write () to this table makes this row 'commitable'
    """

    checktype (table, UnicodeType)
    checktype (row, UnicodeType)

    cState = self.__getState (table, row)
    if cState is not None and cState == 'commitable':
      raise StateChangeError, (table, row)

    self.__setState (table, row, 'initialized')


  # ---------------------------------------------------------------------------
  # Create state information for a given table/row
  # ---------------------------------------------------------------------------

  def __setState (self, table, row, state):
    """
    """

    checktype (table, UnicodeType)
    checktype (row, UnicodeType)

    if not self.__state.has_key (table):
      self.__state [table] = {}

    rows = self.__state [table]
    rows [row] = state


  # ---------------------------------------------------------------------------
  # Return the current state of a row 
  # ---------------------------------------------------------------------------

  def __getState (self, table, row):
    """
    This function returns the current state of a row
    @param table: name of the table to get state information for
    @param row: gnue-id of the row to get state information for

    @return: state information for the requested row
    """

    checktype (table, UnicodeType)
    checktype (row, UnicodeType)

    if self.__state.has_key (table) and self.__state [table].has_key (row):
      return self.__state [table] [row]

    else:
      return None


  # ---------------------------------------------------------------------------
  # Touch a record in the cache
  # ---------------------------------------------------------------------------

  def touch (self, table, row, dirty = True):
    """
    This function touches a record in the cache. If has no state information it
    will be stet to 'initializing' and if the dirty flag is set and state is
    already 'initialized' the record get's 'commitable'

    @param table: name of the table to be touched
    @param row: gnue_id of the row to be touched
    @param dirty: boolean flag to state wether to make the record dirty or not
    """

    # Update state information of the row
    state = self.__getState (table, row)

    if state is None:
      state = 'initializing'
      self.__setState (table, row, state)

    if dirty and state == 'initialized':
      self.__setState (table, row, 'commitable')


  # ---------------------------------------------------------------------------
  # Make the given row in a table to be treated as 'clean'
  # ---------------------------------------------------------------------------

  def makeClean (self, table, row):
    """
    This function makes a row of a table 'clean'. It will be moved from the
    dirty into the clean cache

    @param table: name of the table
    @param row: gnue_id of the row to be moved
    """

    if self.__new.has_key (table) and self.__new [table].has_key (row):
      if not self.__old.has_key (table):
        self.__old [table] = {}

      self.__old [table] [row] = self.__new [table] [row]

    self.__removeFromDict (self.__new, table, row)
    self.__removeFromDict (self.__state, table, row)


  # ---------------------------------------------------------------------------
  # Remove a row of a table completely from the cache
  # ---------------------------------------------------------------------------

  def remove (self, table, row):
    """
    This function removes the given row of the table completely from the cache,
    no matter wether it's dirty or not.

    @param table: name of the table
    @param row: gnue_id of the row to be removed from the cache
    """

    self.__removeFromDict (self.__new, table, row)
    self.__removeFromDict (self.__old, table, row)
    self.__removeFromDict (self.__state, table, row)


  # ---------------------------------------------------------------------------
  # Remove a row of a table from a given cache-dictionary
  # ---------------------------------------------------------------------------

  def __removeFromDict (self, dictionary, table, row):
    """
    This function removes a row of a table from the given cache-dictionary. If
    the specified row was the last row of the table cache, the table-dictionary
    will be removed too.

    @param dictionary: cache-dictionary: dict [table][row][field]
    @param table: name of the table to remove a row from
    @param row: gnue_id of the row to be removed
    """

    if dictionary.has_key (table) and dictionary [table].has_key (row):
      del dictionary [table] [row]

    if dictionary.has_key (table) and not len (dictionary [table].keys ()):
      del dictionary [table]


  # ---------------------------------------------------------------------------
  # Have a look if the cache is really in a clean state
  # ---------------------------------------------------------------------------

  def _assertClean (self):
    """
    This function iterates over all 'new' records in the cache and verifies if
    they're in a clean state.
    """

    for table in self.__new.keys ():
      for row in self.__new [table].keys ():
        if not self.status (table, row) in ["initialized"]:
          raise InvalidCacheError, (table, row, self.status (table, row))

    


# =============================================================================
# Helper methods
# =============================================================================

# -----------------------------------------------------------------------------
# Create a datasource object
# -----------------------------------------------------------------------------

def _createDatasource (connections, database, content, order = None):

  # build table list, field list, and join condition
  master = []
  tables = []
  fields = []
  conditions = None
  for (alias, (table, fk_alias, fk_field, fieldlist)) in content.items ():
    if alias:
      if not fk_alias:
        master.append ((None, table + ' ' + alias))
      fields.extend ([alias + '.' + field for field in fieldlist])
    else:
      master.append ((None, table))
      fields.extend (fieldlist)

    if fk_alias:
      table = u"LEFT JOIN %s %s ON %s.gnue_id = %s.%s" % \
          (table, alias, alias, fk_alias, fk_field)
      tables.append ((alias, table))

  # After building up the table sequences we need to make sure they're sorted
  # by their alias, otherwise the left outer joins won't work.
  master.sort ()
  tables.sort ()

  # prepare attributes of the datasource
  attributes = {}
  attributes ['name']      = ''
  attributes ['database']  = database
  attributes ['table']     = "%s %s" % \
      (string.join ([m [1] for m in master], ','),
       string.join ([t [1] for t in tables], ' '))
  attributes ['primarykey'] = 'gnue_id'

  # give the backend a hint that it's working for appserver :)
  datacon = connections.getConnection (database)
  datacon.parameters ['appserver'] = True

  if order is not None:
    if order:
      attributes ['order_by'] = order

  # create the datasource
  datasource = GDataSource.DataSourceWrapper (
    connections = connections,
    attributes = attributes,
    fields = fields)

  if conditions:
    datasource.setCondition (conditions)

  # enable unicode mode for the datasource
  datasource._dataObject._unicodeMode = 1

  return datasource

# -----------------------------------------------------------------------------
# Create an empty result set
# -----------------------------------------------------------------------------

def _createEmptyResultSet (connections, database, table, fields):

  content = {None: (table, None, None, fields)}
  datasource = _createDatasource (connections, database, content)
  return datasource.createEmptyResultSet ()

# -----------------------------------------------------------------------------
# Create a result set with data
# -----------------------------------------------------------------------------

def _createResultSet (connections, database, content, conditions, order):

  datasource = _createDatasource (connections, database, content, order)
  condition  = _conditionTree (conditions)
  try:
    return datasource.createResultSet (condition)

  finally:
    # If we have a condition, make sure to have all it's elements collectable
    # by the garbage collector
    if condition is not None:
      condition.breakReferences ()


# -----------------------------------------------------------------------------
# Create a condition tree of a given condition
# -----------------------------------------------------------------------------

def _conditionTree (conditions):

  if isinstance (conditions, ListType):
    cTree = GConditions.buildTreeFromList (conditions)

  elif isinstance (conditions, DictType):
    cTree = GConditions.buildConditionFromDict (conditions)

  else:
    cTree = conditions

  return cTree


# -----------------------------------------------------------------------------
# Create a result set containing only one row, identified by the gnue_id
# -----------------------------------------------------------------------------

def _find (connections, database, table, row, fields):

  content = {None: (table, None, None, fields)}
  condition = GConditions.buildConditionFromDict ({'gnue_id': row})
  resultSet = _createResultSet (connections, database, content, condition, [])
  resultSet.firstRecord ()
  return resultSet

# =============================================================================
# Session class
# =============================================================================

class connection:
  """
  This class encapsulates a connection to the database where data is cached on
  connection level. This means that if one query modifies data, another query
  using the same connection reads the new version even if the changes are not
  committed yet.
  """

  # ---------------------------------------------------------------------------
  # Initialize
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, database):
    checktype (connections, GConnections.GConnections)
    checktype (database, StringType)

    self.__connections = connections
    self.__database    = database
    self.__cache       = _cache ()
    self.__inserted    = []
    self.__deleted     = []

    self.__confirmedCache   = None
    self.__confirmedInserts = []
    self.__confirmedDeletes = []

    self.__uuidType = gConfig ('uuidtype').lower ()

  # ---------------------------------------------------------------------------
  # Create a recordset from a query
  # ---------------------------------------------------------------------------

  def query (self, content, conditions, order):
    """
    Executes a query and returns a recordset object.

    @param content: A dictionary of tuples defining the content of the query.
        The format of the dictionary is {alias: (table, fk_alias, fk_field,
        fields)}.
        
        'alias' is a name for the table that is unique within the query (useful
        if two different references point to the same table).  The alias may
        be None if the query references only a single table.

        'table' is the name of the table.
        
        'fk_alias' is the alias of the table containing the foreign key (i.e.
        the table that references this table), and 'fk_field' is the field name
        of the foreign key (i.e. of the referencing field).  The primary key is
        of course always 'gnue_id'. For a single element in the dictionary
        (namely the main table), 'fk_alias' and 'fk_field' both are None.
        Note that an inner join will be executet, that means only results where
        all references can be resolved (and are non-NULL) will be returned.

        Finally, 'fields' is a list of field names to be included in the query.
        All these fields will be fetched from the database and cached, so that
        subsequential access to those fields won't trigger another access to
        the database backend.

        Alias names, table names, and field names all must be Unicode strings.

    @param conditions: The conditions. May be a GCondition tree Field values in
        conditions must be in native Python type; in case of strings they must
        be Unicode.  Field names in conditions must be Unicode.  In case
        aliases are defined in the content paramter, field names in conditions
        must be in the format 'alias.field'.

    @param order: A list of Unicode strings telling the field names to order
        by.  In case aliases are defined in the content parameter, these field
        names must be in the format 'alias.field'.
    """
    checktype (content, DictType)
    for (content_key, content_value) in content.items ():
      checktype (content_key, [NoneType, UnicodeType])
      checktype (content_value, TupleType)
      (table, fk_alias, fk_field, fields) = content_value
      checktype (table, UnicodeType)
      checktype (fk_alias, [NoneType, UnicodeType])
      checktype (fk_field, [NoneType, UnicodeType])
      checktype (fields, ListType)
      for fields_element in fields: checktype (fields_element, UnicodeType)

    return recordset (self.__cache, self.__connections, self.__database,
                      content, _conditionTree (conditions), order)

  # ---------------------------------------------------------------------------
  # Generate a new object id
  # ---------------------------------------------------------------------------

  def __generateId (self):

    if self.__uuidType == 'time':
      return UUID.generateTimeBased ()
    else:
      return UUID.generateRandom ()


  # ---------------------------------------------------------------------------
  # Create a new record
  # ---------------------------------------------------------------------------

  def insertRecord (self, table):
    """
    Inserts a new record. A 'gnue_id' is assigned automatically.

    Table must be a unicode string.
    """
    checktype (table, UnicodeType)

    id = self.__generateId ()
    r = record (self.__cache, self.__connections, self.__database, table, id)
    self.__cache.write (table, id, u'gnue_id', None, 0)  # old id is None
    self.__cache.write (table, id, u'gnue_id', id, 1)    # new id

    self.__inserted.append ((table, id))
    return r

  # ---------------------------------------------------------------------------
  # Delete a record
  # ---------------------------------------------------------------------------

  def deleteRecord (self, table, row):
    """
    Deletes the given record (acutally marks it for deletion on commit). All
    data of the record will stay available until commit, but the field
    'gnue_id' will seem to have a value of None.

    Table and row must be unicode strings.
    """
    checktype (table, UnicodeType)
    checktype (row, UnicodeType)

    if not self.__cache.has (table, row, u'gnue_id'):    # not yet in cache
      self.__cache.write (table, row, u'gnue_id', row, 0)
    self.__cache.write (table, row, u'gnue_id', None, 1)
    self.__deleted.append ((table, row))

  # ---------------------------------------------------------------------------
  # Find a record
  # ---------------------------------------------------------------------------

  def findRecord (self, table, row, fields):
    """
    Loads a record from the database.  All fields given in 'fields' are fetched
    from the database and cached, so that subsequential access to those fields
    won't trigger another access to the db backend.

    This method won't query the db backend for data which is already cached.

    Table and row must be unicode strings, fields must be a list of unicode
    strings.
    """
    checktype (table, UnicodeType)
    checktype (row, UnicodeType)
    checktype (fields, ListType)
    for fields_element in fields:
      checktype (fields_element, UnicodeType)

    uncachedFields = []
    for field in fields:
      if not self.__cache.has (table, row, field):
        uncachedFields.append(field)

    if uncachedFields == [] or \
        self.__cache.status (table, row) in ['initializing', 'initialized',
                                             'inserted']:
      # already cached, no need to load from database
      r = record (self.__cache, self.__connections, self.__database, table, row)
      r._cache (u'gnue_id', row)
    else:
      # not yet cached, need to load from database
      resultSet = _find (self.__connections, self.__database, table, row,
                         uncachedFields)
      try:
        if resultSet.current is None:
          return None

        r = record (self.__cache, self.__connections, self.__database, table,
                    row)
        r._fill (None, uncachedFields, resultSet.current)

      finally:
        resultSet.close ()

    return r


  # ---------------------------------------------------------------------------
  # Write all changes back to the database
  # ---------------------------------------------------------------------------

  def commit (self):
    """
    Write all dirty data to the database backend by a single transaction that
    is committed immediately. This operation invalidates the cache.
    """

    tables = self.__cache.dirtyTables ()

    # first perform all inserts
    for (table, row) in self.__inserted [:]:
      if self.__cache.status (table, row) == 'inserted':
        fields = tables [table] [row]
        resultSet = _createEmptyResultSet (self.__connections,
                                           self.__database,
                                           table, fields.keys ())
        try:
          resultSet.insertRecord ()

          for (field, value) in fields.items ():
             resultSet.current.setField (field, value)

          resultSet.post ()

        finally:
          resultSet.close ()

        self.__inserted.remove ((table, row))

        if (table, row) in self.__confirmedInserts:
          self.__confirmedInserts.remove ((table, row))

        self.__cache.makeClean (table, row)

    # second perform all updates
    for (table, rows) in tables.items ():
      for (row, fields) in rows.items ():
        status = self.__cache.status (table, row)

        if status == 'changed':
          # TODO: gnue-common should provide a method for updating a record
          # without reading it first. Until that is done, we have to create a
          # temporary resultSet for every record we update
          resultSet = _find (self.__connections, self.__database, table, row,
                             [u'gnue_id'] + fields.keys ())

          try:
            for (field, value) in fields.items ():
              resultSet.current.setField (field, value)

            resultSet.post ()

          finally:
            resultSet.close ()

          self.__cache.makeClean (table, row)

    # perform all deletes
    for (table, row) in self.__deleted:
      # TODO: gnue-common should provide a method for deleting a record
      # without reading it first. Until that is done, we have to create a
      # temporary resultSet for every record we delete
      resultSet = _find (self.__connections, self.__database, table, row,
                         [u'gnue_id'])
      try:
        resultSet.current.delete ()
        resultSet.post ()

      finally:
        resultSet.close ()

      self.__cache.remove (table, row)

    self.__deleted = []
    self.__confirmedDeletes = []

    # Assert
    self.__cache._assertClean ()

    # Commit the whole transaction
    self.__connections.commitAll ()

    # The transaction has ended. Changes from other transactions could become
    # valid in this moment, so we have to clear the cache.
    self.__cache.clear (True)


  # ---------------------------------------------------------------------------
  # Undo all changes
  # ---------------------------------------------------------------------------

  def rollback (self):
    """
    Undo all uncommitted changes.
    """

    self.__inserted = []
    self.__deleted  = []
    self.__confirmedInserts = []
    self.__confirmedDeletes = []

    # Send the rollback to the database. Although we have (most probably) not
    # written anything yet, we have to tell the database that a new transaction
    # starts now, so that commits from other sessions become valid now for us
    # too.
    self.__connections.rollbackAll ()

    # The transaction has ended. Changes from other transactions could become
    # valid in this moment, so we have to clear the whole cache.
    self.__cache.clear ()
    self.__confirmedCache = None

  # ---------------------------------------------------------------------------
  # Close the connection
  # ---------------------------------------------------------------------------

  def close (self):
    """
    Close the connection to the database backend.
    """

    self.__connections.closeAll ()


  # ---------------------------------------------------------------------------
  # confirm all recent changes
  # ---------------------------------------------------------------------------

  def confirmChanges (self):
    """
    This function confirms all changes to be safe, so a subsequent call of the
    cancelChanges () function restores to this state.
    """

    # Doing a deepcopy here is a 'no-go' for performance. If we're creating
    # larger transaction (with a lot of records) this call will slow down
    # everything with every record added.
    self.__confirmedCache   = copy.deepcopy (self.__cache)
    self.__confirmedInserts = self.__inserted [:]
    self.__confirmedDeletes = self.__deleted [:]



  # ---------------------------------------------------------------------------
  # revoke all changes up the the last confirm
  # ---------------------------------------------------------------------------

  def cancelChanges (self):
    """
    This function revokes all changes up to the last call of the
    confirmChanges () function.
    """
    if self.__confirmedCache is not None:
      self.__cache = copy.deepcopy (self.__confirmedCache)
    else:
      self.__cache.clear ()

    self.__inserted = self.__confirmedInserts [:]
    self.__deleted  = self.__confirmedDeletes [:]


# =============================================================================
# Recordset class
# =============================================================================

class recordset:
  """
  This class manages the result of a query. An instance of this class can be
  created via the connection.query() method.
  """

  # ---------------------------------------------------------------------------
  # Initialize
  # ---------------------------------------------------------------------------

  def __init__ (self, cache, connections, database, content, conditions,
                order):
    self.__cache       = cache
    self.__connections = connections
    self.__database    = database
    self.__content     = content
    self.__order       = order

    # make sure gnue_id is selected for all tables
    for (alias, (table, fk_alias, fk_field, fields)) in self.__content.items():
      fields.append (u'gnue_id')

    self.__add     = []
    self.__added   = 0
    self.__remove  = {}
    self.__mergeWithCache (conditions, order)

    self.__current   = None
    self.__isFirst   = True
    self.__resultSet = _createResultSet (self.__connections, self.__database,
                                         self.__content, conditions, order)

  # ---------------------------------------------------------------------------
  # Return the number of records
  # ---------------------------------------------------------------------------

  def count (self):
    """
    This function returns the number of records in this result set.
    """

    return self.__resultSet.getRecordCount () + \
        len (self.__add) + self.__added - len (self.__remove.keys ())


  # ---------------------------------------------------------------------------
  # Return the next record
  # ---------------------------------------------------------------------------

  def nextRecord (self):
    """
    Returns the next record or None if nothing is left.  If the query affected
    more than one table, the record for the main table (i.e. the table that is
    the root of all other references) is returned.
    """

    # if no more records are scheduled for addition, return the next usable
    # record from the backend
    if not len (self.__add):
      rec = self.__nextBackendRecord ()
      self.__current = None
      return rec

    else:
      if not len (self.__order):
        row = self.__add [0]
        self.__add.remove (row)
        self.__added += 1
        return record (self.__cache, self.__connections, self.__database,
                       self.__getMasterTable (), row)
      else:
        row = self.__add [0]
        rec = self.__nextBackendRecord ()

        if rec is not None:
          rid   = rec.getField (u'gnue_id')
          left  = SortRecord (row, self.__getSortSequence (row, self.__order))
          right = SortRecord (rid, self.__getSortSequence (rid, self.__order))

          useAdd = cmp (left, right) <= 0
        else:
          useAdd = True

        if useAdd:
          self.__add.remove (row)
          self.__added += 1
          return record (self.__cache, self.__connections, self.__database,
                         self.__getMasterTable (), row)
        else:
          self.__current = None
          return rec


  # ---------------------------------------------------------------------------
  # Close the record set
  # ---------------------------------------------------------------------------

  def close (self):
    """
    This function closes a record set which is no longer needed. It closes the
    underlying result set.
    """

    if self.__resultSet is not None:
      self.__resultSet.close ()

    self.__resultSet = None


  # ---------------------------------------------------------------------------
  # Build record objects from current recordSet
  # ---------------------------------------------------------------------------

  def __buildRecords (self):

    result = None
    records = {}                        # the record for each alias

    # iterate through all tables involved in the query
    for (alias, (table, fk_alias, fk_field, fields)) in self.__content.items():

      # find id of the record
      if alias:
        id = self.__resultSet.current [alias + u'.gnue_id']
      else:
        id = self.__resultSet.current [u'gnue_id']

      if id:
        # build the record object
        r = record (self.__cache, self.__connections, self.__database,
                    table, id)
        r._fill (alias, fields, self.__resultSet.current)
        records [alias] = r

        # the table having no join condition is the "main" table
        if not fk_alias:
          result = r

    # iterate again to put the foreign keys in the cache
    for (alias, (table, fk_alias, fk_field, fields)) in self.__content.items():
      if fk_alias:
        if records.has_key (alias):
          value = records [alias].getField (u'gnue_id')
        else:
          value = None

        if records.has_key (fk_alias):
          records [fk_alias]._cache (fk_field, value)

    return result


  # ---------------------------------------------------------------------------
  # Get the next record from the backend source
  # ---------------------------------------------------------------------------

  def __nextBackendRecord (self):
    """
    This function returns the next usable record from the backend. If there's
    a record in the queue from a previous access it will be used first. Every
    record retrieved from the backend will be checked if it is to be removed
    (listed in the remove queue).

    @return: record instance of the next usable backend record or None if no
        more records are available.
    """

    # If there is no record instance left from an earlier call, we fetch one
    if self.__current is None:
      if self.__isFirst:
        beRec = self.__resultSet.firstRecord ()
        self.__isFirst = False
      else:
        beRec = self.__resultSet.nextRecord ()

      if beRec is None:
        return None

      self.__current = self.__buildRecords ()

    # Now, have a look if the instance has to be removed
    if self.__remove.has_key (self.__current.getRow ()):
      self.__current = None
      res = self.__nextBackendRecord ()

    return self.__current


  # ---------------------------------------------------------------------------
  # Get the name of the master table of this recordset
  # ---------------------------------------------------------------------------

  def __getMasterTable (self):
    """
    This function returns the name of the master table of the recordset
    @return: name of the master table (=table without fk_alias)
    """
    result = None

    for (alias, (table, fk_alias, fk_field, fields)) in self.__content.items():
      if not fk_alias:
        result = table
        break

    return result



  # ---------------------------------------------------------------------------
  # Merge cache 
  # ---------------------------------------------------------------------------

  def __mergeWithCache (self, conditions, order):
    """
    """

    # If the master table is not listed in the dirty cache, we have nothing to
    # do, since everything which is in clean cache will not colide with the
    # backend's result
    tables = self.__cache.dirtyTables ()
    master = self.__getMasterTable ()

    if not tables.has_key (master):
      return

    # Iterate over all dirty rows of the master table and stick them into the
    # apropriate filter
    for (row, fields) in tables [master].items ():
      state = self.__cache.status (master, row)

      if state == 'inserted':
        # an inserted (new) row must match the current condition
        self.__addFilter (row, conditions)

      elif state == 'changed':
        # if a changed row has a modified condition we stick it both into the
        # 'append' and the 'remove' filter
        if self.__conditionChanged (row, conditions):
          self.__addFilter (row, conditions)
          self.__removeFilter (row, conditions)
        else:
          # otherwise we have to analyze the sort order
          if self.__sortOrderChanged (row, order):
            # if the sortorder has changed we remove it from the old position
            # and insert it at the new one
            self.__remove [row] = True
            self.__add.append (row)

      elif state == 'deleted':
        # a deleted row must match the current condition with it's *original*
        # values
        self.__removeFilter (row, conditions)

    # If we have a sort order defined, we need to sort all records listed for
    # addition
    if len (order) and len (self.__add):
      self.__sortAddition (order)


  # ---------------------------------------------------------------------------
  # Filter with destination 'add'
  # ---------------------------------------------------------------------------

  def __addFilter (self, row, condition):
    """
    This function implements a filter for rows heading for the 'add'
    destination. Every row must meet the conditions specified in order to get
    in.
    """

    if condition is not None:
      lookup = self.__getLookupDictionary (condition, row)
      if not condition.evaluate (lookup):
        return

    self.__add.append (row)


  # ---------------------------------------------------------------------------
  # Filter with destination 'remove'
  # ---------------------------------------------------------------------------

  def __removeFilter (self, row, condition):
    """
    This function implements a filter for rows heading for the 'remove'
    destination. Every row must match the condition with it's *original* fields
    in order to get in.
    """

    if condition is not None:
      lookup = self.__getLookupDictionary (condition, row, True)
      if not condition.evaluate (lookup):
        return

    self.__remove [row] = True


  # ---------------------------------------------------------------------------
  # Create a lookup dictionary for condition-evaluation
  # ---------------------------------------------------------------------------

  def __getLookupDictionary (self, condition, row, original = False):
    """
    This function creates a dictionary with all fields listed in a given
    condition as keys and their values based on the given row.

    @param condition: GCondition tree with the conditions
    @param row: gnue_id of the master record to fetch values from
    @param original: if True, the original (clean) values will be used,
        otherwise the current values will be used.
    @return: dictionary with fields and their values
    """
    
    result = {}

    for condfield in condition.findChildrenOfType ('GCCField', True, True):
      path  = self.__getPropertyPath (condfield.name)
      result [condfield.name] = self.__getPathValue (row, path, original)

    return result
      

  # ---------------------------------------------------------------------------
  # Get the value for a property specified in a path
  # ---------------------------------------------------------------------------

  def __getPathValue (self, row, path, original = False):
    """
    This function returns the value of the property defined by the given path
    sequence, starting with the first element using a gnue_id of 'row'.
    @param path: sequence of tuples (alias, table, field) describing the
        property.
    @param row: gnue_id of the record to be used for the first path element
    @param original: if True the original (clean) values are returned,
        otherwise the current values will be returned.
    @return: value of the property
    """

    value = None
    for (alias, table, field) in path:
      r = record (self.__cache, self.__connections, self.__database, table, row)

      value = r.getField (field, original)
      row   = value

      if value is None:
        break

    return value


  # ---------------------------------------------------------------------------
  # Check if a field in a condition has changed
  # ---------------------------------------------------------------------------

  def __conditionChanged (self, row, condition):
    """
    This function iterates over all fields of a condition and returns True if a
    field has been changed or False otherwise.

    @param row: gnue_id of the record to check the condition for
    @param condition: GCondition tree with the condition or None
    @return: True if a condition field has been changed, False otherwise
    """

    if condition is not None:
      for condfield in condition.findChildrenOfType ('GCCField', True, True):
        path    = self.__getPropertyPath (condfield.name)
        if self.__fieldIsChanged (row, path):
          return True

    return False


  # ---------------------------------------------------------------------------
  # Check wether a field in the sort order has changed or not
  # ---------------------------------------------------------------------------

  def __sortOrderChanged (self, row, order):
    """
    This function checks if a field in the sort order has been changed.

    @param row: gnue_id of the record to check the sort order for
    @param order: sequence with the sort order tuples
    @return: True if a field has been changed, False otherwise
    """
    
    for field in [f ['name'] for f in order]:
      path = self.__getPropertyPath (field)
      if self.__fieldIsChanged (row, path):
        return True

    return False


  # ---------------------------------------------------------------------------
  # Check if a field has been changed
  # ---------------------------------------------------------------------------

  def __fieldIsChanged (self, row, path):
    """
    This function checks wether a field (described by a path) has been changed
    or not.

    @param row: gnue_id of the record to check the sort order for
    @param path: sequence of tuples (alias, table, field) describing the
        property.
    @return: True if the field has been changed, False otherwise
    """

    # A path of length one is a normal property without any references.
    # In this case if the property hasn't been changed, everything's fine.
    if len (path) == 1:
      (alias, table, field) = path [0]
      if not self.__cache.has (table, row, field, True):
        return False

    # Otherwise we need to compare the original and the current value
    oldvalue = self.__getPathValue (row, path, original = True)
    current  = self.__getPathValue (row, path)

    return cmp (oldvalue, current) != 0


  # ---------------------------------------------------------------------------
  # Create a path to access a given property
  # ---------------------------------------------------------------------------

  def __getPropertyPath (self, name):
    """
    This function creates a path to access a given property based on the
    content-dictionary. 
    @param: property name including alias and fieldname, separated by a dot
    @return: sequence of tuples (alias, tablename, fieldname)
    """

    result = []
    if '.' in name:
      (alias, field) = name.split ('.', 1)
    else:
      (alias, field) = (None, name)

    if self.__content.has_key (alias):
      (table, fk_alias, fk_field, fields) = self.__content [alias]
      # add a tuple to the result
      result.append ((alias, table, field))

      if fk_alias is not None:
        add = self.__getPropertyPath ("%s.%s" % (fk_alias, fk_field))
        # after finishing recursion add all inner tuples to the result,
        # maintaining sort order
        for ix in xrange (len (add)):
          result.insert (ix, add [ix])

    return result


  # ---------------------------------------------------------------------------
  # Sort all records which we have to merge in
  # ---------------------------------------------------------------------------

  def __sortAddition (self, order):
    """
    This function sorts all records from the cache to fit the current sort
    order.
    @param order: sort-sequence to be used
    """

    if len (self.__add) > 1:
      seq = []
      for row in self.__add:
        seq.append (SortRecord (row, self.__getSortSequence (row, order)))

      seq.sort ()
      self.__add = [item.row for item in seq]


  # ---------------------------------------------------------------------------
  # Create a sort sequence for a sort order and a given record
  # ---------------------------------------------------------------------------

  def __getSortSequence (self, row, order):
    """
    This function creates a sequence of tuples (fieldvalue, direction) for the
    record specified by 'row' using the given sortorder.

    @param row: gnue_id of the record to fetch fieldvalues from
    @param order: sequence of (fieldname, order) tuples specifing the sort order
    @return: sort sequence of tuples (value, direction)
    """

    result = []
    for element in order:
      field      = element ['name']
      direction  = element.get ('descending') or False
      ignorecase = element.get ('ignorecase') or False
      value = self.__getPathValue (row, self.__getPropertyPath (field))

      if ignorecase and hasattr (value, 'lower'):
        value = value.lower ()

      result.append ((value, direction))

    return result


# =============================================================================
# Record class
# =============================================================================

class record:
  """
  This class stands for a record in a database table. An instance of this class
  can be created via the recordset.nextRecord() methods, or by the
  connection.findRecord() method.
  """

  # ---------------------------------------------------------------------------
  # Initialize
  # ---------------------------------------------------------------------------

  def __init__ (self, cache, connections, database, table, row):

    self.__cache       = cache
    self.__connections = connections
    self.__database    = database
    self.__table       = table
    self.__row         = row


  # ---------------------------------------------------------------------------
  # Get the gnue_id of the record
  # ---------------------------------------------------------------------------

  def getRow (self):

    return self.__row


  # ---------------------------------------------------------------------------
  # Cache a single value for this record
  # ---------------------------------------------------------------------------

  def _cache (self, field, value):

    if not self.__cache.has (self.__table, self.__row, field):
      self.__cache.write (self.__table, self.__row, field, value, 0)

  # ---------------------------------------------------------------------------
  # Fill the cache for this record with data from a (gnue-common) RecordSet
  # ---------------------------------------------------------------------------

  def _fill (self, alias, fields, RecordSet):

    for field in fields:
      # Never ever override the cache with data from the backend
      if alias:
        fn = alias + '.' + field
      else:
        fn = field
      self._cache (field, RecordSet [fn])

  # ---------------------------------------------------------------------------
  # Get the value for a field
  # ---------------------------------------------------------------------------

  def getField (self, field, original = False):
    """
    Get the value for a field. If the value isn't cached, a new query to the
    database is issued to get the value.

    The field name must be given as a unicode string. The result will be
    returned as the native Python datatype, in case of a string field it will
    be Unicode.
    """
    checktype (field, UnicodeType)

    if original:
      if self.__cache.has (self.__table, self.__row, field, False):
        return self.__cache.read (self.__table, self.__row, field, False)
    else:
      if self.__cache.has (self.__table, self.__row, field):
        # If we find the field in the cache, use it
        return self.__cache.read (self.__table, self.__row, field)

    # if state is one of 'in*' the record is new and cannot exist in the
    # backend.
    if self.status () in ['initializing', 'initialized', 'inserted']:
      return None

    # Not found in cache, so get it from the db
    resultSet = _find (self.__connections, self.__database, self.__table,
                       self.__row, [field])
    try:
      if resultSet.current is not None:
        value = resultSet.current [field]
        self.__cache.write (self.__table, self.__row, field, value, 0)
        return value
      else:
        return None

    finally:
      resultSet.close ()


  # ---------------------------------------------------------------------------
  # Put the value for a field
  # ---------------------------------------------------------------------------

  def putField (self, field, value):
    """
    Put the value for a field.

    The field name must be given as a unicode string, value must be the native
    Python datatype of the field, in case of a string field it must be Unicode.
    """
    checktype (field, UnicodeType)

    self.__cache.write (self.__table, self.__row, field, value, 1)


  # ---------------------------------------------------------------------------
  # Set this record to initialized state
  # ---------------------------------------------------------------------------

  def initialized (self):
    """
    This function marks this record as 'initialized' so following changes will
    make it commitable.
    """

    self.__cache.initialized (self.__table, self.__row)


  # ---------------------------------------------------------------------------
  # Get the state of the current record
  # ---------------------------------------------------------------------------

  def status (self):
    """
    This function is a pass-through to the cached record's state
    """

    return self.__cache.status (self.__table, self.__row)


  # ---------------------------------------------------------------------------
  # Touch a record in the cache
  # ---------------------------------------------------------------------------

  def touch (self, dirty = True):
    """
    This function touches the underlying record in the cache.
    """

    self.__cache.touch (self.__table, self.__row, dirty)

  # ---------------------------------------------------------------------------
  # Give an official string represenation of the record instance
  # ---------------------------------------------------------------------------

  def __repr__ (self):
    """
    This function returns an official string representation of the record
    instance.
    """

    return "<record of table '%s', row '%s'>" % (self.__table, self.__row)
      

# =============================================================================
# This class implements a sortable hull for record instances
# =============================================================================

class SortRecord:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, row, sorting = None):

    self.nullFirstAsc = gConfig ('null_first_asc')
    self.nullFirstDsc = gConfig ('null_first_dsc')
    self.row          = row
    self.sorting      = sorting


  # ---------------------------------------------------------------------------
  # Comparison of two instances
  # ---------------------------------------------------------------------------

  def __cmp__ (self, other):
    """
    This function implements the comparison of two instances.
    """

    # If this or the other instance has no order-by rule, just do the
    # default-compare for instances
    if self.sorting is None or other.sorting is None:
      return cmp (self, other)

    # If both instance have an order-by rule, they must match in length
    if len (self.sorting) != len (other.sorting):
      raise OrderBySequenceError, \
          u_("Order-by sequence mismatch: '%(self)s' and '%(other)s'") \
          % {'self': self.sorting, 'other': other.sorting}

    for ix in xrange (len (self.sorting)):
      (left, descending)  = self.sorting [ix]
      (right, rightOrder) = other.sorting [ix]

      if descending != rightOrder:
        raise OrderBySequenceError, \
            u_("Order-by sequence element has different directions: "
               "'%(self)s', '%(other)s'") \
            % {'self': self.sorting [ix], 'other': other.sorting [ix]}

      noneOpt = self.nullFirstAsc
      if descending:
        (left, right) = (right, left)
        noneOpt = not self.nullFirstDsc

      if None in [left, right]:
        if not noneOpt:
          (left, right) = (right, left)
        result = cmp (left, right)
      else:
        result = cmp (left, right)

      if result != 0:
        return result

    # If no field gave a result, the two instances are treated to be equal
    return 0


  # ---------------------------------------------------------------------------
  # Official string representation
  # ---------------------------------------------------------------------------

  def __repr__ (self):
    """
    This function returns an official string representation of the sort record
    """

    return "<SortRecord for row %s at %s>" % (self.row, hex (id (self)))


# =============================================================================
# Self test code
# =============================================================================

if __name__ == '__main__':

  from gnue.common.apps import GClientApp
  from gnue.appserver import geasConfiguration

  app = GClientApp.GClientApp (application = 'appserver', defaults =
      geasConfiguration.ConfigOptions)

  print 'create connection object ...',
  c = connection (app.connections, 'gnue')
  print 'Ok'

  print 'connection.query for existing records ...'
  content = {u't0': (u'address_person', None, None, [u'address_name']),
             u't1': (u'address_country', u't0', u'address_country',
                     [u'address_name'])}
  rs = c.query (content, None, [(u't0.address_name', True)])
  print 'Ok'

  print 'recordset.nextRecord ...',
  r = rs.nextRecord ()
  print 'Ok'

  print 'record.getField with prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'record.getField with non-prefetched data ...',
  print repr (r.getField (u'address_city'))

  print 'connection.findRecord for joined table ...',
  r = c.findRecord (u'address_country', r.getField (u'address_country'), [])
  print 'Ok'

  print 'record.getField of joined table with prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'record.getField of joined table with non-prefetched data ...',
  print repr (r.getField (u'address_code'))

  print 'recordset.nextRecord ...',
  r = rs.nextRecord ()
  print 'Ok'

  print 'record.getField with prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'connection.insertRecord ...',
  r = c.insertRecord (u'address_person')
  r.initialized ()
  print 'Ok'

  print 'record.getField ...',
  id = r.getField (u'gnue_id')
  print repr (id)

  print 'record.putField ...',
  r.putField (u'address_name', u'New Person')
  print 'Ok'

  print 'record.getField of inserted data ...',
  print repr (r.getField (u'address_name'))

  print 'connection.commit with an inserted record ...',
  c.commit ()
  print 'Ok'

  print 'closing record set ...',
  rs.close ()
  print 'Ok'


  print 'connection.query for previously inserted record ...',
  content = {None: (u'address_person', None, None, [u'address_name'])}
  rs = c.query (content,
                ['eq', ['field', u'address_name'],
                       ['const', u'New Person']], None)
  print 'Ok'

  print 'recordset.nextRecord ...',
  r = rs.nextRecord ()
  print 'Ok'

  print 'record.getField with prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'record.putField of prefetched data ...',
  r.putField (u'address_name', u'New Name')
  print 'Ok'

  print 'record.putField of non-prefetched data ...',
  r.putField (u'address_city', u'New City')
  print 'Ok'

  print 'record.getField of changed data ...',
  print repr (r.getField (u'address_name'))

  print 'connection.findRecord for previously changed record ...',
  r = c.findRecord (u'address_person', id, [u'address_name'])
  print 'Ok'

  print 'record.getField of changed data, independent query ...',
  print repr (r.getField (u'address_city'))

  print 'connection.commit with a changed record ...',
  c.commit ()
  print 'Ok'

  print 'record.getField of prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'connection.deleteRecord ...',
  c.deleteRecord (u'address_person', id)
  print 'Ok'

  print 'record.getField of deleted uncommitted record, prefetched ...',
  print repr (r.getField (u'address_name'))

  print 'record.getField of deleted uncommitted record, non-prefetched ...',
  print repr (r.getField (u'address_city'))

  print 'connection.commit with a deleted record ...',
  c.commit ()
  print 'Ok'

  print 'closing result set ...',
  rs.close ()
  print 'Ok'

  print 'check if the record is really gone now ...',
  content = {None: (u'address_person', None, None, [u'address_name'])}
  rs = c.query (content,
                ['eq', ['field', u'address_city'],
                       ['const', u'New City']],
                None)
  if rs.nextRecord () != None:
    raise Exception
  print 'Ok'

  print 'closing result set ...',
  rs.close ()
  print 'Ok'

  print 'connection.close ...',
  c.close ()
  print 'Ok'

  print 'Thank you for playing!'
