#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# $Id: $

from gnue.common.apps import errors

BASE_TYPES = ['boolean', 'date', 'datetime', 'id', 'number', 'string', 'time']


# =============================================================================
# Exceptions
# =============================================================================

class ValidationError (errors.ApplicationError):
  pass

class TypeNameError (ValidationError):
  def __init__ (self, typename):
    msg = u_("'%s' is not a valid type") % typename
    ValidationError.__init__ (self, msg)

class TypeFormatError (ValidationError):
  pass

class ParameterValidationError (ValidationError):
  def __init__ (self, parameter, message):
    msg = u_("Parameter %(parameter)s: %(error)s") \
          % {"parameter": parameter,
             "error"    : message}
    ValidationError.__init__ (self, msg)


# -----------------------------------------------------------------------------
# Check if a combination of typename, length and scale is valid
# -----------------------------------------------------------------------------

def verifyBasetype (typename, length, scale):
  """
  This function verifies a given typename with length and scale. If this
  combination makes no sense a TypeFormatError will be raised. If typename
  is no valid base type a TypeNameError will be raised.
  """
  if typename is None:
    raise TypeNameError, repr (typename)

  if not typename in BASE_TYPES:
    raise TypeNameError, (typename)

  if typename == 'string':
    if scale is not None and scale:
      raise TypeFormatError, u_("string does not support 'scale'")

  if typename in ['id', 'date', 'time', 'datetime', 'boolean']:
    if length is not None and length:
      raise TypeFormatError, u_("%s does not support 'length'") % typename
    if scale is not None and scale:
      raise TypeFormatError, u_("%s does not support 'scale'") % typename

  if typename == 'number':
    if scale is not None and scale:
      if length is None or not length:
        raise TypeFormatError, u_("number without 'length'")


# -----------------------------------------------------------------------------
# Check if a given reference is valid
# -----------------------------------------------------------------------------

def verifyReference (reference, length, scale, modules):
  """
  This function verifies if @reference is a valid reference type. @length and
  @scale must be None or zero and the given module dictionary must
  have a classe named 'reference'.
  @param reference: the classname to be verified
  @param length: the length of the type definition
  @param scale: the length of the type definition
  @param modules: module dictionary to be used for lookups
  @return: None if reference is an invalid classname, or the class object
      otherwise.
  """
  for module in modules.values ():
    if module.classes.has_key (reference):
      if length is not None and length:
        raise TypeFormatError, u_("Reference types must not have a 'length'")
      if scale is not None and scale:
        raise TypeFormatError, u_("Reference types must not have a 'scale'")

      return module.classes [reference]

  return None


# -----------------------------------------------------------------------------
# Verify a given type
# -----------------------------------------------------------------------------

def verifyType (typename, length, scale, modules):
  """
  This function verifies a given type, length and scale combination, optionally
  using the given module dictionary for lookups of references.
  """
  ref = verifyReference (typename, length, scale, modules)
  if ref is not None:
    return ref

  else:
    verifyBasetype (typename, length, scale)
    return None


# -----------------------------------------------------------------------------
# Verify a procedure definition
# -----------------------------------------------------------------------------
def verifyProcedure (aProc):
  """
  This function checks the resulttype of a procedure definition, and all
  parameter types (if available).
  """

  # If a result type is specified, check it
  if aProc.gnue_type is not None:
    verifyBasetype (aProc.gnue_type, aProc.gnue_length, aProc.gnue_scale)

  else:
    # otherwise there must not be anything concerning a result type
    if aProc.gnue_length is not None and aProc.gnue_length:
      raise TypeFormatError, u_("%s: Procedure has no result, but a 'length' "
                                "is specified.") % aProc.fullName
    if aProc.gnue_scale is not None and aProc.gnue_scale:
      raise TypeFormatError, u_("%s: Procedure has no result, but a 'scale' "
                                "is specified.") % aProc.fullName

  # verify all given parameter types
  for pa in aProc.parameters.values ():
    try:
      verifyBasetype (pa.gnue_type, pa.gnue_length, pa.gnue_scale)

    except ValidationError, vErr:
      raise ParameterValidationError, (pa.fullName, vErr.message)




# -----------------------------------------------------------------------------
# unit self test code
# -----------------------------------------------------------------------------

if __name__ == '__main__':
  import sys

  from gnue.appserver.test import testApp

  sm = testApp ().getSessionManager ()

  def check (name, length = None, scale = None):
    tstr = name
    if length is not None or scale is not None:
      tstr += "("
      if length is not None:
        tstr += "%d" % length
      if scale is not None:
        tstr += ",%d" % scale
      tstr += ")"

    try:
      verifyType (name, length, scale, sm.classes)
      print tstr, "Ok"

    except:
      print tstr
      print "  %s: %s" % (sys.exc_info () [0], sys.exc_info () [1])


  check ('string')
  check ('string', 5)
  check ('string', 5, 2)
  check ('date')
  check ('Date', 5)
  check ('date', 5, 0)
  check ('time')
  check ('time', 0, 1)
  check ('datetime')
  check ('datetime', 5,1)
  check ('boolean')
  check ('boolean', 5)
  check ('number')
  check ('number', 0, 3)
  check ('number', 5)
  check ('number', 5, 1)
  check ('address_country')
  check ('address_country', None, 3)
  check ('foo_bar')
  check (None)
