# GNU Enterprise Application Server - Class Repository: Namespace
#
# Copyright 2003-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Namespace.py 6851 2005-01-03 20:59:28Z jcater $

import string
from Base import ClassRepositoryError

# =============================================================================
# Exceptions
# =============================================================================

class InvalidNameError (ClassRepositoryError):
  def __init__ (self, name):
    msg = u_("'%s' is not a valid, fully qualified identifier") % name
    ClassRepositoryError.__init__ (self, msg)


# =============================================================================
# Namespace constants
# =============================================================================

NSID_INVALID  = -1
NSID_MODULE   =  1
NSID_CLASS    =  2
NSID_PROPERTY =  3

# -----------------------------------------------------------------------------
# Split a fully qualified name into a tuple (namespace, identifier)
# -----------------------------------------------------------------------------
def splitName (name):
  parts = string.split (name, '_')

  if len (parts) == 1:
    return ('', name)

  elif len (parts) == 2:
    return (parts [0], parts [1])

  raise InvalidNameError, (name)

# -----------------------------------------------------------------------------
# Create a fully qualified name from a namespace and an identifier
# -----------------------------------------------------------------------------
def createName (namespace, identifier):
  return "%s_%s" % (namespace, identifier)

# -----------------------------------------------------------------------------
# Try to figure out which NSID_* constant applies to identifier
# TODO: implement some methods to recognize class-/typeinformation about
#       the classrepository itself.
# -----------------------------------------------------------------------------
def getNamespaceId (identifier):
  # a valid identifier must contain at least one character
  if not identifier:
    return NSID_INVALID

  # a module-identifier must not contain '.'- or '_'-characters; i.e. 'gnue'
  if not 1 in [char in '._' for char in identifier]:
    return NSID_MODULE

  # class-identifiers are formed out of two module-identifieres separated by an
  # underscore, i.e. 'gnue_module'
  parts = string.split (identifier, '_')
  if len (parts) == 2:
    if getNamespaceId (parts [0]) == NSID_MODULE and \
       getNamespaceId (parts [1]) == NSID_MODULE:
      return NSID_CLASS

  # property-identifiers are fromed out of two class-identifiers separated by a
  # single dot, i.e. 'gnue_module.gnue_id'
  parts = string.split (identifier, '.')
  if len (parts) == 2:
    if getNamespaceId (parts [0]) == NSID_CLASS and \
       getNamespaceId (parts [1]) == NSID_CLASS:
      return NSID_PROPERTY

  # since no pattern matched until here, we must assume the identifier as
  # invalid.
  return NSID_INVALID
