# GNU Enterprise Application Server - Class Repository: Class
#
# Copyright 2003-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Class.py 7104 2005-03-07 16:49:25Z johannes $

import sys
from Base import *
from Property import *
from Procedure import *
from Namespace import createName, splitName
from helpers import ValidationError


# =============================================================================
# Exceptions
# =============================================================================

class ClassNotFoundError (ClassRepositoryError):
  """
  Referenced class not found in repository
  """
  def __init__ (self, classname):
    msg = u_("Class '%s' not found in class repository") % classname
    ClassRepositoryError.__init__ (self, msg)


# =============================================================================
# Dictionary with classes
# =============================================================================
class ClassDict (BaseDictionary):
  """
  This class implements a dictionary of classes in the class repository.
  """

  # ---------------------------------------------------------------------------
  # Create a dictionary with all classes [from a specified module]
  # ---------------------------------------------------------------------------

  def __init__ (self, session, moduleDict, module = None, predefines = None):
    BaseDictionary.__init__ (self, session, 'gnue_class')

    self.modules    = moduleDict
    self.__module   = module

    # Note: this 'predefines' is a RepositoryDefinition
    if predefines is not None:
      for cDict in predefines.classes ():
        cMod     = self.modules.find (cDict ['gnue_module'])
        proplist = predefines.properties (cDict ['gnue_id'])

        aClass = Class (session, self, cMod, None, cDict, proplist)
        self._items [aClass.fullName.lower ()] = aClass

    # if this class dictionary is for a special module, we populate it with
    # all classes, either from a global class dictionary or a fresh reload
    if module is not None:
      if self.modules.classdict is not None:
        # copy the predefined class dictionary 
        for item in self.modules.classdict.values ():
          if item.module.fullName == module.fullName:
            self._items [item.fullName.lower ()] = item
      else:
        # fetch a fresh copy from the datasource
        #
        # NOTE: this won't happen very often, since it is only called if a
        # module requests a list of all classes without having it's own
        # ClassDict-instance.
        gDebug (2, "FORCE classdict.reload () in constructor ?!")
        self.reload ()


  # ---------------------------------------------------------------------------
  # Reload the dictionary and run nested reloads for all classes 
  # ---------------------------------------------------------------------------

  def reload (self):
    """
    This function reloads all available classes of a module, and all their
    properties and procedures.
    """
    gDebug (2, "ClassDict::reload ()")

    BaseDictionary.reload (self)

    gProp = PropertyDict (self._session, None, modules = self.modules)
    gProp.reload (idAsKey = True)

    gParam = ParameterDict (self._session, None)
    gParam.reload (idAsKey = True)

    gProc = ProcedureDict (self._session, None, modules = self.modules)
    gProc.reload (idAsKey = True)

    for proc in gProc.values ():
      for param in gParam.values ():
        if param.gnue_procedure.gnue_id == proc.gnue_id:
          proc.parameters [param.fullName] = param

    for aClass in self.values ():
      for prop in gProp.values ():
        if prop.gnue_class.gnue_id == aClass.gnue_id:
          aClass.properties [prop.fullName] = prop

      for proc in gProc.values ():
        if proc.gnue_class.gnue_id == aClass.gnue_id:
          aClass.procedures [proc.fullName] = proc

    gDebug (2, "End of ClassDict::Reload ()")


  # ---------------------------------------------------------------------------
  # Create a new instance for a dictionary-item
  # ---------------------------------------------------------------------------

  def _getNewItem (self, aObject):
    """
    Create a new instance of a class and reload it's properties and procedures.
    """
    module = self.modules.find (aObject.gnue_module.objectId)
    return Class (self._session, self, module, aObject,
                  {"gnue_id": aObject.objectId})


  # ---------------------------------------------------------------------------
  # Get an apropriate reload ()-condition
  # ---------------------------------------------------------------------------

  def _getReloadCondition (self):
    """
    If this class dictionary is bound to a module, the reload condition matches
    all classes of the bound module, otherwise no condition is given.
    """
    if self.__module is not None:
      result = ['eq', ['field', u'gnue_module'],
                    ['const', self.__module.gnue_id]]
    else:
      result = []

    return gLeave (6, result)


  # ---------------------------------------------------------------------------
  # Create a condition to retrieve the klass specified by 'key'
  # ---------------------------------------------------------------------------

  def _getSingleCondition (self, key):
    """
    A single class is identified by it's modules gnue_id and the classname.
    """
    (moduleName, className) = splitName (key)
    module = self.modules [moduleName]

    return ['and', ['eq', ['field', u'gnue_module'], ['const', module.gnue_id]],
                   ['eq', ['field', u'gnue_name'], ['const', className]]]


  # ---------------------------------------------------------------------------
  # Create a key-error exception
  # ---------------------------------------------------------------------------

  def _itemNotFoundError (self, key):
    """
    Return a ClassNotFoundError if an item cannot be found.
    """
    return ClassNotFoundError (key)


  # ---------------------------------------------------------------------------
  # On find () this dictionary will prepare the following columns ()
  # ---------------------------------------------------------------------------

  def _getColumns (self):
    """
    Fields to be fetched on reloads.
    """
    result = [u"gnue_module", u"gnue_name", u"gnue_filter"]
    return gLeave (6, result)


  # ---------------------------------------------------------------------------
  # Find a listed class by its id
  # ---------------------------------------------------------------------------

  def find (self, classId):
    """
    This function searches for a class which gnue_id is @classId.
    """
    for klass in self.values ():
      if klass.gnue_id == classId:
        return klass
    return None


# =============================================================================
# An implementation of a single class
# =============================================================================
class Class (BaseObject):
  
  # ---------------------------------------------------------------------------
  # Wrapping a business object class
  # ---------------------------------------------------------------------------

  def __init__ (self, session, classDict, module, object, predefs = None,
                propDefList = None):
    BaseObject.__init__ (self, session, 'gnue_class', object, predefs)

    self.module     = module
    self.fullName   = createName (module.gnue_name, self.gnue_name)
    self.table      = self.fullName
    self.classes    = classDict

    self.properties = PropertyDict (session, self, propDefList)
    self.procedures = ProcedureDict (session, self)


  # ---------------------------------------------------------------------------
  # Return a property by name
  # ---------------------------------------------------------------------------

  def findProp (self, name):
    """
    Find a property @name.
    """
    return self.properties [name]


  # ---------------------------------------------------------------------------
  # Return a procedure by name
  # ---------------------------------------------------------------------------

  def findProc (self, name):
    """
    Find a procedure @name.
    """
    return self.procedures [name]


  # ---------------------------------------------------------------------------
  # Validate a class definition
  # ---------------------------------------------------------------------------

  def validate (self):
    for p in self.properties.values ():
      try:
        p.validate ()

      except ValidationError:
        del self.properties [p.fullName]
        (name, msg) = errors.getException () [1:3]
        gDebug (2, "%s.%s: %s: %s" % (self.fullName, p.fullName, name, msg))

    for p in self.procedures.values ():
      try:
        p.validate ()

      except ValidationError:
        del self.procedures [p.fullName]
        (name, msg) = errors.getException () [1:3]
        gDebug (2, "%s.%s: %s: %s" % (self.fullName, p.fullName, name, msg))

    self.properties.loadCalculated (self.procedures)


  # ---------------------------------------------------------------------------
  # Replace references
  # ---------------------------------------------------------------------------

  def replaceReferences (self, module):
    """
    This class updates all references of a class to it's module object and
    requests all properties and procedures to do update their references too.
    """
    BaseObject.replaceReferences (self, module)

    me = self._getObject ()
    for prop in self.properties.values ():
      prop.replaceReferences (module)
      prop.replaceReferences (me)

    for proc in self.procedures.values ():
      proc.replaceReferences (module)
      proc.replaceReferences (me)


  # ---------------------------------------------------------------------------
  # For a class to be complete, make sure properties and procedures are loaded
  # ---------------------------------------------------------------------------

  def complete (self):
    """
    After adding a new instance to the class dictionary make sure it's
    properties and procedures are up to date too
    """
    gDebug (2, "Completing class %s" % self.fullName)
    self.properties.reload ()
    self.procedures.reload ()

    self.validate ()

