# GNU Enterprise Application Server - Session Manager Object
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: geasSessionManager.py 5636 2004-04-05 11:09:29Z reinhard $

from types import *

import geasSession
import geasAuthentication
import classrep

from gnue.common.datasources import GConnections

# =============================================================================
# Session Manager class
# =============================================================================

class geasSessionManager:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, connections):
    self._connections = connections
    self._sessNo = 0
    self._sessions = {}
    self._buildInternalSession ()
    classrep.init (self)

    # TODO: load default authagent depending on config setting
    self._authAdapter = geasAuthentication.geasDBAuthAgent (self._internalSession)
    self._langRuntimes = {} 

  # ---------------------------------------------------------------------------
  # Build an internal session
  # ---------------------------------------------------------------------------

  def _buildInternalSession (self):
    self._internalSession = geasSession.geasSession (self._connections,
      geasAuthentication.geasAuthAgent(), self, 0)
    self._internalSession.login (None,None) # fake login
    self._sessions [0] = self._internalSession

  # ---------------------------------------------------------------------------
  # Find a session from session ID
  # ---------------------------------------------------------------------------

  def _getSession (self, sess_id):
    if self._sessions.has_key (sess_id):
      return self._sessions [sess_id]
    else:
      raise gException, u_("Can't find a session with ID '%s'") % sess_id

  # ===========================================================================
  # official API functions
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Open the connection
  # ---------------------------------------------------------------------------

  def open (self, authentication):

    # TODO use a better session ID than this one
    self._sessNo = self._sessNo + 1

    loginHandler = self._connections._loginHandler
    loginOptions = self._connections._loginOptions
    location     = self._connections._location
    
    conn = GConnections.GConnections(location, loginHandler, loginOptions)
   
    sess = geasSession.geasSession (conn, self._authAdapter, self,
                                    self._sessNo)

    sess.login (authentication ['user'], authentication ['password'])
    self._sessions [self._sessNo] = sess
    return self._sessNo

  # ---------------------------------------------------------------------------
  # Close the connection
  # ---------------------------------------------------------------------------

  def close (self, session_id, commit):

    s = self._getSession (session_id)
    if commit:
      s.commit ()
    s.logout ()
    del self._sessions [session_id]

  # ---------------------------------------------------------------------------
  # Commit current transaction
  # ---------------------------------------------------------------------------

  def commit (self, session_id):

    s = self._getSession (session_id)
    s.commit ()

  # ---------------------------------------------------------------------------
  # Rollback current transaction
  # ---------------------------------------------------------------------------

  def rollback (self, session_id):

    s = self._getSession (session_id)
    s.rollback ()

  # ---------------------------------------------------------------------------
  # Build list from query
  # ---------------------------------------------------------------------------

  def request (self, session_id, classname, conditions, sortorder,
               propertylist):

    s = self._getSession (session_id)
    return s.request (classname, conditions, sortorder, propertylist)

  # ---------------------------------------------------------------------------
  # Count number of objects in list
  # ---------------------------------------------------------------------------

  def count (self, session_id, list_id):

    s = self._getSession (session_id)
    return s.count (list_id);

  # ---------------------------------------------------------------------------
  # Fetch objects from list
  # ---------------------------------------------------------------------------

  def fetch (self, session_id, list_id, start, count, close=0):

    s = self._getSession (session_id)
    return s.fetch (list_id, start, count)

  # ---------------------------------------------------------------------------
  # Load objects from ids
  # ---------------------------------------------------------------------------

  def load (self, session_id, classname, obj_id_list, propertylist):

    s = self._getSession (session_id)
    return s.load (classname, obj_id_list, propertylist)

  # ---------------------------------------------------------------------------
  # Store objects
  # ---------------------------------------------------------------------------

  def store (self, session_id, classname, obj_id_list, propertylist, data):

    s = self._getSession (session_id)
    return s.store (classname, obj_id_list, propertylist, data)

  # ---------------------------------------------------------------------------
  # Delete objects
  # ---------------------------------------------------------------------------

  def delete (self, session_id, classname, obj_id_list):

    s = self._getSession (session_id)
    s.delete (classname, obj_id_list)

  # ---------------------------------------------------------------------------
  # Call procedure
  # ---------------------------------------------------------------------------

  def call (self, session_id, classname, obj_id_list, procedurename,
            parameters):

    s = self._getSession (session_id)
    return s.call (classname, obj_id_list, procedurename, parameters)
