# GNU Enterprise Application Server - Class Repository: Procedure
#
# Copyright 2003-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Procedure.py 5469 2004-03-22 20:51:25Z johannes $

from Base import *
from Namespace import createName, splitName
from Parameter import *

# =============================================================================
# Exceptions
# =============================================================================

class ProcedureNotFoundError (ClassRepositoryError):
  def __init__ (self, classname, procedure):
    msg = _("Class '%(classname)s' has no procedure '%(procedure)s'") % \
           { "classname": classname, "procedure": procedure }
    ClassRepositoryError.__init__ (self, msg)


# =============================================================================
# Dictionary with all properties of a given class
# =============================================================================
class ProcedureDict (BaseDictionary):

  # ---------------------------------------------------------------------------
  # Construct a Procedure-Dictionary for class aClass
  # ---------------------------------------------------------------------------
  def __init__ (self, session, aClass, predefs = None):
    BaseDictionary.__init__ (self, session, 'gnue_procedure')

    self.__class  = aClass
    self.__module = self.__class.module


  # ---------------------------------------------------------------------------
  # Reload the dictionary
  # ---------------------------------------------------------------------------

  def reload (self):
    BaseDictionary.reload (self)
    for aProc in self.values ():
      aProc.parameters.reload ()


  # ---------------------------------------------------------------------------
  # Create a new instance of a dictionary item
  # ---------------------------------------------------------------------------
  def _getNewItem (self, aObject):
    pMod = self.__module.modules.find (aObject.gnue_module.objectId)
    aProc = Procedure (self._session, pMod, aObject, 
                      {"gnue_id": aObject.objectId})
    aProc.parameters.reload ()
    return aProc


  # ---------------------------------------------------------------------------
  # A reload () returns only the procedures of the class
  # ---------------------------------------------------------------------------
  def _getReloadCondition (self):
    return [["eq", ""], ["field", "gnue_class"], 
            ["const", self.__class.gnue_id]]


  # ---------------------------------------------------------------------------
  # Get a condition to match a given procedure
  # ---------------------------------------------------------------------------
  def _getSingleCondition (self, key):
    procName = splitName (key) [1]

    return [["eq", ""], ["field", "gnue_class"], 
            ["const", self.__class.gnue_id],
            ["eq", ""], ["field", "gnue_name"], ["const", procName]]


  # ---------------------------------------------------------------------------
  # Get a list of columns to be prepared on a find () 
  # ---------------------------------------------------------------------------
  def _getColumns (self):
    return [u"gnue_module", u"gnue_class", u"gnue_name", u"gnue_language"]

  # ---------------------------------------------------------------------------
  # Create a key-not-found exception
  # ---------------------------------------------------------------------------
  def _itemNotFoundError (self, key):
    return ProcedureNotFoundError (self.__class.fullName, key)


# =============================================================================
# A Business Object Procedure
# =============================================================================
class Procedure (BaseObject):

  # ---------------------------------------------------------------------------
  # Construct a new procedure from module, class and business object
  # ---------------------------------------------------------------------------
  def __init__ (self, session, module, object, pDefs = None):
    BaseObject.__init__ (self, session, 'gnue_procedure', object, pDefs)

    self.module   = module
    self.fullName = createName (module.gnue_name, self.gnue_name)

    self.parameters = ParameterDict (session, self)

