//  
//  Copyright (C) 2009 GNOME Do
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
// 
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
// 
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
// 

using System;
using System.Collections.Generic;
using System.Linq;
using System.IO;
using System.Text.RegularExpressions;

using Cairo;
using Gdk;
using GLib;
using NDesk.DBus;

using Do.Platform;
using Do.Interface;

using Docky.Interface;

namespace BatteryMonitor
{
	
	
	public class BatteryMonitor : AbstractDockletItem
	{
		const string BattInfoPath = "/proc/acpi/battery/BAT0/info";
		const string BattStatePath = "/proc/acpi/battery/BAT0/state";
		
		const string BottomSvg  = "battery_bottom.svg";
		const string InsideSvg  = "battery_inside_{0}.svg";
		const string PluggedSvg = "battery_plugged.svg";
		const string TopSvg     = "battery_top.svg";
		
		int max_capacity;
		int current_capacity;
		uint timer;
		
		Regex number_regex;
		
		double Capacity {
			get {
				return (double) current_capacity / max_capacity;
			}
		}
		
		int RoundedCapacity {
			get {
				return (int) (Math.Round (Capacity, 1) * 100);
			}
		}
		
		public override string Name {
			get {
				return "Battery Monitor";
			}
		}
		
		public BatteryMonitor ()
		{
			number_regex = new Regex ("[0-9]+");
			
			max_capacity = GetBatteryCapacity ();
			
			UpdateBattStat ();
			
			timer = GLib.Timeout.Add (20 * 1000, UpdateBattStat);
		}
		
		public override void Enable ()
		{
			Services.System.OnBatteryChanged += HandleBatteryChanged;
		}
		
		void HandleBatteryChanged (object sender, EventArgs args)
		{
			RedrawIcon ();
		}
		
		int GetBatteryCapacity ()
		{
			int capacity = 1;
			
			if (File.Exists (BattInfoPath)) {
				using (StreamReader reader = new StreamReader (BattInfoPath)) {
					string line;
					while (!reader.EndOfStream) {
						line = reader.ReadLine ();
						if (!line.StartsWith ("last full capacity"))
							continue;
						
						try {
							capacity = Convert.ToInt32 (number_regex.Matches (line) [0].Value);
						} catch { }
						break;
					}
				}
			}
			
			return capacity;
		}
		
		bool UpdateBattStat ()
		{
			string capacity = null;
			string chargeState = null;
			
			if (!File.Exists (BattStatePath)) {
				SetText ("No Battery Found");
				current_capacity = 0;
				max_capacity = 1;
				
				RedrawIcon ();
				return true;
			}
			
			using (StreamReader reader = new StreamReader (BattStatePath)) {
				string line;
				while (!reader.EndOfStream) {
					if (!string.IsNullOrEmpty (capacity) && !string.IsNullOrEmpty (chargeState))
						break;
					
					line = reader.ReadLine ();
					if (line.StartsWith ("remaining capacity")) {
						capacity = line;
						continue;
					}
					
					if (line.StartsWith ("charging state"))
						chargeState = line;
				}
			}
			
			try {
				current_capacity = Convert.ToInt32 (number_regex.Matches (capacity) [0].Value);
			} catch { }
			
			RedrawIcon ();
			SetText (string.Format ("{0:0.0}%", Capacity * 100));
			return true;
		}
		
		void RenderSvgOnContext (Context cr, string resource, int size, int x, int y)
		{
			Gdk.Pixbuf pbuf = Do.Interface.IconProvider.PixbufFromIconName (resource, size);
			if (pbuf == null)
				return;
			
			CairoHelper.SetSourcePixbuf (cr, pbuf, x + ((pbuf.Width - size) / 2), y + ((pbuf.Height - size) / 2));
			cr.Paint ();
		}
		
		protected override Surface MakeIconSurface (Cairo.Surface similar, int size)
		{
			Surface sr = similar.CreateSimilar (similar.Content, size, size);
			using (Context cr = new Context (sr)) {
				RenderSvgOnContext (cr, BottomSvg + "@" + GetType ().Assembly.FullName, size, 0, 0);
				if (RoundedCapacity > 0)
					RenderSvgOnContext (cr, string.Format (InsideSvg, RoundedCapacity) + "@" + GetType ().Assembly.FullName, size, 0, 0);
				RenderSvgOnContext (cr, TopSvg + "@" + GetType ().Assembly.FullName, size, 0, 0);
				if (!Services.System.GetOnBatteryPower ()) {
					RenderSvgOnContext (cr, PluggedSvg + "@" + GetType ().Assembly.FullName, size, 0, 0);
				}
			}
			
			return sr;
		}
		
		public override void Dispose ()
		{
			Services.System.OnBatteryChanged -= HandleBatteryChanged;
			GLib.Source.Remove (timer);
			base.Dispose ();
		}
	}
}
