/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"
#include "gnome-cmd-prepare-xfer-dialog.h"
#include "gnome-cmd-dir-funcs.h"
#include "dir_pool.h"
#include "gnome-cmd-xfer.h"
#include "gnome-cmd-connection-funcs.h"
#include "gnome-cmd-dir-funcs.h"
#include "gnome-cmd-file-funcs.h"
#include "gnome-cmd-file-selector-funcs.h"
#include "gnome-cmd-main-win-funcs.h"
#include "widget-factory.h"

static GnomeDialogClass *parent_class = NULL;



static void
on_ok (GtkButton *button, GnomeCmdPrepareXferDialog *dialog)
{
	GnomeCmdDir *dest_dir;
	GnomeVFSURI *dest_dir_uri;
	gchar *dest_path, *dest_fn, *dest_dir_uri_str;
	
	dest_fn = NULL;
	dest_path = gtk_entry_get_text (GTK_ENTRY (dialog->dest_dir_entry));

	if (!dest_path || strcmp (dest_path, "") == 0) {
		gnome_error_dialog (_("Destination path not specified"));
		return;
	}

	if (!g_path_is_absolute (dest_path)) {
		GnomeCmdDir *cwd = gnome_cmd_file_selector_get_directory (
			gnome_cmd_main_win_get_active_fs (main_win));
		gchar *tmp = g_concat_dir_and_file (
			gnome_cmd_dir_get_path (cwd), dest_path);
		g_free (dest_path);
		dest_path = tmp;
	}
	
	if (g_list_length (dialog->src_files) == 1) {
		if (dest_path[strlen(dest_path)-1] != '/') {
			gchar *tmp = g_dirname (dest_path);
			dest_dir_uri = gnome_vfs_uri_append_path (
				dialog->dest_baseuri, tmp);
			g_free (tmp);
			dest_fn = g_strdup (g_basename (dest_path));
		}
		else {
			dest_dir_uri = gnome_vfs_uri_append_path (
				dialog->dest_baseuri, dest_path);
			dest_fn = NULL;
		}
	}
	else
		dest_dir_uri = gnome_vfs_uri_append_path (
			dialog->dest_baseuri, dest_path);
	
	dest_dir_uri_str = gnome_vfs_uri_to_string (dest_dir_uri, 0);
	
	dest_dir = dir_pool_get (dest_dir_uri_str);
	g_free (dest_dir_uri_str);
	gnome_vfs_uri_unref (dest_dir_uri);

	gtk_widget_hide (GTK_WIDGET (dialog));

	gnome_cmd_dir_ref (dest_dir);
	gnome_cmd_xfer_start (dialog->src_files,
						  dest_dir,
						  dialog->src_fl,
						  dest_fn,
						  dialog->xferOptions,
						  dialog->xferOverwriteMode,
						  NULL, NULL);

	gnome_cmd_file_list_unref (dialog->src_files);
	g_free (dest_path);
}


static void
on_cancel (GtkButton *button, gpointer user_data)
{
	GnomeCmdPrepareXferDialog *dialog = GNOME_CMD_PREPARE_XFER_DIALOG (user_data);

	gtk_widget_hide (GTK_WIDGET (dialog));
}


static gboolean
on_dest_dir_entry_keypressed (GtkEntry *entry,
							  GdkEventKey *event,
							  GnomeCmdPrepareXferDialog *dialog)
{
	if (event->keyval == GDK_Return) {
		gtk_signal_emit_by_name (GTK_OBJECT (dialog->ok_button), "clicked", dialog, NULL);
		return TRUE;
	}

	return FALSE;
}





/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdPrepareXferDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}


static void
init (GnomeCmdPrepareXferDialog *dialog)
{	
	GtkWidget *dialog_vbox;
	GtkWidget *dest_dir_vbox;
	GtkWidget *dest_dir_fileentry;
	GtkWidget *options_hbox;

	
	gtk_window_set_policy (GTK_WINDOW (dialog), FALSE, TRUE, FALSE);
	gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_CENTER);
	gtk_widget_set_usize (GTK_WIDGET (dialog), 450, -1);

	dialog_vbox = GNOME_DIALOG (dialog)->vbox;
	gtk_object_set_data (GTK_OBJECT (dialog), "dialog_vbox", dialog_vbox);
	gtk_widget_show (dialog_vbox);


	
	/* dest dir */
	dest_dir_vbox = create_vbox (GTK_WIDGET (dialog), FALSE, 0);

	dialog->dest_dir_frame = create_category (GTK_WIDGET (dialog), dest_dir_vbox, "");
	gtk_box_pack_start (GTK_BOX (dialog_vbox), dialog->dest_dir_frame, TRUE, TRUE, 0);


	dest_dir_fileentry = create_file_entry (GTK_WIDGET (dialog), "file-entry", "");
	gtk_box_pack_start (GTK_BOX (dest_dir_vbox), dest_dir_fileentry, FALSE, FALSE, 0);
	dialog->dest_dir_entry = gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (dest_dir_fileentry));

	gtk_signal_connect (GTK_OBJECT (dialog->dest_dir_entry), "key-press-event",
						GTK_SIGNAL_FUNC (on_dest_dir_entry_keypressed), dialog);



	/* options */
	options_hbox = create_hbox (GTK_WIDGET (dialog), TRUE, 6);
	gtk_box_pack_start (GTK_BOX (dialog_vbox), options_hbox, TRUE, TRUE, 0);
	
	dialog->left_vbox = create_vbox (GTK_WIDGET (dialog), FALSE, 0);
	dialog->right_vbox = create_vbox (GTK_WIDGET (dialog), FALSE, 0);

	dialog->left_vbox_frame = create_category (GTK_WIDGET (dialog), dialog->left_vbox, "");
	gtk_container_add (GTK_CONTAINER (options_hbox), dialog->left_vbox_frame);
	
	dialog->right_vbox_frame = create_category (GTK_WIDGET (dialog), dialog->right_vbox, "");
	gtk_container_add (GTK_CONTAINER (options_hbox), dialog->right_vbox_frame);
	

	
	/* buttons */
	gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_OK);
	dialog->ok_button = g_list_last (GNOME_DIALOG (dialog)->buttons)->data;
	gtk_widget_ref (dialog->ok_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "dialog->ok_button",
							  dialog->ok_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->ok_button);
	GTK_WIDGET_SET_FLAGS (dialog->ok_button, GTK_CAN_DEFAULT);

	gnome_dialog_append_button (GNOME_DIALOG (dialog), GNOME_STOCK_BUTTON_CANCEL);
	dialog->cancel_button = g_list_last (GNOME_DIALOG (dialog)->buttons)->data;
	gtk_widget_ref (dialog->cancel_button);
	gtk_object_set_data_full (GTK_OBJECT (dialog), "dialog->cancel_button",
							  dialog->cancel_button, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->cancel_button);
	GTK_WIDGET_SET_FLAGS (dialog->cancel_button, GTK_CAN_DEFAULT);

	gtk_widget_grab_default (dialog->ok_button);

	gtk_signal_connect_after (GTK_OBJECT (dialog->ok_button), "clicked", on_ok, dialog);
	gtk_signal_connect_after (GTK_OBJECT (dialog->cancel_button), "clicked", on_cancel, dialog);
}




/***********************************
 * Public functions
 ***********************************/

GtkType
gnome_cmd_prepare_xfer_dialog_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdPrepareXferDialog",
			sizeof (GnomeCmdPrepareXferDialog),
			sizeof (GnomeCmdPrepareXferDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}


GtkWidget*
gnome_cmd_prepare_xfer_dialog_new (GnomeCmdFileList *from,
								   GnomeCmdFileSelector *to)
{
	gchar *dest_str;
	GnomeCmdConnection *con;
	GnomeCmdPrepareXferDialog *dialog = gtk_type_new (gnome_cmd_prepare_xfer_dialog_get_type ());

	dialog->src_files = gnome_cmd_file_list_get_selected_files (from);
	gnome_cmd_file_list_ref (dialog->src_files);
	dialog->dest_dir = gnome_cmd_file_selector_get_directory (to);
	dialog->src_fl = from;

	con = gnome_cmd_file_selector_get_connection (to);
	dialog->dest_baseuri = gnome_cmd_connection_get_baseuri (con);

	if (g_list_length (dialog->src_files) == 1) {
		const gchar *path = gnome_cmd_dir_get_path (dialog->dest_dir);
		GnomeCmdFile *finfo = (GnomeCmdFile*)dialog->src_files->data;

		if (path[strlen (path) - 1] == '/')
			dest_str = g_strdup_printf ("%s%s", path, finfo->info->name);
		else
			dest_str = g_strdup_printf ("%s/%s", path, finfo->info->name);
	}
	else
		dest_str = g_strdup (gnome_cmd_dir_get_path (dialog->dest_dir));
	
	gtk_entry_set_text (GTK_ENTRY (dialog->dest_dir_entry), dest_str);
	g_free (dest_str);

	gtk_widget_grab_focus (GTK_WIDGET (dialog->dest_dir_entry));
	
	return GTK_WIDGET (dialog);
}
