-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/tools/files.adb,v $
--  Description     : Export the contents of a data base                     --
--  Author          : Michael Erdmann                                        --
--  Created         : 5.4.2003                                               --
--  Last Modified By: $Author: merdmann $                                    --
--  Last Modified On: $Date: 2004/03/09 09:56:56 $                           --
--  Status          : $State: Exp $                                          --
--                                                                           --
--  Copyright (C) 2000 - 2003 Michael Erdmann                                --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--                                                                           --
--  Export the contents of a table into a simple file format. This file      --
--  format i intended for importing or processing by other gnade tools       --
--                                                                           --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  References                                                               --
--  ==========                                                               --
--  None                                                                     --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Strings.Unbounded;     use Ada.Strings.Unbounded;
with Ada.Streams;               use Ada.Streams;
with Ada.Streams.Stream_IO;     use Ada.Streams.Stream_IO;
with Tools;                     use Tools;
with Util;                      use Util;

package body Files is

   -----------------
   -- Header_Type --
   -----------------
   type Header_Type is record
          Contents      : Contents_Type := Null_Contents_Type;
	  Major_Version : Integer := 1;
	  Minor_Version : Integer := 0;
	  Patch_Level   : Integer := 0;
          Length        : Integer := 0;
      end record;

   -----------------
   -- Object_Data --
   -----------------
   type Object_Data_Type is record
         Contents : Contents_Type := Null_Contents_Type;
         Stream   : Stream_Access ;
         File     : File_Type;
         Header   : Header_Type;
         Vector   : String_Array_Access;
	 Field    : String_Array_Access;
      end record;

   --********************************************************************--
   --                I N T E R N A L    M E T H O D S                    --
   --********************************************************************--

   ----------
   -- Info --
   ----------
   procedure Info(
      S : in String ) is
   begin
      if Opt_Verbose then
         Report(S);
      end if;
   end Info;

   -----------------
   -- Read_Header --
   -----------------
   procedure Read_Header(
      This : in out Object ) is
      -- Read in the header information of an import/export file
      Data : Object_Data_Access renames This.Data;
      H    : Header_Type        renames Data.Header;
   begin
      H := Header_Type'Input( Data.Stream );
      Info("Contents: " & Contents_Type'Image( H.Contents ) );

      case H.Contents is
         when Database_Export =>
            Data.Vector := new String_Array( 1..H.Length );
            Data.Field  := new String_Array( 1..H.Length );

            for i in 1..H.Length loop
               Data.Field(i) := Unbounded_String'Input( Data.Stream );
            end loop;
         when others =>
            null;
      end case;

   end Read_Header;

   ------------------
   -- Write_Header --
   ------------------
   procedure Write_Header(
      This : in out Object ) is
      -- this procedure write the correct header layout based on the
      -- document type. This procedure has allays be in line with the
      -- Read_Header procedure,
      Data : Object_Data_Access renames This.Data;
      H    : Header_Type        renames Data.Header;
   begin
      H.Contents := Data.Contents;

      Header_Type'Output( Data.Stream, H );

      case H.Contents is
         when Database_Export =>
            for i in 1..H.Length loop
               Unbounded_String'Output( Data.Stream, Data.Field(i) );
            end loop;
         when others =>
            null;
      end case;
   end Write_Header;

   --********************************************************************--
   --                                                                    --
   --********************************************************************--

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      This : in out Object ) is
      -- Initialize the object
   begin
      This.Data := new Object_Data_Type;
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      This : in out Object ) is
      -- finalize the object and all related resources.
      Data : Object_Data_Access renames This.Data;
   begin
      if Data /= null then
         null;
      end if;
      Data := null;
   end Finalize;

   ------------
   -- Fields --
   ------------
   procedure Fields(
      This : in out Object;
      Info : in String_Array_Access ) is
      -- set the fields
      Data : Object_Data_Access renames This.Data;
      H    : Header_Type renames Data.Header;
   begin
      H.Length := Info.all'Length;

      Data.Field  := new String_Array( 1..H.Length );
      Data.Vector := new String_Array( 1..H.Length );

      for I in Info.all'Range loop
         Data.Field(I)  := Info(I);
	 Data.Vector(I) := Null_Unbounded_String;
      end loop;

   end Fields;

   ------------
   -- Fields --
   ------------
   function Fields(
      This : in Object ) return String_Array_Access is
      Data : Object_Data_Access renames This.Data;
   begin
      return Data.Field;
   end Fields;

   -----------------
   -- End_of_File --
   -----------------
   function End_Of_File(
      This : in Object ) return Boolean is
      Data : Object_Data_Access renames This.Data;
   begin
      return Stream_IO.End_Of_File( Data.File ) ;
   end End_Of_File;

   ----------
   -- Open --
   ----------
   procedure Open(
      This     : in out Object;
      Name     : in String;
      Contents : Contents_Type := Database_Export  ) is
      -- Open the file to be imported. The file has to exist, otherwise
      -- an exception is raised.
      Data : Object_Data_Access renames This.Data;
   begin
      Open( File => Data.File, Name => Name, Mode => In_File );
      Data.Stream := Stream_IO.Stream( Data.File );

      Read_Header( This );

      Data.Contents := Data.Header.Contents;
      if Data.Contents /= Contents then
         raise Unexpected_File_Type;
      end if;

   exception
      when Others =>
         raise File_Not_Existing;
   end Open;

   ------------
   -- Create --
   ------------
   procedure Create(
      This     : in out Object;
      Name     : in String := "";
      Contents : in Contents_Type := Database_Export ) is
      -- create a new file based on the file object
      Data : Object_Data_Access renames This.Data;
   begin
      Stream_IO.Create( File =>Data.File, Name=>Name, Mode=>Stream_IO.Out_File);
      Data.Stream   := Stream_IO.Stream( Data.File );
      Data.Contents := Contents;

      Write_Header( This );
   end Create;

   -----------
   -- Close --
   -----------
   procedure Close(
      This : in out Object ) is
      -- close the file
      Data : Object_Data_Access renames This.Data;
   begin
      Stream_IO.Close( Data.File );
   end Close;

   ----------
   -- Read --
   ----------
   function Read(
      This : in  Object ) return String_Array_Access is
      -- Read a record from the import file and return the associated vector
      Data : Object_Data_Access renames This.Data;
      H    : Header_Type renames Data.Header;
   begin
      for I in 1..H.Length loop
            Data.Vector(I) := Unbounded_String'Input( Data.Stream );
      end loop;

      return Data.Vector;
   exception
      when Others =>
         return Null_String_Array;
   end Read;

   -----------
   -- Write --
   -----------
   procedure Write(
      This  : in out Object;
      Items : in String_Array_Access ) is
      Data  : Object_Data_Access renames This.Data;
      H     : header_Type renames Data.Header;
   begin
      for i in 1..H.Length loop
         Unbounded_String'Output( Data.Stream, Items(i) );
      end loop;
   end Write;

end Files;
