-------------------------------------------------------------------------------
--									     --
--  Filename        : $Source: /cvsroot/gnade/gnade/tools/exp.adb,v $
--  Description     : Export the contents of a data base                     --
--  Author          : Michael Erdmann                                        --
--  Created         : 5.4.2003                                               --
--  Last Modified By: $Author: merdmann $				     --
--  Last Modified On: $Date: 2003/05/10 16:24:03 $			     --
--  Status          : $State: Exp $					     --
--									     --
--  Copyright (C) 2000 - 2003 Michael Erdmann                                --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--                                                                           --
--  Export the contents of a table into a simple file format. This file      --
--  format i intended for importing or processing by other gnade tools       --
--       								     --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  References                                                               --
--  ==========                                                               --
--  None                                                                     --
--                                                                           --
-------------------------------------------------------------------------------
with Ada.Text_IO;		use Ada.Text_IO;
with Ada.Strings.Unbounded;     use Ada.Strings.Unbounded;
with Ada.Command_Line;          use Ada.Command_Line;
with Files;                     use Files;	       
with ODBC;      		use ODBC;
with Tools;			use Tools;
with Util;                      use Util;

procedure Exp is
   Version     : constant String := "$Id: exp.adb,v 1.4 2003/05/10 16:24:03 merdmann Exp $";

   DB          : ODBC.Object;
   Login       : Unbounded_String;
   Pwd         : Unbounded_String;
   Source      : Unbounded_String;
   Argc        : Natural := 1;
   Export_File : Files.Object;
 
   ---------------------------
   -- Export_Arguments_Type --
   ---------------------------
   type Export_Arguments_Type is record
         Name      : Unbounded_String := Null_Unbounded_String;
	 File_Name : Unbounded_String := Null_Unbounded_String;
      end record;

   Null_Argument : constant Export_Arguments_Type := (
        Name      => Null_Unbounded_String,
        File_Name => Null_Unbounded_String);

   Export_Args : Export_Arguments_Type ;

   ------------
   -- Tables --
   ------------
   package Tables is new Bucket( 
      Bucket_type => Export_Arguments_Type );

   ----------
   -- Help --
   ----------
   procedure Help is
   begin
      Put_Line("usage : " );
      Put_Line("    exp -u <login> [ -l <passwd> ] <source> table [-f name] ... " );
      Put_Line("");
   end Help;

   ------------
   -- Export --
   ------------
   function Export(
      Name : in Unbounded_String; 
      T    : in Unbounded_String ) return Natural is 
      -- export the contents of a table into the given output stream 
      R    : Result_Record := null;
      N    : Natural := 0;
   begin
      Query( DB, "select * from " & To_String(T) );
      R := Fetch( DB );

      -- prepare the header information from the query
      declare
         F : String_Array_Access := new String_Array( 1..R'Length );
      begin
         for i in R'Range loop
     	    F(i) := To_Unbounded_String(Field_Name( DB, i ));
         end loop;
	 Fields( Export_File, F );
      end ;
        
      Create( Export_File, To_String(Name), Contents => Database_Export );

      while R /= null loop
         N := N + 1;
	 Write( Export_File, R );

	 Free( R );
	 R := Fetch( DB );	
      end loop;

      Close( Export_File );
      
      return N; 
   end Export;

begin 
   Initialize(DB);

   -- Handle the command line arguments
   while Argc in 1..Argument_Count loop
      declare
         Arg : constant String := Argument( Argc );
      begin
         if Arg = "-u" then
            Argc := Argc + 1;
            Login := To_Unbounded_String( Argument( Argc ) );
         elsif Arg = "-p" then
            Argc := Argc + 1;
            Pwd := To_Unbounded_String( Argument( Argc ) );
	 elsif Arg = "-f" then
	    Argc := Argc + 1;
	    Export_Args.File_Name := To_Unbounded_String( Argument( Argc ) );
         elsif Arg = "-h" then
	    Help;
	    return;
	 elsif Arg = "-v" then
	    Opt_Verbose := True;
	 elsif Arg = "-q" then
	    Util.Quiet  := True;
	    Opt_Verbose := False;
	 else
	    if Source = Null_Unbounded_String then
	    	Source := To_Unbounded_String( Arg );
	    else 
	        Export_Args.Name := To_Unbounded_String( Arg );
		Tables.Add( Export_Args );

		Export_Args := Null_Argument;
	    end if;
	 end if;
         Argc := Argc + 1;
      end;
   end loop;

   if not Quiet then
      Report("GNADE Table Export, Version " & Tools.Version);
      Report("Copyright (C) 2003, Michael Erdmann (michael.erdmann@snafu.de)");
      Report("");
   end if;     

   if Source = Null_Unbounded_String then
      Error("source not specified");
      Error("");
      Help;
      Set_Exit_Status(Failure);
      return;
   end if;

 
   -- connect to the data base
   Authorization( DB, To_String(Login), To_String(Pwd) );
   Data_Source( DB, To_String(Source) );
   Connect( DB );
   
   Export_Args := Tables.Get( First => True );
   loop
      declare
         File_Name : Unbounded_String renames Export_Args.File_Name;
	 Tab_Name  : Unbounded_String renames Export_Args.Name;
	 Records   : Natural := 0;
      begin
         Initialize( Export_File ); 

   	 if File_Name = Null_Unbounded_String then
	    File_Name := Export_Args.Name & ".exp";
	 end if;

         Records := Export(  File_Name, Tab_Name );    
	 Report("Table " & To_String( Tab_Name ) & ", " &
	        "exported " & Natural'Image(Records) & " record(s)." );

         Finalize( Export_File );
      exception
         when Others =>
	    Error("Exception when exporting " & To_String(Tab_Name) );
	    Report_DB_Error( DB );
	    Close( Export_File );
	    Finalize( Export_File );
      end;
      exit when Tables.End_of_Bucket;

      Export_Args := Tables.Get;
   end loop;
 
   --Finalize( DB );
   Set_Exit_Status( Success );

exception
   when Others =>
      Report_DB_Error( DB );   
      Set_Exit_Status( Failure ); 
end Exp;

