------------------------------------------------------------------------------
--  Thin Ada95 binding to OCI (Oracle Call Interface)                    --
--  Copyright (C) 2000-2003 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-debug_allocation.adb,v 1.6 2004/01/28 11:25:22 vagul Exp $

with Ada.Text_IO;

with Interfaces.C;

with OCI.Thread;

package body OCI.Thick.Debug_Allocation is

   type Integer_Parameters is array (0 .. 16) of Integer;

   type Integer_Parameters_Access is access all Integer_Parameters;

   function Callback
     (ctxp       : in DVoid;
      hndlp      : in OCIHandle;
      htype      : in Ub4;
      fcode      : in Ub4;
      when_call  : in Ub1;
      returnCode : in SWord;
      errnop     : access Sb4;
      arglist    : in Integer_Parameters_Access)
      return     SWord;
   pragma Convention (C, Callback);

   OCIHandleAlloc_FCode     : constant := 2;
   OCIHandleFree_FCode      : constant := 3;
   OCIDescriptorAlloc_FCode : constant := 4;
   OCIDescriptorFree_FCode  : constant := 5;

   HandleAlloc     : aliased constant String := "HandleAlloc";
   HandleFree      : aliased constant String := "HandleFree";
   DescriptorAlloc : aliased constant String := "DescriptorAlloc";
   DescriptorFree  : aliased constant String := "DescriptorFree";

   type String_Access is access constant String;

   Started : Boolean := False;

   subtype Func_Code is Integer
      range OCIHandleAlloc_FCode .. OCIDescriptorFree_FCode;

   subtype Handle_Type is Integer range OCI_HTYPE_FIRST .. OCI_HTYPE_LAST;

   subtype Descriptor_Type is Integer range OCI_DTYPE_FIRST .. OCI_DTYPE_LAST;

   subtype Call_Counter is Natural;

   Func_Names : array (Func_Code) of String_Access :=
     (OCIHandleAlloc_FCode     => HandleAlloc'Access,
      OCIHandleFree_FCode      => HandleFree'Access,
      OCIDescriptorAlloc_FCode => DescriptorAlloc'Access,
      OCIDescriptorFree_FCode  => DescriptorFree'Access);

   Statistic : array (Func_Code) of Call_Counter := (others => 0);

   Error_Statistic : array (Func_Code) of Call_Counter := (others => 0);

   Handle_Allocated : array (Handle_Type) of Call_Counter := (others => 0);

   Handle_Freed : array (Handle_Type) of Call_Counter := (others => 0);

   Descriptor_Allocated : array (Descriptor_Type) of Call_Counter
     := (others => 0);

   Descriptor_Freed : array (Descriptor_Type) of Call_Counter
     := (others => 0);

   protected Oper is
      procedure Increment (Item : in out Call_Counter);
   end Oper;

   --------------
   -- Callback --
   --------------

   function Callback
     (ctxp       : in DVoid;
      hndlp      : in OCIHandle;
      htype      : in Ub4;
      fcode      : in Ub4;
      when_call  : in Ub1;
      returnCode : in SWord;
      errnop     : access Sb4;
      arglist    : in Integer_Parameters_Access)
      return     SWord
   is
      use Interfaces.C;
      Alloc_type : Integer renames arglist (2);
      Free_type : Integer renames arglist (1);

      procedure Warning (Output : String) renames Ada.Text_IO.Put_Line;

   begin
      if errnop.all = OCI_SUCCESS then
         Oper.Increment (Statistic (Func_Code (fcode)));
      else
         Oper.Increment (Error_Statistic (Func_Code (fcode)));
      end if;

      case fcode is
      when OCIHandleAlloc_FCode =>
         if Alloc_Type in Handle_Type then
            Oper.Increment (Handle_Allocated (Alloc_type));
         else
            Warning ("Alloc unrecognized handlee ");
         end if;
      when OCIHandleFree_FCode =>
         if Free_Type in Handle_Type then
            Oper.Increment (Handle_Freed (Free_type));
         else
            Warning ("Free unrecognized handle.");
         end if;
      when OCIDescriptorAlloc_FCode =>
         if Alloc_type in Descriptor_Type then
            Oper.Increment (Descriptor_Allocated (Alloc_type));
         else
            Warning ("Alloc unrecognized descriptor.");
         end if;
      when OCIDescriptorFree_FCode  =>
         if Free_Type in Descriptor_Type then
            Oper.Increment (Descriptor_Freed (Free_type));
         else
            Warning ("Free unrecognized descriptor.");
         end if;
      when others =>
         Warning ("Callback for unregistered function call.");
      end case;

      return OCI_CONTINUE;
   end Callback;

   ----------
   -- Oper --
   ----------

   protected body Oper is

      procedure Increment (Item : in out Call_Counter) is
      begin
         Item := Item + 1;
      end Increment;

   end Oper;

   ----------------
   -- Print_Info --
   ----------------

   procedure Print_Info is
      use Ada.Text_IO;
   begin
      for I in Func_Code loop
         Put_Line (Func_Names (I).all &
            Integer'Image (Statistic (I)) &
            Integer'Image (Error_Statistic (I)));
      end loop;

      Put_Line ("Handles     "
        & Integer'Image
            (Statistic (OCIHAndleAlloc_FCode)
             - Statistic (OCIHAndleFree_FCode)));

      Put_Line ("Descriptors "
        & Integer'Image
            (Statistic (OCIDescriptorAlloc_FCode)
             - Statistic (OCIDescriptorFree_FCode)));

      for I in Handle_Type loop
        Put_Line
           (Integer'Image (I) & '.' &
            Integer'Image (Handle_Allocated (I)) & '-' &
            Integer'Image (Handle_Freed (I)) & '=' &
            Integer'Image (Handle_Allocated (I) - Handle_Freed (I)));
      end loop;
      for I in Descriptor_Type loop
        Put_Line (
            Integer'Image (I) & '.' &
            Integer'Image (Descriptor_Allocated (I)) & '-' &
            Integer'Image (Descriptor_Freed (I)) & '=' &
            Integer'Image (Descriptor_Allocated (I) - Descriptor_Freed (I)));
      end loop;
   end Print_Info;

   -----------
   -- Start --
   -----------

   procedure Start is
      Rc    : SWord;
      Env   : constant OCIEnv   := Thread.Environment;
      Error : constant OCIError := Thread.Error;
   begin
      if Started then
         raise Already_Started;
      else
         for J in Func_Code'Range loop
            --  Very strange, but GNAT 3.15p not optimized executable
            --  raises PROGRAM_ERROR : EXCEPTION_ACCESS_VIOLATION
            --  here if we do not put dummy output before
            --  Lib.OCIUserCallbackRegister call.

            Ada.Text_IO.Put_Line (Func_Names (J).all);

            Rc := Lib.OCIUserCallbackRegister
                    (hndlp     => OCIHandle (Env),
                     htype     => OCI_HTYPE_ENV,
                     ehndlp    => Error,
                     callback  => Callback'Address,
                     ctxp      => Empty_Handle,
                     fcode     => Ub4 (J),
                     when_call => OCI_UCBTYPE_EXIT);
            Check_Error (Rc);
         end loop;
      end if;
   end Start;

end OCI.Thick.Debug_Allocation;
