/* gmpc-last.fm (GMPC plugin)
 * Copyright (C) 2006-2009 Qball Cow <qball@sarine.nl>
 * Project homepage: http://gmpcwiki.sarine.nl/
 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libmpd/debug_printf.h>
#include <gmpc/plugin.h>
#include <gmpc/gmpc_easy_download.h>
#include <gmpc/metadata.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <config.h>

#define LASTFM_API_KEY "ec1cdd08d574e93fa6ef9ad861ae795a" 
#define LASTFM_API_ROOT "http://ws.audioscrobbler.com/2.0/"



static char *__lastfm_art_get_artist_similar(char *nartist);


static int lastfm_get_enabled()
{
	return cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "enable", TRUE);
}
static void lastfm_set_enabled(int enabled)
{
	cfg_set_single_value_as_int(config, "cover-lastfm", "enable", enabled);
}

static int lastfm_fetch_cover_priority(void){
	return cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "priority", 80);
}
static void lastfm_fetch_cover_priority_set(int priority){
	cfg_set_single_value_as_int(config, "cover-lastfm", "priority", priority);
}

static xmlNodePtr get_first_node_by_name(xmlNodePtr xml, gchar *name) {
    if(name == NULL) return NULL;
	if(xml) {
		xmlNodePtr c = xml->xmlChildrenNode;
		for(;c;c=c->next) {
			if(c->name && xmlStrEqual(c->name, (xmlChar *) name))
				return c;
		}
	}
	return NULL;
}

static char *__lastfm_art_xml_get_image(char *data,int size, char *type)
{
	if(size  == 0)
		return NULL;
	/* Hack to check for xml file */
    if(data == NULL) return NULL;
	if(data[0] != '<')
		return NULL;
	gchar *url = NULL;
	xmlDocPtr doc = xmlParseMemory(data,size);
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
        if(root)
        {
            /* loop through all albums */
            xmlNodePtr cur = get_first_node_by_name(root,type);
            if(cur)
            {
                int type = 0;
                xmlNodePtr cur2 = cur->xmlChildrenNode;
                for(;cur2;cur2 = cur2->next)
                {
                    if(cur2->name)
                    {
                        if (xmlStrEqual(cur2->name, (xmlChar *)"image"))
                        {

                            xmlChar *temp = xmlGetProp(cur2, (xmlChar *)"size");
                            if(temp)
                            {
                                /**
                                 * We want large image, but if that is not available, get the medium one 
                                 */
                                if(xmlStrEqual(temp, (xmlChar *)"medium") && type < 2)
                                {
                                    xmlChar *xurl = xmlNodeGetContent(cur2);
                                    url = g_strdup((char *)xurl);
                                    xmlFree(xurl);
                                    type = 2;
                                }else if(xmlStrEqual(temp, (xmlChar *)"large"))
                                {
                                    xmlChar *xurl = xmlNodeGetContent(cur2);
                                    if(url) g_free(url);
                                    url = g_strdup((char *)xurl);
                                    xmlFree(xurl);
                                    type = 3;
                                }
                                xmlFree(temp);
                            }
                        }
                    }
                }
            }
        }
		xmlFreeDoc(doc);
		
	}
	return url;
}

static int __lastfm_art_get_artist_image(mpd_Song *song,char *nartist, char **path)
{

	gmpc_easy_download_struct data= {NULL, 0,-1,NULL, NULL};
	int found = META_DATA_UNAVAILABLE;
	char furl[1024];


	snprintf(furl,1024,LASTFM_API_ROOT"?method=artist.getinfo&artist=%s&api_key=%s", nartist,LASTFM_API_KEY);

	if(gmpc_easy_download(furl, &data))
	{
		char *url = __lastfm_art_xml_get_image(data.data, data.size, "artist");
		gmpc_easy_download_clean(&data);
		if(url && url[0] != '\0' && strstr(url, "noartist_") == NULL) 
		{
			gmpc_easy_download(url, &data);
			if(data.size){
				FILE *fp = NULL;
				char *imgpath = NULL;
				imgpath = gmpc_get_metadata_filename(META_ARTIST_ART, song, "jpg");
				fp = g_fopen(imgpath, "wb");
				if(fp)
				{
                    size_t size;
    			    size = fwrite(data.data, sizeof(char), data.size,fp);
                    if(size != data.size){
                        debug_printf(DEBUG_ERROR, "Something went wrong writing image to file");
                    }
					fclose(fp);
				}
				*path = imgpath;
				gmpc_easy_download_clean(&data);
				found = META_DATA_AVAILABLE;
                debug_printf(DEBUG_INFO,"Found cover art, returning\n");
			}
		}
		if(url)g_free(url);

	}

	return found;
}
/*
 * Get 20 artists
 */
static char *__lastfm_art_xml_get_artist_similar(char *data,int size)
{
	char *retv = NULL;
	GString *string;
	if(size  == 0)
		return NULL;
    if(data == NULL) return NULL;
	if(data[0] != '<')
		return NULL;
	string = g_string_new("");
	xmlDocPtr doc = xmlParseMemory(data,size);
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
		xmlNodePtr cur = get_first_node_by_name(root, "similarartists");
        if(cur)
        {
            xmlNodePtr cur2 = cur->xmlChildrenNode;
            for(;cur2;cur2=cur2->next)
            {
                if(xmlStrEqual(cur2->name, (xmlChar *)"artist"))
                {
                    xmlNodePtr cur3 = cur2->xmlChildrenNode;
                    for(;cur3;cur3=cur3->next)
                    {
                        if(xmlStrEqual(cur3->name, (xmlChar *)"name"))
                        {
                            xmlChar *temp = xmlNodeGetContent(cur3);
                            g_string_append_printf(string, "%s\n", temp);
                            xmlFree(temp);
                        }
                    }
                }
            }
        }
		xmlFreeDoc(doc);
		
	}
	if(string->len>0)
		retv = string->str;
	g_string_free(string, FALSE);
	return retv;
}

/*
 * Get's similar list 
 */
gchar *__lastfm_art_get_artist_similar(char *nartist)
{

	gmpc_easy_download_struct data= {NULL, 0,-1,NULL, NULL};
	char furl[1024];
	char *artist = gmpc_easy_download_uri_escape(nartist);
	snprintf(furl,1024,LASTFM_API_ROOT"?method=artist.getsimilar&artist=%s&api_key=%s", artist,LASTFM_API_KEY);
	debug_printf(DEBUG_INFO, "furl: %s\n", furl);
	g_free(artist);
	if(gmpc_easy_download(furl, &data))
	{
		char *result = __lastfm_art_xml_get_artist_similar(data.data,data.size);
		gmpc_easy_download_clean(&data);
		return result;
	}
	return NULL;
}
/*
 * Get 20Songs 
 */
static char *__lastfm_art_xml_get_song_similar(char *data,int size)
{
	char *retv = NULL;
	GString *string;
	if(size  == 0)
		return NULL;
    if(data == NULL) return NULL;
    if(data[0] != '<')
		return NULL;
	string = g_string_new("");
	xmlDocPtr doc = xmlParseMemory(data,size);
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
		xmlNodePtr cur = get_first_node_by_name(root, "similartracks");
        if(cur)
        {
            xmlNodePtr cur2 = cur->xmlChildrenNode;
            for(;cur2;cur2=cur2->next)
            {
                if(xmlStrEqual(cur2->name, (xmlChar *)"track"))
                {
                    xmlNodePtr cur3 = cur2->xmlChildrenNode;
                    xmlChar *artist = NULL;
                    xmlChar *title = NULL;
                    for(;cur3;cur3=cur3->next)
                    {
                        if(xmlStrEqual(cur3->name, (xmlChar *)"name"))
                        {
                            xmlChar *temp = xmlNodeGetContent(cur3);
                            title = temp; 
                        }
                        else if (xmlStrEqual(cur3->name, (xmlChar *)"artist")) 
                        {
                            xmlNodePtr cur4 = get_first_node_by_name(cur3, "name");
                            if(cur4){
                                xmlChar *temp = xmlNodeGetContent(cur4);
                                artist = temp; 
                            }
                        }
                    }
                    if(artist && title) {

                        g_string_append_printf(string, "%s::%s\n", artist, title);
                    }
                    if(artist) xmlFree(artist);
                    if(title) xmlFree(title);
                }
            }
        }
		xmlFreeDoc(doc);
		
	}
	if(string->len>0)
		retv = string->str;
	g_string_free(string, FALSE);
	return retv;
}

/*
 * Get's similar list 
 */
static gchar *__lastfm_art_get_song_similar(char *nartist, char *ntitle)
{

	gmpc_easy_download_struct data= {NULL, 0,-1,NULL, NULL};
	char furl[1024];
	char *artist = gmpc_easy_download_uri_escape(nartist);
    char *title =  gmpc_easy_download_uri_escape(ntitle);
	snprintf(furl,1024,LASTFM_API_ROOT"?method=track.getsimilar&artist=%s&track=%s&api_key=%s", artist,title,LASTFM_API_KEY);
	debug_printf(DEBUG_INFO, "furl: %s\n", furl);
	g_free(artist);
    g_free(title);
	if(gmpc_easy_download(furl, &data))
	{
		char *result = __lastfm_art_xml_get_song_similar(data.data,data.size);
		gmpc_easy_download_clean(&data);
		return result;
	}
	return NULL;
}



/** 
 * Get album image
 */
static int __lastfm_art_get_album_image(mpd_Song *song,char *artist, char *album, char **path)
{

	gmpc_easy_download_struct data= {NULL, 0,-1,NULL, NULL};
	int found = META_DATA_UNAVAILABLE;
	char furl[1024];
	snprintf(furl,1024,LASTFM_API_ROOT"?method=album.getinfo&artist=%s&album=%s&api_key=%s", artist,album,LASTFM_API_KEY);
	debug_printf(DEBUG_INFO, "furl: %s\n", furl);
	if(gmpc_easy_download(furl, &data))
	{
		char *url = __lastfm_art_xml_get_image(data.data, data.size,"album");
		gmpc_easy_download_clean(&data);
		if(url && url[0] != '\0' &&  strstr(url, "noartist_") == NULL && strstr(url, "noimage") == NULL) 
		{
			gmpc_easy_download(url, &data);
			/* Don't use images under 900 bytes */
			if(data.size > 900){
				FILE *fp = NULL;
				char *imgpath = NULL;
				imgpath = gmpc_get_metadata_filename(META_ALBUM_ART, song, "jpg");
				fp = g_fopen(imgpath, "wb");
				if(fp)
				{
					size_t size = fwrite(data.data, sizeof(char), data.size,fp);
                    if(size != data.size)
                    {
                        debug_printf(DEBUG_ERROR, "Something went wrong writing image to file");
                    }
					fclose(fp);
				}
				*path = imgpath;
				gmpc_easy_download_clean(&data);
				found = META_DATA_AVAILABLE;
				debug_printf(DEBUG_INFO, "Found album art for: %s %s\n", artist, album);
			}
		}
		if(url)g_free(url);
	}
	return found;
}
/**
 * Get artist info 
 */


static char * __lastfm_art_xml_get_artist_bio(char *data , int size )
{
	xmlDocPtr doc = xmlParseMemory(data,size);
	char *info=NULL;
	if(doc)
	{
		xmlNodePtr root = xmlDocGetRootElement(doc);
		xmlNodePtr bio = get_first_node_by_name(get_first_node_by_name(get_first_node_by_name(root,"artist"),"bio"),"content");
		if(bio)
		{
			xmlChar *temp = xmlNodeGetContent(bio);
			info = g_strdup((char *)temp);
			xmlFree(temp);
		}
	}
	xmlFreeDoc(doc);
	
	return info;
}
static gchar *__lastfm_art_get_artist_txt(mpd_Song *song)
{

	gmpc_easy_download_struct data= {NULL, 0,-1,NULL, NULL};
	char furl[1024];
	char *artist = gmpc_easy_download_uri_escape(song->artist);
	/* Create the uri */
	snprintf(furl,1024, LASTFM_API_ROOT"?method=artist.getinfo&artist=%s&api_key=%s", artist,LASTFM_API_KEY);
	/* free artist name */
	g_free(artist);
	/* Download */
	if(gmpc_easy_download(furl, &data))
	{
		gchar *imgpath = NULL;
		/* Parse the data and try to get bio. */
		gchar *result = __lastfm_art_xml_get_artist_bio(data.data,data.size);
		/* Cleanup download struct */
		gmpc_easy_download_clean(&data);
		/* If there is a result store it. */
		if(result)
		{
			FILE *fp = NULL;
            imgpath = gmpc_get_metadata_filename(META_ARTIST_TXT, song, "jpg");
			fp = g_fopen(imgpath, "w");
			if(fp)
			{
				int j=0,depth=0;;
				for(j=0; j < strlen(result);j++)
				{
					if(result[j] == '<') depth++;
					else if(result[j] == '>' && depth) depth--;
					else if(depth == 0)
					{
						/* Quick and dirty html unescape*/
						if(strncasecmp(&result[j], "&lt;", 4) == 0){
							fputc('<', fp);
							j+=3;
						}else if (strncasecmp(&result[j], "&gt;", 4) == 0){
							fputc('>', fp);
							j+=3;
						}else if (strncasecmp(&result[j], "&quot;", 6) == 0){
							fputc('"', fp);
							j+=5;
						}else if (strncasecmp(&result[j], "&amp;", 5) == 0){
							fputc('&', fp);
							j+=4;
						}
						else
							fputc(result[j],fp);
					}
				}
				fclose(fp);
			}
			/* If we failed to open the file, return NULL */
			else 
			{
				g_free(imgpath);
				imgpath = NULL;
			}
			g_free(result);
		}
		return imgpath;
	}
	return NULL;
}

/** other */
static int lastfm_fetch_cover_album_art(mpd_Song *song, char **path)
{
	int retval;
	char *artist = gmpc_easy_download_uri_escape(song->artist);
	char *album = gmpc_easy_download_uri_escape(song->album);
	debug_printf(DEBUG_INFO, "Trying to fetch: %s:%s\n", artist, album);
	retval = __lastfm_art_get_album_image(song,artist,album, path);
	g_free(artist);
	g_free(album);
	return retval;
}

static int lastfm_fetch_cover_art(mpd_Song *song, char **path)
{
	int retval;
	char *artist = gmpc_easy_download_uri_escape(song->artist);
	retval = __lastfm_art_get_artist_image(song,artist,path);
	g_free(artist);
	return retval;
}

static int lastfm_fetch_get_image(mpd_Song *song,MetaDataType type, char **path)
{
	int result = 0;
	if(song->artist == NULL || lastfm_get_enabled() == FALSE)
	{
		return META_DATA_UNAVAILABLE;
	}

	if(type == META_ARTIST_ART  && cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-art-artist", TRUE))
	{
		/** So we don't do to much queries a second */
		/* this is nicer then waiting one second, because now the result is directly available, only a new query < 1 second is delayed
		 */
		result = lastfm_fetch_cover_art(song, path);
		return result;
	}
	else if (type == META_ALBUM_ART && cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-art-album", TRUE))
	{
		result = lastfm_fetch_cover_album_art(song, path);
		return result;
	}
	else if (type == META_ARTIST_SIMILAR && cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-similar-artist", TRUE))
	{
		*path = __lastfm_art_get_artist_similar(song->artist);
		if(*path)
		{
			return META_DATA_AVAILABLE;	
		}
		if(*path) g_free(*path);
	}
	else if (type == META_SONG_SIMILAR && cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-similar-song", TRUE))
	{
        if(song->artist && song->title)
        {
            *path = __lastfm_art_get_song_similar(song->artist,song->title);
            if(*path)
            {
                return META_DATA_AVAILABLE;	
            }
            if(*path) g_free(*path);
        }
	}
	/* Fetch artist info */
	else if (type == META_ARTIST_TXT && cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-biography-artist", TRUE))
	{
		/* Wait for the max 1x a second lock. */
		*path = __lastfm_art_get_artist_txt(song);
		/* If there is a path, return it */
		if(*path)
		{
			return META_DATA_AVAILABLE;	
		}
	}
	return META_DATA_UNAVAILABLE; 
}	

/**
 * Preferences 
 */
static void pref_destroy(GtkWidget *con)
{
    GtkWidget *child = gtk_bin_get_child(GTK_BIN(con));
    if(child) {
        gtk_container_remove(GTK_CONTAINER(con), child);
    }
}

static void pref_enable_fetch(GtkWidget *con, gpointer data)
{
    MetaDataType type = GPOINTER_TO_INT(data);
    int state = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(con));
    switch(type) {
        case META_ARTIST_ART:
            cfg_set_single_value_as_int(config, "cover-lastfm", "fetch-art-artist",state); 
            break;
        case META_ALBUM_ART:
            cfg_set_single_value_as_int(config, "cover-lastfm", "fetch-art-album",state); 
            break;
        case META_ARTIST_SIMILAR:
            cfg_set_single_value_as_int(config, "cover-lastfm", "fetch-similar-artist",state); 
            break;
        case META_SONG_SIMILAR:
            cfg_set_single_value_as_int(config, "cover-lastfm", "fetch-similar-song",state); 
            break;                                                                                 
        case META_ARTIST_TXT:
            cfg_set_single_value_as_int(config, "cover-lastfm", "fetch-biography-artist",state); 
            break;                                                                                 
        default:
            break;
    }
}


static void pref_construct(GtkWidget *con)
{
    GtkWidget *frame,*enable_ck,*vbox;
    GtkWidget *a_a_ck, *a_b_ck, *a_s_ck,*c_a_ck, *s_s_ck;

    /**
     * Enable/Disable checkbox
     */
    frame = gtk_frame_new("");
    gtk_label_set_markup(GTK_LABEL(gtk_frame_get_label_widget(GTK_FRAME(frame))), "<b>Fetch</b>");
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_NONE);
    vbox = gtk_vbox_new(FALSE,6);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);
    gtk_container_add(GTK_CONTAINER(frame), vbox);

    /* Fetch artist art */
    a_a_ck = gtk_check_button_new_with_label("artist images");    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(a_a_ck),
        cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-art-artist", TRUE)); 
    gtk_box_pack_start(GTK_BOX(vbox), a_a_ck, FALSE, TRUE, 0);
    g_signal_connect(G_OBJECT(a_a_ck), "toggled", G_CALLBACK(pref_enable_fetch), GINT_TO_POINTER(META_ARTIST_ART));

    /* Fetch artist text*/
    a_b_ck = gtk_check_button_new_with_label("artist biography");    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(a_b_ck),
        cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-biography-artist", TRUE)); 
    gtk_box_pack_start(GTK_BOX(vbox), a_b_ck, FALSE, TRUE, 0);
    g_signal_connect(G_OBJECT(a_b_ck), "toggled", G_CALLBACK(pref_enable_fetch), GINT_TO_POINTER(META_ARTIST_TXT));

    /* Fetch album art */
    a_s_ck = gtk_check_button_new_with_label("similar artists");    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(a_s_ck),
        cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-similar-artist", TRUE)); 
    gtk_box_pack_start(GTK_BOX(vbox), a_s_ck, FALSE, TRUE, 0);
    g_signal_connect(G_OBJECT(a_s_ck), "toggled", G_CALLBACK(pref_enable_fetch), GINT_TO_POINTER(META_ARTIST_SIMILAR));

    /* Fetch artist art */
    c_a_ck = gtk_check_button_new_with_label("album cover");    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(c_a_ck),
        cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-art-album", TRUE)); 
    gtk_box_pack_start(GTK_BOX(vbox), c_a_ck, FALSE, TRUE, 0);
    g_signal_connect(G_OBJECT(c_a_ck), "toggled", G_CALLBACK(pref_enable_fetch), GINT_TO_POINTER(META_ALBUM_ART));


    /* Fetch artist art */
    s_s_ck = gtk_check_button_new_with_label("similar songs");    
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(s_s_ck),
        cfg_get_single_value_as_int_with_default(config, "cover-lastfm", "fetch-similar-song", TRUE)); 
    gtk_box_pack_start(GTK_BOX(vbox), s_s_ck, FALSE, TRUE, 0);
    g_signal_connect(G_OBJECT(s_s_ck), "toggled", G_CALLBACK(pref_enable_fetch), GINT_TO_POINTER(META_SONG_SIMILAR));



    if(!lastfm_get_enabled()) {
        gtk_widget_set_sensitive(GTK_WIDGET(vbox), FALSE);
    }

    gtk_widget_show_all(frame);
    gtk_container_add(GTK_CONTAINER(con), frame);
}

gmpcPrefPlugin pref = {
    .construct      = pref_construct,
    .destroy        = pref_destroy
};

/**
 * Metadata Plugin
 */
gmpcMetaDataPlugin lf_cover = {
	.get_priority   = lastfm_fetch_cover_priority,
    .set_priority   = lastfm_fetch_cover_priority_set,
	.get_image      = lastfm_fetch_get_image
};

int plugin_api_version = PLUGIN_API_VERSION;

gmpcPlugin plugin = {
	.name           = "Last FM metadata fetcher",
	.version        = {PLUGIN_MAJOR_VERSION,PLUGIN_MINOR_VERSION,PLUGIN_MICRO_VERSION},
	.plugin_type    = GMPC_PLUGIN_META_DATA,
	.metadata       = &lf_cover,
    .pref           = &pref,
	.get_enabled    = lastfm_get_enabled,
	.set_enabled    = lastfm_set_enabled
};
