/*
** Copyright (C) 20 June 2000 Jonas Munsin <jmunsin@iki.fi>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
** 
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
** 
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software 
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>

#include "common_gtk.h"
#include "multisession.h"
#include "cdrecord_options.h"
#include "iso9660.h"
#include "isosize.h"
#include "isoinfo.h"
#include "globals.h"

static void add_info_item(char *expl, char *val, GtkWidget *clist) {
	char *text[2];

	text[0] = expl;
	text[1] = val;
	gtk_clist_append(GTK_CLIST(clist), text);
	g_free(val);
}

static void add_time_item(char *s, char t[], GtkWidget *clist) {
	struct iso9660_ltime *time;

	time = (struct iso9660_ltime *) t;
	add_info_item(s, g_strdup_printf("%c%c%c%c-%c%c-%c%c %c%c:%c%c:%c%c (%i)",
				time->lt_year[0], time->lt_year[1], time->lt_year[2], time->lt_year[3],
				time->lt_month[0], time->lt_month[1],
				time->lt_day[0],
				time->lt_day[1],
				time->lt_hour[0],
				time->lt_hour[1],
				time->lt_minute[0],
				time->lt_minute[1],
				time->lt_second[0],
				time->lt_second[1],
				time->lt_gmtoff[0]), clist);
}

/* Ok, there is probably some really good crossplatform reasons for the casting
 * macros in Joergs isosize.c, but it's late and I don't have time to get them
 * 100% correct... There is definetely something wrong here, but I'm too tired.
 * Btw, I have no idea what most of these fields are, I just picked
 * out those that seemed to be useful...
 */
static void add_cd_info(struct iso9660_pr_voldesc *vp, GtkWidget *clist) {
	int bigendian;
	int x = 1;

	if(*(char *)&x == 1)
		bigendian = FALSE;
	else
		bigendian = TRUE;

	/* type */
	add_info_item("id", g_strndup(vp->vd_id, 5), clist);
	add_info_item("version", g_strdup_printf("%u", GET_UBYTE(vp->vd_version)), clist);
	/* unused1 */
	add_info_item("system_id", g_strndup(vp->vd_system_id, 32), clist);
	add_info_item("volume_id", g_strndup(vp->vd_volume_id, 32), clist);
	/* unused2 */
	if (bigendian)
		add_info_item("volume_space_size", g_strdup_printf("%lu",
					GET_INT(&vp->vd_volume_space_size[0])), clist);
	else
		add_info_item("volume_space_size", g_strdup_printf("%lu",
					GET_INT(&vp->vd_volume_space_size[4])), clist);
	/* unused3 */
	if (bigendian)
		add_info_item("volume_set_size", g_strdup_printf("%u",
					GET_SHORT(&vp->vd_volume_set_size[0])), clist);
	else
		add_info_item("volume_set_size", g_strdup_printf("%u",
					GET_SHORT(&vp->vd_volume_set_size[2])), clist);
	if (bigendian)
		add_info_item("volume_seq_number", g_strdup_printf("%u",
					GET_SHORT(&vp->vd_volume_seq_number[0])), clist);
	else
		add_info_item("volume_seq_number", g_strdup_printf("%u",
					GET_SHORT(&vp->vd_volume_seq_number[2])), clist);
	if (bigendian)
		add_info_item("lbsize", g_strdup_printf("%u",
					GET_SHORT(&vp->vd_lbsize[0])), clist);
	else
		add_info_item("lbsize", g_strdup_printf("%u",
					GET_SHORT(&vp->vd_lbsize[2])), clist);
	if (bigendian) {
		unsigned long size = GET_INT(&vp->vd_volume_space_size[0])*GET_SHORT(&vp->vd_lbsize[0]);
		add_info_item("size", g_strdup_printf("%lu B, %lu KB, %lu MB",
					size, size/1024, size/(1024*1024)), clist);
	} else {
		unsigned long size = GET_INT(&vp->vd_volume_space_size[4])*GET_SHORT(&vp->vd_lbsize[2]);
		add_info_item("size", g_strdup_printf("%lu B, %lu KB, %lu MB",
					size, size/1024, size/(1024*1024)), clist);
	}
	/* path_table_size */
	/* pos_path_table_l */
	/* opt_pos_path_table_l */
	/* pos_path_table_m */
	/* opt_pos_path_table_m */
	/* root_dir */
	add_info_item("volume_set_id", g_strndup(vp->vd_volume_set_id, 128), clist);
	add_info_item("publisher_id", g_strndup(vp->vd_publisher_id, 128), clist);
	add_info_item("data_preparer_id", g_strndup(vp->vd_data_preparer_id, 128), clist);
	add_info_item("application_id", g_strndup(vp->vd_application_id, 128), clist);
	add_info_item("copyr_file_id", g_strndup(vp->vd_copyr_file_id, 37), clist);
	add_info_item("abstr_file_id", g_strndup(vp->vd_abstr_file_id, 37), clist);
	add_info_item("bibl_file_id", g_strndup(vp->vd_bibl_file_id, 37), clist);
	add_time_item("create_time", vp->vd_create_time, clist);
	add_time_item("mod_time", vp->vd_mod_time, clist);
	add_time_item("expiry_time", vp->vd_expiry_time, clist);
	add_time_item("effective_time", vp->vd_effective_time, clist);
	/* file_struct_vers */
	/* reserved1 */
	/* application_use */
	/* fill */
}

static void open_cd_info(struct iso9660_pr_voldesc *vp, char *path) {
	GtkWidget *clist, *win, *vbox, *scrolled_window;
	char *title;

	win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_default_size(GTK_WINDOW(win), 600, 360);

	title = g_strdup_printf(_("Info about: %s"), path);
	gtk_window_set_title(GTK_WINDOW(win), title);
	g_free(title);
	gtk_signal_connect_object(GTK_OBJECT(win), "destroy",
			(GtkSignalFunc)gtk_widget_destroy, GTK_OBJECT(win));

	vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_container_add(GTK_CONTAINER(win), vbox);
	gtk_widget_show(vbox);
 
	clist = gtk_clist_new(2);
	gtk_widget_show(clist);
	gtk_clist_set_column_title(GTK_CLIST(clist), 0, _("Description"));
	gtk_clist_set_column_title(GTK_CLIST(clist), 1, _("Value"));
	gtk_clist_set_column_width(GTK_CLIST(clist), 0, 100);
	gtk_clist_set_column_width(GTK_CLIST(clist), 1, 100);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
	gtk_clist_column_titles_passive(GTK_CLIST(clist));
	gtk_clist_column_titles_show(GTK_CLIST(clist));

	scrolled_window = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(scrolled_window);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_window),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_window), clist);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled_window, TRUE, TRUE, 0);

	add_cd_info(vp, clist);

	gtk_widget_show(win);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 1, TRUE);
}

void verbose_info_clicked(GtkWidget *widget, gpointer data) {
	struct iso9660_pr_voldesc *vp;
	char *from;
	int fd;

	if (1 == GPOINTER_TO_INT(data))
		from = g_strdup(gtk_entry_get_text(GTK_ENTRY(iso_device_path)));
	else
		from = g_strdup(gtk_entry_get_text(GTK_ENTRY(image_path)));
	if (-1 == (fd = open(from, O_RDONLY | O_NONBLOCK))) {
		char *m;
		m = g_strdup_printf(_("Error opening %s"), from);
		alert_user_of_error(m);
		g_free(m);
		g_free(from);
		return;
	}
	if (read_will_block_cd(fd)) {
		/* Reading audio cds or empty stations will block, so lets not */
		char *m;
		m = g_strdup_printf(_("Did not try to read ISO information because:\n%s does not seem to be a valid data CD or file"), from);
		alert_user_of_error(m);
		g_free(m);
		g_free(from);
		return;
	}

	vp = get_voldesc(fd);
	close(fd);
	if (NULL == vp) {
		char *m;
		m = g_strdup_printf(_("Error reading iso header from %s"), from);
		alert_user_of_error(m);
		g_free(m);
		g_free(from);
		return;
	}

	open_cd_info(vp, from);

	free(vp);
	g_free(from);
}

