/***************************************************************************

  pcode_temp.h

  The p-code disassembler

  (c) 2000-2004 Benot Minisini <gambas@freesurf.fr>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 1, or (at your option)
  any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***************************************************************************/


#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "gb_common.h"
#include "gb_error.h"

#ifdef PROJECT_COMP
#include "gb_limit.h"
#include "gbc_compile.h"
#endif

#include "gb_pcode.h"

/*#define DEBUG*/

PUBLIC short PCODE_dump(short addr, PCODE *code)
{
  static const char *op_comp[] = { "=", "<>", ">", "<=", "<", ">=", "IS", "CASE" };
  static const char *op_arith[] = { "+", "-" , "*", "/", "NEG", "\\", "MOD", "POW", "AND", "OR", "NOT", "XOR", "&", "LIKE", "&/" };

  int j;
  unsigned short op;
  unsigned short digit;
  long value;
  #ifdef PROJECT_COMP
  long index;
  TABLE *table;
  bool trans;
  #endif
  int ncode;

  op = *code;

  switch (op & 0xFF00)
  {
    case C_PUSH_UNKNOWN: case C_POP_UNKNOWN:
    case C_PUSH_INTEGER:
    case C_JUMP: case C_JUMP_IF_TRUE: case C_JUMP_IF_FALSE:
    case C_NEXT: case C_JUMP_NEXT:
    case C_TRY:

      ncode = 2;
      break;

    case C_PUSH_LONG:

      ncode = 3;
      break;

    default:

      ncode = 1;
  }

  printf("%04d : ", addr);

  for (j = 0; j < ncode; j++)
    printf(" %04hX", code[j]);

  for (j = 0; j < (3 - ncode); j++)
    printf("     ");

  printf("  ");

  digit = (op >> 12);
  value = op & 0xFFF;
  if (value >= 0x800) value |= 0xFFFFF000;

  switch (digit)
  {
    #ifdef PROJECT_COMP

    case 0xF:
      printf("PUSH QUICK %d", (short)value);
      break;

    case 0xE:
      printf("PUSH CONST %d", (short)value);

      switch(JOB->class->constant[value].type.t.id)
      {
        case T_STRING:
          table = JOB->class->string;
          trans = FALSE;
          break;

        case T_CSTRING:
          table = JOB->class->string;
          trans = TRUE;
          break;

        default:
          table = JOB->class->table;
          trans = FALSE;
          break;
      }

      if (trans)
        printf(" (\"%s\")", TABLE_get_symbol_name(table, JOB->class->constant[value].value));
      else
        printf(" \"%s\"", TABLE_get_symbol_name(table, JOB->class->constant[value].value));

      break;

    case 0xD: case 0xC:
      printf("%s %s ", (digit == 0xD ? "POP" : "PUSH"), (value & 0x800) ? "STATIC" : "DYNAMIC");
      index = ((value & 0x800) ?  JOB->class->stat[value & 0x7FF].index : JOB->class->dyn[value & 0x7FF].index);
      printf("%s", TABLE_get_symbol_name(JOB->class->table, index));
      break;

    case 0xB:
      if (value & 0x800)
      {
        printf("PUSH FUNCTION ");
        index = JOB->class->function[value & 0x7FF].name;
      }
      else
      {
        printf("PUSH CLASS ");
        index = JOB->class->class[value];
      }
      printf("%s", TABLE_get_symbol_name(JOB->class->table, index));
      break;

    #else

    case 0xF:
      printf("PUSH QUICK %d", (short)value);
      break;

    case 0xE:
      printf("PUSH CONST %d", (short)value);
      break;

    case 0xD:
      printf("POP %s %ld", (value & 0x800) ? "STATIC" : "DYNAMIC", value & 0x7FF);
      break;

    case 0xC:
      printf("PUSH %s %ld", (value & 0x800) ? "STATIC" : "DYNAMIC", value & 0x7FF);
      break;

    case 0xB:
      printf("PUSH %s %ld", (value & 0x800) ? "FUNCTION" : "CLASS", value & 0x7FF);
      break;

    #endif

    default:

      digit = op & 0xFF00;
      value = op & 0xFF;
      if (value >= 0x80) value |= 0xFFFFFF00;

      if (digit >= C_PUSH_LOCAL && digit < C_QUIT)
        printf("PUSH ");
      else if (digit >= C_POP_LOCAL && digit < C_BREAK)
        printf("POP ");

      switch(digit)
      {
        case C_PUSH_LOCAL: case C_POP_LOCAL:
          if (value >= 0)
            printf("LOCAL %d", (short)value);
          else
            printf("PARAM %d", (short)value);
          break;

        case C_POP_CTRL:
          printf("CTRL %d", (short)value);
          break;

        case C_POP_OPTIONAL:
          printf("OPTIONAL %d", (short)value);
          break;

        case C_PUSH_UNKNOWN: case C_POP_UNKNOWN:
          value = code[1];
          #ifdef PROJECT_COMP
          printf("UNKNOWN %s", TABLE_get_symbol_name(JOB->class->table, JOB->class->unknown[value]));
          #else
          printf("UNKNOWN %d", (short)value);
          #endif

          break;

        case C_PUSH_SPECIAL:
          printf("SPECIAL %d", (short)value);
          break;

        case C_PUSH_EXTERN:
          printf("EXTERN %d", (short)value);
          break;

        case C_PUSH_EVENT:
          printf("EVENT %d", (short)value);
          break;

        case C_PUSH_ARRAY: case C_POP_ARRAY:
          printf("ARRAY (%d)", (short)value);
          break;

        case C_CALL:
          printf("CALL ");
          if (value & CODE_CALL_VARIANT)
            printf("VARIANT ");
          if (value & CODE_CALL_VOID)
            printf("VOID ");

          printf("(%d)", (short)value & 0x3F);
          break;

        case C_PUSH_INTEGER:
          value = code[1];
          printf("PUSH SHORT %d", (short)value);
          break;

        case C_PUSH_LONG:
          value = *((long *)&code[1]);
          printf("PUSH INTEGER %ld", value);
          break;

        case C_PUSH_BOOLEAN:
          printf("PUSH %s", value ? "TRUE" : "FALSE");
          break;

        case C_PUSH_ME:
          printf("PUSH ME");
          break;

        case C_PUSH_LAST:
          printf("PUSH LAST");
          break;

        case C_PUSH_NULL:
          printf("PUSH NULL");
          break;

        case C_JUMP: case C_JUMP_IF_TRUE: case C_JUMP_IF_FALSE:
          value = code[1];
          printf("JUMP%s %04d",
            (digit == C_JUMP ? "" :
             digit == C_JUMP_IF_TRUE ? " IF TRUE" :
             digit == C_JUMP_IF_FALSE ? " IF FALSE" : "??"),
            (short)(addr + value + 2));
          break;

        case C_JUMP_FIRST:

          printf("JUMP FIRST LOCAL %d", (short)value);
          break;

        case C_JUMP_NEXT:

          printf("JUMP NEXT ");

          value = code[1];
          printf("%04d ", (short)(addr + value + 2));

          break;

        case C_FIRST:

          printf("ENUM FIRST LOCAL %d", (short)value);
          break;

        case C_NEXT:

          printf("ENUM NEXT ");
          if (value & 0xFF)
            printf("DROP ");
          value = code[1];
          printf("%04d ", (short)(addr + value + 2));
          break;

        case C_DROP:
          printf("DROP (%d)", (short)value);
          break;

        case C_DUP:
          printf("DUP");
          break;

        case C_NEW:
          printf("NEW ");
          if (value & CODE_NEW_EVENT)
            printf("EVENT ");
          if (value & CODE_NEW_ARRAY)
            printf("ARRAY ");
          printf("(%d)", (short)value & 0x3F);
          break;

        case C_BREAK:
          printf("BREAK");
          break;

        case C_RETURN:
          printf("RETURN (%d)", (short)value);
          break;

        case C_QUIT:
          if (value)
            printf("STOP");
          else
            printf("QUIT");
          break;

        case C_PUSH_CHAR:
          printf("PUSH CHAR (%d)", (short)value);
          break;

        case C_PUSH_VOID:
          printf("PUSH VOID");
          break;

        case C_EVENT:
          if (value == 2)
            printf("STOP EVENT");
          else
            printf("EVENT %s", value ? "ON" : "OFF");
          break;

        case C_PUSH_RETURN:
          printf("PUSH RETURN");
          break;

        case C_TRY:
          value = code[1];
          printf("TRY %04d", (short)(addr + value + 2));
          break;

        case C_END_TRY:
          printf("END TRY");
          break;

        case C_CATCH:
          printf("CATCH");
          break;

        default:
          digit = (digit >> 8);
          if (digit >= CODE_FIRST_SUBR)
          {
            #ifdef PROJECT_COMP
            printf("SUBR %s ", SUBR_get_from_opcode(digit - CODE_FIRST_SUBR, (short)value & 0x3F)->name);
            #else
            printf("SUBR #%d ", digit - CODE_FIRST_SUBR);
            #endif

            if (value & CODE_CALL_VARIANT)
              printf("VARIANT ");
            if (value & CODE_CALL_VOID)
              printf("VOID ");

            printf("(%d)", (short)value & 0x3F);
          }
          else if (digit >= 0x28 && digit <= 0x2F)
            printf("%s (%d)", op_comp[digit - 0x28], (short)value);
          else if (digit >= 0x30 && digit <= 0x3E)
            printf("%s (%d)", op_arith[digit - 0x30], (short)value);
          else
            printf("ILLEGAL");
      }
  }

  printf("\n");
  return ncode;
}


