#include <string.h>
#include <ctype.h>

#include <gdk/gdk.h>
#include <gtk/gtkplug.h>

#include <gtkplugin.h>

#include <debug.h>
#include <util.h>
#include <conversation.h>

#include "ge_blist.h"
#include "state_ui.h"
#include "state.h"

GHashTable *encryption_state_table; /* name -> EncryptionState */

/* Helper function: */
static void reset_state_struct(const GaimAccount* account,
                               const gchar* name,
                               EncryptionState* state);

void GE_state_init() {
   encryption_state_table = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, g_free);
}

void GE_state_delete() {
   g_hash_table_destroy(encryption_state_table);
}

EncryptionState* GE_get_state(GaimConversation* conv) {
   if (conv == NULL) return NULL;

   EncryptionState *state = gaim_conversation_get_data(conv, "GE_state");

   if (state == NULL) {
      state = g_malloc(sizeof(EncryptionState));
      gaim_conversation_set_data(conv, "GE_state", state);

      // should probably change this to use some prefs info rather than our buddy list stuff
      state->outgoing_encrypted =
         GE_get_buddy_default_autoencrypt(gaim_conversation_get_account(conv),
                                          gaim_conversation_get_name(conv));
      
      if (conv == NULL) {
         return NULL;
      }

      state->has_been_notified =
         GE_get_default_notified(gaim_conversation_get_account(conv),
                                 gaim_conversation_get_name(conv));
      
      state->incoming_encrypted = FALSE;
      state->is_capable = FALSE;      
   }
   return state;
}

void GE_reset_state(GaimConversation* conv) {
   EncryptionState *state;

   if (conv == NULL) return;

   state = GE_get_state(conv);

   reset_state_struct(gaim_conversation_get_account(conv), gaim_conversation_get_name(conv), state);
}

void GE_free_state(GaimConversation* conv) {
   EncryptionState *state;

   if (conv == NULL) return;

   state = GE_get_state(conv);

   if (state) {
      g_free(state);
   }
}

static void reset_state_struct(const GaimAccount* account, const gchar* name,
                               EncryptionState* state) {

   state->outgoing_encrypted = GE_get_buddy_default_autoencrypt(account, name);
   state->has_been_notified = GE_get_default_notified(account, name);

   state->incoming_encrypted = FALSE;
   state->is_capable = FALSE;
}

void GE_set_tx_encryption(GaimConversation* conv, gboolean do_encrypt) {
   EncryptionState *state;

   if (conv == NULL) return;

   state = GE_get_state(conv);

   if (state->outgoing_encrypted != do_encrypt) {
      state->outgoing_encrypted = do_encrypt;
      GE_sync_state(conv);
   }
}

void GE_set_capable(GaimConversation* conv, gboolean cap) {
   EncryptionState *state;

   if (conv == NULL) return;

   state = GE_get_state(conv);

   if (state->is_capable != cap) {
      state->is_capable = cap;
      if (state->outgoing_encrypted == FALSE) {
         GE_sync_state(conv);
      }
   }
}

void GE_set_rx_encryption(GaimConversation *conv, gboolean incoming_encrypted) {
   EncryptionState *state;

   if (conv == NULL) return;

   state = GE_get_state(conv);

   if (state->incoming_encrypted != incoming_encrypted) {
      state->incoming_encrypted = incoming_encrypted;
      GE_set_rx_encryption_icon(conv, incoming_encrypted);
   }
}

gboolean GE_get_tx_encryption(GaimConversation *conv) {
   EncryptionState *state = GE_get_state(conv);

   if (state == NULL) return FALSE;

   return state->outgoing_encrypted;
}

void GE_sync_state(GaimConversation *conv) {
   EncryptionState *state;

   if (conv == NULL) return;

   state = GE_get_state(conv);
   GE_set_rx_encryption_icon(conv, state->incoming_encrypted);
   GE_set_tx_encryption_icon(conv, state->outgoing_encrypted, state->is_capable);
}

gboolean GE_has_been_notified(GaimConversation *conv) {
   EncryptionState *state = GE_get_state(conv);

   if (state == NULL) return TRUE;

   return state->has_been_notified;
}

void GE_set_notified(GaimConversation *conv, gboolean new_state) {
   EncryptionState *state;

   if (conv == NULL) return;

   state = GE_get_state(conv);

   state->has_been_notified = new_state;
}


gboolean GE_get_default_notified(const GaimAccount *account, const gchar* name) {
   /* Most protocols no longer have a notify message, since they don't do HTML */
   
   /* The only special case here is Oscar/TOC: If the other user's name is all */
   /* digits, then they're ICQ, so we pretend that we already notified them    */

   const char* protocol_id = gaim_account_get_protocol_id(account);
   
   if (strcmp(protocol_id, "prpl-toc") == 0 || strcmp(protocol_id, "prpl-oscar") == 0) {
      
      while(*name != 0) {
         if (!isdigit(*name++)) {
            /* not ICQ */
            return FALSE;
         }
      }
      /* Hrm.  must be ICQ */
      return TRUE;
   }

   /* default to notifying next time, if protocol allows it */

   return FALSE;
}

   
