/* GAdmin-OpenVPN - An easy to use GTK+ frontend for the openvpn server.
 * Copyright (C) 2008 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include <gtk/gtk.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "widgets.h"
#include "gettext.h"
#include "reread_conf.h"
#include "show_info.h"
#include "functions.h"
#include "system_defines.h"
#include "apply_server_settings.h"
#include "allocate.h"
#include "commented.h"
#include "commands.h"
#include "populate_server_settings.h"
#include "populate_conf_tab.h"


extern int activated;

/* Check input length validity */
int has_value(gchar *input)
{
    int have_value = 0;

    if( input!=NULL && strlen(input) > 2 )
      have_value = 1;

    return have_value;
}


/* Make firestarter/iptables accept tap+ interfaces */
void add_firestarter_user_pre()
{
    FILE *fp;
    long file_size = 0;
    char *line;
    gchar *conf;
    int have_tap_lines = 0;

    if((fp=fopen(FIRESTARTER_USER_PRE, "r"))==NULL)
    {
	/* Dont show a popup */
	return;
    }
    fseek(fp, 0, SEEK_END);
    file_size = ftell(fp);
    rewind(fp);

    line = allocate(file_size);

    if( file_size > 1 )
    while(fgets(line, file_size, fp)!=NULL)
    {
        if( line==NULL || strlen(line) < 10 || commented(line) )
            continue;

        if( cmplowercase(line, "-a input -i tap+ -j accept") )
          have_tap_lines++;

        if( cmplowercase(line, "-a output -o tap+ -j accept") )
          have_tap_lines++;

        if( cmplowercase(line, "-a forward -i tap+ -j accept") )
          have_tap_lines++;
    }
    fclose(fp);
    free(line);

    /* Dont add the tap+ lines if they already exist */
    if( have_tap_lines >= 3 )
      return;

    /* Append the lines. Does Firestarter allow broadcast traffic ? Fix if it doesnt  */
    conf = g_strdup_printf("$IPT -A INPUT -i tap+ -j ACCEPT\n$IPT -A OUTPUT -o tap+ -j ACCEPT\n$IPT -A FORWARD -i tap+ -j ACCEPT\n");
    if((fp=fopen(FIRESTARTER_USER_PRE, "a"))==NULL)
    {
	/* Show a popup */
	g_free(conf);
	return;
    }
    fputs(conf, fp);
    fclose(fp);
    g_free(conf);
}



void apply_server_settings(struct w *widgets)
{
    /* Writes the server configuration and sysinit script.
       Writes a client configuration suited for this server. */
    FILE *fp;
    int i = 0;
    gint active_index;
    gchar *info, *cmd;
    gchar *sysinit_script_path, *sysinit_start_script;
    gchar *conf, *openvpn_server_conf, *openvpn_client_conf;
    gchar *firestarter_start, *firestarter_stop;
    gchar *non_firestarter_start, *non_firestarter_stop, *lan_net_addr;

    /* Entries */
    G_CONST_RETURN gchar *external_iface_name;
    G_CONST_RETURN gchar *external_iface_address;
    G_CONST_RETURN gchar *lan_iface_name;
    G_CONST_RETURN gchar *lan_iface_address;
    G_CONST_RETURN gchar *lan_subnet_mask;
    G_CONST_RETURN gchar *lan_broadcast_address;
    G_CONST_RETURN gchar *client_address_range;

    /* Spin buttons */
    G_CONST_RETURN gchar *server_port;
    G_CONST_RETURN gchar *max_clients;
    G_CONST_RETURN gchar *log_level;
    G_CONST_RETURN gchar *ping_interval;
    G_CONST_RETURN gchar *ping_timeout;
    G_CONST_RETURN gchar *server_user;
    G_CONST_RETURN gchar *server_group;

    /* Combos */
    gchar *proto=NULL, *tap_dev=NULL, *require_auth=NULL;
    gchar *cipher=NULL, *client_to_client=NULL;
    gchar *redirect_gateway=NULL, *compression=NULL;
    gchar *share_internet=NULL, *client_auth_line=NULL;

    /* Certificate settings are saved when generating certificates */

    /* Entries */
    external_iface_name    = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[0]));
    external_iface_address = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[1]));
    lan_iface_name         = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[2]));
    lan_iface_address      = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[3]));
    lan_subnet_mask        = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[4]));
    lan_broadcast_address  = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[5]));
    client_address_range   = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[6]));
    server_user            = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[7]));
    server_group           = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_entry[8]));

    /* Spin buttons */
    server_port   = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[0]));
    max_clients   = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[1]));
    log_level     = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[2]));
    ping_interval = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[3]));
    ping_timeout  = gtk_entry_get_text(GTK_ENTRY(widgets->server_set_spinbutton[4]));


    /* Add the server group if it doesnt exist */
    if( ! group_exists(server_group) )
    {
	cmd = g_strdup_printf("%s '%s'", ADDGROUP, server_group);
	if( ! run_command(cmd) )
	{
	    info = g_strdup_printf(_("Error adding group: %s\n"), server_group);
	    show_info(info);
	    g_free(info);
	    return;
	}
	g_free(cmd);
    }
    /* Add the server user to the server group if it doesnt exist */
    if( ! user_exists(server_user) )
    {
	cmd = g_strdup_printf("%s '%s' -g '%s' -d /dev/null -c '%s' -s /dev/null", ADDUSER, server_user, server_group, "GAdmin-OpenVPN-Server");
	if( ! run_command(cmd) )
	{
	    info = g_strdup_printf(_("Failed adding user: %s\n"), server_user);
	    show_info(info);
	    g_free(info);
	    return;
	}
	g_free(cmd);
    }


    /* Combos */

    /* Protocol */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[0]));
    if( active_index == 0 )
      proto = g_strdup_printf("proto tcp");
    else
      proto = g_strdup_printf("proto udp");

    /* Tap device */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[1]));
    if( active_index == 0 )
      tap_dev = g_strdup_printf("tap0");
    if( active_index == 1 )
      tap_dev = g_strdup_printf("tap1");
    if( active_index == 2 )
      tap_dev = g_strdup_printf("tap2");
    if( active_index == 3 )
      tap_dev = g_strdup_printf("tap3");

    /* Require authentication. Also adds username-as-common-name so per user config is possible. */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[2]));
    if( active_index == 0 )
    {
        require_auth = g_strdup_printf("\n"); /* "No" */
        client_auth_line = g_strdup_printf("\n");
    }
    else
    {
        require_auth = g_strdup_printf("plugin %s login\nusername-as-common-name\n", OPENVPN_AUTH_PAM_PLUGIN); /* "Yes" */
        client_auth_line = g_strdup_printf("auth-user-pass %s/client/passfile\n", OPENVPN_SYSCONF_DIR);
    }

    /* Cipher */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[3]));
    if( active_index == 0 )
      cipher = g_strdup_printf("cipher BF-CBC\n");       /* BlowFish */
    if( active_index == 1 )
      cipher = g_strdup_printf("cipher AES-128-CBC\n");  /* AES 128 */
    if( active_index == 2 )
      cipher = g_strdup_printf("cipher AES-256-CBC\n");  /* AES 256 */
    if( active_index == 3 )
      cipher = g_strdup_printf("cipher DES-EDE3-CBC\n"); /* Triple-DES */

    /* Client-to-client */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[4]));
    if( active_index == 0 )
      client_to_client = g_strdup_printf("\n"); /* "No" */
    else
      client_to_client = g_strdup_printf("client-to-client\n");

    /* Redirect-gateway */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[5]));
    if( active_index == 0 )
      redirect_gateway = g_strdup_printf("\n");
    else
      redirect_gateway = g_strdup_printf("push \"redirect-gateway\"\n");

    /* Compression */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[6]));
    if( active_index == 0 )
      compression = g_strdup_printf("\n");
    else
      compression = g_strdup_printf("comp-lzo\n");


    /* Chroot is standard */


    /* Add Firestarter iptables tap+ settings to the user-pre file if it exists */
    add_firestarter_user_pre();


    /* Sysinit script parts to add if internet sharing is specified. */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[7]));
    if( active_index == 1 )
    {
	/* Get the LAN network address. Snip at the last dot. */
	lan_net_addr = g_strdup_printf("%s", lan_iface_address);
	for(i=strlen(lan_net_addr)-1; lan_net_addr[i]!='\0'; i--)
	if( lan_net_addr[i]=='.')
	{
	    lan_net_addr[i+1]='0';
	    lan_net_addr[i+2]='\0';
	    break;
	}
	share_internet = g_strconcat(
	IP_FORWARD_CMD_START, "\n",
	"iptables -t nat -A POSTROUTING -s ", lan_net_addr, "/", lan_subnet_mask, " -o eth0\n",
	NULL);

	g_free(lan_net_addr);
    }
    else
      share_internet = g_strdup_printf("%s\n", IP_FORWARD_CMD_STOP);


    /* Sysinit script parts to add if Firestarter is used. */
    active_index = gtk_combo_box_get_active(GTK_COMBO_BOX(widgets->server_set_combo[8]));
    if( active_index == 1 )
    {
	firestarter_start = g_strconcat(
	"# Start Firestarter if its not running.\n",
	"if [ ! -e /var/lock/subsys/firestarter -o -e /var/lock/firestarter ]; then\n",
	"    ", FIRESTARTER_BINARY, " -s\n",
	"fi\n",
	NULL);

	firestarter_stop = g_strconcat(
	"# Stop Firestarter if its running.\n",
	"if [ -e /var/lock/subsys/firestarter -o -e /var/lock/firestarter ]; then\n",
	"    ", FIRESTARTER_BINARY, " -p\n",
	"fi\n",
	NULL);
	
	non_firestarter_start = g_strdup_printf("\n");
	non_firestarter_stop  = g_strdup_printf("\n");
    }
    else
    {
	/* Instead of firestarter configuring iptables, we do it. */
	non_firestarter_start = g_strconcat(
	"# Run some iptable rules for the LAN network and VPN-clients.\n",
	"iptables -A INPUT  -i tap+ -j ACCEPT\n",
	"iptables -A OUTPUT -o tap+ -j ACCEPT\n",
	"iptables -A FORWARD -i tap+ -j ACCEPT\n",
	share_internet,
	NULL);

	/* Disallow input and output on tap interfaces.
	   Restart the LAN network interface. Doesnt work otherwise. */
	non_firestarter_stop = g_strconcat(
	"iptables -D INPUT  -i tap+ -j ACCEPT\n",
	"iptables -D OUTPUT -o tap+ -j ACCEPT\n",
	"iptables -D FORWARD -i tap+ -j ACCEPT\n",
	"ifconfig ", lan_iface_name, "down\n",
	"ifconfig ", lan_iface_name, "up\n",
	share_internet,
	NULL);
	/* Dont remove any internet sharing. */
	
	firestarter_start = g_strdup_printf("\n");
	firestarter_stop  = g_strdup_printf("\n");
    }
    g_free(share_internet);

    /* Write the openvpn server configuration file:
       /etc/gadmin-openvpn/server/gadmin-openvpn-server.conf */
    openvpn_server_conf = g_strdup_printf("%s/server/gadmin-openvpn-server.conf", OPENVPN_SYSCONF_DIR);
    if((fp=fopen(openvpn_server_conf, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Can not write openvpn server configuration here: \n%s\n"), openvpn_server_conf);
	show_info(info);
	g_free(info);
	g_free(openvpn_server_conf);
	g_free(proto);
	g_free(tap_dev);
	g_free(require_auth);
	g_free(cipher);
	g_free(client_to_client);
	g_free(redirect_gateway);
	g_free(compression);
	g_free(firestarter_start);
	g_free(firestarter_stop);
	g_free(non_firestarter_start);
	g_free(non_firestarter_stop);
	g_free(client_auth_line);
        return;
    }

    conf = g_strconcat(
    "local ", external_iface_address, "\n",
    "port ", server_port, "\n",
    proto, "\n",
    "dev ", tap_dev, "\n",
    "ca   certs/cacert.pem\n",
    "cert certs/cert.pem\n",
    "key  certs/key.pem\n",
    "dh   certs/dh.pem\n",
    "tls-auth certs/ta.key 0\n",
    "tls-server\n",
    require_auth,
    cipher,
    client_to_client,
    redirect_gateway,
    compression,
    "chroot ", OPENVPN_SYSCONF_DIR, "/server\n",
    "server-bridge ", lan_iface_address," ", lan_subnet_mask," ", client_address_range,"\n",
    "duplicate-cn\n",
    "keepalive ", ping_interval, " ", ping_timeout, "\n",
    "max-clients ", max_clients, "\n",
    "user ", server_user, "\n",
    "group ", server_group, "\n",
    "persist-key\n",
    "persist-tun\n",
    "log    /etc/gadmin-openvpn/server/openvpn-server.log\n",
    "status /etc/gadmin-openvpn/server/openvpn-server-status.log\n",
    "verb ", log_level, "\n",
    "mute 20\n",
    NULL);

    fputs(conf, fp);
    fclose(fp);
    g_free(openvpn_server_conf);


    /* Write the client configuration */
    openvpn_client_conf = g_strdup_printf("%s/server/client/gadmin-openvpn-client.conf", OPENVPN_SYSCONF_DIR);
    if((fp=fopen(openvpn_client_conf, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Can not write openvpn client configuration here: \n%s\n"), openvpn_client_conf);
	show_info(info);
	g_free(info);
	g_free(openvpn_client_conf);
	g_free(proto);
	g_free(tap_dev);
	g_free(require_auth);
	g_free(cipher);
	g_free(client_to_client);
	g_free(redirect_gateway);
	g_free(compression);
	g_free(firestarter_start);
	g_free(firestarter_stop);
	g_free(non_firestarter_start);
	g_free(non_firestarter_stop);
	g_free(client_auth_line);
        return;
    }

    conf = g_strconcat(
    "client\n",
    "tls-client\n",
    "remote ", external_iface_address, " ", server_port, "\n",
    proto, "\n",
    "dev ", tap_dev, "\n",
    "ca   certs/cacert.pem\n",
    "cert certs/cert.pem\n",
    "key  certs/key.pem\n",
    "dh   certs/dh.pem\n",
    "tls-auth certs/ta.key 1\n",
    client_auth_line,
    cipher,
    compression,
    "chroot ", OPENVPN_SYSCONF_DIR, "/client\n",
    "user ", server_user, "\n",
    "group ", server_group, "\n",
    "persist-key\n",
    "persist-tun\n",
    "log    /etc/gadmin-openvpn/client/openvpn-client.log\n",
    "status /etc/gadmin-openvpn/client/openvpn-client-status.log\n",
    "verb ", log_level, "\n",
    "mute 20\n",
    NULL);

    fputs(conf, fp);
    fclose(fp);
    g_free(openvpn_client_conf);
    g_free(client_auth_line);



    g_free(proto);
    g_free(require_auth);
    g_free(cipher);
    g_free(client_to_client);
    g_free(redirect_gateway);
    g_free(compression);
    /* tap_dev is freed after the init script below... */



    /* Write the servers sysinit script according to the settings */
    sysinit_start_script = g_strconcat(
    "#!/bin/bash\n",
    "#\n",
    "#\n",
    "# chkconfig: 2345 10 30\n",
    "#\n",
    "# processname: gadmin-openvpn-server\n",
    "# description: This shell script takes care of starting and stopping \\n",
    "# gadmin-openvpn-server in bridge mode on chkconfig-based systems.\n",
    "#\n",
    "### BEGIN INIT INFO\n",
    "# Provides: gadmin-openvpn-server\n",
    "# Required-Start: network NetworkManager\n",
    "# Required-Stop: NetworkManager\n",
    "# Default-Start: 2 3 4 5\n",
    "# Default-Stop: 0 1 6\n",
    "# Short-Description: start and stop gadmin-openvpn-server\n",
    "# Description: GAdmin-OpenVPN-Server starts and stops a bridged VPN server.\n",
    "### END INIT INFO\n",
    "\n",
    "\n",
    "# Define Bridge Interface\n",
    "br=\"br0\"\n",
    "\n",
    "# Define list of TAP interfaces to be bridged,\n",
    "# for example tap=\"tap0 tap1 tap2\".\n",
    "tap=\"", tap_dev, "\"\n",
    "\n",
    "# LAN Interface to be bridged with TAP interface(s) above.\n",
    "# Should not be the same LAN network address as connecting clients have.\n",
    "external_iface_name=\"", external_iface_name, "\"\n",
    "lan_iface_name=\"", lan_iface_name, "\"\n",
    "lan_iface_ip=\"", lan_iface_address, "\"\n",
    "lan_iface_netmask=\"", lan_subnet_mask, "\"\n",
    "lan_iface_broadcast=\"", lan_broadcast_address, "\"\n",
    "\n",
    "\n",
    "case \"$1\" in\n",
    "		start)\n",
    "\n",
    firestarter_stop,
    "\n",
    "modprobe tun\n",
    "modprobe bridge\n",
    "\n",
    "for t in $tap; do\n",
    "    openvpn --mktun --dev $t\n",
    "done\n",
    "\n",
    "brctl addbr $br\n",
    "brctl addif $br $lan_iface_name\n",
    "\n",
    "for t in $tap; do\n",
    "    brctl addif $br $t\n",
    "done\n",
    "\n",
    "for t in $tap; do\n",
    "    ifconfig $t 0.0.0.0 promisc up\n",
    "done\n",
    "\n",
    "ifconfig $lan_iface_name 0.0.0.0 promisc up\n",
    "\n",
    "ifconfig $br $lan_iface_ip netmask $lan_iface_netmask broadcast $lan_iface_broadcast\n",
    "\n",
    "openvpn --daemon --writepid /var/run/openvpn/openvpn.pid \\\n",
    "--config ", OPENVPN_SYSCONF_DIR, "/server/gadmin-openvpn-server.conf --cd ", OPENVPN_SYSCONF_DIR, "/server\n",
    "\n",
    firestarter_start,
    non_firestarter_start,
    ";;\n",
    "\n",
    "		stop)\n",
    "\n",
    "killall -9 openvpn\n",
    "\n",
    non_firestarter_stop,
    "\n",
    "ifconfig $br down\n",
    "brctl delbr $br\n",
    "\n",
    "for t in $tap; do\n",
    "    openvpn --rmtun --dev $t\n",
    "done\n",
    ";;\n",
    "		*)\n",
    "echo \"usage gadmin-openvpn-server {start|stop}\"\n",
    "exit 1\n",
    ";;\n",
    "esac\n",
    "\n",
    "exit 0\n",
    "\n",
    NULL);

    /* Put the script with the others as: /etc/rc.d/init.d/gadmin-openvpn-server  */
    sysinit_script_path = g_strdup_printf("%s/gadmin-openvpn-server", SYSINIT_SCRIPTS_DIR);
    if((fp=fopen(sysinit_script_path, "w+"))==NULL)
    {
	info = g_strdup_printf(_("Can not write openvpn server sysinit script here: \n%s\n"), sysinit_script_path);
	show_info(info);
	g_free(info);
	g_free(sysinit_start_script);
	g_free(sysinit_script_path);
	g_free(tap_dev);
	g_free(firestarter_start);
	g_free(firestarter_stop);
	g_free(non_firestarter_start);
	g_free(non_firestarter_stop);
        return;
    }
    fputs(sysinit_start_script, fp);
    fclose(fp);

    /* Chmod the script to 755 */
    cmd = g_strdup_printf("chmod 755 %s", sysinit_script_path);
    if( ! run_command(cmd) )
    {
	printf("Show popup, error chmodding sysinit script.\n");
    }
    g_free(cmd);

    g_free(sysinit_start_script);
    g_free(sysinit_script_path);

    g_free(tap_dev);
    g_free(firestarter_start);
    g_free(firestarter_stop);
    g_free(non_firestarter_start);
    g_free(non_firestarter_stop);

    /* Populate the server settings */
    populate_server_settings(widgets);

    /* Populate the configuration tab */
    populate_conf_tab(widgets);

    if( activated )
      reread_conf(widgets);
}
