/*
 * Copyright 2012 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

import QtQuick 2.0
import Ubuntu.Components 0.1
import Ubuntu.Components.ListItems 0.1 as ListItem
import Ubuntu.Components.Popups 0.1
import Friends 0.2

Item {
    id: details
    height: visible ? childrenRect.height + units.gu(1) : 0

    FriendsDispatcher {
        id: friends
        onSendComplete: {
            if (success) {
                sendSpinner.visible = false;
                entryArea.text = "";
                console.log ("Send completed successfully");
            } else {
                sendSpinner.visible = false;
                PopupUtils.open(errorPopover, sendButton);
                console.log ("Send failed: " + errorMessage);
            }
        }
    }

    Item {
        id: pictureWrapper
        anchors.right: parent.right
        width: parent.width - units.gu(8)
        height: childrenRect.height

        Image {
            id: pictureImage
            visible: linkPicture.length > 0 && status == Image.Ready
            source: Qt.resolvedUrl(linkPicture)
            asynchronous: true
            fillMode: Image.PreserveAspectCrop
            sourceSize.width: units.dp(128)
            sourceSize.height: units.dp(128)
            MouseArea {
                anchors.fill: pictureImage
                onClicked: {
                    if (linkUrl.length > 0)
                    {
                        Qt.openUrlExternally(linkUrl);
                    }
                }
            }
        }
    }

    Image {
        id: mapimage
        width: visible ? Math.round(Math.min(parent.width * 0.8, units.gu(20))) : 0
        height: visible ? Math.round(Math.min(parent.width * 0.8, units.gu(20))) : 0
        anchors.horizontalCenter: parent.horizontalCenter
        anchors.top: pictureWrapper.bottom
        source: {
            return "http://open.mapquestapi.com/staticmap/v4/getmap?size=" +
                Math.round(Math.min(parent.width * 0.8, units.gu(20))) + "," +
                Math.round(Math.min(parent.width * 0.8, units.gu(20))) +
                "&zoom=12&center=" + latitude + "," + longitude

        }
        visible: location.length > 0
    }

    ThreadView {
        id: threadView
        width: parent.width
        height: childrenRect.height

        anchors {
            top: mapimage.bottom
            left: parent.left
            right: parent.right
        }
    }

    Item {
        id: entry
        property var lastCursorPosition
        height: childrenRect.height
        anchors {
            top: threadView.bottom
            left: parent.left
            right: parent.right
            topMargin: units.gu(2)
        }

        TextArea {
            id: entryArea
            anchors {
                left: parent.left
                leftMargin: units.gu(2)
                right: sendButton.left
                rightMargin: units.gu(1)
                bottomMargin: units.gu(1.5)
            }
            // send message if return was pressed
            Keys.onReturnPressed: sendButton.clicked(null)
            Keys.onEscapePressed: text = ""
            height: units.gu(4)
            placeholderText: i18n.tr("Reply")
            autoSize: true
            maximumLineCount: 0
            color: text.length < 140 ? "gray" : "red"
            textFormat: TextEdit.PlainText
            contentWidth: width - units.gu(5)
            onActiveFocusChanged: {
                if (activeFocus) {
                    if (service == "twitter" || service == "identica") {
                        if (entryArea.text.search("@"+senderNick) == -1) {
                            entryArea.text = "@"+senderNick+" ";
                            cursorPosition = text.length;
                        }
                    }
                }
            }

            onTextChanged: {
                var enteredText = text.substring(entry.lastCursorPosition, cursorPosition);
                if (enteredText.substring(0,4) == "http") {
                    var shortUrl = friends.urlShorten(enteredText);
                    if (shortUrl.length > 4) {
                        text = text.replace (enteredText, shortUrl);
                        cursorPosition = text.length;
                    }
                }
                entry.lastCursorPosition = cursorPosition;
            }

            Item {
                id: counter
                anchors {
                    right: entryArea.right
                    top: entryArea.top
                    rightMargin: units.gu(1)
                }
                height: units.gu(4)
                width: childrenRect.width
                visible: entryArea.text.length > 0 ? true : false
                Text {
                    text: 140 - entryArea.text.length
                    font.bold: true
                    opacity: 0.3
                }
            }
            Button {
                id: clearButton
                anchors {
                    right: entryArea.right
                    bottom: entryArea.bottom
                    rightMargin: units.gu(1)
                }
                width: units.gu(2)
                height: units.gu(2)
                visible: entryArea.text.length > 0 ? true : false
                iconSource: "images/clear-search.png"
                color: "transparent"
                opacity: 0.3
                onClicked: {
                    entryArea.text = "";
                }
            }
        }

        Button {
            id: sendButton
            anchors {
                right: parent.right
                rightMargin: units.gu(1)
                bottom: entryArea.bottom
            }
            width: units.gu(9)
            height: units.gu(4)
            text: i18n.tr("Send")
            color: main.headerColor
            onClicked: {
                console.log ("Reply to " + messageId + " from " + accountId + ": " + entryArea.text);
                sendSpinner.visible = true
                friends.sendReplyAsync(accountId, messageId, entryArea.text);
            }
        }

        ActivityIndicator {
            id: sendSpinner
            anchors.centerIn: entryArea
            visible: false
            running: visible
        }
    }

    Component {
        id: errorPopover
        Popover {
            Column {
                anchors {
                    left: parent.left
                    top: parent.top
                    right: parent.right
                }
                ListItem.Standard {
                    text: i18n.tr("An error occured")
                }
            }
        }
    }
}
