# 24 "cil/src/logic/logic_lexer.mll"
 

  open Logic_parser
  open Lexing
  open Logic_ptree

  type state = Normal | Test

  let state_stack = Stack.create ()

  let () = Stack.push Normal state_stack

  let get_state () = try Stack.top state_stack with Stack.Empty -> Normal

  let pop_state () = try ignore (Stack.pop state_stack) with Stack.Empty -> ()

  exception Error of (int * int) * string

  let loc lexbuf = (lexeme_start lexbuf, lexeme_end lexbuf)

  let lex_error lexbuf s =
    raise (Error (loc lexbuf, "lexical error, " ^ s))

  let find_utf8 =
    let h = Hashtbl.create 97 in
    List.iter (fun (i,t) -> Hashtbl.add h i t)
      [ Utf8_logic.forall, FORALL;
        Utf8_logic.exists, EXISTS;
        Utf8_logic.eq, EQ;
        Utf8_logic.neq, NE;
        Utf8_logic.le, LE;
        Utf8_logic.ge, GE;
        Utf8_logic.implies,IMPLIES;
        Utf8_logic.iff, IFF;
        Utf8_logic.conj, AND;
        Utf8_logic.disj, OR;
        Utf8_logic.neg, NOT;
        Utf8_logic.x_or, HATHAT;
      ];

    fun s -> try Hashtbl.find h s
    with Not_found -> IDENTIFIER s

  let identifier =
    let all_kw = Hashtbl.create 37 in
    let c_kw = Hashtbl.create 37 in
    List.iter
      (fun (i,t,flag) ->
         Hashtbl.add all_kw i t;
         if flag then Hashtbl.add c_kw i t
      )
      [
        "assert", ASSERT, false;
        "assigns", ASSIGNS, false;
        "assumes", ASSUMES, false;
        "axiom", AXIOM, false;
        "axiomatic", AXIOMATIC, false;
        "behavior", BEHAVIOR, false;
        "behaviors", BEHAVIORS, false;
	"case", CASE, true;
        "char", CHAR, true;
        "complete", COMPLETE, false;
        "decreases", DECREASES, false;
        "disjoint", DISJOINT, false;
        "double", DOUBLE, true;
        "else", ELSE, true;
        "ensures", ENSURES, false ;
        "enum", ENUM, true;
        "float", FLOAT, true;
        "for", FOR, true;
        "if", IF, true;
	"inductive", INDUCTIVE, false;
        "int", INT, true;
        "integer", INTEGER, true;
        "invariant", INVARIANT, false;
        "global",    GLOBAL, false;
        "label", LABEL, false;
        "lemma", LEMMA, false;
        "logic", LOGIC, false;
        "long", LONG, true;
        "loop", LOOP, false;
        "pragma", PRAGMA, false;
        "predicate", PREDICATE, false;
        "reads", READS, false;
        "real", REAL, true;
        "requires", REQUIRES, false;
        "short", SHORT, true;
        "signed", SIGNED, true;
        "sizeof", SIZEOF, true;
        "slice", SLICE, false;
	"impact", IMPACT, false;
        "struct", STRUCT, true;
        "terminates", TERMINATES, false;
        "type", TYPE, false;
        "union", UNION, true;
        "unsigned", UNSIGNED, true;
        "variant", VARIANT, false;
        "void", VOID, true;
      ];
    fun s ->
      try
        Hashtbl.find (if Logic_const.is_kw_c_mode () then c_kw else all_kw) s
      with Not_found ->
        try
          if Hashtbl.find Logic_const.typenames s then
            TYPENAME s
          else IDENTIFIER s
        with Not_found -> IDENTIFIER s

  let bs_identifier =
    let h = Hashtbl.create 97 in
    List.iter (fun (i,t) -> Hashtbl.add h i t)
      [
        "\\at", AT;
        "\\base_addr", BASE_ADDR;
        "\\block_length", BLOCK_LENGTH;
        "\\empty", EMPTY;
        "\\exists", EXISTS;
        "\\false", FALSE;
        "\\forall", FORALL;
        "\\fresh", FRESH;
        "\\from", FROM;
        "\\inter", INTER;
        "\\lambda", LAMBDA;
        "\\max", IDENTIFIER "\\max";
        "\\min", IDENTIFIER "\\min";
        "\\nothing", NOTHING;
        "\\null", NULL;
        "\\numof", IDENTIFIER "\\numof";
        "\\old", OLD;
        "\\product", IDENTIFIER "\\product";
        "\\result", RESULT;
        "\\separated", SEPARATED;
        "\\sum", IDENTIFIER "\\sum";
        "\\true", TRUE;
        "\\type", BSTYPE;
        "\\typeof", TYPEOF;
        "\\union", BSUNION;
        "\\valid", VALID;
        "\\valid_index", VALID_INDEX;
        "\\valid_range", VALID_RANGE;
      ];
    fun lexbuf ->
      let s = lexeme lexbuf in
      try Hashtbl.find h s
      with Not_found ->
        lex_error lexbuf ("illegal escape sequence " ^ s)


  let int_of_digit chr =
    match chr with
        '0'..'9' -> (Char.code chr) - (Char.code '0')
      | 'a'..'f' -> (Char.code chr) - (Char.code 'a') + 10
      | 'A'..'F' -> (Char.code chr) - (Char.code 'A') + 10
      | _ -> assert false

  (* Update lexer buffer. *)
  let update_newline_loc lexbuf =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_lnum = pos.Lexing.pos_lnum + 1;
	Lexing.pos_bol = pos.Lexing.pos_cnum;
      }

  (* Update lexer buffer. *)
  let update_line_loc lexbuf line absolute chars =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_lnum = if absolute then line else pos.Lexing.pos_lnum + line;
	Lexing.pos_bol = pos.Lexing.pos_cnum - chars;
      }

  (* Update lexer buffer. *)
  let update_file_loc lexbuf file =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_fname = file;
      }

  (* Update lexer buffer. *)
  let update_bol_loc lexbuf bol =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_bol = bol;
      }

# 193 "cil/src/logic/logic_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\195\255\196\255\000\000\202\255\203\255\204\255\205\255\
    \206\255\207\255\196\000\215\255\217\255\165\000\222\255\223\255\
    \202\000\230\255\231\255\199\000\233\255\234\255\201\000\139\000\
    \227\000\208\000\206\000\240\255\004\001\225\000\006\001\254\000\
    \036\001\109\001\198\001\000\002\236\000\254\255\030\001\091\001\
    \252\255\253\255\075\002\188\002\127\002\128\002\129\002\161\002\
    \246\255\189\002\191\002\192\002\241\255\158\002\077\001\208\000\
    \187\002\001\003\039\003\211\000\212\000\114\003\024\003\245\255\
    \242\255\243\255\150\003\175\003\244\255\204\003\214\003\124\003\
    \243\003\185\003\239\000\254\003\027\004\059\004\069\004\225\255\
    \224\000\239\255\199\255\201\255\228\000\228\000\238\255\237\255\
    \236\255\208\255\232\255\200\255\228\255\220\255\198\255\211\255\
    \200\000\251\255\093\004\097\001\166\004\189\004\190\001\121\004\
    \186\000\221\004\255\255\202\000\212\000\031\003\224\001\220\002\
    \092\001";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\058\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\042\000\255\255\255\255\036\000\255\255\255\255\
    \026\000\255\255\255\255\031\000\255\255\255\255\020\000\037\000\
    \034\000\041\000\046\000\255\255\060\000\028\000\060\000\008\000\
    \008\000\005\000\005\000\060\000\039\000\255\255\000\000\255\255\
    \255\255\255\255\004\000\255\255\255\255\255\255\255\255\009\000\
    \255\255\255\255\255\255\014\000\255\255\012\000\255\255\008\000\
    \007\000\255\255\006\000\006\000\007\000\010\000\255\255\255\255\
    \255\255\255\255\255\255\011\000\255\255\255\255\011\000\255\255\
    \012\000\255\255\029\000\011\000\255\255\011\000\255\255\255\255\
    \045\000\255\255\255\255\255\255\043\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\004\000\001\000\255\255\000\000\001\000\255\255\
    \004\000\002\000\255\255\255\255\255\255\255\255\003\000\255\255\
    \255\255";
  Lexing.lex_default = 
   "\003\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\255\255\000\000\000\000\
    \255\255\000\000\000\000\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\000\000\049\000\255\255\045\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\039\000\
    \000\000\000\000\255\255\049\000\045\000\045\000\045\000\045\000\
    \000\000\049\000\049\000\049\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\000\000\000\000\000\000\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \097\000\000\000\255\255\255\255\255\255\255\255\255\255\097\000\
    \255\255\255\255\000\000\255\255\255\255\040\000\111\000\111\000\
    \041\000";
  Lexing.lex_trans = 
   "\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\038\000\037\000\001\000\038\000\038\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \038\000\022\000\028\000\027\000\021\000\011\000\024\000\030\000\
    \009\000\008\000\014\000\015\000\020\000\019\000\029\000\036\000\
    \032\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\016\000\017\000\025\000\026\000\010\000\018\000\
    \038\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\034\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\005\000\035\000\004\000\013\000\033\000\
    \001\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\007\000\023\000\006\000\012\000\001\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\001\000\
    \002\000\095\000\094\000\093\000\092\000\090\000\089\000\088\000\
    \091\000\087\000\083\000\080\000\082\000\084\000\255\255\074\000\
    \255\255\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\039\000\055\000\079\000\081\000\059\000\
    \060\000\085\000\086\000\107\000\098\000\055\000\255\255\038\000\
    \059\000\060\000\038\000\038\000\053\000\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \108\000\040\000\000\000\000\000\055\000\000\000\038\000\059\000\
    \060\000\000\000\000\000\054\000\000\000\055\000\000\000\000\000\
    \059\000\060\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\053\000\055\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\038\000\000\000\
    \050\000\000\000\046\000\054\000\000\000\041\000\106\000\000\000\
    \000\000\054\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\000\000\000\000\055\000\000\000\000\000\000\000\000\000\
    \062\000\055\000\062\000\000\000\057\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\000\000\
    \000\000\054\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\055\000\000\000\000\000\057\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \040\000\000\000\000\000\000\000\033\000\000\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \043\000\255\255\000\000\000\000\255\255\044\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \000\000\000\000\041\000\000\000\255\255\000\000\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\000\000\000\000\000\000\000\000\033\000\000\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\040\000\037\000\000\000\000\000\042\000\
    \000\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\255\255\048\000\
    \047\000\000\000\042\000\255\255\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\255\255\255\255\
    \048\000\255\255\255\255\000\000\064\000\000\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \000\000\000\000\000\000\046\000\046\000\046\000\255\255\052\000\
    \255\255\051\000\052\000\066\000\065\000\255\255\000\000\000\000\
    \255\255\053\000\065\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\046\000\041\000\000\000\
    \054\000\000\000\000\000\066\000\065\000\000\000\000\000\060\000\
    \000\000\000\000\065\000\000\000\000\000\000\000\000\000\000\000\
    \060\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \050\000\050\000\000\000\050\000\050\000\000\000\000\000\000\000\
    \054\000\000\000\000\000\000\000\000\000\000\000\000\000\060\000\
    \037\000\106\000\000\000\000\000\037\000\000\000\000\000\000\000\
    \060\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\000\000\000\000\000\000\000\000\037\000\
    \000\000\110\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\000\000\000\000\000\000\000\000\000\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\000\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\000\000\000\000\
    \000\000\000\000\000\000\059\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\059\000\000\000\000\000\255\255\
    \255\255\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\000\000\000\000\
    \000\000\000\000\000\000\059\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\059\000\000\000\000\000\000\000\
    \000\000\255\255\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\000\000\000\000\
    \063\000\000\000\000\000\000\000\255\255\255\255\063\000\255\255\
    \255\255\073\000\000\000\073\000\000\000\000\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \063\000\000\000\000\000\000\000\255\255\000\000\063\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\000\000\069\000\068\000\000\000\071\000\
    \000\000\071\000\000\000\068\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \000\000\000\000\000\000\000\000\069\000\068\000\000\000\000\000\
    \000\000\000\000\000\000\068\000\068\000\000\000\000\000\255\255\
    \000\000\000\000\068\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\000\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \000\000\065\000\000\000\000\000\068\000\000\000\000\000\065\000\
    \000\000\000\000\068\000\076\000\068\000\000\000\078\000\000\000\
    \078\000\000\000\068\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\000\000\000\000\000\000\
    \000\000\065\000\000\000\000\000\000\000\000\000\000\000\065\000\
    \000\000\000\000\000\000\076\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\077\000\077\000\041\000\
    \000\000\068\000\037\000\106\000\041\000\000\000\000\000\068\000\
    \000\000\000\000\000\000\000\000\000\000\099\000\099\000\099\000\
    \099\000\099\000\099\000\099\000\099\000\000\000\000\000\000\000\
    \000\000\037\000\000\000\000\000\041\000\000\000\000\000\000\000\
    \000\000\068\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \000\000\105\000\105\000\105\000\105\000\105\000\105\000\105\000\
    \105\000\105\000\105\000\000\000\000\000\100\000\000\000\000\000\
    \000\000\041\000\000\000\000\000\000\000\000\000\041\000\041\000\
    \000\000\000\000\000\000\041\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\041\000\000\000\000\000\000\000\041\000\
    \000\000\041\000\000\000\041\000\000\000\100\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \000\000\000\000\000\000\000\000\000\000\104\000\000\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\000\000\000\000\000\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\105\000\105\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\010\000\010\000\013\000\016\000\019\000\022\000\023\000\
    \016\000\024\000\025\000\026\000\025\000\025\000\028\000\029\000\
    \030\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\036\000\055\000\074\000\080\000\059\000\
    \060\000\084\000\085\000\104\000\096\000\055\000\028\000\038\000\
    \059\000\060\000\038\000\038\000\031\000\030\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \107\000\108\000\255\255\255\255\055\000\255\255\038\000\059\000\
    \060\000\255\255\255\255\031\000\255\255\055\000\255\255\255\255\
    \059\000\060\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\032\000\031\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\038\000\255\255\
    \028\000\255\255\030\000\031\000\255\255\039\000\112\000\255\255\
    \255\255\032\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \032\000\255\255\255\255\031\000\255\255\255\255\255\255\255\255\
    \054\000\032\000\054\000\255\255\032\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\255\255\
    \255\255\032\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \032\000\099\000\099\000\099\000\099\000\099\000\099\000\099\000\
    \099\000\032\000\255\255\255\255\032\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \096\000\255\255\255\255\255\255\033\000\255\255\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \034\000\110\000\255\255\255\255\110\000\034\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\255\255\110\000\255\255\028\000\255\255\030\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\255\255\255\255\255\255\255\255\034\000\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\039\000\112\000\255\255\255\255\035\000\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\255\255\255\255\255\255\
    \255\255\044\000\045\000\046\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\044\000\045\000\
    \046\000\255\255\042\000\047\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\043\000\049\000\
    \047\000\050\000\051\000\255\255\053\000\255\255\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \255\255\255\255\255\255\044\000\045\000\046\000\043\000\049\000\
    \110\000\050\000\051\000\053\000\053\000\111\000\255\255\255\255\
    \111\000\056\000\053\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\047\000\111\000\255\255\
    \056\000\255\255\255\255\053\000\053\000\255\255\255\255\056\000\
    \255\255\255\255\053\000\255\255\255\255\255\255\255\255\255\255\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \043\000\049\000\255\255\050\000\051\000\255\255\255\255\255\255\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\056\000\
    \109\000\109\000\255\255\255\255\109\000\255\255\255\255\255\255\
    \056\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\255\255\255\255\255\255\255\255\109\000\
    \255\255\109\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\255\255\255\255\255\255\255\255\255\255\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\255\255\255\255\
    \255\255\255\255\255\255\058\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\058\000\255\255\255\255\044\000\
    \045\000\046\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \058\000\058\000\058\000\058\000\058\000\058\000\255\255\255\255\
    \255\255\255\255\255\255\058\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\058\000\255\255\255\255\255\255\
    \255\255\047\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\255\255\255\255\
    \061\000\255\255\255\255\255\255\043\000\049\000\061\000\050\000\
    \051\000\066\000\255\255\066\000\255\255\255\255\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \061\000\255\255\255\255\255\255\111\000\255\255\061\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\255\255\067\000\067\000\255\255\069\000\
    \255\255\069\000\255\255\067\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \255\255\255\255\255\255\255\255\067\000\067\000\255\255\255\255\
    \255\255\255\255\255\255\067\000\070\000\255\255\255\255\109\000\
    \255\255\255\255\070\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\255\255\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \255\255\072\000\255\255\255\255\070\000\255\255\255\255\072\000\
    \255\255\255\255\070\000\075\000\075\000\255\255\076\000\255\255\
    \076\000\255\255\075\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\255\255\255\255\255\255\
    \255\255\072\000\255\255\255\255\255\255\255\255\255\255\072\000\
    \255\255\255\255\255\255\075\000\075\000\255\255\255\255\255\255\
    \255\255\255\255\075\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\077\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\098\000\
    \255\255\077\000\103\000\103\000\098\000\255\255\255\255\077\000\
    \255\255\255\255\255\255\255\255\255\255\098\000\098\000\098\000\
    \098\000\098\000\098\000\098\000\098\000\255\255\255\255\255\255\
    \255\255\103\000\255\255\255\255\098\000\255\255\255\255\255\255\
    \255\255\077\000\255\255\255\255\255\255\255\255\255\255\077\000\
    \255\255\103\000\103\000\103\000\103\000\103\000\103\000\103\000\
    \103\000\103\000\103\000\255\255\255\255\098\000\255\255\255\255\
    \255\255\098\000\255\255\255\255\255\255\255\255\098\000\098\000\
    \255\255\255\255\255\255\098\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\098\000\255\255\255\255\255\255\098\000\
    \255\255\098\000\255\255\098\000\255\255\098\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \255\255\255\255\255\255\255\255\255\255\103\000\255\255\100\000\
    \100\000\100\000\100\000\100\000\100\000\101\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\101\000\101\000\
    \101\000\101\000\101\000\101\000\255\255\255\255\255\255\100\000\
    \100\000\100\000\100\000\100\000\100\000\105\000\105\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\101\000\101\000\
    \101\000\101\000\101\000\101\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\103\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \007\000\000\000\000\000\000\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\007\000\
    \000\000\000\000\000\000\010\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\001\000\000\000\000\000\000\000\004\000\
    \004\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\034\000\255\255\255\255\255\255\000\000\
    \034\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_code = 
   "\255\001\255\255\002\255\255\000\002\255\000\001\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 3 (-1) ;   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 236 "cil/src/logic/logic_lexer.mll"
           ( token lexbuf )
# 748 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 237 "cil/src/logic/logic_lexer.mll"
         ( update_newline_loc lexbuf; token lexbuf )
# 753 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 238 "cil/src/logic/logic_lexer.mll"
                      ( update_newline_loc lexbuf; token lexbuf )
# 758 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 239 "cil/src/logic/logic_lexer.mll"
                     ( token lexbuf )
# 763 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 241 "cil/src/logic/logic_lexer.mll"
                       ( bs_identifier lexbuf )
# 768 "cil/src/logic/logic_lexer.ml"

  | 5 ->
# 242 "cil/src/logic/logic_lexer.mll"
                        ( let s = lexeme lexbuf in identifier s )
# 773 "cil/src/logic/logic_lexer.ml"

  | 6 ->
# 244 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 778 "cil/src/logic/logic_lexer.ml"

  | 7 ->
# 245 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 783 "cil/src/logic/logic_lexer.ml"

  | 8 ->
# 246 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 788 "cil/src/logic/logic_lexer.ml"

  | 9 ->
let
# 247 "cil/src/logic/logic_lexer.mll"
                 prelude
# 794 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 247 "cil/src/logic/logic_lexer.mll"
                                                   content
# 799 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) (lexbuf.Lexing.lex_curr_pos + -1) in
# 248 "cil/src/logic/logic_lexer.mll"
      (
        let b = Buffer.create 5 in
        Buffer.add_string b prelude;
        let lbf = Lexing.from_string content in
        CONSTANT (IntConstant (chr b lbf ^ "'"))
      )
# 808 "cil/src/logic/logic_lexer.ml"

  | 10 ->
# 254 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (FloatConstant (lexeme lexbuf)) )
# 813 "cil/src/logic/logic_lexer.ml"

  | 11 ->
# 255 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (FloatConstant (lexeme lexbuf)) )
# 818 "cil/src/logic/logic_lexer.ml"

  | 12 ->
# 256 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (FloatConstant (lexeme lexbuf)) )
# 823 "cil/src/logic/logic_lexer.ml"

  | 13 ->
let
# 259 "cil/src/logic/logic_lexer.mll"
            n
# 829 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -2) in
# 259 "cil/src/logic/logic_lexer.mll"
                            ( lexbuf.lex_curr_pos <- lexbuf.lex_curr_pos - 2;
                              CONSTANT (IntConstant n) )
# 834 "cil/src/logic/logic_lexer.ml"

  | 14 ->
let
# 262 "cil/src/logic/logic_lexer.mll"
                prelude
# 840 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 262 "cil/src/logic/logic_lexer.mll"
                                                 content
# 845 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) (lexbuf.Lexing.lex_curr_pos + -1) in
# 263 "cil/src/logic/logic_lexer.mll"
      ( STRING_LITERAL (prelude.[0] = 'L',content) )
# 849 "cil/src/logic/logic_lexer.ml"

  | 15 ->
# 264 "cil/src/logic/logic_lexer.mll"
                            ( hash lexbuf )
# 854 "cil/src/logic/logic_lexer.ml"

  | 16 ->
# 265 "cil/src/logic/logic_lexer.mll"
                            ( IMPLIES )
# 859 "cil/src/logic/logic_lexer.ml"

  | 17 ->
# 266 "cil/src/logic/logic_lexer.mll"
                            ( IFF )
# 864 "cil/src/logic/logic_lexer.ml"

  | 18 ->
# 267 "cil/src/logic/logic_lexer.mll"
                            ( AND )
# 869 "cil/src/logic/logic_lexer.ml"

  | 19 ->
# 268 "cil/src/logic/logic_lexer.mll"
                            ( OR )
# 874 "cil/src/logic/logic_lexer.ml"

  | 20 ->
# 269 "cil/src/logic/logic_lexer.mll"
                            ( NOT )
# 879 "cil/src/logic/logic_lexer.ml"

  | 21 ->
# 270 "cil/src/logic/logic_lexer.mll"
                            ( DOLLAR )
# 884 "cil/src/logic/logic_lexer.ml"

  | 22 ->
# 271 "cil/src/logic/logic_lexer.mll"
                            ( COMMA )
# 889 "cil/src/logic/logic_lexer.ml"

  | 23 ->
# 272 "cil/src/logic/logic_lexer.mll"
                            ( ARROW )
# 894 "cil/src/logic/logic_lexer.ml"

  | 24 ->
# 273 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Test state_stack; QUESTION )
# 899 "cil/src/logic/logic_lexer.ml"

  | 25 ->
# 274 "cil/src/logic/logic_lexer.mll"
                            ( SEMICOLON )
# 904 "cil/src/logic/logic_lexer.ml"

  | 26 ->
# 275 "cil/src/logic/logic_lexer.mll"
                            ( match get_state() with
                                  Normal  -> COLON
                                | Test -> pop_state(); COLON2
                            )
# 912 "cil/src/logic/logic_lexer.ml"

  | 27 ->
# 279 "cil/src/logic/logic_lexer.mll"
                            ( COLONCOLON )
# 917 "cil/src/logic/logic_lexer.ml"

  | 28 ->
# 280 "cil/src/logic/logic_lexer.mll"
                            ( DOT )
# 922 "cil/src/logic/logic_lexer.ml"

  | 29 ->
# 281 "cil/src/logic/logic_lexer.mll"
                            ( DOTDOT )
# 927 "cil/src/logic/logic_lexer.ml"

  | 30 ->
# 282 "cil/src/logic/logic_lexer.mll"
                            ( DOTDOTDOT )
# 932 "cil/src/logic/logic_lexer.ml"

  | 31 ->
# 283 "cil/src/logic/logic_lexer.mll"
                            ( MINUS )
# 937 "cil/src/logic/logic_lexer.ml"

  | 32 ->
# 284 "cil/src/logic/logic_lexer.mll"
                            ( PLUS )
# 942 "cil/src/logic/logic_lexer.ml"

  | 33 ->
# 285 "cil/src/logic/logic_lexer.mll"
                            ( STAR )
# 947 "cil/src/logic/logic_lexer.ml"

  | 34 ->
# 286 "cil/src/logic/logic_lexer.mll"
                            ( AMP )
# 952 "cil/src/logic/logic_lexer.ml"

  | 35 ->
# 287 "cil/src/logic/logic_lexer.mll"
                            ( HATHAT )
# 957 "cil/src/logic/logic_lexer.ml"

  | 36 ->
# 288 "cil/src/logic/logic_lexer.mll"
                            ( HAT )
# 962 "cil/src/logic/logic_lexer.ml"

  | 37 ->
# 289 "cil/src/logic/logic_lexer.mll"
                            ( PIPE )
# 967 "cil/src/logic/logic_lexer.ml"

  | 38 ->
# 290 "cil/src/logic/logic_lexer.mll"
                            ( TILDE )
# 972 "cil/src/logic/logic_lexer.ml"

  | 39 ->
# 291 "cil/src/logic/logic_lexer.mll"
                            ( SLASH )
# 977 "cil/src/logic/logic_lexer.ml"

  | 40 ->
# 292 "cil/src/logic/logic_lexer.mll"
                            ( PERCENT )
# 982 "cil/src/logic/logic_lexer.ml"

  | 41 ->
# 293 "cil/src/logic/logic_lexer.mll"
                            ( LT )
# 987 "cil/src/logic/logic_lexer.ml"

  | 42 ->
# 294 "cil/src/logic/logic_lexer.mll"
                            ( GT )
# 992 "cil/src/logic/logic_lexer.ml"

  | 43 ->
# 295 "cil/src/logic/logic_lexer.mll"
                            ( LE )
# 997 "cil/src/logic/logic_lexer.ml"

  | 44 ->
# 296 "cil/src/logic/logic_lexer.mll"
                            ( GE )
# 1002 "cil/src/logic/logic_lexer.ml"

  | 45 ->
# 297 "cil/src/logic/logic_lexer.mll"
                            ( EQ )
# 1007 "cil/src/logic/logic_lexer.ml"

  | 46 ->
# 298 "cil/src/logic/logic_lexer.mll"
                            ( EQUAL )
# 1012 "cil/src/logic/logic_lexer.ml"

  | 47 ->
# 299 "cil/src/logic/logic_lexer.mll"
                            ( NE )
# 1017 "cil/src/logic/logic_lexer.ml"

  | 48 ->
# 300 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LPAR )
# 1022 "cil/src/logic/logic_lexer.ml"

  | 49 ->
# 301 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RPAR )
# 1027 "cil/src/logic/logic_lexer.ml"

  | 50 ->
# 302 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LBRACE )
# 1032 "cil/src/logic/logic_lexer.ml"

  | 51 ->
# 303 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RBRACE )
# 1037 "cil/src/logic/logic_lexer.ml"

  | 52 ->
# 304 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LSQUARE )
# 1042 "cil/src/logic/logic_lexer.ml"

  | 53 ->
# 305 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RSQUARE )
# 1047 "cil/src/logic/logic_lexer.ml"

  | 54 ->
# 306 "cil/src/logic/logic_lexer.mll"
                            ( LTCOLON )
# 1052 "cil/src/logic/logic_lexer.ml"

  | 55 ->
# 307 "cil/src/logic/logic_lexer.mll"
                            ( COLONGT )
# 1057 "cil/src/logic/logic_lexer.ml"

  | 56 ->
# 308 "cil/src/logic/logic_lexer.mll"
                            ( LTLT )
# 1062 "cil/src/logic/logic_lexer.ml"

  | 57 ->
# 309 "cil/src/logic/logic_lexer.mll"
                            ( GTGT )
# 1067 "cil/src/logic/logic_lexer.ml"

  | 58 ->
let
# 310 "cil/src/logic/logic_lexer.mll"
                 c
# 1073 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 310 "cil/src/logic/logic_lexer.mll"
                            ( find_utf8 c )
# 1077 "cil/src/logic/logic_lexer.ml"

  | 59 ->
# 311 "cil/src/logic/logic_lexer.mll"
                            ( EOF )
# 1082 "cil/src/logic/logic_lexer.ml"

  | 60 ->
# 312 "cil/src/logic/logic_lexer.mll"
        ( lex_error lexbuf ("illegal character " ^ lexeme lexbuf) )
# 1087 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and chr buffer lexbuf =
    __ocaml_lex_chr_rec buffer lexbuf 96
and __ocaml_lex_chr_rec buffer lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 316 "cil/src/logic/logic_lexer.mll"
      ( let s = lexeme lexbuf in
        let real_s = String.sub s 2 (String.length s - 2) in
        let rec add_one_char s =
          let l = String.length s in
          if l = 0 then ()
          else
          let h = int_of_digit s.[0] in
          let c,s =
            if l = 1 then (h,"")
            else
              (16*h + int_of_digit s.[1],
               String.sub s 2 (String.length s - 2))
          in
          Buffer.add_char buffer (Char.chr c); add_one_char s
        in add_one_char real_s; chr buffer lexbuf
      )
# 1113 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 333 "cil/src/logic/logic_lexer.mll"
      ( let s = lexeme lexbuf in
        let real_s = String.sub s 1 (String.length s - 1) in
        let rec value i s =
          if s = "" then i
          else value (8*i+int_of_digit s.[0])
            (String.sub s 1 (String.length s -1))
        in let c = value 0 real_s in
        Buffer.add_char buffer (Char.chr c); chr buffer lexbuf
      )
# 1126 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 343 "cil/src/logic/logic_lexer.mll"
      ( Buffer.add_char buffer
          (match (lexeme lexbuf).[1] with
               'a' -> '\007'
             | 'b' -> '\b'
             | 'f' -> '\012'
             | 'n' -> '\n'
             | 'r' -> '\r'
             | 't' -> '\t'
             | '\'' -> '\''
             | '"' -> '"'
             | '?' -> '?'
             | '\\' -> '\\'
             | _ -> assert false
          ); chr buffer lexbuf)
# 1144 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 357 "cil/src/logic/logic_lexer.mll"
        ( Buffer.contents buffer )
# 1149 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 358 "cil/src/logic/logic_lexer.mll"
       ( Buffer.add_string buffer (lexeme lexbuf); chr buffer lexbuf )
# 1154 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_chr_rec buffer lexbuf __ocaml_lex_state

and hash lexbuf =
    __ocaml_lex_hash_rec lexbuf 103
and __ocaml_lex_hash_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 361 "cil/src/logic/logic_lexer.mll"
        ( update_newline_loc lexbuf; token lexbuf)
# 1165 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 362 "cil/src/logic/logic_lexer.mll"
             ( hash lexbuf)
# 1170 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 363 "cil/src/logic/logic_lexer.mll"
              ( (* We are seeing a line number. This is the number for the
                   * next line *)
                 let s = Lexing.lexeme lexbuf in
                 let lineno = try
                   int_of_string s
                 with Failure ("int_of_string") ->
                   (* the int is too big. *)
                   Cil.warnLoc (lexbuf.lex_start_p, lexbuf.lex_curr_p)
                     "Bad line number in preprocessed file: %s"  s;
                   (-1)
                 in
                 update_line_loc lexbuf (lineno - 1) true 0;
                  (* A file name may follow *)
		  file lexbuf )
# 1188 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 377 "cil/src/logic/logic_lexer.mll"
                ( hash lexbuf )
# 1193 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 378 "cil/src/logic/logic_lexer.mll"
            ( endline lexbuf)
# 1198 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_hash_rec lexbuf __ocaml_lex_state

and file lexbuf =
    __ocaml_lex_file_rec lexbuf 109
and __ocaml_lex_file_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 381 "cil/src/logic/logic_lexer.mll"
                      ( update_newline_loc lexbuf; token lexbuf)
# 1209 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 382 "cil/src/logic/logic_lexer.mll"
                  ( file lexbuf)
# 1214 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 384 "cil/src/logic/logic_lexer.mll"
            (
              let n = Lexing.lexeme lexbuf in
              let n1 = String.sub n 1
                ((String.length n) - 2) in
              update_file_loc lexbuf n1;
	      endline lexbuf
            )
# 1225 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 392 "cil/src/logic/logic_lexer.mll"
      ( endline lexbuf)
# 1230 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_file_rec lexbuf __ocaml_lex_state

and endline lexbuf =
    __ocaml_lex_endline_rec lexbuf 112
and __ocaml_lex_endline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 395 "cil/src/logic/logic_lexer.mll"
                ( update_newline_loc lexbuf; token lexbuf)
# 1241 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 396 "cil/src/logic/logic_lexer.mll"
                                ( EOF )
# 1246 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 397 "cil/src/logic/logic_lexer.mll"
      ( endline lexbuf)
# 1251 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_endline_rec lexbuf __ocaml_lex_state

;;

# 399 "cil/src/logic/logic_lexer.mll"
 

  open Format

  let copy_lexbuf dest_lexbuf src_loc =
    update_file_loc dest_lexbuf src_loc.pos_fname;
    update_line_loc dest_lexbuf src_loc.pos_lnum true 0;
    let bol = src_loc.Lexing.pos_cnum - src_loc.Lexing.pos_bol in
    update_bol_loc dest_lexbuf (-bol)

  let start_pos lexbuf =
    let pos = lexeme_start_p lexbuf in
    pos.Lexing.pos_cnum - pos.Lexing.pos_bol

  let end_pos lexbuf =
    let pos = lexeme_end_p lexbuf in
    pos.Lexing.pos_cnum - pos.Lexing.pos_bol

  let parse_from_location f (loc, s : Lexing.position * string) =
    let lb = from_string s in
    copy_lexbuf lb loc;
    try
      f token lb
    with
      | Parsing.Parse_error as _e ->
          let start, stop = start_pos lb,end_pos lb in
          Cil.error_loc (
	    lb.lex_curr_p.Lexing.pos_fname,
	    lb.lex_curr_p.Lexing.pos_lnum,start,stop)
            "syntax error while parsing annotation@.";
          Logic_const.exit_kw_c_mode ();
          raise Parsing.Parse_error

      | Error ((b,e), m) ->
          Cil.error_loc (
	    lb.lex_curr_p.Lexing.pos_fname,
	    lb.lex_curr_p.Lexing.pos_lnum,
            b - lb.lex_curr_p.Lexing.pos_bol,
            e - lb.lex_curr_p.Lexing.pos_bol)
            "%s@."
            m;
          Logic_const.exit_kw_c_mode ();
          raise Parsing.Parse_error
      | Logic_const.Not_well_formed (loc, m) ->
          Cil.error_loc
            ((fst loc).Lexing.pos_fname,
             (fst loc).Lexing.pos_lnum,
             (fst loc).Lexing.pos_cnum - (fst loc).Lexing.pos_bol,
             (snd loc).Lexing.pos_cnum - (snd loc).Lexing.pos_bol)
            "%s@." m;
          Logic_const.exit_kw_c_mode ();
          raise Parsing.Parse_error


  let lexpr = parse_from_location Logic_parser.lexpr

  let annot = parse_from_location Logic_parser.annot

  let spec = parse_from_location Logic_parser.spec


# 1319 "cil/src/logic/logic_lexer.ml"
