/**************************************************************************

   Fotoxx      edit photos and manage collections

   Copyright 2007-2014 Michael Cornelison
   Source URL: http://kornelix.com/fotoxx
   Contact: kornelix@posteo.de

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program. If not, see http://www.gnu.org/licenses/.

***************************************************************************/

#define EX extern                                                          //  enable extern declarations
#include "fotoxx.h"                                                        //  (variables in fotoxx.h are refs)

/**************************************************************************
   Fotoxx image edit - Edit menu functions
***************************************************************************/


/**************************************************************************/

//  Rotate image right or left 90 degrees to upright the image.
//  This is not an edit transaction: file is replaced and re-opened.

zdialog  *zdrotate90;

void m_rotate90(GtkWidget *, cchar *menu)                                  //  v.13.08
{
   void turnfunc(GtkWidget *, cchar *menu);

   if (! curr_file) return;                                                //  v.13.09
   F1_help_topic = "rotate";
   
   if (checkpend("all")) return;                                           //  v.13.12
   Fmenulock = 1;

   zdrotate90 = zdialog_new(ZTX("Rotate Image 90º"),Mwin,null);
   zdialog_add_widget(zdrotate90,"vbox","vb1","dialog",0,"space=3");
   GtkWidget *tbar = create_toolbar(zdialog_widget(zdrotate90,"vb1"),32);
   gtk_toolbar_set_style(GTK_TOOLBAR(tbar),GTK_TOOLBAR_BOTH);
   add_toolbar_button(tbar,"left",ZTX("rotate -90º"),"rotate-left.png",turnfunc);
   add_toolbar_button(tbar,"right",ZTX("rotate +90º"),"rotate-right.png",turnfunc);
   add_toolbar_button(tbar,"cancel",0,"cancel.png",turnfunc);
   zdialog_resize(zdrotate90,200,0);
   zdialog_run(zdrotate90,0,"save");
   zdialog_wait(zdrotate90);
   zdialog_free(zdrotate90);

   Fmenulock = 0;
   return;
}


void turnfunc(GtkWidget *, cchar *menu)
{
   int      angle = 0;

   if (strEqu(menu,"cancel")) {
      zdialog_destroy(zdrotate90);
      return;
   }

   if (strEqu(menu,"right")) angle = +90;
   if (strEqu(menu,"left")) angle = -90;

   mutex_lock(&Fpixmap_lock);                                              //  block window updates
   E0pxm = PXM_load(curr_file,1);                                          //  load poss. 16-bit image         v.12.10
   if (E0pxm) {
      E3pxm = PXM_rotate(E0pxm,angle);                                     //  rotate
      PXM_free(E0pxm);
      E0pxm = E3pxm;
      E3pxm = 0;
   }
   mutex_unlock(&Fpixmap_lock);

   if (strEqu(curr_file_type,"other"))                                     //  if unsupported type, use jpg
      strcpy(curr_file_type,"jpg");

   f_save(curr_file,curr_file_type,curr_file_bpc);                         //  save file with curr. edits applied
   f_open_saved();

   return;
}


/**************************************************************************/

//  Rotate image right or left 90 degrees to upright the image.
//  This version is for clicked gallery thumbnails only.
//  This is not an edit transaction: file is replaced on disk.

void m_rotate90_thumb(GtkWidget *, cchar *)                                //  v.13.10
{
   void  turnfunc_thumb(GtkWidget *, cchar *menu);

   int      err, Nth = 0;
   char     *turn_file;

   if (! clicked_file) return;
   turn_file = clicked_file;
   clicked_file = 0;
   
   if (checkpend("all")) return;                                           //  v.13.12

   err = f_open(turn_file,Nth,0,1);                                        //  open clicked image file
   zfree(turn_file);
   if (err) return;

   Fmenulock = 1;                                                          //  block other edit funcs       v.13.12

   zdrotate90 = zdialog_new(ZTX("Rotate Image 90º"),Mwin,null);
   zdialog_add_widget(zdrotate90,"vbox","vb1","dialog",0,"space=3");
   GtkWidget *tbar = create_toolbar(zdialog_widget(zdrotate90,"vb1"),32);
   gtk_toolbar_set_style(GTK_TOOLBAR(tbar),GTK_TOOLBAR_BOTH);
   add_toolbar_button(tbar,"left",ZTX("rotate -90º"),"rotate-left.png",turnfunc_thumb);
   add_toolbar_button(tbar,"right",ZTX("rotate +90º"),"rotate-right.png",turnfunc_thumb);
   add_toolbar_button(tbar,"cancel",0,"cancel.png",turnfunc_thumb);
   zdialog_resize(zdrotate90,200,0);
   zdialog_run(zdrotate90,0,"save");
   zdialog_wait(zdrotate90);
   zdialog_free(zdrotate90);

   Fmenulock = 0;                                                          //  v.13.12
   gallery(curr_file,"paint");
   m_viewmode(0,"G");
   return;
}


void turnfunc_thumb(GtkWidget *, cchar *menu)
{
   int      angle = 0;

   if (strEqu(menu,"cancel")) {
      zdialog_destroy(zdrotate90);
      return;
   }

   if (strEqu(menu,"right")) angle = +90;
   if (strEqu(menu,"left")) angle = -90;
   zdialog_destroy(zdrotate90);

   E0pxm = PXM_load(curr_file,1);                                          //  load poss. 16-bit image
   if (E0pxm) {
      E3pxm = PXM_rotate(E0pxm,angle);                                     //  rotate
      PXM_free(E0pxm);
      E0pxm = E3pxm;
      E3pxm = 0;
   }

   if (strEqu(curr_file_type,"other"))                                     //  if unsupported type, use jpg
      strcpy(curr_file_type,"jpg");

   f_save(curr_file,curr_file_type,curr_file_bpc);                         //  save rotated file
   f_open_saved();

   strcpy(curr_file_type,f_save_type);                                     //  update curr_file_xxx from f_save_xxx
   curr_file_size = f_save_size;
   PXM_free(E0pxm);

   return;
}


/**************************************************************************/

//  trim (crop) and/or rotate an image                                     //  combine trim and rotate      v.13.10

namespace trimrotate
{
// int      trimrect[4];                                                   //  current trim rectangle (GLOBAL fotoxx.h)
// int      trimsize[2];                                                   //  current trim size (GLOBAL fotoxx.h)
   int      ptrimrect[4];                                                  //  prior trim rectangle (this image)
   int      ptrimsize[2];                                                  //  prior trim size (previous image)
   float    trimR;                                                         //  trim ratio, width/height
   float    rotate_goal, rotate_angle;                                     //  target and actual rotation
   int      rotatecrop1 = 0, rotatecrop2;                                  //  default rotate with no autocrop
   int      E0ww, E0hh, E3ww, E3hh;                                        //  full size image and preview size
   float    FPR;                                                           //  full size / preview ratio
   int      Fguidelines, guidelineX, guidelineY;                           //  horz/vert guidelines for rotate
   int      Fcorner, KBcorner, Frotate;
   int      Fmax = 0;

   editfunc EFtrimrotate;

   void   initz(int autotrim);
   void   dialog();
   int    dialog_event(zdialog *zd, cchar *event);
   void   trim_customize();
   void   mousefunc();
   void   KBfunc(int key);
   void   trim_limits();
   void   trim_darkmargins();
   void   trim_final();
   void   rotate_func();
   void   trim_fullsize();
   void   drawlines();
}


//  menu function

void m_trimrotate(GtkWidget *, cchar *menu)
{
   using namespace trimrotate;

   int         ii, xmargin, ymargin;
   cchar       *trim_message = ZTX("Trim: drag middle to move, drag corners to resize");
   cchar       *rotate_mess = ZTX("Minor rotate: drag right edge with mouse");
   char        text[20];
   zdialog     *zd;

   F1_help_topic = "trim_rotate";

   EFtrimrotate.menufunc = m_trimrotate;                                   //  menu function
   EFtrimrotate.funcname = "trim_rotate";
   EFtrimrotate.FprevReq = 1;                                              //  use preview
   EFtrimrotate.Frestart = 1;                                              //  allow restart
   EFtrimrotate.mousefunc = mousefunc;
   if (! edit_setup(EFtrimrotate)) return;                                 //  setup edit
   
   E0ww = E0pxm->ww;                                                       //  full-size image dimensions
   E0hh = E0pxm->hh;
   E3ww = E3pxm->ww;                                                       //  preview image (possibly smaller)
   E3hh = E3pxm->hh;
   FPR = 1.0 * (E0ww + E0hh) / (E3ww + E3hh);                              //  full image / preview ratio

   if (menu && strEqu(menu,"autotrim")) {                                  //  autotrim: keep preset trim rectangle
      trimsize[0] = trimrect[2] - trimrect[0];                             //  (full image scale)
      trimsize[1] = trimrect[3] - trimrect[1];
      trimR = 1.0 * trimsize[0] / trimsize[1];                             //  trim ratio = width/height
   }

   else
   {
      trimsize[0] = 0.8 * E0ww;                                            //  initial trim rectangle, 80% image size
      trimsize[1] = 0.8 * E0hh;
      xmargin = 0.5 * (E0ww - trimsize[0]);                                //  set balanced margins
      ymargin = 0.5 * (E0hh - trimsize[1]);
      trimrect[0] = xmargin;
      trimrect[2] = E0ww - xmargin;
      trimrect[1] = ymargin;
      trimrect[3] = E0hh - ymargin;
      trimR = 1.0 * trimsize[0] / trimsize[1];                             //  trim ratio = width/height
   }

   trimrect[0] = trimrect[0] / FPR;                                        //  use preview scale from now on
   trimrect[1] = trimrect[1] / FPR;
   trimrect[2] = trimrect[2] / FPR;
   trimrect[3] = trimrect[3] / FPR;

   ptrimrect[0] = 0;                                                       //  set prior trim rectangle
   ptrimrect[1] = 0;                                                       //    = 100% of image
   ptrimrect[2] = E3ww;
   ptrimrect[3] = E3hh;

   rotate_goal = rotate_angle = 0;                                         //  initially no rotation
   rotatecrop2 = rotatecrop1;                                              //  rotate autocrop setting

/***
       ________________________________________________________
      |                                                        |
      |   Trim: drag middle to move, drag corners to resize    |
      |                                                        |
      |  width [____|-|+]  height [____|-|+]   ratio 1.5       |
      |  trim size: [Max] [Prev] [Invert]   [x] Lock Ratio     |
      |  [1:1] [2:1] [3:2] [4:3] [16:9] [gold] [Customize]     |
      |                                                        |
      |    Minor rotate: drag right edge with mouse            |
      |  Rotate: degrees [____|-+]    [x] auto-trim            |
      |  [left -90]  [right +90]                               |
      |                                                        |
      |                        [Grid] [Apply] [Done] [Cancel]  |
      |________________________________________________________|

***/

   zd = zdialog_new(ZTX("Trim/Rotate"),Mwin,Bgrid,Bapply,Bdone,Bcancel,null);
   EFtrimrotate.zd = zd;

   zdialog_add_widget(zd,"label","labtrim","dialog",trim_message,"space=3");

   zdialog_add_widget(zd,"hbox","hb1","dialog");
   zdialog_add_widget(zd,"label","labW","hb1",ZTX("width"),"space=5");
   zdialog_add_widget(zd,"spin","width","hb1","20|20000|1|1000");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=5");
   zdialog_add_widget(zd,"label","labH","hb1",ZTX("height"),"space=5");
   zdialog_add_widget(zd,"spin","height","hb1","20|20000|1|600");
   zdialog_add_widget(zd,"label","space","hb1",0,"space=5");
   zdialog_add_widget(zd,"label","labR","hb1",ZTX("ratio"),"space=5");
   zdialog_add_widget(zd,"label","ratio","hb1","1.67   ");

   zdialog_add_widget(zd,"hbox","hb2","dialog");
   zdialog_add_widget(zd,"label","lab2","hb2",ZTX("trim size:"),"space=5");
   zdialog_add_widget(zd,"button","max","hb2",Bmax,"space=5");
   zdialog_add_widget(zd,"button","prev","hb2",Bprev,"space=5");
   zdialog_add_widget(zd,"button","invert","hb2",Binvert,"space=5");
   zdialog_add_widget(zd,"check","lock","hb2",ZTX("Lock Ratio"),"space=15");

   zdialog_add_widget(zd,"hbox","hb3","dialog");                                       //  ratio buttons inserted here
   for (ii = 0; ii < 6; ii++)
      zdialog_add_widget(zd,"button",trimbuttons[ii],"hb3",trimbuttons[ii],"space=5");
   zdialog_add_widget(zd,"button","custom","hb3",ZTX("Customize"),"space=5");          //  and [custom] button
   
   zdialog_add_widget(zd,"hsep","sep1","dialog",0,"space=3");

   zdialog_add_widget(zd,"label","labrotmess","dialog",rotate_mess,"space=3");

   zdialog_add_widget(zd,"hbox","hb5","dialog");
   zdialog_add_widget(zd,"label","labrotate","hb5",ZTX("Rotate: degrees"),"space=5");
   zdialog_add_widget(zd,"spin","degrees","hb5","-360|360|0.1|0");
   zdialog_add_widget(zd,"check","rotatecrop","hb5",ZTX("auto-trim"),"space=20");

   zdialog_add_widget(zd,"hbox","hb6","dialog");
   zdialog_add_widget(zd,"label","lableft","hb6",Bleft,"space=5");
   zdialog_add_widget(zd,"button","-90","hb6","-90");
   zdialog_add_widget(zd,"label","space","hb6",0,"space=10");
   zdialog_add_widget(zd,"label","labr","hb6",Bright,"space=5");
   zdialog_add_widget(zd,"button","+90","hb6","+90");

   zd = EFtrimrotate.zd;

   zdialog_stuff(zd,"width",trimsize[0]);                                  //  stuff width, height, ratio
   zdialog_stuff(zd,"height",trimsize[1]);
   sprintf(text,"%.2f  ",trimR);
   zdialog_stuff(zd,"ratio",text);
   zdialog_stuff(zd,"degrees",0);
   zdialog_stuff(zd,"rotatecrop",rotatecrop1);                             //  preserve rotate autocrop setting

   takeMouse(mousefunc,dragcursor);                                        //  connect mouse function
   currgrid = 1;                                                           //  use trim/rotate grid            v.13.11
   PXM_free(E9pxm);
   E9pxm = PXM_copy(E3pxm);
   Fzoom = 0;
   trim_darkmargins();

   zdialog_run(zd,dialog_event,"save");                                    //  run dialog - parallel
   
   if (Fmax) {
      Fmax = 0;
      zdialog_send_event(zd,"max");
   }

   return;
}


//  dialog event and completion callback function

int trimrotate::dialog_event(zdialog *zd, cchar *event)
{
   using namespace trimrotate;

   static int  flip = 0;
   int         width, height, delta;
   int         ii, rlock;
   float       r1, r2, ratio = 0;
   char        text[20];
   cchar       *pp;
   
   if (strEqu(event,"done")) zd->zstat = 3;                                //  apply and quit               v.13.12
   if (strEqu(event,"enter")) zd->zstat = 3;                               //  v.14.03

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) {                                                //  grid
         zd->zstat = 0;                                                    //  keep dialog active
         m_gridlines(0,"grid 1");
         return 1;
      }

      if (zd->zstat == 2) {                                                //  apply
         trim_fullsize();                                                  //  use full-size images
         edit_done(0);
         Fmax = 1;
         m_trimrotate(0,0);                                                //  restart edit
         return 1;
      }

      if (zd->zstat == 3) {                                                //  done
         trim_fullsize();                                                  //  use full-size images
         edit_done(0);
         currgrid = 0;                                                     //  restore normal grid settings
         return 1;
      }

      draw_toplines(2);                                                    //  erase trim rectangle
      erase_topcircles();                                                  //    and target circle
      currgrid = 0;                                                        //  restore normal grid settings
      edit_cancel(0);                                                      //  cancel or kill
      return 1;
   }

   if (strEqu(event,"custom")) {
      edit_cancel(0);                                                      //  cancel edit
      trim_customize();                                                    //  customize dialog
      m_trimrotate(0,0);                                                   //  restart edit
      return 1;
   }

   if (strEqu(event,"focus")) {
      takeMouse(mousefunc,dragcursor);                                     //  connect mouse function
      return 1;
   }

   if (strEqu(event,"prev"))                                               //  use width/height from prior        v.14.01
   {                                                                       //    image trim 
      width = ptrimsize[0];
      height = ptrimsize[1];
      if (width < 20 || height < 20) return 1;                             //  no value established, ignore
      if (width > E0ww) width = E0ww;
      if (height > E0hh) height = E0hh;
      zdialog_stuff(zd,"width",width);
      zdialog_stuff(zd,"height",height);
      event = "width";                                                     //  process same as manual inputs
   }

   if (strstr("width height",event))                                       //  get direct width/height inputs
   {
      zdialog_fetch(zd,"width",width);                                     //  full image scale
      zdialog_fetch(zd,"height",height);

      if (width < trimsize[0])                                             //  increase full scale change as needed
         while (int(width/FPR) == int(trimsize[0]/FPR)) width--;           //    to make a change at preview scale

      if (width > trimsize[0])
         while (int(width/FPR) == int(trimsize[0]/FPR)) width++;

      if (height < trimsize[1])
         while (int(height/FPR) == int(trimsize[1]/FPR)) height--;

      if (height > trimsize[1])
         while (int(height/FPR) == int(trimsize[1]/FPR)) height++;

      zdialog_fetch(zd,"lock",rlock);                                      //  lock ratio on/off

      width = width / FPR;                                                 //  preview scale
      height = height / FPR;

      if (strEqu(event,"width")) {
         if (width > E3ww) width = E3ww;
         if (rlock) {                                                      //  ratio locked
            height = width / trimR + 0.5;                                  //  try to keep ratio
            if (height > E3hh) height = E3hh;
         }
      }

      if (strEqu(event,"height")) {
         if (height > E3hh) height = E3hh;
         if (rlock) {
            width = height * trimR + 0.5;
            if (width > E3ww) width = E3ww;
         }
      }

      flip = 1 - flip;                                                     //  alternates 0, 1, 0, 1 ...

      delta = width - int(trimsize[0]/FPR);

      if (delta > 0) {                                                     //  increased width
         trimrect[0] = trimrect[0] - delta / 2;                            //  left and right sides equally
         trimrect[2] = trimrect[2] + delta / 2;
         if (delta % 2) {                                                  //  if increase is odd
            trimrect[0] = trimrect[0] - flip;                              //    add 1 alternatively to each side
            trimrect[2] = trimrect[2] + 1 - flip;
         }
         if (trimrect[0] < 0) {                                            //  add balance to upper limit
            trimrect[2] = trimrect[2] - trimrect[0];
            trimrect[0] = 0;
         }
         if (trimrect[2] > E3ww) {                                         //  add balance to lower limit
            trimrect[0] = trimrect[0] - trimrect[2] + E3ww;
            trimrect[2] = E3ww;
         }
      }

      if (delta < 0) {                                                     //  decreased width
         trimrect[0] = trimrect[0] - delta / 2;
         trimrect[2] = trimrect[2] + delta / 2;
         if (delta % 2) {
            trimrect[0] = trimrect[0] + flip;
            trimrect[2] = trimrect[2] - 1 + flip;
         }
      }

      delta = height - int(trimsize[1]/FPR);

      if (delta > 0) {                                                     //  increased height
         trimrect[1] = trimrect[1] - delta / 2;                            //  top and bottom sides equally
         trimrect[3] = trimrect[3] + delta / 2;
         if (delta % 2) {                                                  //  if increase is odd
            trimrect[1] = trimrect[1] - flip;                              //    add 1 alternatively to each side
            trimrect[3] = trimrect[3] + 1 - flip;
         }
         if (trimrect[1] < 0) {
            trimrect[3] = trimrect[3] - trimrect[1];
            trimrect[1] = 0;
         }
         if (trimrect[3] > E3hh) {
            trimrect[1] = trimrect[1] - trimrect[3] + E3hh;
            trimrect[3] = E3hh;
         }
      }

      if (delta < 0) {                                                     //  decreased height
         trimrect[1] = trimrect[1] - delta / 2;
         trimrect[3] = trimrect[3] + delta / 2;
         if (delta % 2) {
            trimrect[1] = trimrect[1] + flip;
            trimrect[3] = trimrect[3] - 1 + flip;
         }
      }

      if (trimrect[0] < 0) trimrect[0] = 0;                                //  keep within limits
      if (trimrect[2] > E3ww) trimrect[2] = E3ww;                          //  use ww not ww-1
      if (trimrect[1] < 0) trimrect[1] = 0;
      if (trimrect[3] > E3hh) trimrect[3] = E3hh;

      width = trimrect[2] - trimrect[0];                                   //  new width and height
      height = trimrect[3] - trimrect[1];

      if (width > E3ww) width = E3ww;                                      //  limit to actual size      v.13.11
      if (height > E3hh) height = E3hh;

      width = width * FPR;                                                 //  full image scale
      height = height * FPR;

      trimsize[0] = width;                                                 //  new trim size
      trimsize[1] = height;

      zdialog_stuff(zd,"width",width);                                     //  update dialog values
      zdialog_stuff(zd,"height",height);

      if (! rlock)                                                         //  set new ratio if not locked
         trimR = 1.0 * trimsize[0] / trimsize[1];

      sprintf(text,"%.2f  ",trimR);                                        //  stuff new ratio
      zdialog_stuff(zd,"ratio",text);

      trim_darkmargins();                                                  //  show trim area in image
      return 1;
   }

   if (strEqu(event,"max"))                                                //  maximize trim rectangle            v.14.01
   {
      trimrect[0] = 0;
      trimrect[1] = 0;
      trimrect[2] = E3ww;
      trimrect[3] = E3hh;

      width = E3ww * FPR;                                                  //  full scale trim size
      height = E3hh * FPR;

      trimsize[0] = width;                                                 //  new trim size
      trimsize[1] = height;

      zdialog_stuff(zd,"width",width);                                     //  update dialog values
      zdialog_stuff(zd,"height",height);

      trimR = 1.0 * trimsize[0] / trimsize[1];

      sprintf(text,"%.2f  ",trimR);                                        //  stuff new ratio
      zdialog_stuff(zd,"ratio",text);
      zdialog_stuff(zd,"lock",0);                                          //  set no ratio lock

      trim_darkmargins();                                                  //  show trim area in image
      return 1;
   }

   if (strEqu(event,"invert"))                                             //  invert width/height dimensions     v.14.01
      ratio = 1.0 / trimR;                                                 //  mark ratio changed

   for (ii = 0; ii < 6; ii++)                                              //  trim ratio buttons
      if (strEqu(event,trimbuttons[ii])) break;
   if (ii < 6) {
      r1 = r2 = ratio = 0;
      pp = strField(trimratios[ii],':',1);
      if (pp) r1 = atof(pp);
      pp = strField(trimratios[ii],':',2);
      if (pp) r2 = atof(pp);
      if (r1 > 0 && r2 > 0) ratio = r1/r2;
      if (ratio < 0.1 || ratio > 10) ratio = 1.0;
      if (! ratio) return 1;
      zdialog_stuff(zd,"lock",1);                                          //  assume lock is wanted
      trimR = ratio;
   }
   
   if (ratio)                                                              //  ratio was changed
   {
      trimR = ratio;

      if (trimrect[2] - trimrect[0] > trimrect[3] - trimrect[1])
         trimrect[3] = trimrect[1] + (trimrect[2] - trimrect[0]) / trimR;  //  adjust smaller dimension
      else
         trimrect[2] = trimrect[0] + (trimrect[3] - trimrect[1]) * trimR;

      if (trimrect[2] > E3ww) {                                            //  if off the right edge,
         trimrect[2] = E3ww;                                               //  adjust height
         trimrect[3] = trimrect[1] + (trimrect[2] - trimrect[0]) / trimR;
      }

      if (trimrect[3] > E3hh) {                                            //  if off the bottom edge,
         trimrect[3] = E3hh;                                               //  adjust width
         trimrect[2] = trimrect[0] + (trimrect[3] - trimrect[1]) * trimR;
      }

      trimsize[0] = FPR * (trimrect[2] - trimrect[0]);                     //  new rectangle dimensions
      trimsize[1] = FPR * (trimrect[3] - trimrect[1]);

      zdialog_stuff(zd,"width",trimsize[0]);                               //  stuff width, height, ratio
      zdialog_stuff(zd,"height",trimsize[1]);
      sprintf(text,"%.2f  ",trimR);
      zdialog_stuff(zd,"ratio",text);

      trim_darkmargins();                                                  //  update trim area in image
      return 1;
   }
   
   if (strstr("+90 -90 degrees",event))                                    //  rotate action
   {
      if (strEqu(event,"+90"))                                             //  90 deg. CW
         rotate_goal += 90;

      if (strEqu(event,"-90"))                                             //  90 deg. CCW
         rotate_goal -= 90;

      if (strEqu(event,"degrees"))                                         //  degrees adjustment
         zdialog_fetch(zd,"degrees",rotate_goal);

      if (rotate_goal > 180) rotate_goal -= 360;                           //  keep within -180 to +180 deg.
      if (rotate_goal < -180) rotate_goal += 360;
      if (fabsf(rotate_goal) < 0.01) rotate_goal = 0;                      //  = 0 within my precision
      zdialog_stuff(zd,"degrees",rotate_goal);

      rotate_func();                                                       //  E3 is rotated E1
   }

   if (strEqu(event,"rotatecrop")) {
      zdialog_fetch(zd,"rotatecrop",rotatecrop1);                          //  new autocrop setting
      rotate_func();                                                       //  E3 is rotated E1
   }

   return 1;
}


//  dialog to get custom trim button names and corresponding ratios

void trimrotate::trim_customize()
{
   using namespace trimrotate;

   char        text[20], blab[8], rlab[8];
   float       r1, r2, ratio;
   cchar       *pp;
   int         ii, zstat;

   zdialog *zd = zdialog_new(ZTX("Trim Buttons"),Mwin,Bdone,Bcancel,null);
   zdialog_add_widget(zd,"hbox","hbb","dialog",0,"homog|space=3");
   zdialog_add_widget(zd,"hbox","hbr","dialog",0,"homog|space=3");
   zdialog_add_widget(zd,"label","label","hbb",ZTX("label"),"space=3");
   zdialog_add_widget(zd,"label","ratio","hbr",ZTX("ratio"),"space=3");

   strcpy(blab,"butt-0");
   strcpy(rlab,"ratio-0");

   for (ii = 0; ii < 6; ii++)                                              //  add current trimbuttons to dialog
   {
      blab[5] = '0' + ii;
      rlab[6] = '0' + ii;
      zdialog_add_widget(zd,"entry",blab,"hbb",trimbuttons[ii],"scc=6");
      zdialog_add_widget(zd,"entry",rlab,"hbr",trimratios[ii],"scc=6");
   }

   zdialog_run(zd,0,"mouse");                                              //  run dialog
   zstat = zdialog_wait(zd);                                               //  wait for complete

   if (zstat == 1)                                                         //  done
   {
      for (ii = 0; ii < 6; ii++)                                           //  get new button names
      {
         blab[5] = '0' + ii;
         zdialog_fetch(zd,blab,text,12);
         strTrim2(text);
         if (! *text) continue;
         zfree(trimbuttons[ii]);
         trimbuttons[ii] = zstrdup(text);
      }

      for (ii = 0; ii < 6; ii++)                                           //  get new ratios
      {
         rlab[6] = '0' + ii;
         zdialog_fetch(zd,rlab,text,12);
         strTrim2(text);
         r1 = r2 = ratio = 0;
         pp = strField(text,':',1);
         if (! pp) continue;
         r1 = atof(pp);
         pp = strField(text,':',2);
         if (! pp) continue;
         r2 = atof(pp);
         if (r1 > 0 && r2 > 0) ratio = r1/r2;
         if (ratio < 0.1 || ratio > 10) continue;
         zfree(trimratios[ii]);
         trimratios[ii] = zstrdup(text);
      }
   }

   zdialog_free(zd);                                                       //  kill dialog
   save_params();                                                          //  save parameters

   return;
}


//  trim/rotate mouse function

void trimrotate::mousefunc()
{
   using namespace trimrotate;

   int         mpx, mpy, xdrag, ydrag;
   int         moveall = 0;
   int         dx, dy, dd, d1, d2, d3, d4;
   zdialog     *zd = EFtrimrotate.zd;
   
   if (! LMclick && ! RMclick && ! Mxdrag && ! Mydrag) {                   //  no click or drag
      Fcorner = Frotate = 0;
      return;
   }

   if (LMclick)                                                            //  add vertical and horizontal
   {                                                                       //    lines at click position       v.13.10
      LMclick = 0;
      Fcorner = Frotate = 0;
      Fguidelines = 1;
      guidelineX = Mxclick;
      guidelineY = Myclick;
      trim_darkmargins();
      Fpaint2();
      return;
   }

   if (RMclick)                                                            //  remove the lines
   {
      RMclick = 0;
      Fcorner = Frotate = 0;
      Fguidelines = 0;
      trim_darkmargins();
      Fpaint2();
      return;
   }

   if (Mxdrag || Mydrag)                                                   //  drag
   {
      mpx = Mxdrag;
      mpy = Mydrag;
      xdrag = Mxdrag - Mxdown;
      ydrag = Mydrag - Mydown;
      Mxdown = Mxdrag;                                                     //  reset drag origin
      Mydown = Mydrag;
      Mxdrag = Mydrag = 0;
   }

   if (Frotate) goto mouse_rotate;                                         //  continue rotate in progress

   else if (Fcorner)
   {
      if (Fcorner == 1) { trimrect[0] = mpx; trimrect[1] = mpy; }          //  continue dragging corner with mouse
      if (Fcorner == 2) { trimrect[2] = mpx; trimrect[1] = mpy; }
      if (Fcorner == 3) { trimrect[2] = mpx; trimrect[3] = mpy; }
      if (Fcorner == 4) { trimrect[0] = mpx; trimrect[3] = mpy; }
   }

   else
   {
      moveall = 1;
      dd = 0.2 * (trimrect[2] - trimrect[0]);                              //  test if mouse is in the broad
      if (mpx < trimrect[0] + dd) moveall = 0;                             //    middle of the rectangle
      if (mpx > trimrect[2] - dd) moveall = 0;
      dd = 0.2 * (trimrect[3] - trimrect[1]);
      if (mpy < trimrect[1] + dd) moveall = 0;
      if (mpy > trimrect[3] - dd) moveall = 0;
   }

   if (moveall) {                                                          //  yes, move the whole rectangle
      trimrect[0] += xdrag;
      trimrect[2] += xdrag;
      trimrect[1] += ydrag;
      trimrect[3] += ydrag;
   }

   else if (! Fcorner)                                                     //  no, find closest corner to mouse
   {
      dx = mpx - trimrect[0];
      dy = mpy - trimrect[1];
      d1 = sqrt(dx*dx + dy*dy);                                            //  distance from NW corner

      dx = mpx - trimrect[2];
      dy = mpy - trimrect[1];
      d2 = sqrt(dx*dx + dy*dy);                                            //  NE

      dx = mpx - trimrect[2];
      dy = mpy - trimrect[3];
      d3 = sqrt(dx*dx + dy*dy);                                            //  SE

      dx = mpx - trimrect[0];
      dy = mpy - trimrect[3];
      d4 = sqrt(dx*dx + dy*dy);                                            //  SW

      Fcorner = 1;                                                         //  NW
      dd = d1;
      if (d2 < dd) { Fcorner = 2; dd = d2; }                               //  NE
      if (d3 < dd) { Fcorner = 3; dd = d3; }                               //  SE
      if (d4 < dd) { Fcorner = 4; dd = d4; }                               //  SW

      if (E3ww - mpx < dd && mpx > 0.9 * E3ww)                             //  mouse closer to right edge than corner
         if (mpy > 0.3 * E3hh && mpy < 0.7 * E3hh)                         //  if far from corners assume rotate
            goto mouse_rotate;                                             //  v.14.03

      if (Fcorner == 1) { trimrect[0] = mpx; trimrect[1] = mpy; }          //  move this corner to mouse
      if (Fcorner == 2) { trimrect[2] = mpx; trimrect[1] = mpy; }
      if (Fcorner == 3) { trimrect[2] = mpx; trimrect[3] = mpy; }
      if (Fcorner == 4) { trimrect[0] = mpx; trimrect[3] = mpy; }
   }
   
   KBcorner = Fcorner;                                                     //  remember last corner moved

   trim_limits();                                                          //  check margin limits and adjust if req. 
   trim_darkmargins();                                                     //  show trim area in image

   return;

// ------------------------------------------------------------------------

   mouse_rotate:

   Frotate = 1;
   Fcorner = 0;

   rotate_goal += 30.0 * ydrag / E3ww;                                     //  convert radians to degrees, reduced
   if (rotate_goal > 180) rotate_goal -= 360;                              //  keep within -180 to +180 deg.
   if (rotate_goal < -180) rotate_goal += 360;
   if (fabsf(rotate_goal) < 0.01) rotate_goal = 0;                         //  = 0 within my precision

   zdialog_stuff(zd,"degrees",rotate_goal);

   rotate_func();                                                          //  E3 is rotated E1
   return;
}


//  Keyboard function
//  KB arrow keys tweak the last selected corner 

void trimrotate::KBfunc(int key)                                           //  v.14.06
{
   using namespace trimrotate;
   
   int      xstep, ystep;
   
   if (! Fcorner) Fcorner = KBcorner;
   
   xstep = ystep = 0;
   if (key == GDK_KEY_Left) xstep = -1;
   if (key == GDK_KEY_Right) xstep = +1;
   if (key == GDK_KEY_Up) ystep = -1;
   if (key == GDK_KEY_Down) ystep = +1;

   if (Fcorner == 1) {                                                     //  NW
      trimrect[0] += xstep;
      trimrect[1] += ystep;
   }

   if (Fcorner == 2) {                                                     //  NE
      trimrect[2] += xstep;
      trimrect[1] += ystep;
   }

   if (Fcorner == 3) {                                                     //  SE
      trimrect[2] += xstep;
      trimrect[3] += ystep;
   }

   if (Fcorner == 4) {                                                     //  SW
      trimrect[0] += xstep;
      trimrect[3] += ystep;
   }

   trim_limits();                                                          //  check margin limits and adjust if req. 
   trim_darkmargins();                                                     //  show trim area in image

   return;
}


//  check new margins for sanity and enforce locked aspect ratio

void trimrotate::trim_limits()
{
   using namespace trimrotate;

   int         rlock, chop;
   float       drr;
   char        text[20];
   zdialog     *zd = EFtrimrotate.zd;

   if (trimrect[0] > trimrect[2]-10) trimrect[0] = trimrect[2]-10;         //  sanity limits
   if (trimrect[1] > trimrect[3]-10) trimrect[1] = trimrect[3]-10;

   zdialog_fetch(zd,"lock",rlock);                                         //  w/h ratio locked
   if (rlock && Fcorner) {
      if (Fcorner < 3)
         trimrect[3] = trimrect[1] + 1.0 * (trimrect[2] - trimrect[0]) / trimR;
      else
         trimrect[1] = trimrect[3] - 1.0 * (trimrect[2] - trimrect[0]) / trimR;
   }

   chop = 0;
   if (trimrect[0] < 0) {                                                  //  look for off the edge
      trimrect[0] = 0;                                                     //    after corner move
      chop = 1;
   }

   if (trimrect[2] > E3ww) { 
      trimrect[2] = E3ww;
      chop = 2;
   }

   if (trimrect[1] < 0) {
      trimrect[1] = 0;
      chop = 3;
   }

   if (trimrect[3] > E3hh) {
      trimrect[3] = E3hh;
      chop = 4;
   }

   if (rlock && chop) {                                                    //  keep ratio if off edge
      if (chop < 3)
         trimrect[3] = trimrect[1] + 1.0 * (trimrect[2] - trimrect[0]) / trimR;
      else
         trimrect[2] = trimrect[0] + 1.0 * (trimrect[3] - trimrect[1]) * trimR;
   }

   if (trimrect[0] > trimrect[2]-10) trimrect[0] = trimrect[2]-10;         //  sanity limits
   if (trimrect[1] > trimrect[3]-10) trimrect[1] = trimrect[3]-10;

   if (trimrect[0] < 0) trimrect[0] = 0;                                   //  keep within visible area
   if (trimrect[2] > E3ww) trimrect[2] = E3ww;
   if (trimrect[1] < 0) trimrect[1] = 0;
   if (trimrect[3] > E3hh) trimrect[3] = E3hh;

   trimsize[0] = FPR * (trimrect[2] - trimrect[0]);                        //  new rectangle dimensions
   trimsize[1] = FPR * (trimrect[3] - trimrect[1]);

   drr = 1.0 * trimsize[0] / trimsize[1];                                  //  new w/h ratio
   if (! rlock) trimR = drr;

   zdialog_stuff(zd,"width",trimsize[0]);                                  //  stuff width, height, ratio
   zdialog_stuff(zd,"height",trimsize[1]);
   sprintf(text,"%.2f  ",trimR);
   zdialog_stuff(zd,"ratio",text);
   
   return;
}


//  Darken image pixels outside of current trim margins.
//  messy logic: update pixmaps only for changed pixels to increase speed

void trimrotate::trim_darkmargins()
{
   using namespace trimrotate;

   int      ox1, oy1, ox2, oy2;                                            //  outer trim rectangle
   int      nx1, ny1, nx2, ny2;                                            //  inner trim rectangle
   int      px, py;
   float    *pix1, *pix3;

   if (trimrect[0] < 0) trimrect[0] = 0;                                   //  keep within visible area
   if (trimrect[2] > E3ww) trimrect[2] = E3ww;
   if (trimrect[1] < 0) trimrect[1] = 0;
   if (trimrect[3] > E3hh) trimrect[3] = E3hh;

   if (ptrimrect[0] < trimrect[0]) ox1 = ptrimrect[0];                     //  outer rectangle
   else ox1 = trimrect[0];
   if (ptrimrect[2] > trimrect[2]) ox2 = ptrimrect[2];
   else ox2 = trimrect[2];
   if (ptrimrect[1] < trimrect[1]) oy1 = ptrimrect[1];
   else oy1 = trimrect[1];
   if (ptrimrect[3] > trimrect[3]) oy2 = ptrimrect[3];
   else oy2 = trimrect[3];

   if (ptrimrect[0] > trimrect[0]) nx1 = ptrimrect[0];                     //  inner rectangle
   else nx1 = trimrect[0];
   if (ptrimrect[2] < trimrect[2]) nx2 = ptrimrect[2];
   else nx2 = trimrect[2];
   if (ptrimrect[1] > trimrect[1]) ny1 = ptrimrect[1];
   else ny1 = trimrect[1];
   if (ptrimrect[3] < trimrect[3]) ny2 = ptrimrect[3];
   else ny2 = trimrect[3];

   ox1 -= 2;                                                               //  expand outer rectangle
   oy1 -= 2;
   ox2 += 2;
   oy2 += 2;
   nx1 += 2;                                                               //  reduce inner rectangle
   ny1 += 2;
   nx2 -= 2;
   ny2 -= 2;

   if (ox1 < 0) ox1 = 0;
   if (oy1 < 0) oy1 = 0;
   if (ox2 > E3ww) ox2 = E3ww;
   if (oy2 > E3hh) oy2 = E3hh;

   if (nx1 < ox1) nx1 = ox1;
   if (ny1 < oy1) ny1 = oy1;
   if (nx2 > ox2) nx2 = ox2;
   if (ny2 > oy2) ny2 = oy2;

   if (nx2 < nx1) nx2 = nx1;
   if (ny2 < ny1) ny2 = ny1;

   for (py = oy1; py < ny1; py++)                                          //  top band of pixels
   for (px = ox1; px < ox2; px++)
   {
      pix1 = PXMpix(E9pxm,px,py);  
      pix3 = PXMpix(E3pxm,px,py);

      if (px < trimrect[0] || px > trimrect[2] || py < trimrect[1] || py > trimrect[3]) {
         pix3[0] = pix1[0] / 2;                                            //  outside trim margins
         pix3[1] = pix1[1] / 2;                                            //  50% brightness
         pix3[2] = pix1[2] / 2;
      }
      else {
         pix3[0] = pix1[0];                                                //  inside trim margins
         pix3[1] = pix1[1];                                                //  100% brightness
         pix3[2] = pix1[2];
      }
   }

   for (py = oy1; py < oy2; py++)                                          //  right band
   for (px = nx2; px < ox2; px++)
   {
      pix1 = PXMpix(E9pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      if (px < trimrect[0] || px > trimrect[2] || py < trimrect[1] || py > trimrect[3]) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else {
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   for (py = ny2; py < oy2; py++)                                          //  bottom band
   for (px = ox1; px < ox2; px++)
   {
      pix1 = PXMpix(E9pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      if (px < trimrect[0] || px > trimrect[2] || py < trimrect[1] || py > trimrect[3]) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else {
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   for (py = oy1; py < oy2; py++)                                          //  left band
   for (px = ox1; px < nx1; px++)
   {
      pix1 = PXMpix(E9pxm,px,py);
      pix3 = PXMpix(E3pxm,px,py);

      if (px < trimrect[0] || px > trimrect[2] || py < trimrect[1] || py > trimrect[3]) {
         pix3[0] = pix1[0] / 2;
         pix3[1] = pix1[1] / 2;
         pix3[2] = pix1[2] / 2;
      }
      else {
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   Fpaint3(ox1,oy1,ox2-ox1,ny1-oy1);                                       //  4 updated rectangles
   Fpaint3(nx2,oy1,ox2-nx2,oy2-oy1);
   Fpaint3(ox1,ny2,ox2-ox1,oy2-ny2);
   Fpaint3(ox1,oy1,nx1-ox1,oy2-oy1);
   
   drawlines();                                                            //  draw trim rectangle etc.

   ptrimrect[0] = trimrect[0];                                             //  set prior trim rectangle
   ptrimrect[2] = trimrect[2];                                             //    from current trim rectangle
   ptrimrect[1] = trimrect[1];
   ptrimrect[3] = trimrect[3];

   return;
}


//  final trim - cut margins off the full size image
//  E3 is the input image, rotated and with black edges cropped if chosen
//  E3 is the output image from user trim rectangle
//  FPR should be 1.0 and E3 should be full size

void trimrotate::trim_final()
{
   using namespace trimrotate;

   int      px1, py1, px2, py2;
   float    *pix3, *pix9;

   draw_toplines(2);                                                       //  erase trim rectangle
   erase_topcircles();                                                     //    and target circle

   if (trimrect[2] > E3ww) trimrect[2] = E3ww;                             //  final check
   if (trimrect[3] > E3hh) trimrect[3] = E3hh;

   trimsize[0] = trimrect[2] - trimrect[0];                                //  new rectangle dimensions
   trimsize[1] = trimrect[3] - trimrect[1];

   PXM_free(E9pxm);
   E9pxm = PXM_make(trimsize[0],trimsize[1]);                              //  new pixmap with requested size

   for (py1 = trimrect[1]; py1 < trimrect[3]; py1++)                       //  copy E3 (rotated) to new size
   for (px1 = trimrect[0]; px1 < trimrect[2]; px1++)
   {
      px2 = px1 - trimrect[0];
      py2 = py1 - trimrect[1];
      pix3 = PXMpix(E3pxm,px1,py1);
      pix9 = PXMpix(E9pxm,px2,py2);
      pix9[0] = pix3[0];
      pix9[1] = pix3[1];
      pix9[2] = pix3[2];
   }

   PXM_free(E3pxm);
   E3pxm = E9pxm;
   E9pxm = 0;

   E3ww = E3pxm->ww;                                                       //  update E3 dimensions
   E3hh = E3pxm->hh;
   FPR = 1.0 * (E0ww + E0hh) / (E3ww + E3hh);                              //  full image / preview ratio

   CEF->Fmods++;
   CEF->Fsaved = 0;
   
   ptrimsize[0] = trimsize[0] + 0.5;                                       //  save final trim size for poss.     v.14.01
   ptrimsize[1] = trimsize[1] + 0.5;                                       //    recall via [Previous] button

   Fpaint2();
   return;
}


//  rotate function
//  E3 = E1 with rotation and edges cropped if chosen.
//  E9 = E3 copy: used as source for darkened pixels in E3.

void trimrotate::rotate_func()
{
   using namespace trimrotate;

   int         px3, py3, px9, py9;
   int         wwcut, hhcut, ww, hh;
   float       trim_angle, radians;
   float       *pix3, *pix9;
   zdialog     *zd = EFtrimrotate.zd;

   zmainloop();                                                            //  updates before big CPU time     v.13.12

   if (fabs(rotate_goal) < 0.01) {                                         //  no rotation
      rotate_goal = rotate_angle = 0.0;
      zdialog_stuff(zd,"degrees",0.0);
      PXM_free(E3pxm);                                                     //  E1 >> E3
      E3pxm = PXM_copy(E1pxm);
      PXM_free(E9pxm);                                                     //  E1 >> E9
      E9pxm = PXM_copy(E1pxm);
      CEF->Fmods = 0;                                                      //  will be overridden by trim
   }

   if (rotate_goal != rotate_angle || rotatecrop2 != rotatecrop1)
   {
      rotate_angle = rotate_goal;
      rotatecrop2 = rotatecrop1;                                           //  remember autocrop setting

      PXM_free(E9pxm);
      E9pxm = PXM_rotate(E1pxm,rotate_angle);                              //  E9 is rotated E1

      if (rotatecrop1)                                                     //  edge cropping is chosen
      {
         trim_angle = fabsf(rotate_goal);
         while (trim_angle > 45) trim_angle -= 90;
         radians = fabsf(trim_angle / 57.296);
         wwcut = int(E9pxm->hh * sinf(radians) + 1);                       //  amount to crop off edges
         hhcut = int(E9pxm->ww * sinf(radians) + 1);

         if (wwcut > E9pxm->ww / 2 - 10)                                   //  do not cut 100%
            wwcut = E9pxm->ww / 2 - 10;
         if (hhcut > E9pxm->hh / 2 - 10)
            hhcut = E9pxm->hh / 2 - 10;

         ww = E9pxm->ww - 2 * wwcut;
         hh = E9pxm->hh - 2 * hhcut;

         PXM_free(E3pxm);
         E3pxm = PXM_make(ww,hh);                                          //  E3 = cropped E9

         for (py3 = 0; py3 < hh; py3++)                                    //  copy pixels E9 > E3
         for (px3 = 0; px3 < ww; px3++)
         {
            pix3 = PXMpix(E3pxm,px3,py3);
            px9 = px3 + wwcut;
            py9 = py3 + hhcut;
            pix9 = PXMpix(E9pxm,px9,py9);
            pix3[0] = pix9[0];
            pix3[1] = pix9[1];
            pix3[2] = pix9[2];
         }

         PXM_free(E9pxm);                                                  //  E9 = E3
         E9pxm = PXM_copy(E3pxm);
      }

      else {                                                               //  no edge cropping
         PXM_free(E3pxm);                                                  //  E3 = E9
         E3pxm = PXM_copy(E9pxm);
      }
   }

   E3ww = E3pxm->ww;                                                       //  update E3 dimensions
   E3hh = E3pxm->hh;

   if (trimrect[2] > E3ww) trimrect[2] = E3ww;                             //  contract trim rectangle if needed
   if (trimrect[3] > E3hh) trimrect[3] = E3hh;

   trimsize[0] = FPR * (trimrect[2] - trimrect[0]);                        //  new rectangle dimensions
   trimsize[1] = FPR * (trimrect[3] - trimrect[1]);

   zdialog_stuff(zd,"width",trimsize[0]);                                  //  stuff width, height
   zdialog_stuff(zd,"height",trimsize[1]);

   Fpaintnow();
   drawlines();                                                            //  draw trim rectangle etc.

   ptrimrect[0] = 0;                                                       //  set prior trim rectangle
   ptrimrect[1] = 0;
   ptrimrect[2] = E3ww;                                                    //    = 100% of image
   ptrimrect[3] = E3hh;

   return;
}


//  perform final rotate and trim on full-size image

void trimrotate::trim_fullsize()
{
   using namespace trimrotate;

   if (CEF->Fpreview)                                                      //  preview image was used
   {
      CEF->Fpreview = 0;

      PXM_free(E1pxm);                                                     //  free preview images
      PXM_free(E3pxm);
      PXM_free(E9pxm);

      E1pxm = PXM_copy(E0pxm);                                             //  E0 >> E1, full size image

      trimrect[0] = FPR * trimrect[0];                                     //  full scale trim rectangle
      trimrect[1] = FPR * trimrect[1];
      trimrect[2] = FPR * trimrect[2];
      trimrect[3] = FPR * trimrect[3];
      FPR = 1.0;

      trimsize[0] = trimrect[2] - trimrect[0];                             //  final trim size
      trimsize[1] = trimrect[3] - trimrect[1];

      rotate_angle = 0;                                                    //  rotate full size image
      rotate_func();                                                       //  E3 = E9 = rotated and edge-cropped E1
   }

   trim_final();                                                           //  trim E3 from user trim rectangle
   return;
}


//  draw lines on image: trim rectangle, guidelines, gridlines

void trimrotate::drawlines()
{
   using namespace trimrotate;

   int      px, py, radius;

   Ntoplines = 4;                                                          //  outline trim rectangle
   toplinex1[0] = trimrect[0];
   topliney1[0] = trimrect[1];
   toplinex2[0] = trimrect[2];
   topliney2[0] = trimrect[1];
   toplinex1[1] = trimrect[2];
   topliney1[1] = trimrect[1];
   toplinex2[1] = trimrect[2];
   topliney2[1] = trimrect[3];
   toplinex1[2] = trimrect[2];
   topliney1[2] = trimrect[3];
   toplinex2[2] = trimrect[0];
   topliney2[2] = trimrect[3];
   toplinex1[3] = trimrect[0];
   topliney1[3] = trimrect[3];
   toplinex2[3] = trimrect[0];
   topliney2[3] = trimrect[1];

   if (Fguidelines) {                                                      //  vert/horz rotate guidelines
      Ntoplines = 6;
      toplinex1[4] = guidelineX;
      topliney1[4] = 0;
      toplinex2[4] = guidelineX;
      topliney2[4] = E3hh-1;
      toplinex1[5] = 0;
      topliney1[5] = guidelineY;
      toplinex2[5] = E3ww-1;
      topliney2[5] = guidelineY;
   }

   draw_toplines(1);                                                       //  draw trim rectangle and guidelines

   erase_topcircles();
   px = 0.5 * (ptrimrect[0] + ptrimrect[2]);
   py = 0.5 * (ptrimrect[1] + ptrimrect[3]);
   radius = 0.02 * Mscale * (E3ww + E3hh);
   Fpaint3(px-radius,py-radius,2*radius,2*radius);                         //  bugfix                             v.14.01

   px = 0.5 * (trimrect[0] + trimrect[2]);
   py = 0.5 * (trimrect[1] + trimrect[3]);
   add_topcircle(px,py,radius/2,1);
   add_topcircle(px,py,radius/2+1,2);
   draw_topcircles();
   
   if (gridsettings[currgrid][GON]) Fpaint2();                             //  refresh gridlines, delayed         v.14.01

   return;
}


/**************************************************************************/

//  auto-trim image - set trim rectangle to exclude black margins
//                    left over from rotate or warp functions

void m_autotrim(GtkWidget *, cchar *)
{
   int      px1, py1, px2, py2;
   int      qx1, qy1, qx2, qy2;
   int      qx, qy, step1, step2;
   int      area1, area2, Fgrow = 0;
   float    *ppix;

   if (! curr_file) return;
   if (! E0pxm) {
      E0pxm = PXM_load(curr_file,1);                                       //  create E0 image for my use      v.13.11
      if (! E0pxm) return;
   }

   if (checkpend("edit lock")) return;                                     //  v.13.12
   Fmenulock = 1;
   Ffuncbusy++;  

   px1 = 0.4 * Fpxb->ww;                                                   //  select small rectangle in the middle
   py1 = 0.4 * Fpxb->hh;
   px2 = 0.6 * Fpxb->ww;
   py2 = 0.6 * Fpxb->hh;

   step1 = 0.02 * (Fpxb->ww + Fpxb->hh);                                   //  start with big search steps
   step2 = 0.2 * step1;                                                    //  v.13.11
   if (step2 < 1) step2 = 1;

   while (true)
   {
      while (true)
      {
         Fgrow = 0;

         area1 = (px2 - px1) * (py2 - py1);                                //  area of current selection rectangle
         area2 = area1;

         for (qx1 = px1-step1; qx1 <= px1+step1; qx1 += step2)             //  loop, vary NW and SE corners +/-
         for (qy1 = py1-step1; qy1 <= py1+step1; qy1 += step2)
         for (qx2 = px2-step1; qx2 <= px2+step1; qx2 += step2)
         for (qy2 = py2-step1; qy2 <= py2+step1; qy2 += step2)
         {
            if (qx1 < 0) continue;                                         //  check image limits
            if (qy1 < 0) continue;
            if (qx1 > 0.5 * Fpxb->ww) continue;
            if (qy1 > 0.5 * Fpxb->hh) continue;
            if (qx2 > Fpxb->ww-1) continue;
            if (qy2 > Fpxb->hh-1) continue;
            if (qx2 < 0.5 * Fpxb->ww) continue;
            if (qy2 < 0.5 * Fpxb->hh) continue;

            ppix = PXMpix(E0pxm,qx1,qy1);                                  //  check 4 corners are not
            if (ppix[0] < 1 && ppix[1] < 1 && ppix[2] < 1) continue;       //    in the black margin zones
            ppix = PXMpix(E0pxm,qx2,qy1);
            if (ppix[0] < 1 && ppix[1] < 1 && ppix[2] < 1) continue;       //  check all colors             v.13.11
            ppix = PXMpix(E0pxm,qx2,qy2);
            if (ppix[0] < 1 && ppix[1] < 1 && ppix[2] < 1) continue;
            ppix = PXMpix(E0pxm,qx1,qy2);
            if (ppix[0] < 1 && ppix[1] < 1 && ppix[2] < 1) continue;

            area2 = (qx2 - qx1) * (qy2 - qy1);                             //  look for larger enclosed area
            if (area2 <= area1) continue;

            for (qx = qx1; qx < qx2; qx++) {                               //  check top/bottom sides don't intersect
               ppix = PXMpix(E0pxm,qx,qy1);                                //    the black margin zones
               if (ppix[0] < 1 && ppix[1] < 1 && ppix[2] < 1) break;
               ppix = PXMpix(E0pxm,qx,qy2);
               if (ppix[0] < 1 && ppix[1] < 1 && ppix[2] < 1) break;
            }
            if (qx < qx2) continue;

            for (qy = qy1; qy < qy2; qy++) {                               //  also left/right sides
               ppix = PXMpix(E0pxm,qx1,qy);
               if (ppix[0] < 1 && ppix[1] < 1 && ppix[2] < 1) break;
               ppix = PXMpix(E0pxm,qx2,qy);
               if (ppix[0] < 1 && ppix[1] < 1 && ppix[2] < 1) break;
            }
            if (qy < qy2) continue;

            Fgrow = 1;                                                     //  successfully grew the rectangle
            px1 = qx1;                                                     //  new bigger rectangle coordinates
            py1 = qy1;                                                     //    for the next search iteration
            px2 = qx2;
            py2 = qy2;
            goto breakout;                                                 //  leave all 4 loops      v.13.08
         }

      breakout:
         if (! Fgrow) break;
         zmainloop();                                                      //  v.13.11
      }

      if (step2 == 1) break;                                               //  done

      step1 = 0.6 * step1;                                                 //  reduce search step size
      if (step1 < 2) step1 = 2;
      step2 = 0.2 * step1;
      if (step2 < 1) step2 = 1;
   }

   trimrect[0] = px1;                                                      //  set parameters for trim function
   trimrect[2] = px2;
   trimrect[1] = py1;
   trimrect[3] = py2;

   Fmenulock = 0;                                                          //  v.13.12 
   Ffuncbusy--;  

   m_trimrotate(0,"autotrim");                                             //  perform regular trim with
   return;                                                                 //    pre-set margins
}


/**************************************************************************/

//  Resize (rescale) image
//
//  Output pixels are composites of input pixels, e.g. 2/3 size means
//  that 3x3 input pixels are mapped into 2x2 output pixels, and an
//  image size of 1000 x 600 becomes 667 x 400.

int      resize_ww0, resize_hh0;                                           //  original size
int      resize_ww1, resize_hh1;                                           //  new size
editfunc    EFresize;

void   resize_initz(zdialog *zd);
int    resize_dialog_event(zdialog *zd, cchar *event);
void * resize_thread(void *);


void m_resize(GtkWidget *, cchar *)
{
   cchar  *lockmess = ZTX("Lock aspect ratio");

   F1_help_topic = "resize";

   EFresize.menufunc = m_resize;
   EFresize.funcname = "resize";
   EFresize.Frestart = 1;                                                  //  allow restart             v.13.04
   EFresize.threadfunc = resize_thread;                                    //  thread function
   if (! edit_setup(EFresize)) return;                                     //  setup edit

/****
                  pixels     percent
          width   [______]   [______]
          height  [______]   [______]

          presets [2/3] [1/2] [1/3] [1/4] [Prev]

          [_] lock aspect ratio

                            [done] [cancel]
****/

   zdialog *zd = zdialog_new(ZTX("Resize Image"),Mwin,Bdone,Bcancel,null);
   EFresize.zd = zd;
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"vbox","vb11","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb12","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vb13","hb1",0,"homog|space=5");
   zdialog_add_widget(zd,"label","placeholder","vb11",0);
   zdialog_add_widget(zd,"label","labw","vb11",Bwidth);
   zdialog_add_widget(zd,"label","labh","vb11",Bheight);
   zdialog_add_widget(zd,"label","labpix","vb12","pixels");
   zdialog_add_widget(zd,"spin","wpix","vb12","20|20000|1|20");
   zdialog_add_widget(zd,"spin","hpix","vb12","20|20000|1|20");
   zdialog_add_widget(zd,"label","labpct","vb13",Bpercent);
   zdialog_add_widget(zd,"spin","wpct","vb13","1|500|0.1|100");
   zdialog_add_widget(zd,"spin","hpct","vb13","1|500|0.1|100");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","preset","hb2",Bpresets,"space=5");
   zdialog_add_widget(zd,"button","b 3/4","hb2","3/4");
   zdialog_add_widget(zd,"button","b 2/3","hb2","2/3");
   zdialog_add_widget(zd,"button","b 1/2","hb2","1/2");
   zdialog_add_widget(zd,"button","b 1/3","hb2","1/3");
   zdialog_add_widget(zd,"button","b 1/4","hb2","1/4");
   zdialog_add_widget(zd,"button","prev","hb2",Bprev);
   zdialog_add_widget(zd,"hbox","hb3","dialog",0,"space=5");
   zdialog_add_widget(zd,"check","lock","hb3",lockmess,"space=5");

   resize_initz(zd);

   zdialog_run(zd,resize_dialog_event,"save");                             //  run dialog
   return;
}


//  initialization for new image

void resize_initz(zdialog *zd)
{
   resize_ww0 = E1pxm->ww;                                                 //  original width, height
   resize_hh0 = E1pxm->hh;
   zdialog_stuff(zd,"wpix",resize_ww0);
   zdialog_stuff(zd,"hpix",resize_hh0);
   zdialog_stuff(zd,"lock",1);
   return;
}


//  dialog event and completion callback function

int resize_dialog_event(zdialog *zd, cchar *event)
{
   int         lock;
   float       wpct1, hpct1;

   if (strEqu(event,"done")) zd->zstat = 1;                                //  apply and quit               v.13.12
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  v.14.03

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) {                                                //  done
         editresize[0] = resize_ww1;                                       //  remember size used
         editresize[1] = resize_hh1;
         edit_done(0);
      }
      else edit_cancel(0);                                                 //  cancel or kill

      Fzoom = 0;
      return 1;
   }

   if (strEqu(event,"focus")) return 1;                                    //  ignore focus                 v.13.09

   zdialog_fetch(zd,"wpix",resize_ww1);                                    //  get all widget values
   zdialog_fetch(zd,"hpix",resize_hh1);
   zdialog_fetch(zd,"wpct",wpct1);
   zdialog_fetch(zd,"hpct",hpct1);
   zdialog_fetch(zd,"lock",lock);

   if (strEqu(event,"b 3/4")) {
      resize_ww1 = (3 * resize_ww0 + 3) / 4;
      resize_hh1 = (3 * resize_hh0 + 3) / 4;
   }

   if (strEqu(event,"b 2/3")) {
      resize_ww1 = (2 * resize_ww0 + 2) / 3;
      resize_hh1 = (2 * resize_hh0 + 2) / 3;
   }

   if (strEqu(event,"b 1/2")) {
      resize_ww1 = (resize_ww0 + 1) / 2;
      resize_hh1 = (resize_hh0 + 1) / 2;
   }

   if (strEqu(event,"b 1/3")) {
      resize_ww1 = (resize_ww0 + 2) / 3;
      resize_hh1 = (resize_hh0 + 2) / 3;
   }

   if (strEqu(event,"b 1/4")) {
      resize_ww1 = (resize_ww0 + 3) / 4;
      resize_hh1 = (resize_hh0 + 3) / 4;
   }

   if (strEqu(event,"prev")) {                                             //  use previous resize values
      resize_ww1 = editresize[0];
      resize_hh1 = editresize[1];
   }

   if (strEqu(event,"wpct"))                                               //  width % - set pixel width
      resize_ww1 = int(wpct1 / 100.0 * resize_ww0 + 0.5);

   if (strEqu(event,"hpct"))                                               //  height % - set pixel height
      resize_hh1 = int(hpct1 / 100.0 * resize_hh0 + 0.5);

   if (lock && event[0] == 'w')                                            //  preserve width/height ratio
      resize_hh1 = int(resize_ww1 * (1.0 * resize_hh0 / resize_ww0) + 0.5);
   if (lock && event[0] == 'h')
      resize_ww1 = int(resize_hh1 * (1.0 * resize_ww0 / resize_hh0) + 0.5);

   if (resize_ww1 < 20 || resize_hh1 < 20) return 1;                       //  refuse tiny size       v.12.11

   hpct1 = 100.0 * resize_hh1 / resize_hh0;                                //  set percents to match pixels
   wpct1 = 100.0 * resize_ww1 / resize_ww0;

   zdialog_stuff(zd,"wpix",resize_ww1);                                    //  index all widget values
   zdialog_stuff(zd,"hpix",resize_hh1);
   zdialog_stuff(zd,"wpct",wpct1);
   zdialog_stuff(zd,"hpct",hpct1);

   signal_thread();                                                        //  update image, no wait for idle

   return 1;
}


//  do the resize job

void * resize_thread(void *)
{
   while (true)
   {
      thread_idle_loop();                                                  //  wait for signal

      mutex_lock(&Fpixmap_lock);
      PXM_free(E3pxm);
      E3pxm = PXM_rescale(E1pxm,resize_ww1,resize_hh1);                    //  rescale the edit image
      if (! E3pxm) thread_exit();                                          //  memory failure      v.13.03
      CEF->Fmods++;
      CEF->Fsaved = 0;
      mutex_unlock(&Fpixmap_lock);
      Fpaint2();
      zsleep(0.1);                                                         //  v.13.03
   }

   return 0;
}


/**************************************************************************/

//  flip an image horizontally or vertically

editfunc    EFflip;

void m_flip(GtkWidget *, cchar *)
{
   int flip_dialog_event(zdialog *zd, cchar *event);

   F1_help_topic = "flip_image";

   EFflip.menufunc = m_flip;
   EFflip.funcname = "flip";
   EFflip.Frestart = 1;                                                    //  allow restart             v.13.04
   if (! edit_setup(EFflip)) return;                                       //  setup edit

   zdialog *zd = zdialog_new(ZTX("Flip"),Mwin,Bdone,Bcancel,null);
   EFflip.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=3");
   zdialog_add_widget(zd,"button","horz","hb1",ZTX("horizontal"),"space=5");
   zdialog_add_widget(zd,"button","vert","hb1",ZTX("vertical"),"space=5");

   zdialog_run(zd,flip_dialog_event,"save");                               //  run dialog, parallel

   return;
}


//  dialog event and completion callback function

int flip_dialog_event(zdialog *zd, cchar *event)
{
   int flip_horz();
   int flip_vert();

   if (strEqu(event,"done")) zd->zstat = 1;                                //  apply and quit               v.13.12
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  v.14.03

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done(0);
      else edit_cancel(0);
      return 1;
   }

   if (strEqu(event,"horz")) flip_horz();
   if (strEqu(event,"vert")) flip_vert();
   return 1;
}


int flip_horz()
{
   int      px, py;
   float    *pix3, *pix9;

   E9pxm = PXM_copy(E3pxm);

   for (py = 0; py < E3pxm->hh; py++)
   for (px = 0; px < E3pxm->ww; px++)
   {
      pix3 = PXMpix(E3pxm,px,py);                                          //  image9 = flipped image3
      pix9 = PXMpix(E9pxm,E3pxm->ww-1-px,py);
      pix9[0] = pix3[0];
      pix9[1] = pix3[1];
      pix9[2] = pix3[2];
   }

   mutex_lock(&Fpixmap_lock);
   PXM_free(E3pxm);                                                        //  image9 >> image3
   E3pxm = E9pxm;
   E9pxm = 0;
   mutex_unlock(&Fpixmap_lock);

   CEF->Fmods++;
   CEF->Fsaved = 0;
   Fpaint2();
   return 0;
}


int flip_vert()
{
   int      px, py;
   float    *pix3, *pix9;

   E9pxm = PXM_copy(E3pxm);

   for (py = 0; py < E3pxm->hh; py++)
   for (px = 0; px < E3pxm->ww; px++)
   {
      pix3 = PXMpix(E3pxm,px,py);                                          //  image9 = flipped image3
      pix9 = PXMpix(E9pxm,px,E3pxm->hh-1-py);
      pix9[0] = pix3[0];
      pix9[1] = pix3[1];
      pix9[2] = pix3[2];
   }

   mutex_lock(&Fpixmap_lock);
   PXM_free(E3pxm);                                                        //  image9 >> image3
   E3pxm = E9pxm;
   E9pxm = 0;
   mutex_unlock(&Fpixmap_lock);

   CEF->Fmods++;
   CEF->Fsaved = 0;
   Fpaint2();
   return 0;
}


/**************************************************************************/

//  write text on top of the image

namespace writetext
{
   textattr_t  attr;                                                       //  text attributes and image

   char     file[1000] = "";                                               //  file for write_text data
   int      px, py;                                                        //  text position on image
   int      present;                                                       //  flag, write_text present on image

   int   dialog_event(zdialog *zd, cchar *event);                          //  dialog event function
   void  mousefunc();                                                      //  mouse event function
   void  write(int mode);                                                  //  write text on image

   editfunc    EFwritetext;
}


void m_write_text(GtkWidget *, cchar *menu)
{
   using namespace writetext;

   cchar    *intro = ZTX("Enter text, click/drag on image, right click to remove");

   F1_help_topic = "write_text";                                           //  user guide topic

   EFwritetext.menufunc = m_write_text;
   EFwritetext.funcname = "write_text";
   EFwritetext.Farea = 1;                                                  //  select area ignored
   EFwritetext.Frestart = 1;                                               //  allow restart                v.13.05
   if (! edit_setup(EFwritetext)) return;                                  //  setup edit

/***
       ____________________________________________________________
      |               Write Text on Image                          |
      |                                                            |
      |  Enter text, click/drag on image, right click to remove.   |
      |                                                            |
      |  Text [__________________________________________________] |
      |                                                            |
      |  [font] [FreeSans         ]  size [ 44|v]                  |
      |                                                            |
      |            color   transp.   width    angle                |
      |  text     [_____]  [___|v]           [___|v]               |
      |  backing  [_____]  [___|v]                                 |
      |  outline  [_____]  [___|v]  [___|v]                        |
      |  shadow   [_____]  [___|v]  [___|v]  [___|v]               |
      |                                                            |
      |  Text File: [Open] [Save]    Metadata: [Caption] [Comment] |
      |                                                            |
      |                                    [Apply] [Done] [Cancel] |
      |____________________________________________________________|

***/

   zdialog *zd = zdialog_new(ZTX("Write Text on Image"),Mwin,Bapply,Bdone,Bcancel,null);
   EFwritetext.zd = zd;
   EFwritetext.mousefunc = mousefunc;
   EFwritetext.menufunc = m_write_text;                                    //  allow restart          v.13.04

   zdialog_add_widget(zd,"label","intro","dialog",intro,"space=3");
   zdialog_add_widget(zd,"hbox","hbtext","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labtext","hbtext",ZTX("Text"),"space=5");
   zdialog_add_widget(zd,"frame","frtext","hbtext",0,"expand");
   zdialog_add_widget(zd,"edit","text","frtext","text","expand|wrap");     //  v.13.12

   zdialog_add_widget(zd,"hbox","hbfont","dialog",0,"space=5");
   zdialog_add_widget(zd,"button","fontbutt","hbfont",Bfont);
   zdialog_add_widget(zd,"entry","fontname","hbfont","FreeSans","space=2|expand");
   zdialog_add_widget(zd,"label","labsize","hbfont",Bsize,"space=2");
   zdialog_add_widget(zd,"spin","fontsize","hbfont","8|500|1|40");

   zdialog_add_widget(zd,"hbox","hbcol","dialog");
   zdialog_add_widget(zd,"vbox","vbcol1","hbcol",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vbcol2","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol3","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol4","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol5","hbcol",0,"homog|space=2");

   zdialog_add_widget(zd,"label","space","vbcol1");
   zdialog_add_widget(zd,"label","labtext","vbcol1",ZTX("text"));
   zdialog_add_widget(zd,"label","labback","vbcol1",ZTX("backing"));
   zdialog_add_widget(zd,"label","laboutln","vbcol1",ZTX("outline"));
   zdialog_add_widget(zd,"label","labshadow","vbcol1",ZTX("shadow"));

   zdialog_add_widget(zd,"label","labcol","vbcol2",ZTX("Color"));
   zdialog_add_widget(zd,"colorbutt","fgcolor","vbcol2","0|0|0");
   zdialog_add_widget(zd,"colorbutt","bgcolor","vbcol2","255|255|255");
   zdialog_add_widget(zd,"colorbutt","tocolor","vbcol2","255|0|0");
   zdialog_add_widget(zd,"colorbutt","shcolor","vbcol2","255|0|0");

   zdialog_add_widget(zd,"label","labcol","vbcol3",ZTX("Transparency"));
   zdialog_add_widget(zd,"spin","fgtransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"spin","bgtransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"spin","totransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"spin","shtransp","vbcol3","0|100|1|0");

   zdialog_add_widget(zd,"label","labw","vbcol4",ZTX("Width"));
   zdialog_add_widget(zd,"label","space","vbcol4");
   zdialog_add_widget(zd,"label","space","vbcol4");
   zdialog_add_widget(zd,"spin","towidth","vbcol4","0|30|1|0");
   zdialog_add_widget(zd,"spin","shwidth","vbcol4","0|50|1|0");

   zdialog_add_widget(zd,"label","labw","vbcol5",Bangle);
   zdialog_add_widget(zd,"spin","angle","vbcol5","-180|180|0.5|0");
   zdialog_add_widget(zd,"label","space","vbcol5");
   zdialog_add_widget(zd,"label","space","vbcol5");
   zdialog_add_widget(zd,"spin","shangle","vbcol5","-180|180|1|0");

   zdialog_add_widget(zd,"hbox","hbtext","dialog",0,"space=8");
   zdialog_add_widget(zd,"label","labtf","hbtext",ZTX("Text File:"),"space=3");
   zdialog_add_widget(zd,"button","loadtext","hbtext",Bopen);
   zdialog_add_widget(zd,"button","savetext","hbtext",Bsave);

   zdialog_add_widget(zd,"label","space","hbtext",0,"space=3");            //  v.13.12
   zdialog_add_widget(zd,"label","labmeta","hbtext",ZTX("Metadata:"),"space=3");
   zdialog_add_widget(zd,"button","caption","hbtext",ZTX("Caption"));
   zdialog_add_widget(zd,"button","comment","hbtext",ZTX("Comment"));
   
   zdialog_restore_inputs(zd);                                             //  restore prior inputs

   zdialog_fetch(zd,"text",attr.text,1000);                                //  get defaults or prior inputs
   zdialog_fetch(zd,"fontname",attr.font,80);
   zdialog_fetch(zd,"fontsize",attr.size);
   zdialog_fetch(zd,"angle",attr.angle);
   zdialog_fetch(zd,"fgcolor",attr.color[0],20);
   zdialog_fetch(zd,"bgcolor",attr.color[1],20);
   zdialog_fetch(zd,"tocolor",attr.color[2],20);
   zdialog_fetch(zd,"shcolor",attr.color[3],20);
   zdialog_fetch(zd,"fgtransp",attr.transp[0]);
   zdialog_fetch(zd,"bgtransp",attr.transp[1]);
   zdialog_fetch(zd,"totransp",attr.transp[2]);
   zdialog_fetch(zd,"shtransp",attr.transp[3]);
   zdialog_fetch(zd,"towidth",attr.towidth);
   zdialog_fetch(zd,"shwidth",attr.shwidth);
   zdialog_fetch(zd,"shangle",attr.shangle);

   gentext(&attr);                                                         //  initial text

   takeMouse(mousefunc,dragcursor);                                        //  connect mouse function
   present = 0;                                                            //  no text on image yet
   px = py = -1;                                                           //  no position defined yet

   zdialog_run(zd,dialog_event,"save");                                    //  run dialog, parallel
   return;
}


//  dialog event and completion callback function

int writetext::dialog_event(zdialog *zd, cchar *event)
{
   using namespace writetext;

   GtkWidget   *font_dialog;
   char        font[100];                                                  //  font name and size
   int         size;
   char        *pp;
   cchar       *keyname;
   char        **keyvals;

   if (strEqu(event,"done")) zd->zstat = 2;                                //  apply and quit               v.13.12

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                //  Apply, commit present text to image
         write(1);                                                         //  v.12.03
         present = 0;                                                      //  (no old text to erase)
         px = py = -1;
         zd->zstat = 0;
         edit_done(0);                                                     //  finish
         PXM_free(attr.pxm_text);                                          //  free memory                  v.12.10
         PXM_free(attr.pxm_transp);
         m_write_text(0,0);                                                //  start again
         return 1;
      }

      if (zd->zstat == 2 && CEF->Fmods) edit_done(0);                      //  Done, complete pending edit
      else edit_cancel(0);                                                 //  Cancel or kill
      PXM_free(attr.pxm_text);                                             //  free memory                  v.12.10
      PXM_free(attr.pxm_transp);
      return 1;
   }

   if (strEqu(event,"focus")) {                                            //  toggle mouse capture
      takeMouse(mousefunc,dragcursor);                                     //  connect mouse function
      return 1;
   }

   if (strEqu(event,"loadtext"))                                           //  load zdialog fields from file
   {
      load_text(zd);
      zdialog_fetch(zd,"text",attr.text,1000);                             //  get all zdialog fields
      zdialog_fetch(zd,"fontname",attr.font,80);
      zdialog_fetch(zd,"fontsize",attr.size);
      zdialog_fetch(zd,"angle",attr.angle);
      zdialog_fetch(zd,"fgcolor",attr.color[0],20);
      zdialog_fetch(zd,"bgcolor",attr.color[1],20);
      zdialog_fetch(zd,"tocolor",attr.color[2],20);
      zdialog_fetch(zd,"shcolor",attr.color[3],20);
      zdialog_fetch(zd,"fgtransp",attr.transp[0]);
      zdialog_fetch(zd,"bgtransp",attr.transp[1]);
      zdialog_fetch(zd,"totransp",attr.transp[2]);
      zdialog_fetch(zd,"shtransp",attr.transp[3]);
      zdialog_fetch(zd,"towidth",attr.towidth);
      zdialog_fetch(zd,"shwidth",attr.shwidth);
      zdialog_fetch(zd,"shangle",attr.shangle);
   }

   if (strEqu(event,"savetext")) {                                         //  save zdialog fields to file
      save_text(zd);
      return 1;
   }
   
   if (strstr("caption comment",event)) {                                  //  get caption or comment metadata    v.13.12
      if (strEqu(event,"caption")) keyname = iptc_caption_key;
      else keyname = exif_comment_key;
      keyvals = exif_get(curr_file,&keyname,1);
      if (! *keyvals) return 1;
      if (strlen(*keyvals) > 999) *keyvals[999] = 0;
      repl_1str(*keyvals,attr.text,"\\n","\n");                            //  replace "\n" with newlines
      zfree(*keyvals);
      zdialog_stuff(zd,"text",attr.text);                                  //  stuff dialog with metadata
   }

   if (strEqu(event,"text"))                                               //  get text from dialog
      zdialog_fetch(zd,"text",attr.text,1000);

   if (strEqu(event,"fontbutt")) {                                         //  select new font
      snprintf(font,100,"%s %d",attr.font,attr.size);
      font_dialog = gtk_font_chooser_dialog_new(ZTX("select font"),MWIN);
      gtk_font_chooser_set_font(GTK_FONT_CHOOSER(font_dialog),font);
      gtk_dialog_run(GTK_DIALOG(font_dialog));
      pp = gtk_font_chooser_get_font(GTK_FONT_CHOOSER(font_dialog));
      gtk_widget_destroy(font_dialog);

      if (pp) {                                                            //  should have "fontname nn"
         strncpy0(font,pp,100);
         g_free(pp);
         pp = font + strlen(font);
         while (*pp != ' ') pp--;
         if (pp > font) {
            size = atoi(pp);
            if (size < 8) size = 8;
            zdialog_stuff(zd,"fontsize",size);
            attr.size = size;
            *pp = 0;
            strncpy0(attr.font,font,80);                                   //  get fontname = new font name
            zdialog_stuff(zd,"fontname",font);
         }
      }
   }

   if (strEqu(event,"fontsize"))                                           //  new font size
      zdialog_fetch(zd,"fontsize",attr.size);

   if (strEqu(event,"angle"))
      zdialog_fetch(zd,"angle",attr.angle);

   if (strEqu(event,"fgcolor"))                                            //  foreground (text) color
      zdialog_fetch(zd,"fgcolor",attr.color[0],20);

   if (strEqu(event,"bgcolor"))                                            //  background color
      zdialog_fetch(zd,"bgcolor",attr.color[1],20);

   if (strEqu(event,"tocolor"))                                            //  text outline color
      zdialog_fetch(zd,"tocolor",attr.color[2],20);

   if (strEqu(event,"shcolor"))                                            //  text shadow color
      zdialog_fetch(zd,"shcolor",attr.color[3],20);

   if (strEqu(event,"fgtransp"))                                           //  foreground transparency
      zdialog_fetch(zd,"fgtransp",attr.transp[0]);

   if (strEqu(event,"bgtransp"))                                           //  background transparency
      zdialog_fetch(zd,"bgtransp",attr.transp[1]);

   if (strEqu(event,"totransp"))                                           //  text outline transparency
      zdialog_fetch(zd,"totransp",attr.transp[2]);

   if (strEqu(event,"shtransp"))                                           //  text shadow transparency
      zdialog_fetch(zd,"shtransp",attr.transp[3]);

   if (strEqu(event,"towidth"))                                            //  text outline width
      zdialog_fetch(zd,"towidth",attr.towidth);

   if (strEqu(event,"shwidth"))                                            //  text shadow width
      zdialog_fetch(zd,"shwidth",attr.shwidth);

   if (strEqu(event,"shangle"))                                            //  text shadow angle
      zdialog_fetch(zd,"shangle",attr.shangle);

   gentext(&attr);                                                         //  build text image from text and attributes
   write(1);                                                               //  write on image
   zmainloop();

   return 1;
}


//  mouse function, set position for text on image

void writetext::mousefunc()
{
   using namespace writetext;

   if (LMclick) {                                                          //  left mouse click
      px = Mxclick;                                                        //  new text position on image
      py = Myclick;
      write(1);                                                            //  erase old, write new text on image
   }

   if (RMclick) {                                                          //  right mouse click
      write(2);                                                            //  erase old text, if any
      px = py = -1;                                                        //  v.12.03
   }

   if (Mxdrag || Mydrag)                                                   //  mouse dragged
   {
      px = Mxdrag;                                                         //  new text position on image
      py = Mydrag;
      write(1);                                                            //  erase old, write new text on image
   }

   LMclick = RMclick = Mxdrag = Mydrag = 0;                                //  v.12.11
   return;
}


//  write text on image at designated location
//  mode: 1  erase old and write to new position
//        2  erase old and write nothing

void writetext::write(int mode)                                            //  v.12.02
{
   using namespace writetext;

   float       *pix1, *pix2, *pix3;
   int         px1, py1, px3, py3, done;
   float       e3part;

   static int  orgx1, orgy1, ww1, hh1;                                     //  old text image overlap rectangle
   int         orgx2, orgy2, ww2, hh2;                                     //  new overlap rectangle

   if (present)                                                            //  v.13.03
   {
      for (py3 = orgy1; py3 < orgy1 + hh1; py3++)                          //  erase prior text image
      for (px3 = orgx1; px3 < orgx1 + ww1; px3++)                          //  replace E3 pixels with E1 pixels
      {                                                                    //    in prior overlap rectangle
         if (px3 < 0 || px3 >= E3pxm->ww) continue;
         if (py3 < 0 || py3 >= E3pxm->hh) continue;
         pix1 = PXMpix(E1pxm,px3,py3);
         pix3 = PXMpix(E3pxm,px3,py3);
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   done = 0;
   if (mode == 2) done = 1;                                                //  erase only
   if (! *attr.text) done = 1;                                             //  no text defined
   if (px < 0 && py < 0) done = 1;                                         //  no position defined

   if (done) {
      if (present) {
         Fpaint3(orgx1,orgy1,ww1,hh1);                                     //  update window to erase old text
         present = 0;                                                      //  mark no text present
      }
      return;
   }

   ww2 = attr.pxm_text->ww;                                                //  text image
   hh2 = attr.pxm_text->hh;

   orgx2 = px - ww2/2;                                                     //  copy-to image3 location
   orgy2 = py - hh2/2;

   for (py1 = 0; py1 < hh2; py1++)                                         //  loop all pixels in text image
   for (px1 = 0; px1 < ww2; px1++)
   {
      px3 = orgx2 + px1;                                                   //  copy-to image3 pixel
      py3 = orgy2 + py1;

      if (px3 < 0 || px3 >= E3pxm->ww) continue;                           //  omit parts beyond edges
      if (py3 < 0 || py3 >= E3pxm->hh) continue;

      pix1 = PXMpix(attr.pxm_text,px1,py1);                                //  copy-from text pixel
      pix2 = PXMpix(attr.pxm_transp,px1,py1);                              //  copy-from transparency
      pix3 = PXMpix(E3pxm,px3,py3);                                        //  copy-to image pixel

      e3part = pix2[0] / 256.0;                                            //  image part visible through text

      pix3[0] = pix1[0] + e3part * pix3[0];                                //  combine text part + image part
      pix3[1] = pix1[1] + e3part * pix3[1];
      pix3[2] = pix1[2] + e3part * pix3[2];
   }

   if (present) {
      Fpaint3(orgx1,orgy1,ww1,hh1);                                        //  update window to erase old text
      present = 0;
   }
                                                                           //  (updates together to reduce flicker)
   Fpaint3(orgx2,orgy2,ww2,hh2);                                           //  update window for new text

   CEF->Fmods++;                                                           //  increase mod count           v.12.02
   CEF->Fsaved = 0;
   present = 1;                                                            //  mark text is present

   orgx1 = orgx2;                                                          //  remember overlap rectangle
   orgy1 = orgy2;                                                          //    for next call
   ww1 = ww2;
   hh1 = hh2;
   return;
}


//  load text and attributes from a file

void load_text(zdialog *zd)                                                //  v.13.09
{
   FILE        *fid;
   int         err, nn;
   char        *pp, *pp2, *file, buff[1200];
   cchar       *dialogtitle = "load text data from a file";
   textattr_t  attr;

   file = zgetfile(dialogtitle,"file",writetext_dirk);                     //  get input file from user
   if (! file) return;

   fid = fopen(file,"r");                                                  //  open for read
   if (! fid) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      zfree(file);
      return;
   }

   pp = fgets_trim(buff,1200,fid);                                         //  read text string
   if (! pp) goto badfile;
   if (strnNeq(pp,"text string: ",13)) goto badfile;
   pp += 13;
   if (strlen(pp) < 2) goto badfile;
   repl_Nstrs(pp,attr.text,"\\n","\n",null);                               //  replace "\n" with newline char.

   pp = fgets_trim(buff,1200,fid);                                         //  read font and size
   if (! pp) goto badfile;
   if (strnNeq(pp,"font: ",6)) goto badfile;
   pp += 6;
   strTrim(pp);
   pp2 = strstr(pp,"size: ");
   if (! pp2) goto badfile;
   *pp2 = 0;
   pp2 += 6;
   strncpy0(attr.font,pp,80);
   attr.size = atoi(pp2);
   if (attr.size < 8) goto badfile;

   pp = fgets_trim(buff,1200,fid);                                         //  read text attributes
   if (! pp) goto badfile;
   nn = sscanf(pp,"attributes: %f  %s %s %s %s  %d %d %d %d  %d %d %d",
            &attr.angle, attr.color[0], attr.color[1], attr.color[2], attr.color[3],
            &attr.transp[0], &attr.transp[1], &attr.transp[2], &attr.transp[3],
            &attr.towidth, &attr.shwidth, &attr.shangle);
   if (nn != 12) goto badfile;

   err = fclose(fid);
   if (err) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      zfree(file);
      return;
   }

   zdialog_stuff(zd,"text",attr.text);                                     //  stuff zdialog fields
   zdialog_stuff(zd,"fontname",attr.font);
   zdialog_stuff(zd,"fontsize",attr.size);
   zdialog_stuff(zd,"angle",attr.angle);
   zdialog_stuff(zd,"fgcolor",attr.color[0]);
   zdialog_stuff(zd,"bgcolor",attr.color[1]);
   zdialog_stuff(zd,"tocolor",attr.color[2]);
   zdialog_stuff(zd,"shcolor",attr.color[3]);
   zdialog_stuff(zd,"fgtransp",attr.transp[0]);
   zdialog_stuff(zd,"bgtransp",attr.transp[1]);
   zdialog_stuff(zd,"totransp",attr.transp[2]);
   zdialog_stuff(zd,"shtransp",attr.transp[3]);
   zdialog_stuff(zd,"towidth",attr.towidth);
   zdialog_stuff(zd,"shwidth",attr.shwidth);
   zdialog_stuff(zd,"shangle",attr.shangle);
   return;

badfile:                                                                   //  project file had a problem
   fclose(fid);
   zmessageACK(Mwin,0,ZTX("text file is defective"));
   printz("buff: %s\n",buff);
}


//  save text to a file

void save_text(zdialog *zd)                                                //  v.13.09
{
   cchar       *dialogtitle = "save text data to a file";
   FILE        *fid;
   char        *file, text2[1200];
   textattr_t  attr;

   file = zgetfile(dialogtitle,"save",writetext_dirk);                     //  get output file from user
   if (! file) return;

   fid = fopen(file,"w");                                                  //  open for write
   if (! fid) {
      zmessageACK(Mwin,0,"%s",strerror(errno));
      zfree(file);
      return;
   }

   zdialog_fetch(zd,"text",attr.text,1000);                                //  get text and attributes from zdialog
   zdialog_fetch(zd,"fontname",attr.font,80);
   zdialog_fetch(zd,"fontsize",attr.size);
   zdialog_fetch(zd,"angle",attr.angle);
   zdialog_fetch(zd,"fgcolor",attr.color[0],20);
   zdialog_fetch(zd,"bgcolor",attr.color[1],20);
   zdialog_fetch(zd,"tocolor",attr.color[2],20);
   zdialog_fetch(zd,"shcolor",attr.color[3],20);
   zdialog_fetch(zd,"fgtransp",attr.transp[0]);
   zdialog_fetch(zd,"bgtransp",attr.transp[1]);
   zdialog_fetch(zd,"totransp",attr.transp[2]);
   zdialog_fetch(zd,"shtransp",attr.transp[3]);
   zdialog_fetch(zd,"towidth",attr.towidth);
   zdialog_fetch(zd,"shwidth",attr.shwidth);
   zdialog_fetch(zd,"shangle",attr.shangle);

   repl_Nstrs(attr.text,text2,"\n","\\n",null);                            //  replace newlines with "\n"

   fprintf(fid,"text string: %s\n",text2);

   fprintf(fid,"font: %s  size: %d \n",attr.font, attr.size);

   fprintf(fid,"attributes: %.4f  %s %s %s %s  %d %d %d %d  %d %d %d \n",
            attr.angle, attr.color[0], attr.color[1], attr.color[2], attr.color[3],
            attr.transp[0], attr.transp[1], attr.transp[2], attr.transp[3],
            attr.towidth, attr.shwidth, attr.shangle);

   fclose(fid);
   return;
}


/**************************************************************************/

//  Create a graphic image with text, white on black, any font, any angle.
//  Add outline and shadow colors. Apply transparencies.

int gentext(textattr_t *attr)                                              //  v.13.10
{
   PXM * gentext_outline(textattr_t *, PXM *);
   PXM * gentext_shadow(textattr_t *, PXM *);

   PangoFontDescription    *pfont;
   PangoLayout             *playout;
   cairo_surface_t         *surface;
   cairo_t                 *cr;

   float          angle;
   int            fgred, fggreen, fgblue;
   int            bgred, bggreen, bgblue;
   int            tored, togreen, toblue;
   int            shred, shgreen, shblue;
   float          fgtransp, bgtransp, totransp, shtransp;

   PXM            *pxm_temp1, *pxm_temp2, *pxm_temp3;
   uint8          *cairo_data, *cpix;
   float          *pix1, *pix2, *pix3;
   int            px, py, ww, hh;
   char           font[100];                                               //  font name and size
   int            red, green, blue;
   float          fgpart, topart, shpart, bgpart;

   zthreadcrash();                                                         //  thread usage not allowed

   if (! *attr->text) strcpy(attr->text,"null");                           //  v.13.10

   snprintf(font,100,"%s %d",attr->font,attr->size);                       //  font name and size together
   angle = attr->angle;                                                    //  text angle, degrees
   attr->sinT = sin(angle/57.296);                                         //  trig funcs for text angle
   attr->cosT = cos(angle/57.296);
   fgred = atoi(strField(attr->color[0],'|',1));                           //  get text foreground color
   fggreen = atoi(strField(attr->color[0],'|',2));
   fgblue = atoi(strField(attr->color[0],'|',3));
   bgred = atoi(strField(attr->color[1],'|',1));                           //  get text background color
   bggreen = atoi(strField(attr->color[1],'|',2));
   bgblue = atoi(strField(attr->color[1],'|',3));
   tored = atoi(strField(attr->color[2],'|',1));                           //  get text outline color
   togreen = atoi(strField(attr->color[2],'|',2));
   toblue = atoi(strField(attr->color[2],'|',3));
   shred = atoi(strField(attr->color[3],'|',1));                           //  get text shadow color
   shgreen = atoi(strField(attr->color[3],'|',2));
   shblue = atoi(strField(attr->color[3],'|',3));
   fgtransp = 0.01 * attr->transp[0];                                      //  get transparencies
   bgtransp = 0.01 * attr->transp[1];                                      //  text, background, outline, shadow
   totransp = 0.01 * attr->transp[2];
   shtransp = 0.01 * attr->transp[3];

   pfont = pango_font_description_from_string(font);                       //  make layout with text
   playout = gtk_widget_create_pango_layout(Cdraw,null);
   pango_layout_set_font_description(playout,pfont);
   pango_layout_set_text(playout,attr->text,-1);

   pango_layout_get_pixel_size(playout,&ww,&hh);
   ww += 2 + 0.2 * attr->size;                                             //  compensate bad font metrics
   hh += 2 + 0.1 * attr->size;
   attr->tww = ww;                                                         //  save text image size before rotate
   attr->thh = hh;

   surface = cairo_image_surface_create(CAIRO_FORMAT_RGB24,ww,hh);         //  cairo output image
   cr = cairo_create(surface);
   pango_cairo_show_layout(cr,playout);                                    //  write text layout to image

   cairo_data = cairo_image_surface_get_data(surface);                     //  get text image pixels

   pxm_temp1 = PXM_make(ww,hh);                                            //  create PXM

   for (py = 0; py < hh; py++)                                             //  copy text image to PXM
   for (px = 0; px < ww; px++)
   {
      cpix = cairo_data + 4 * (ww * py + px);                              //  pango output is monocolor
      pix2 = PXMpix(pxm_temp1,px,py);
      pix2[0] = cpix[3];                                                   //  use red [0] for text intensity
      pix2[1] = pix2[2] = 0;
   }

   pango_font_description_free(pfont);                                     //  free resources
   g_object_unref(playout);
   cairo_destroy(cr);
   cairo_surface_destroy(surface);

   pxm_temp2 = gentext_outline(attr,pxm_temp1);                            //  add text outline if any
   if (pxm_temp2) {                                                        //    using green [1] for outline intensity
      PXM_free(pxm_temp1);
      pxm_temp1 = pxm_temp2;
   }

   pxm_temp2 = gentext_shadow(attr,pxm_temp1);                             //  add text shadow color if any
   if (pxm_temp2) {                                                        //    using blue [2] for shadow intensity
      PXM_free(pxm_temp1);
      pxm_temp1 = pxm_temp2;
   }

   if (fabsf(angle) > 0.1) {                                               //  rotate text if wanted
      pxm_temp2 = PXM_rotate(pxm_temp1,angle);
      PXM_free(pxm_temp1);
      pxm_temp1 = pxm_temp2;
   }

   ww = pxm_temp1->ww;                                                     //  text image input PXM
   hh = pxm_temp1->hh;
   pxm_temp2 = PXM_make(ww,hh);                                            //  text image output PXM
   pxm_temp3 = PXM_make(ww,hh);                                            //  output transparency map

   for (py = 0; py < hh; py++)                                             //  loop all pixels in text image
   for (px = 0; px < ww; px++)
   {
      pix1 = PXMpix(pxm_temp1,px,py);                                      //  copy-from pixel (text + outline + shadow)
      pix2 = PXMpix(pxm_temp2,px,py);                                      //  copy-to pixel
      pix3 = PXMpix(pxm_temp3,px,py);                                      //  copy-to transparency

      fgpart = pix1[0] / 256.0;
      topart = pix1[1] / 256.0;
      shpart = pix1[2] / 256.0;
      bgpart = (1.0 - fgpart - topart - shpart);
      fgpart = fgpart * (1.0 - fgtransp);
      topart = topart * (1.0 - totransp);
      shpart = shpart * (1.0 - shtransp);
      bgpart = bgpart * (1.0 - bgtransp);

      red = fgpart * fgred + topart * tored + shpart * shred + bgpart * bgred;
      green = fgpart * fggreen + topart * togreen + shpart * shgreen + bgpart * bggreen;
      blue = fgpart * fgblue + topart * toblue + shpart * shblue + bgpart * bgblue;

      pix2[0] = red;                                                       //  output total red, green blue
      pix2[1] = green;
      pix2[2] = blue;

      pix3[0] = 255 * (1.0 - fgpart - topart - shpart - bgpart);           //  image part visible through text
   }

   PXM_free(pxm_temp1);
   PXM_free(attr->pxm_text);                                               //  v.14.07
   PXM_free(attr->pxm_transp);
   attr->pxm_text = pxm_temp2;
   attr->pxm_transp = pxm_temp3;
   return 0;
}


//  add an outline color to the text character edges
//  red color [0] is original monocolor text
//  use green color [1] for added outline

PXM * gentext_outline(textattr_t *attr, PXM *pxm1)                         //  v.13.09
{
   PXM         *pxm2;
   int         toww, ww1, hh1, ww2, hh2;
   int         px, py, dx, dy;
   float       *pix1, *pix2, theta;

   toww = attr->towidth;                                                   //  text outline color width
   if (toww == 0) return 0;                                                //  zero

   ww1 = pxm1->ww;                                                         //  input PXM dimensions
   hh1 = pxm1->hh;
   ww2 = ww1 + toww * 2;                                                   //  add margins for outline width
   hh2 = hh1 + toww * 2;
   pxm2 = PXM_make(ww2,hh2);                                               //  output PXM

   for (py = 0; py < hh1; py++)                                            //  copy text pixels to outline pixels
   for (px = 0; px < ww1; px++)                                            //    displaced by outline width
   {
      pix1 = PXMpix(pxm1,px,py);
      pix2 = PXMpix(pxm2,px+toww,py+toww);
      pix2[0] = pix1[0];
   }

   theta = 0.7 / toww;
   for (theta = 0; theta < 6.3; theta += 0.7/toww)                         //  displace outline pixels in all directions
   {
      dx = roundf(toww * sinf(theta));
      dy = roundf(toww * cosf(theta));

      for (py = 0; py < hh1; py++)
      for (px = 0; px < ww1; px++)
      {
         pix1 = PXMpix(pxm1,px,py);
         pix2 = PXMpix(pxm2,px+toww+dx,py+toww+dy);
         if (pix2[1] < pix1[0] - pix2[0])                                  //  compare text to outline brightness
            pix2[1] = pix1[0] - pix2[0];                                   //  brighter part is outline pixel
      }
   }

   return pxm2;                                                            //  pix2[0] / pix2[1] = text / outline
}


//  add a shadow to the text character edges
//  red color [0] is original monocolor text intensity
//  green color [1] is added outline if any
//  use blue color [2] for added shadow

PXM * gentext_shadow(textattr_t *attr, PXM *pxm1)                          //  v.13.09
{
   PXM         *pxm2;
   int         shww, ww1, hh1, ww2, hh2;
   int         px, py, dx, dy;
   float       *pix1, *pix2, theta;

   shww = attr->shwidth;                                                   //  text shadow width
   if (shww == 0) return 0;                                                //  zero

   ww1 = pxm1->ww;                                                         //  input PXM dimensions
   hh1 = pxm1->hh;
   ww2 = ww1 + shww * 2;                                                   //  add margins for shadow width
   hh2 = hh1 + shww * 2;
   pxm2 = PXM_make(ww2,hh2);                                               //  output PXM

   for (py = 0; py < hh1; py++)                                            //  copy text pixels to shadow pixels
   for (px = 0; px < ww1; px++)                                            //    displaced by shadow width
   {
      pix1 = PXMpix(pxm1,px,py);
      pix2 = PXMpix(pxm2,px+shww,py+shww);
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
   }

   theta = (90 - attr->shangle) / 57.3;                                    //  degrees to radians, 0 = to the right
   dx = roundf(shww * sinf(theta));
   dy = roundf(shww * cosf(theta));

   for (py = 0; py < hh1; py++)                                            //  displace text by shadow width
   for (px = 0; px < ww1; px++)
   {
      pix1 = PXMpix(pxm1,px,py);
      pix2 = PXMpix(pxm2,px+shww+dx,py+shww+dy);
      if (pix2[2] < pix1[0] + pix1[1] - pix2[0] - pix2[1])                 //  compare text+outline to shadow pixels
         pix2[2] = pix1[0] + pix1[1] - pix2[0] - pix2[1];                  //  brighter part is shadow pixel
   }

   return pxm2;                                                            //  pix2[0] / pix2[1] / pix2[2]
}                                                                          //    = text / outline / shadow brightness


/**************************************************************************/

//  write a line or arrow on top of the image

namespace writeline
{
   lineattr_t  attr;                                                       //  line/arrow attributes and image

   int      px, py;                                                        //  line position on image
   int      present;                                                       //  flag, write_line present on image

   int   dialog_event(zdialog *zd, cchar *event);                          //  dialog event function
   void  mousefunc();                                                      //  mouse event function
   void  write(int mode);                                                  //  write line on image

   editfunc    EFwriteline;
}


void m_write_line(GtkWidget *, cchar *menu)
{
   using namespace writeline;

   cchar    *intro = ZTX("Enter line or arrow properties in dialog, \n"
                         "click/drag on image, right click to remove");

   F1_help_topic = "add_lines";                                            //  user guide topic

   EFwriteline.menufunc = m_write_line;
   EFwriteline.funcname = "write_line";
   EFwriteline.Farea = 1;                                                  //  select area ignored
   if (! edit_setup(EFwriteline)) return;                                  //  setup edit

/***
       ____________________________________________________________
      |               Write Line or Arrow on Image                 |
      |                                                            |
      |  Enter line or arrow properties in dialog,                 |
      |  click/drag on image, right click to remove.               |
      |                                                            |
      |  Line length [___|+-]  width [___|+-]                      |
      |  Arrow head  [x] left   [x] right                          |
      |                                                            |
      |            color   transp.   width     angle               |
      |  line     [_____]  [___|+-]            [___|+-]            |
      |  backing  [_____]  [___|+-]                                |
      |  outline  [_____]  [___|+-]  [___|+-]                      |
      |  shadow   [_____]  [___|+-]  [___|+-]  [___|+-]            |
      |                                                            |
      |                                    [Apply] [Done] [Cancel] |
      |____________________________________________________________|

***/

   zdialog *zd = zdialog_new(ZTX("Write Line or Arrow on Image"),Mwin,Bapply,Bdone,Bcancel,null);
   EFwriteline.zd = zd;
   EFwriteline.mousefunc = mousefunc;
   EFwriteline.menufunc = m_write_line;                                    //  allow restart          v.13.04

   zdialog_add_widget(zd,"label","intro","dialog",intro,"space=3");

   zdialog_add_widget(zd,"hbox","hbline","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","lablength","hbline",ZTX("Line length"),"space=5");
   zdialog_add_widget(zd,"spin","length","hbline","2|9999|1|20");
   zdialog_add_widget(zd,"label","space","hbline",0,"space=10");
   zdialog_add_widget(zd,"label","labwidth","hbline",ZTX("width"),"space=5");
   zdialog_add_widget(zd,"spin","width","hbline","1|99|1|2");

   zdialog_add_widget(zd,"hbox","hbarrow","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labarrow","hbarrow",ZTX("Arrow head"),"space=5");
   zdialog_add_widget(zd,"check","larrow","hbarrow",Bleft);
   zdialog_add_widget(zd,"label","space","hbarrow",0,"space=10");
   zdialog_add_widget(zd,"check","rarrow","hbarrow",Bright);

   zdialog_add_widget(zd,"hbox","hbcol","dialog");
   zdialog_add_widget(zd,"vbox","vbcol1","hbcol",0,"homog|space=5");
   zdialog_add_widget(zd,"vbox","vbcol2","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol3","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol4","hbcol",0,"homog|space=2");
   zdialog_add_widget(zd,"vbox","vbcol5","hbcol",0,"homog|space=2");

   zdialog_add_widget(zd,"label","space","vbcol1");
   zdialog_add_widget(zd,"label","labline","vbcol1",ZTX("line"));
   zdialog_add_widget(zd,"label","labback","vbcol1",ZTX("backing"));
   zdialog_add_widget(zd,"label","laboutln","vbcol1",ZTX("outline"));
   zdialog_add_widget(zd,"label","labshadow","vbcol1",ZTX("shadow"));

   zdialog_add_widget(zd,"label","labcol","vbcol2",ZTX("Color"));
   zdialog_add_widget(zd,"colorbutt","fgcolor","vbcol2","0|0|0");
   zdialog_add_widget(zd,"colorbutt","bgcolor","vbcol2","255|255|255");
   zdialog_add_widget(zd,"colorbutt","tocolor","vbcol2","255|0|0");
   zdialog_add_widget(zd,"colorbutt","shcolor","vbcol2","255|0|0");

   zdialog_add_widget(zd,"label","labcol","vbcol3",ZTX("Transparency"));
   zdialog_add_widget(zd,"spin","fgtransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"spin","bgtransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"spin","totransp","vbcol3","0|100|1|0");
   zdialog_add_widget(zd,"spin","shtransp","vbcol3","0|100|1|0");

   zdialog_add_widget(zd,"label","labw","vbcol4",ZTX("Width"));
   zdialog_add_widget(zd,"label","space","vbcol4");
   zdialog_add_widget(zd,"label","space","vbcol4");
   zdialog_add_widget(zd,"spin","towidth","vbcol4","0|30|1|0");
   zdialog_add_widget(zd,"spin","shwidth","vbcol4","0|50|1|0");

   zdialog_add_widget(zd,"label","labw","vbcol5",Bangle);
   zdialog_add_widget(zd,"spin","angle","vbcol5","-180|180|0.5|0");
   zdialog_add_widget(zd,"label","space","vbcol5");
   zdialog_add_widget(zd,"label","space","vbcol5");
   zdialog_add_widget(zd,"spin","shangle","vbcol5","-180|180|1|0");

   zdialog_restore_inputs(zd);                                             //  restore prior inputs

   zdialog_fetch(zd,"length",attr.length);                                 //  get defaults or prior inputs
   zdialog_fetch(zd,"width",attr.width);
   zdialog_fetch(zd,"larrow",attr.larrow);
   zdialog_fetch(zd,"rarrow",attr.rarrow);
   zdialog_fetch(zd,"angle",attr.angle);
   zdialog_fetch(zd,"fgcolor",attr.color[0],20);
   zdialog_fetch(zd,"bgcolor",attr.color[1],20);
   zdialog_fetch(zd,"tocolor",attr.color[2],20);
   zdialog_fetch(zd,"shcolor",attr.color[3],20);
   zdialog_fetch(zd,"fgtransp",attr.transp[0]);
   zdialog_fetch(zd,"bgtransp",attr.transp[1]);
   zdialog_fetch(zd,"totransp",attr.transp[2]);
   zdialog_fetch(zd,"shtransp",attr.transp[3]);
   zdialog_fetch(zd,"towidth",attr.towidth);
   zdialog_fetch(zd,"shwidth",attr.shwidth);
   zdialog_fetch(zd,"shangle",attr.shangle);

   genline(&attr);                                                         //  generate initial line

   takeMouse(mousefunc,dragcursor);                                        //  connect mouse function
   present = 0;                                                            //  no line on image yet
   px = py = -1;                                                           //  no position defined yet

   zdialog_run(zd,dialog_event,"save");                                    //  run dialog, parallel
   return;
}


//  dialog event and completion callback function

int writeline::dialog_event(zdialog *zd, cchar *event)
{
   using namespace writeline;

   if (strEqu(event,"done")) zd->zstat = 2;                                //  apply and quit               v.13.12

   if (zd->zstat)
   {
      if (zd->zstat == 1) {                                                //  Apply, commit present line to image
         write(1);                                                         //  v.12.03
         present = 0;                                                      //  (no old line to erase)
         px = py = -1;
         zd->zstat = 0;
         edit_done(0);                                                     //  finish
         PXM_free(attr.pxm_line);                                               //  free memory                  v.12.10
         PXM_free(attr.pxm_transp);
         m_write_line(0,0);                                                //  start again
         return 1;
      }

      if (zd->zstat == 2 && CEF->Fmods) edit_done(0);                      //  Done, complete pending edit
      else edit_cancel(0);                                                 //  Cancel or kill
      PXM_free(attr.pxm_line);                                                  //  free memory                  v.12.10
      PXM_free(attr.pxm_transp);
      return 1;
   }

   if (strEqu(event,"focus")) {                                            //  toggle mouse capture
      takeMouse(mousefunc,dragcursor);                                     //  connect mouse function
      return 1;
   }
   
   if (strEqu(event,"length"));                                            //  line length
      zdialog_fetch(zd,"length",attr.length);

   if (strEqu(event,"width"));                                             //  line width
      zdialog_fetch(zd,"width",attr.width);

   if (strEqu(event,"larrow"));                                            //  left arrow head
      zdialog_fetch(zd,"larrow",attr.larrow);

   if (strEqu(event,"rarrow"));                                            //  right arrow head
      zdialog_fetch(zd,"rarrow",attr.rarrow);

   if (strEqu(event,"angle"))                                              //  line angle
      zdialog_fetch(zd,"angle",attr.angle);

   if (strEqu(event,"fgcolor"))                                            //  foreground (line) color
      zdialog_fetch(zd,"fgcolor",attr.color[0],20);

   if (strEqu(event,"bgcolor"))                                            //  background color
      zdialog_fetch(zd,"bgcolor",attr.color[1],20);

   if (strEqu(event,"tocolor"))                                            //  line outline color
      zdialog_fetch(zd,"tocolor",attr.color[2],20);

   if (strEqu(event,"shcolor"))                                            //  line shadow color
      zdialog_fetch(zd,"shcolor",attr.color[3],20);

   if (strEqu(event,"fgtransp"))                                           //  foreground transparency
      zdialog_fetch(zd,"fgtransp",attr.transp[0]);

   if (strEqu(event,"bgtransp"))                                           //  background transparency
      zdialog_fetch(zd,"bgtransp",attr.transp[1]);

   if (strEqu(event,"totransp"))                                           //  line outline transparency
      zdialog_fetch(zd,"totransp",attr.transp[2]);

   if (strEqu(event,"shtransp"))                                           //  line shadow transparency
      zdialog_fetch(zd,"shtransp",attr.transp[3]);

   if (strEqu(event,"towidth"))                                            //  line outline width
      zdialog_fetch(zd,"towidth",attr.towidth);

   if (strEqu(event,"shwidth"))                                            //  line shadow width
      zdialog_fetch(zd,"shwidth",attr.shwidth);

   if (strEqu(event,"shangle"))                                            //  line shadow angle
      zdialog_fetch(zd,"shangle",attr.shangle);

   genline(&attr);                                                         //  build line image from attributes
   write(1);                                                               //  write on image
   zmainloop();

   return 1;
}


//  mouse function, set position for line on image

void writeline::mousefunc()
{
   using namespace writeline;

   if (LMclick) {                                                          //  left mouse click
      px = Mxclick;                                                        //  new line position on image
      py = Myclick;
      write(1);                                                            //  erase old, write new line on image
   }

   if (RMclick) {                                                          //  right mouse click
      write(2);                                                            //  erase old line, if any
      px = py = -1;                                                        //  v.12.03
   }

   if (Mxdrag || Mydrag)                                                   //  mouse dragged
   {
      px = Mxdrag;                                                         //  new line position on image
      py = Mydrag;
      write(1);                                                            //  erase old, write new line on image
   }

   LMclick = RMclick = Mxdrag = Mydrag = 0;                                //  v.12.11
   return;
}


//  write line on image at designated location
//  mode: 1  erase old and write to new position
//        2  erase old and write nothing

void writeline::write(int mode)                                            //  v.12.02
{
   using namespace writeline;

   float       *pix1, *pix2, *pix3;
   int         px1, py1, px3, py3, done;
   float       e3part;

   static int  orgx1, orgy1, ww1, hh1;                                     //  old line image overlap rectangle
   int         orgx2, orgy2, ww2, hh2;                                     //  new overlap rectangle

   if (present)                                                            //  v.13.03
   {
      for (py3 = orgy1; py3 < orgy1 + hh1; py3++)                          //  erase prior line image
      for (px3 = orgx1; px3 < orgx1 + ww1; px3++)                          //  replace E3 pixels with E1 pixels
      {                                                                    //    in prior overlap rectangle
         if (px3 < 0 || px3 >= E3pxm->ww) continue;
         if (py3 < 0 || py3 >= E3pxm->hh) continue;
         pix1 = PXMpix(E1pxm,px3,py3);
         pix3 = PXMpix(E3pxm,px3,py3);
         pix3[0] = pix1[0];
         pix3[1] = pix1[1];
         pix3[2] = pix1[2];
      }
   }

   done = 0;
   if (mode == 2) done = 1;                                                //  erase only
   if (px < 0 && py < 0) done = 1;                                         //  no position defined

   if (done) {
      if (present) {
         Fpaint3(orgx1,orgy1,ww1,hh1);                                     //  update window to erase old line
         present = 0;                                                      //  mark no line present
      }
      return;
   }

   ww2 = attr.pxm_line->ww;                                                     //  line image buffer
   hh2 = attr.pxm_line->hh;

   orgx2 = px - ww2/2;                                                     //  copy-to image3 location
   orgy2 = py - hh2/2;

   for (py1 = 0; py1 < hh2; py1++)                                         //  loop all pixels in line image
   for (px1 = 0; px1 < ww2; px1++)
   {
      px3 = orgx2 + px1;                                                   //  copy-to image3 pixel
      py3 = orgy2 + py1;

      if (px3 < 0 || px3 >= E3pxm->ww) continue;                           //  omit parts beyond edges
      if (py3 < 0 || py3 >= E3pxm->hh) continue;

      pix1 = PXMpix(attr.pxm_line,px1,py1);                                     //  copy-from line pixel
      pix2 = PXMpix(attr.pxm_transp,px1,py1);                                   //  copy-from transparency
      pix3 = PXMpix(E3pxm,px3,py3);                                        //  copy-to image pixel

      e3part = pix2[0] / 256.0;                                            //  image part visible through line

      pix3[0] = pix1[0] + e3part * pix3[0];                                //  combine line part + image part
      pix3[1] = pix1[1] + e3part * pix3[1];
      pix3[2] = pix1[2] + e3part * pix3[2];
   }

   if (present) {
      Fpaint3(orgx1,orgy1,ww1,hh1);                                        //  update window to erase old line
      present = 0;
   }
                                                                           //  (updates together to reduce flicker)
   Fpaint3(orgx2,orgy2,ww2,hh2);                                           //  update window for new line

   CEF->Fmods++;                                                           //  increase mod count           v.12.02
   CEF->Fsaved = 0;
   present = 1;                                                            //  mark line is present

   orgx1 = orgx2;                                                          //  remember overlap rectangle
   orgy1 = orgy2;                                                          //    for next call
   ww1 = ww2;
   hh1 = hh2;
   return;
}


/**************************************************************************/

//  Create a graphic image of a line or arrow, white on black, any size, 
//  any angle. Add outline and shadow colors.

int genline(lineattr_t *attr)                                              //  v.14.07
{
   PXM * genline_outline(lineattr_t *, PXM *);
   PXM * genline_shadow(lineattr_t *, PXM *);

   cairo_surface_t         *surface;
   cairo_t                 *cr;

   float          angle;
   int            fgred, fggreen, fgblue;
   int            bgred, bggreen, bgblue;
   int            tored, togreen, toblue;
   int            shred, shgreen, shblue;
   float          fgtransp, bgtransp, totransp, shtransp;

   PXM            *pxm_temp1, *pxm_temp2, *pxm_temp3;
   uint8          *cairo_data, *cpix;
   float          *pix1, *pix2, *pix3;
   float          length, width;
   int            px, py, ww, hh;
   int            red, green, blue;
   float          fgpart, topart, shpart, bgpart;

   zthreadcrash();                                                         //  thread usage not allowed

   angle = attr->angle;                                                    //  line angle, degrees
   attr->sinT = sin(angle/57.296);                                         //  trig funcs for line angle
   attr->cosT = cos(angle/57.296);
   fgred = atoi(strField(attr->color[0],'|',1));                           //  get line foreground color
   fggreen = atoi(strField(attr->color[0],'|',2));
   fgblue = atoi(strField(attr->color[0],'|',3));
   bgred = atoi(strField(attr->color[1],'|',1));                           //  get line background color
   bggreen = atoi(strField(attr->color[1],'|',2));
   bgblue = atoi(strField(attr->color[1],'|',3));
   tored = atoi(strField(attr->color[2],'|',1));                           //  get line outline color
   togreen = atoi(strField(attr->color[2],'|',2));
   toblue = atoi(strField(attr->color[2],'|',3));
   shred = atoi(strField(attr->color[3],'|',1));                           //  get line shadow color
   shgreen = atoi(strField(attr->color[3],'|',2));
   shblue = atoi(strField(attr->color[3],'|',3));
   fgtransp = 0.01 * attr->transp[0];                                      //  get transparencies
   bgtransp = 0.01 * attr->transp[1];                                      //  line, background, outline, shadow
   totransp = 0.01 * attr->transp[2];
   shtransp = 0.01 * attr->transp[3];
   
   length = attr->length;                                                  //  line dimensions
   width = attr->width;
   
   ww = length + 2 * width + 20;                                           //  create cairo surface
   hh = width + 20;                                                        //    with margins all around

   if (attr->larrow || attr->rarrow)                                       //  wider if arrow head used
      hh += width * 4;
   
   attr->lww = ww;
   attr->lhh = hh;
   
   surface = cairo_image_surface_create(CAIRO_FORMAT_RGB24,ww,hh);
   cr = cairo_create(surface);
   
   cairo_set_antialias(cr,CAIRO_ANTIALIAS_BEST);
   cairo_set_line_width(cr,width);
   cairo_set_line_cap(cr,CAIRO_LINE_CAP_ROUND);
   
   cairo_move_to(cr, 10, hh/2.0);                                          //  draw line in middle of surface
   cairo_line_to(cr, length+10, hh/2.0);
   
   if (attr->larrow) {                                                     //  add arrow heads if req. 
      cairo_move_to(cr, 10, hh/2.0);
      cairo_line_to(cr, 10 + 2 * width, hh/2.0 - 2 * width);
      cairo_move_to(cr, 10, hh/2.0);
      cairo_line_to(cr, 10 + 2 * width, hh/2.0 + 2 * width);
   }

   if (attr->rarrow) {
      cairo_move_to(cr, length+10, hh/2.0);
      cairo_line_to(cr, length+10 - 2 * width, hh/2.0 - 2 * width);
      cairo_move_to(cr, length+10, hh/2.0);
      cairo_line_to(cr, length+10 - 2 * width, hh/2.0 + 2 * width);
   }

   cairo_stroke(cr);

   cairo_data = cairo_image_surface_get_data(surface);                     //  cairo image pixels

   pxm_temp1 = PXM_make(ww,hh);                                            //  create PXM

   for (py = 0; py < hh; py++)                                             //  copy image to PXM
   for (px = 0; px < ww; px++)
   {
      cpix = cairo_data + 4 * (ww * py + px);                              //  pango output is monocolor
      pix2 = PXMpix(pxm_temp1,px,py);
      pix2[0] = cpix[3];                                                   //  use red [0] for line intensity
      pix2[1] = pix2[2] = 0;
   }

   cairo_destroy(cr);                                                      //  free resources
   cairo_surface_destroy(surface);

   pxm_temp2 = genline_outline(attr,pxm_temp1);                            //  add line outline if any
   if (pxm_temp2) {                                                        //    using green [1] for outline intensity
      PXM_free(pxm_temp1);
      pxm_temp1 = pxm_temp2;
   }

   pxm_temp2 = genline_shadow(attr,pxm_temp1);                             //  add line shadow color if any
   if (pxm_temp2) {                                                        //    using blue [2] for shadow intensity
      PXM_free(pxm_temp1);
      pxm_temp1 = pxm_temp2;
   }

   if (fabsf(angle) > 0.1) {                                               //  rotate line if wanted
      pxm_temp2 = PXM_rotate(pxm_temp1,angle);
      PXM_free(pxm_temp1);
      pxm_temp1 = pxm_temp2;
   }

   ww = pxm_temp1->ww;                                                     //  line image input PXM
   hh = pxm_temp1->hh;
   pxm_temp2 = PXM_make(ww,hh);                                            //  line image output PXM
   pxm_temp3 = PXM_make(ww,hh);                                            //  output transparency map

   for (py = 0; py < hh; py++)                                             //  loop all pixels in line image
   for (px = 0; px < ww; px++)
   {
      pix1 = PXMpix(pxm_temp1,px,py);                                      //  copy-from pixel (line + outline + shadow)
      pix2 = PXMpix(pxm_temp2,px,py);                                      //  copy-to pixel
      pix3 = PXMpix(pxm_temp3,px,py);                                      //  copy-to transparency

      fgpart = pix1[0] / 256.0;
      topart = pix1[1] / 256.0;
      shpart = pix1[2] / 256.0;
      bgpart = (1.0 - fgpart - topart - shpart);
      fgpart = fgpart * (1.0 - fgtransp);
      topart = topart * (1.0 - totransp);
      shpart = shpart * (1.0 - shtransp);
      bgpart = bgpart * (1.0 - bgtransp);

      red = fgpart * fgred + topart * tored + shpart * shred + bgpart * bgred;
      green = fgpart * fggreen + topart * togreen + shpart * shgreen + bgpart * bggreen;
      blue = fgpart * fgblue + topart * toblue + shpart * shblue + bgpart * bgblue;
      
      pix2[0] = red;                                                       //  output total red, green blue
      pix2[1] = green;
      pix2[2] = blue;

      pix3[0] = 255 * (1.0 - fgpart - topart - shpart - bgpart);           //  image part visible through line
   }

   PXM_free(pxm_temp1);
   PXM_free(attr->pxm_line);
   PXM_free(attr->pxm_transp);
   attr->pxm_line = pxm_temp2;
   attr->pxm_transp = pxm_temp3;
   return 0;
}


//  add an outline color to the line edges
//  red color [0] is original monocolor line
//  use green color [1] for added outline

PXM * genline_outline(lineattr_t *attr, PXM *pxm1)                         //  v.13.09
{
   PXM         *pxm2;
   int         toww, ww1, hh1, ww2, hh2;
   int         px, py, dx, dy;
   float       *pix1, *pix2, theta;

   toww = attr->towidth;                                                   //  line outline color width
   if (toww == 0) return 0;                                                //  zero

   ww1 = pxm1->ww;                                                         //  input PXM dimensions
   hh1 = pxm1->hh;
   ww2 = ww1 + toww * 2;                                                   //  add margins for outline width
   hh2 = hh1 + toww * 2;
   pxm2 = PXM_make(ww2,hh2);                                               //  output PXM

   for (py = 0; py < hh1; py++)                                            //  copy line pixels to outline pixels
   for (px = 0; px < ww1; px++)                                            //    displaced by outline width
   {
      pix1 = PXMpix(pxm1,px,py);
      pix2 = PXMpix(pxm2,px+toww,py+toww);
      pix2[0] = pix1[0];
   }

   theta = 0.7 / toww;
   for (theta = 0; theta < 6.3; theta += 0.7/toww)                         //  displace outline pixels in all directions
   {
      dx = roundf(toww * sinf(theta));
      dy = roundf(toww * cosf(theta));

      for (py = 0; py < hh1; py++)
      for (px = 0; px < ww1; px++)
      {
         pix1 = PXMpix(pxm1,px,py);
         pix2 = PXMpix(pxm2,px+toww+dx,py+toww+dy);
         if (pix2[1] < pix1[0] - pix2[0])                                  //  compare line to outline brightness
            pix2[1] = pix1[0] - pix2[0];                                   //  brighter part is outline pixel
      }
   }

   return pxm2;                                                            //  pix2[0] / pix2[1] = line / outline
}


//  add a shadow to the line edges
//  red color [0] is original monocolor line intensity
//  green color [1] is added outline if any
//  use blue color [2] for added shadow

PXM * genline_shadow(lineattr_t *attr, PXM *pxm1)                          //  v.13.09
{
   PXM         *pxm2;
   int         shww, ww1, hh1, ww2, hh2;
   int         px, py, dx, dy;
   float       *pix1, *pix2, theta;

   shww = attr->shwidth;                                                   //  line shadow width
   if (shww == 0) return 0;                                                //  zero

   ww1 = pxm1->ww;                                                         //  input PXM dimensions
   hh1 = pxm1->hh;
   ww2 = ww1 + shww * 2;                                                   //  add margins for shadow width
   hh2 = hh1 + shww * 2;
   pxm2 = PXM_make(ww2,hh2);                                               //  output PXM

   for (py = 0; py < hh1; py++)                                            //  copy line pixels to shadow pixels
   for (px = 0; px < ww1; px++)                                            //    displaced by shadow width
   {
      pix1 = PXMpix(pxm1,px,py);
      pix2 = PXMpix(pxm2,px+shww,py+shww);
      pix2[0] = pix1[0];
      pix2[1] = pix1[1];
   }

   theta = (90 - attr->shangle) / 57.3;                                    //  degrees to radians, 0 = to the right
   dx = roundf(shww * sinf(theta));
   dy = roundf(shww * cosf(theta));

   for (py = 0; py < hh1; py++)                                            //  displace line by shadow width
   for (px = 0; px < ww1; px++)
   {
      pix1 = PXMpix(pxm1,px,py);
      pix2 = PXMpix(pxm2,px+shww+dx,py+shww+dy);
      if (pix2[2] < pix1[0] + pix1[1] - pix2[0] - pix2[1])                 //  compare line+outline to shadow pixels
         pix2[2] = pix1[0] + pix1[1] - pix2[0] - pix2[1];                  //  brighter part is shadow pixel
   }

   return pxm2;                                                            //  pix2[0] / pix2[1] / pix2[2]
}                                                                          //    = line / outline / shadow brightness


/**************************************************************************/

//  edit plugins menu

void  m_edit_plugins(GtkWidget *, cchar *)                                 //  simplified          v.13.12
{
   int   edit_plugins_event(zdialog *zd, cchar *event);

   int         ii;
   zdialog     *zd;
   
   F1_help_topic = "plugins";

/***
       _________________________________________
      |         Edit Plugins                    |
      |                                         |
      |  select [__________________________|v]  |
      |   edit  [____________________________]  |     Edit with Gimp = gimp
      |                                         |
      |                   [Add] [Remove] [Done] |
      |_________________________________________|

***/

   zd = zdialog_new("Edit Plugins",Mwin,Badd,Bremove,Bdone,null);
   zdialog_add_widget(zd,"hbox","hbm","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labm","hbm",Bselect,"space=5");
   zdialog_add_widget(zd,"combo","select","hbm",0,"space=5");
   zdialog_add_widget(zd,"hbox","hbe","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labe","hbe",Bedit,"space=5");
   zdialog_add_widget(zd,"entry","edit","hbe",0,"space=5|expand");

   for (ii = 0; ii < Nplugins; ii++)                                       //  stuff combo box with available
      zdialog_cb_app(zd,"select",plugins[ii]);
   
   zdialog_run(zd,edit_plugins_event);
   return;
}


//  dialog event function

int  edit_plugins_event(zdialog *zd, cchar *event)
{
   int      ii, jj, cc, zstat;
   char     *pp, menuentry[200];
   char     buff[200];
   FILE     *fid;
   cchar    *errmess = ZTX("format: menu name = command %%s");

   if (strEqu(event,"select")) 
   {
      zdialog_fetch(zd,"select",menuentry,200);
      zdialog_stuff(zd,"edit",menuentry);                                  //  stuff corresp. command in dialog
      return 1;
   }
   
   zstat = zd->zstat;                                                      //  wait for dialog completion
   if (! zstat) return 0;
   
   if (zstat < 1 || zstat > 3) {                                           //  [x] button
      zdialog_free(zd);
      return 1;
   }
   
   if (zstat == 1)                                                         //  add new plugin
   {
      zd->zstat = 0;                                                       //  keep dialog active  bugfix      v.14.05

      if (Nplugins == maxplugins) {
         zmessageACK(Mwin,0,"too many plugins");
         return 1;
      }
      
      zdialog_fetch(zd,"edit",menuentry,200);                              //  validate format:  menu = command %s
      pp = strstr(menuentry," = ");
      if (! pp || pp-menuentry < 3 || strlen(pp) < 6) {
         zmessageACK(Mwin,0,errmess);
         return 1;
      }
      if (! strstr(pp+3," %s")) {
         zmessageACK(Mwin,0,errmess);
         return 1;
      }
      
      cc = pp - menuentry + 1;                                             //  look for duplicate "menu name ="
      for (ii = 0; ii < Nplugins; ii++)
         if (strnEqu(plugins[ii],menuentry,cc)) break;
      if (ii < Nplugins) {
         zdialog_cb_delete(zd,"select",plugins[ii]);                       //  delete old entry from combo box
         Nplugins--;                                                       //  remove old plugin record        v.14.05
         for (jj = ii; jj < Nplugins; jj++)
            plugins[jj] = plugins[jj+1];
      }

      ii = Nplugins++;                                                     //  add plugin record
      plugins[ii] = zstrdup(menuentry);
      zdialog_cb_app(zd,"select",menuentry);                               //  add to combo box
      zdialog_stuff(zd,"select",menuentry);                                //  select this entry
   }
   
   if (zstat == 2)                                                         //  remove current plugin
   {
      zd->zstat = 0;                                                       //  keep dialog active              v.14.05

      zdialog_fetch(zd,"edit",menuentry,200);

      for (ii = 0; ii < Nplugins; ii++)
         if (strEqu(plugins[ii],menuentry)) break;
      if (ii == Nplugins) {
         zmessageACK(Mwin,0,"menu not found");
         return 1;
      }
      
      Nplugins--;                                                          //  remove plugin record
      for (jj = ii; jj < Nplugins; jj++)
         plugins[jj] = plugins[jj+1];
      zdialog_cb_delete(zd,"select",menuentry);                            //  delete entry from combo box
   }
   
   if (zstat == 3)                                                         //  done
   {
      snprintf(buff,199,"%s/plugins",get_zuserdir());                      //  open file for plugins
      fid = fopen(buff,"w");
      if (! fid) {
         zmessageACK(Mwin,0,strerror(errno));
         return 1;
      }
      
      for (int ii = 0; ii < Nplugins; ii++)                                //  save plugins
         if (plugins[ii])
            fprintf(fid,"%s \n",plugins[ii]);
      
      fclose(fid);

      zdialog_free(zd);
      zmessageACK(Mwin,0,ZTX("Restart Fotoxx to update plugin menu"));
   }
   
   return 1;
}


//  process plugin menu selection
//  execute correspinding command using current image file

editfunc    EFplugin;

void  m_run_plugin(GtkWidget *, cchar *menu)
{
   int         ii, jj, err;
   char        *pp = 0, plugincommand[200], pluginfile[200];
   PXM         *pxmtemp;
   zdialog     *zd = 0;

   F1_help_topic = "plugins";

   if (checkpend("lock")) return;                                          //  check nothing pending              v.13.12

   for (ii = 0; ii < Nplugins; ii++)                                       //  search plugins for menu name
   {
      pp = strchr(plugins[ii],'=');                                        //  match menu name to plugin command
      if (! pp) continue;                                                  //  menu name = ...
      *pp = 0;
      jj = strEqu(plugins[ii],menu);
      *pp = '=';
      if (jj) break;
   }

   if (ii == Nplugins) {
      zmessageACK(Mwin,0,"plugin menu not found %s",menu);
      return;
   }
   
   strncpy0(plugincommand,pp+1,200);                                       //  corresp. command
   strTrim2(plugincommand);

   pp = strstr(plugincommand,"%s");
   if (! pp) {
      zmessageACK(Mwin,0,"plugin command has no \"%%s\" ");
      return;
   }

   EFplugin.menufunc = m_run_plugin;
   EFplugin.funcname = menu;
   if (! edit_setup(EFplugin)) return;                                     //  start edit function
   
   Fmenulock = 1;                                                          //  block others

   snprintf(pluginfile,199,"%s/plugfile.tif",tempdir);                     //  /.../fotoxx-nnnnnn/plugfile.tif
   
   err = PXM_TIFF_save(E1pxm,pluginfile,8);                                //  E1 >> plugin_file
   if (err) {
      if (*file_errmess)                                                   //  pass error to user
         zmessageACK(Mwin,0,file_errmess);
      edit_cancel(0);
      goto RETURN;
   }

   zd = zmessage_post(Mwin,0,ZTX("Plugin working ..."));                   //  v.13.12
   
   repl_1str(plugincommand,command,"%s",pluginfile);                       //  v.14.05

   err = shell_ack(command);                                               //  execute plugin command
   if (err) {
      edit_cancel(0);
      goto RETURN;
   }
   
   pxmtemp = TIFF_PXM_load(pluginfile);                                    //  read command output file
   if (! pxmtemp) {
      zmessageACK(Mwin,0,ZTX("plugin failed"));
      edit_cancel(0);
      goto RETURN;
   }
   
   PXM_free(E3pxm);                                                        //  plugin_file >> E3
   E3pxm = pxmtemp;

   CEF->Fmods = 1;                                                         //  assume image was modified
   CEF->Fsaved = 0;
   edit_done(0);

RETURN:
   Fmenulock = 0;
   if (zd) zdialog_free(zd);
   return;
}   


/**************************************************************************/

//  automatic image tuneup without user guidance

float       voodoo_brdist[256];                                            //  pixel count per brightness level
int         voodoo_01, voodoo_99;                                          //  1% and 99% brightness levels (0 - 255)
void        voodoo_distribution();                                         //  compute brightness distribution
void *      voodoo_thread(void *);
editfunc    EFvoodoo;


void m_voodoo(GtkWidget *, cchar *)                                        //  new v.12.06
{
   F1_help_topic = "voodoo";

   EFvoodoo.menufunc = m_voodoo;
   EFvoodoo.funcname = "voodoo";
   EFvoodoo.Farea = 1;                                                     //  select area ignored
   EFvoodoo.threadfunc = voodoo_thread;
   if (! edit_setup(EFvoodoo)) return;                                     //  setup edit
   
   voodoo_distribution();                                                  //  compute brightness distribution
   signal_thread();                                                        //  start update thread
   edit_done(0);                                                           //  edit done
   return;
}


//  compute brightness distribution for image

void voodoo_distribution()
{
   int         px, py, ii;
   int         ww = E1pxm->ww, hh = E1pxm->hh;
   float       bright1;
   float       *pix1;

   for (ii = 0; ii < 256; ii++)                                            //  clear brightness distribution data
      voodoo_brdist[ii] = 0;

   for (py = 0; py < hh; py++)                                             //  compute brightness distribution
   for (px = 0; px < ww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);
      bright1 = pixbright(pix1);
      voodoo_brdist[int(bright1)]++;
   }

   for (ii = 1; ii < 256; ii++)                                            //  cumulative brightness distribution
      voodoo_brdist[ii] += voodoo_brdist[ii-1];                            //   0 ... (ww * hh)
   
   voodoo_01 = 0;
   voodoo_99 = 255;

   for (ii = 0; ii < 256; ii++)                                            //  compute index values (0 - 255)
   {                                                                       //    for darkest 1% and brightest 1%
      if (voodoo_brdist[ii] < 0.01 * ww * hh) voodoo_01 = ii;
      if (voodoo_brdist[ii] < 0.99 * ww * hh) voodoo_99 = ii;
   }
   
   for (ii = 0; ii < 256; ii++)
      voodoo_brdist[ii] = voodoo_brdist[ii]                                //  multiplier per brightness level 
                    / (ww * hh) * 256.0 / (ii + 1);                        //  ( = 1 for a flat distribution)
   return;
}


//  thread function - use multiple working threads

void * voodoo_thread(void *)
{
   void  * voodoo_wthread(void *arg);

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wthread(voodoo_wthread,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion

      CEF->Fmods++; 
      CEF->Fsaved = 0;
      Fpaint2();                                                           //  update window
   }

   return 0;                                                               //  not executed, stop g++ warning
}

void * voodoo_wthread(void *arg)                                           //  worker thread function
{
   int         index = *((int *) (arg));
   int         px, py;
   float       *pix1, *pix3;
   float       bright1, bright2, bright3, cmax;
   float       red1, green1, blue1, red3, green3, blue3;
   float       flat1 = 0.3;                                                //  strength of distribution flatten
   float       flat2 = 1.0 - flat1;
   float       sat1 = 0.3, sat2;                                           //  strength of saturation increase    v.14.06
   float       f1, f2;
   float       expand = 256.0 / (voodoo_99 - voodoo_01 + 1);               //  brightness range expander
   
   for (py = index; py < E1pxm->hh; py += Nwt)                             //  voodoo brightness distribution
   for (px = 0; px < E1pxm->ww; px++)
   {
      pix1 = PXMpix(E1pxm,px,py);                                          //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                          //  output pixel

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      bright2 = 0.25 * red1 + 0.65 * green1 + 0.10 * blue1;                //  input brightness, 0 - 256
      bright2 = (bright2 - voodoo_01) * expand;                            //  expand to clip low / high 1% 
      if (bright2 < 0) bright2 = 0;
      if (bright2 > 255) bright2 = 255;

      bright1 = voodoo_brdist[int(bright2)];                               //  factor for flat output brightness
      bright3 = flat1 * bright1 + flat2;                                   //  attenuate

      f1 = (256.0 - bright2) / 256.0;                                      //  bright2 = 0 - 256  >>  f1 = 1 - 0
      f2 = 1.0 - f1;                                                       //  prevent banding in bright areas
      bright3 = f1 * bright3 + f2;                                         //  tends to 1.0 for brighter pixels

      red3 = red1 * bright3;                                               //  blend new and old brightness
      green3 = green1 * bright3;
      blue3 = blue1 * bright3;
      
      bright3 = 0.333 * (red3 + green3 + blue3);                           //  mean color brightness
      sat2 = sat1 * (256.0 - bright3) / 256.0;                             //  bright3 = 0 - 256  >>  sat2 = sat1 - 0
      red3 = red3 + sat2 * (red3 - bright3);                               //  amplified color, max for dark pixels
      green3 = green3 + sat2 * (green3 - bright3);
      blue3 = blue3 + sat2 * (blue3 - bright3);
      
      if (red3 < 0) red3 = 0;                                              //  stop possible underflow
      if (green3 < 0) green3 = 0;
      if (blue3 < 0) blue3 = 0;

      cmax = red3;                                                         //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 255.9) {
         cmax = 255.9 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  Retouch Combo
//  Adjust all aspects of brightness, contrast, color.
//  Brightness curves are used, overall and per RGB color.
//  Color saturation and white balance adjustments are also available.

namespace combo_names
{
   editfunc    EFcombo;
   int         combo_spc;                                                  //  current spline curve
   float       brightness;                                                 //  brightness input, -1 ... +1
   float       contrast;                                                   //  contrast input, -1 ... +1
   float       wbalR, wbalG, wbalB;                                        //  white balance, white basis standard
   int         wbtemp;                                                     //  illumination temp. input, 1K ... 10K
   float       tempR, tempG, tempB;                                        //  RGB of illumination temp.
   float       combR, combG, combB;                                        //  combined RGB factors
   float       satlevel;                                                   //  saturation input, -1 ... +1 = saturated
   float       areaemph;                                                   //  emphasis input, -1 ... +1 = bright areas
   float       emphcurve[256];                                             //  emphasis per brightness level, 0 ... 1
   int         Fapply = 0;                                                 //  flag, apply dialog controls to image
   int         Fdist = 0;                                                  //  flag, show brightness distribution
   float       amplify = 1.0;                                              //  curve amplifier, 0 ... 2
   char        combo_curves[200] = "";                                     //  previous curves save file
   GtkWidget   *drawwin_dist, *drawwin_scale;                              //  brightness distribution graph widgets
   int         RGBW[4] = { 0, 0, 0, 0 };                                   //     "  colors: red/green/blue/white (all)
}

void  blackbodyRGB(int K, float &R, float &G, float &B);


//  menu function

void m_combo(GtkWidget *, cchar *)
{
   using namespace combo_names;

   int    combo_dialog_event(zdialog* zd, cchar *event);
   void   combo_curvedit(int spc);
   void * combo_thread(void *);

   F1_help_topic = "retouch_combo";

   if (! *combo_curves)                                                    //  initz. previous curves save file
      snprintf(combo_curves,200,"%s/RCprev.curve",saved_curves_dirk);

   EFcombo.menufunc = m_combo;
   EFcombo.funcname = "retouch_combo";
   EFcombo.FprevReq = 1;                                                   //  use preview
   EFcombo.Farea = 2;                                                      //  select area usable
   EFcombo.Frestart = 1;                                                   //  restart allowed
   EFcombo.threadfunc = combo_thread;
   if (! edit_setup(EFcombo)) return;                                      //  setup edit
   
/***
       _____________________________________________________
      |                    Retouch Combo                    |
      |  _________________________________________________  |
      | |                                                 | |              //  5 curves are maintained:
      | |                                                 | |              //  curve 0: current display curve
      | |                                                 | |              //        1: curve for all colors
      | |         curve edit area                         | |              //        2,3,4: red, green, blue
      | |                                                 | |
      | |                                                 | |
      | |_________________________________________________| |
      | |_________________________________________________| |              //  brightness scale: black to white stripe
      |   (o) all  (o) red  (o) green  (o) blue             |              //  select curve to display
      |                                                     |
      |  Amplifier  ================[]=============  Max.   |              //  curve amplifier
      |  Brightness ================[]=============  High   |              //  brightness
      |   Contrast  ================[]=============  High   |              //  contrast
      |   Low Color ====================[]=========  High   |              //  color saturation
      |    Warmer   ====================[]========= Cooler  |              //  color temperature
      |  Dark Areas ==========[]=================== Bright  |              //  color emphasis
      |                                                     | 
      |  [x] Brightness Distribution                        |
      |  [x] Click for white balance or black level         |              //  click gray/white spot for white balance
      |                                                     |
      |  Settings [Load] [Save]                             |
      |                                                     |
      |                      [Reset] [Prev] [Done] [Cancel] |
      |_____________________________________________________|

***/

   EFcombo.zd = zdialog_new(ZTX("Retouch Combo"),Mwin,Breset,Bprev,Bdone,Bcancel,null);
   zdialog *zd = EFcombo.zd;

   zdialog_add_widget(zd,"frame","frameH","dialog",0,"expand");            //  edit-curve and distribution graph 
   zdialog_add_widget(zd,"frame","frameB","dialog");                       //  black to white brightness scale

   zdialog_add_widget(zd,"hbox","hbrgb","dialog");
   zdialog_add_widget(zd,"radio","all","hbrgb",Ball,"space=5");
   zdialog_add_widget(zd,"radio","red","hbrgb",Bred,"space=3");
   zdialog_add_widget(zd,"radio","green","hbrgb",Bgreen,"space=3");
   zdialog_add_widget(zd,"radio","blue","hbrgb",Bblue,"space=3");
   
   zdialog_add_widget(zd,"hbox","hbcolor","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","space","hbcolor",0,"space=5");
   zdialog_add_widget(zd,"vbox","vbcolor1","hbcolor",0,"homog");
   zdialog_add_widget(zd,"label","space","hbcolor",0,"space=3");
   zdialog_add_widget(zd,"vbox","vbcolor2","hbcolor",0,"homog|expand");
   zdialog_add_widget(zd,"label","space","hbcolor",0,"space=3");
   zdialog_add_widget(zd,"vbox","vbcolor3","hbcolor",0,"homog");
   zdialog_add_widget(zd,"label","space","hbcolor",0,"space=5");

   zdialog_add_widget(zd,"label","labamp","vbcolor1",ZTX("Amplifier"));
   zdialog_add_widget(zd,"label","labrite","vbcolor1",ZTX("Brightness"));
   zdialog_add_widget(zd,"label","labcont","vbcolor1",ZTX("Contrast"));
   zdialog_add_widget(zd,"label","labsat1","vbcolor1",ZTX("Low Color"));
   zdialog_add_widget(zd,"label","labtemp1","vbcolor1",ZTX("Warmer"));
   zdialog_add_widget(zd,"label","labarea1","vbcolor1",ZTX("Dark Areas"));

   zdialog_add_widget(zd,"hscale","amplify","vbcolor2","0.0|2.0|0.01|1.0");
   zdialog_add_widget(zd,"hscale","brightness","vbcolor2","-1.0|1.0|0.002|0");
   zdialog_add_widget(zd,"hscale","contrast","vbcolor2","-1.0|1.0|0.002|0");
   zdialog_add_widget(zd,"hscale","satlevel","vbcolor2","-1.0|1.0|0.002|0");
   zdialog_add_widget(zd,"hscale","wbtemp","vbcolor2","2000|8000|1|5000");
   zdialog_add_widget(zd,"hscale","areaemph","vbcolor2","0|1.0|0.002|0.5");

   zdialog_add_widget(zd,"label","labrite2","vbcolor3",ZTX("Max."));
   zdialog_add_widget(zd,"label","labrite2","vbcolor3",ZTX("High"));
   zdialog_add_widget(zd,"label","labcont2","vbcolor3",ZTX("High"));
   zdialog_add_widget(zd,"label","labsat2","vbcolor3",ZTX("High"));
   zdialog_add_widget(zd,"label","labtemp2","vbcolor3",ZTX("Cooler"));
   zdialog_add_widget(zd,"label","labarea2","vbcolor3",ZTX("Bright"));
   
   zdialog_add_widget(zd,"hbox","hbdist","dialog");
   zdialog_add_widget(zd,"check","dist","hbdist",ZTX("Brightness Distribution"),"space=3");
   
   zdialog_add_widget(zd,"hbox","hbclick","dialog");
   zdialog_add_widget(zd,"check","click","hbclick",ZTX("Click for white balance or black level"),"space=3");
   
   zdialog_add_widget(zd,"hbox","hbset","dialog");
   zdialog_add_widget(zd,"label","labset","hbset",ZTX("Settings File"),"space=5");
   zdialog_add_widget(zd,"button","load","hbset",Bload,"space=3");
   zdialog_add_widget(zd,"button","save","hbset",Bsave,"space=3");
   
   GtkWidget *frameH = zdialog_widget(zd,"frameH");                        //  setup edit curves
   spldat *sd = splcurve_init(frameH,combo_curvedit);
   EFcombo.curves = sd;

   sd->Nscale = 1;                                                         //  diagonal fixed line, neutral curve
   sd->xscale[0][0] = 0.00;
   sd->yscale[0][0] = 0.00;
   sd->xscale[1][0] = 1.00;
   sd->yscale[1][0] = 1.00;

   for (int ii = 0; ii < 5; ii++)                                          //  loop curves 0-4
   {
      sd->nap[ii] = 5;                                                     //  initial curves are neutral
      sd->vert[ii] = 0;
      sd->apx[ii][0] = sd->apy[ii][0] = 0.01;                              //  curve 0 = edit curve
      sd->apx[ii][1] = sd->apy[ii][1] = 0.25;                              //  curve 1 = overall brightness
      sd->apx[ii][2] = sd->apy[ii][2] = 0.50;                              //  curve 2 = red adjustment
      sd->apx[ii][3] = sd->apy[ii][3] = 0.75;                              //  curve 3 = green adjustment
      sd->apx[ii][4] = sd->apy[ii][4] = 0.99;                              //  curve 4 = blue adjustment
      splcurve_generate(sd,ii);
   }

   sd->Nspc = 1;                                                           //  only curve 0 is shown 
   combo_spc = 1;                                                          //  current active curve: 1 (all)
   zdialog_stuff(zd,"all",1);                                              //  stuff default selection, all

   drawwin_dist = sd->drawarea;                                            //  setup brightness distr. drawing area
   G_SIGNAL(drawwin_dist,"draw",brdist_drawgraph,RGBW);

   GtkWidget *frameB = zdialog_widget(zd,"frameB");                        //  setup brightness scale drawing area
   drawwin_scale = gtk_drawing_area_new();
   gtk_container_add(GTK_CONTAINER(frameB),drawwin_scale);
   gtk_widget_set_size_request(drawwin_scale,300,12);
   G_SIGNAL(drawwin_scale,"draw",brdist_drawscale,0);

   brightness = 0;                                                         //  neutral brightness                 v.14.02
   contrast = 0;                                                           //  neutral contrast                   v.14.02

   wbalR = wbalG = wbalB = 1.0;                                            //  neutral white balance 
   wbtemp = 5000;                                                          //  neutral illumination temp.
   blackbodyRGB(wbtemp,tempR,tempG,tempB);                                 //  all are 1.0
   combR = combG = combB = 1.0;                                            //  neutral RGB factors

   satlevel = 0;                                                           //  neutral saturation
   
   areaemph = 0.5;                                                         //  even dark/bright area emphasis
   for (int ii = 0; ii < 256; ii++)
      emphcurve[ii] = 1;
   
   zdialog_stuff(zd,"click",0);                                            //  reset mouse click status

   zdialog_resize(zd,300,450);
   zdialog_run(zd,combo_dialog_event,"save");                              //  run dialog - parallel
   
   return;
}


//  dialog event and completion callback function

int combo_dialog_event(zdialog *zd, cchar *event)
{
   using namespace combo_names;

   void  combo_mousefunc();

   spldat      *sd = EFcombo.curves;
   float       c1, c2, c3, xval, yval;
   float       bright0, dbrite, cont0, dcont;
   float       dx, dy;
   int         ii, jj;
   int         Fapply = 0;
   
   if (strEqu(event,"done")) zd->zstat = 3;                                //  apply and quit 
   if (strEqu(event,"enter")) zd->zstat = 3;                               //  v.14.03
   
   if (zd->zstat == 1)                                                     //  [reset]
   {
      zd->zstat = 0;                                                       //  keep dialog active

      zdialog_stuff(zd,"amplify",1);                                       //  neutral amplifier                  v.14.04
      zdialog_stuff(zd,"brightness",0);                                    //  neutral brightness                 v.14.02
      zdialog_stuff(zd,"contrast",0);                                      //  neutral contrast                   v.14.02
      brightness = contrast = 0;

      wbalR = wbalG = wbalB = 1.0;                                         //  neutral white balance
      wbtemp = 5000;                                                       //  neutral illumination temp.
      zdialog_stuff(zd,"wbtemp",wbtemp);
      blackbodyRGB(wbtemp,tempR,tempG,tempB);                              //  all are 1.0
      combR = combG = combB = 1.0;                                         //  neutral RGB factors
      
      satlevel = 0;                                                        //  neutral saturation
      zdialog_stuff(zd,"satlevel",0);

      areaemph = 0.5;                                                      //  even area emphasis
      zdialog_stuff(zd,"areaemph",0.5);
      for (int ii = 0; ii < 256; ii++)                                     //  flat curve
         emphcurve[ii] = 1;

      for (int ii = 0; ii < 5; ii++)                                       //  loop gamma curves 0-4
      {
         sd->nap[ii] = 5;                                                  //  all curves are neutral
         sd->vert[ii] = 0;
         sd->apx[ii][0] = sd->apy[ii][0] = 0.01;
         sd->apx[ii][1] = sd->apy[ii][1] = 0.25;
         sd->apx[ii][2] = sd->apy[ii][2] = 0.50;
         sd->apx[ii][3] = sd->apy[ii][3] = 0.75;
         sd->apx[ii][4] = sd->apy[ii][4] = 0.99;
         splcurve_generate(sd,ii);
      }

      combo_spc = 1;                                                       //  current active curve: 1 (all)
      zdialog_stuff(zd,"all",1);
      edit_reset();                                                        //  restore initial image 
      event = "update";                                                    //  trigger graph update               v.14.04
   }
   
   if (zd->zstat == 2)                                                     //  [prev] restore previous settings
   {
      zd->zstat = 0;                                                       //  keep dialog active
      zdialog_restore_inputs(zd);                                          //  restore previous inputs

      sd->Nspc = 5;                                                        //  5 curves
      splcurve_load(sd,combo_curves);                                      //  load all curves from file
      sd->Nspc = 1;                                                        //  only one curve active
      for (int ii = 0; ii < 5; ii++)
         splcurve_generate(sd,ii);                                         //  v.14.02
      Fapply = 1;                                                          //  trigger apply event
   }

   if (zd->zstat)                                                          //  [done] or [cancel]
   {
      freeMouse();
      if (zd->zstat == 3) {                                                //  [done]
         sd->Nspc = 5;                                                     //  include all 5 curves
         splcurve_save(sd,combo_curves);                                   //  save curves for next time
         edit_done(0);                                                     //  saves zdialog inputs
      }
      else edit_cancel(0);                                                 //  [cancel] or [x]
      return 1;
   }
   
   if (strEqu(event,"load")) {                                             //  load all settings from a file      v.14.05
      sd->Nspc = 5;
      edit_load_widgets(&EFcombo,0);
      sd->Nspc = 1;
   }

   if (strEqu(event,"save")) {                                             //  save all settings to a file        v.14.05
      sd->Nspc = 5;
      edit_save_widgets(&EFcombo);
      sd->Nspc = 1;
   }

   if (strEqu(event,"click")) {                                            //  toggle mouse click input
      zdialog_fetch(zd,"click",ii);
      if (ii) takeMouse(combo_mousefunc,dragcursor);                       //  on: connect mouse function
      else freeMouse();                                                    //  off: free mouse
      return 1;
   }

   if (strEqu(event,"dist")) {                                             //  distribution checkbox
      zdialog_fetch(zd,"dist",Fdist);
      event = "update";                                                    //  trigger graph update               v.14.04
   }
   
   if (strstr("all red green blue",event))                                 //  new choice of curve
   {
      zdialog_fetch(zd,event,ii);
      if (! ii) return 0;                                                  //  button OFF event, wait for ON event

      ii = strcmpv(event,"all","red","green","blue",null);
      combo_spc = ii;                                                      //  active curve: 1, 2, 3, 4

      sd->nap[0] = sd->nap[ii];                                            //  copy active curve nodes to curve 0
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[0][jj] = sd->apx[ii][jj];
         sd->apy[0][jj] = sd->apy[ii][jj];
      }

      splcurve_generate(sd,0);                                             //  regenerate curve 0
      
      for (jj = 0; jj < 1000; jj++)                                        //  copy curve values back
         sd->yval[ii][jj] = sd->yval[0][jj];

      gtk_widget_queue_draw(sd->drawarea);                                 //  draw curve
   }
   
   if (strEqu(event,"brightness"))                                         //  brightness slider, 0 ... 1         v.14.02
   {
      bright0 = brightness;
      zdialog_fetch(zd,"brightness",brightness);
      dbrite = brightness - bright0;
      
      zdialog_stuff(zd,"all",1);                                           //  active curve is "all"
      ii = combo_spc = 1;
      sd->nap[0] = sd->nap[ii];                                            //  copy active curve to curve 0
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[0][jj] = sd->apx[ii][jj];
         sd->apy[0][jj] = sd->apy[ii][jj];
      }

      for (jj = 0; jj < sd->nap[0]; jj++)                                  //  update curve 0 nodes
      {
         dx = sd->apx[0][jj];                                              //  0 ... 0.5 ... 1
         if (dx <= 0.01 || dx >= 0.99) continue;
         dx = 0.5 - fabsf(dx - 0.5);                                       //  0 ... 0.5 ... 0
         dx = 0.5 * sqrtf(dx);                                             //  0 ... 0.35 ... 0  (more rounded shape)
         dy = dx * dbrite;
         dy += sd->apy[0][jj];
         if (dy < 0) dy = 0;
         if (dy > 1) dy = 1;
         sd->apy[0][jj] = dy;
      }

      splcurve_generate(sd,0);                                             //  regenerate curve 0
      
      for (jj = 0; jj < sd->nap[0]; jj++) {                                //  copy curve nodes back
         sd->apx[ii][jj] = sd->apx[0][jj];   
         sd->apy[ii][jj] = sd->apy[0][jj];
      }

      for (jj = 0; jj < 1000; jj++)                                        //  copy curve values back
         sd->yval[ii][jj] = sd->yval[0][jj];

      gtk_widget_queue_draw(sd->drawarea);                                 //  draw curve 0
   }
   
   if (strEqu(event,"contrast"))                                           //  contrast slider, 0 ... 1           v.14.02
   {
      cont0 = contrast;
      zdialog_fetch(zd,"contrast",contrast);
      dcont = contrast - cont0;

      zdialog_stuff(zd,"all",1);                                           //  active curve is "all"
      ii = combo_spc = 1;
      sd->nap[0] = sd->nap[ii];                                            //  copy active curve to curve 0
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[0][jj] = sd->apx[ii][jj];
         sd->apy[0][jj] = sd->apy[ii][jj];
      }

      for (jj = 0; jj < sd->nap[0]; jj++)                                  //  update curve 0 nodes
      {
         dx = sd->apx[0][jj];                                              //  0 ... 0.5 ... 1
         if (dx <= 0.01 || dx >= 0.99) continue;
         if (dx < 0.5) dx = -0.25 + fabsf(dx - 0.25);                      //  0 ... -0.25 ... 0
         else dx = +0.25 - fabsf(dx - 0.75);                               //  0 ... +0.25 ... 0
         dy = dx * dcont;
         dy += sd->apy[0][jj];
         if (dy < 0) dy = 0;
         if (dy > 1) dy = 1;
         sd->apy[0][jj] = dy;
      }

      splcurve_generate(sd,0);                                             //  regenerate curve 0
      
      for (jj = 0; jj < sd->nap[0]; jj++) {                                //  copy curve nodes back
         sd->apx[ii][jj] = sd->apx[0][jj];
         sd->apy[ii][jj] = sd->apy[0][jj];
      }

      for (jj = 0; jj < 1000; jj++)                                        //  copy curve values back
         sd->yval[ii][jj] = sd->yval[0][jj];

      gtk_widget_queue_draw(sd->drawarea);                                 //  draw curve 0
   }
   
   if (Fdist) {                                                            //  distribution enabled
      zdialog_fetch(zd,"red",RGBW[0]);                                     //  get graph color choice
      zdialog_fetch(zd,"green",RGBW[1]);
      zdialog_fetch(zd,"blue",RGBW[2]);
      zdialog_fetch(zd,"all",RGBW[3]);
      if (RGBW[3]) RGBW[0] = RGBW[1] = RGBW[2] = 1;
      RGBW[3] = 0;
   }
   else RGBW[0] = RGBW[1] = RGBW[2] = RGBW[3] = 0;

   if (strEqu(event,"update"))                                             //  thread done or new color choice
      gtk_widget_queue_draw(drawwin_dist);                                 //  update distribution graph

   if (strEqu(event,"blendwidth")) Fapply = 1;                             //  trigger apply event
   if (strstr("amplify brightness contrast",event)) Fapply = 1;
   if (strstr("satlevel areaemph wbtemp",event)) Fapply = 1;
   if (strEqu(event,"load")) Fapply = 1;
   if (! Fapply) return 1;                                                 //  wait for change
   
   zdialog_fetch(zd,"amplify",amplify);                                    //  get curve amplifier setting
   zdialog_fetch(zd,"brightness",brightness);                              //  get brightness setting
   zdialog_fetch(zd,"contrast",contrast);                                  //  get contrast setting
   zdialog_fetch(zd,"satlevel",satlevel);                                  //  get saturation setting

   zdialog_fetch(zd,"wbtemp",wbtemp);                                      //  get illumination temp. setting
   blackbodyRGB(wbtemp,tempR,tempG,tempB);                                 //  generate new temp. adjustments
   
   zdialog_fetch(zd,"areaemph",areaemph);                                  //  get dark/bright area emphasis
      
   c1 = 2 * (1 - areaemph);                                                //  generate area emphasis curve
   if (c1 > 1) c1 = 1;
   c3 = 2 * areaemph;
   if (c3 > 1) c3 = 1;
   c2 = 0.7 * (c1 + c3);
   
   for (ii = 0; ii < 256; ii++) {
      xval = ii / 256.0;
      if (xval <= 0.5)  yval = c1 + 4 * (c2 - c1) * xval * xval;
      else  yval = c3 + 4 * (c2 - c3) * (1 - xval) * (1 - xval);
      if (yval > 1.0) yval = 1.0;
      emphcurve[ii] = yval;
   }

   signal_thread();                                                        //  update the image
   return 1;
}


//  this function is called when a curve is edited

void combo_curvedit(int spc)
{
   using namespace combo_names;

   int      ii = combo_spc, jj;
   spldat   *sd = EFcombo.curves;                                          //  active curve, 1-4
   
   sd->nap[ii] = sd->nap[0];                                               //  copy curve 0 to active curve
   for (jj = 0; jj < sd->nap[0]; jj++) {
      sd->apx[ii][jj] = sd->apx[0][jj];
      sd->apy[ii][jj] = sd->apy[0][jj];
   }

   for (jj = 0; jj < 1000; jj++) 
      sd->yval[ii][jj] = sd->yval[0][jj];

   signal_thread();
   return;
}


//  get nominal white color from mouse click position

void combo_mousefunc()                                                     //  mouse function
{
   using namespace combo_names;

   int         px, py, dx, dy, ii, err;
   float       red, green, blue, rgbmean;
   float       *ppix;
   spldat      *sd = EFcombo.curves;                                       //  active curve, 1-4
   zdialog     *zd = EFcombo.zd;
   char        mousetext[60];
   GdkScreen   *screen;
   GdkDevice   *mouse;

   if (! LMclick) return;
   LMclick = 0;
   
   px = Mxclick;                                                           //  mouse click position
   py = Myclick;
   
   if (px < 2) px = 2;                                                     //  pull back from edge
   if (px > E3pxm->ww-3) px = E3pxm->ww-3;
   if (py < 2) py = 2;
   if (py > E3pxm->hh-3) py = E3pxm->hh-3;
   
   red = green = blue = 0;

   for (dy = -1; dy <= 1; dy++)                                            //  3x3 block around mouse position
   for (dx = -1; dx <= 1; dx++)
   {
      ppix = PXMpix(E1pxm,px+dx,py+dy);                                    //  input image
      red += ppix[0];
      green += ppix[1];
      blue += ppix[2];
   }

   red = red / 9.0;                                                        //  mean RGB levels 
   green = green / 9.0;
   blue = blue / 9.0;

   snprintf(mousetext,60,"3x3 pixels RGB: %.0f %.0f %.0f \n",red,green,blue);
   err = get_mouse_device(Mwin,&screen,&mouse);
   if (err) return;
   poptext_mouse(mousetext,mouse,10,10,0,3);
   
   if (red > 80 && green > 80 && blue > 80)                                //  assume this is a gray/white point
   {
      rgbmean = (red + green + blue) / 3.0;
      wbalR = rgbmean / red;                                               //  = 1.0 if all RGB are equal
      wbalG = rgbmean / green;                                             //  <1/>1 if RGB should be less/more
      wbalB = rgbmean / blue;
      signal_thread();                                                     //  trigger image update
   }
   
   else if (red < 50 && green < 50 && blue < 50)                           //  assume this is a black point
   {
      combo_spc = 1;                                                       //  curve 1 (all) is active curve
      zdialog_stuff(zd,"all",1);

      sd->apx[1][0] = (red + green + blue) / 3.0 / 256.0;                  //  set gamma curve low clipping level
      sd->apy[1][0] = 0.0;                                                 //    from black point pixel

      sd->nap[0] = sd->nap[1];                                             //  copy curve 1 to curve 0
      for (ii = 0; ii < sd->nap[0]; ii++) {
         sd->apx[0][ii] = sd->apx[1][ii];
         sd->apy[0][ii] = sd->apy[1][ii];
      }

      splcurve_generate(sd,0);                                             //  regenerate curve 0
      
      for (ii = 0; ii < 1000; ii++)                                        //  copy curve values back
         sd->yval[1][ii] = sd->yval[0][ii];

      gtk_widget_queue_draw(sd->drawarea);                                 //  draw curve
      signal_thread();                                                     //  update image
   }

   return;
}


//  combo thread function

void * combo_thread(void *arg)
{
   using namespace combo_names;

   void * combo_wthread(void *);
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      combR = wbalR * tempR / 256.0;                                       //  gray standard based on clicked pixel
      combG = wbalG * tempG / 256.0;                                       //    colors and illumination temp.
      combB = wbalB * tempB / 256.0;                                       //      <1/>1 if RGB should be less/more

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wthread(combo_wthread,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion
      
      Fpaint2();                                                           //  update window
      CEF->Fmods++;                                                        //  image3 modified
      CEF->Fsaved = 0;
      
      if (CEF && CEF->zd) {                                                //                                     v.14.06
         zd_thread = CEF->zd;                                              //  signal dialog to update graph      v.14.04
         zd_thread_event = "update";
      }
      else zd_thread = 0;                                                  //  v.14.06
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * combo_wthread(void *arg)                                            //  worker thread function
{
   using namespace combo_names;
   
   int         index = *((int *) arg);
   int         ii, dist = 0, px, py;
   float       *pix1, *pix3;
   float       red1, green1, blue1, maxrgb;
   float       red3, green3, blue3;
   float       pixbrite, F1, F2;
   float       coeff = 1000.0 / 256.0;
   float       dold, dnew, ff;
   spldat      *sd = EFcombo.curves;

   for (py = index; py < E3pxm->hh; py += Nwt)                             //  loop all pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                  //  select area active
         ii = py * E3pxm->ww + px;
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  pixel outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                          //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                          //  output pixel

      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      //  apply white balance and temperature color shift

      red3 = combR * red1;                                                 //  <1/>1 if RGB should be less/more
      green3 = combG * green1;
      blue3 = combB * blue1;

      //  apply saturation color shift

      if (satlevel != 0) {
         pixbrite = 0.333 * (red3 + green3 + blue3);                       //  pixel brightness, 0 to 255.9
         red3 = red3 + satlevel * (red3 - pixbrite);                       //  satlevel is -1 ... +1
         green3 = green3 + satlevel * (green3 - pixbrite);
         blue3 = blue3 + satlevel * (blue3 - pixbrite);
      }

      if (red3 < 0) red3 = 0;                                              //  stop underflow
      if (green3 < 0) green3 = 0;
      if (blue3 < 0) blue3 = 0;

      maxrgb = red3;                                                       //  stop overflow
      if (green3 > maxrgb) maxrgb = green3;
      if (blue3 > maxrgb) maxrgb = blue3;
      if (maxrgb > 255.9) {
         red3 = red3 * 255.9 / maxrgb;
         green3 = green3 * 255.9 / maxrgb;
         blue3 = blue3 * 255.9 / maxrgb;
         maxrgb = 255.9;
      }

      //  apply dark/bright area emphasis curve for color changes
      
      F1 = emphcurve[int(maxrgb)];                                         //  0 ... 1   neutral is flat curve = 1
      if (F1 < 1.0) {
         F2 = 1.0 - F1;
         red3 = F1 * red3 + F2 * red1;
         green3 = F1 * green3 + F2 * green1;
         blue3 = F1 * blue3 + F2 * blue1;
      }

      //  apply brightness/contrast curve 
      
      pixbrite = red3;                                                     //  use max. RGB value
      if (green3 > pixbrite) pixbrite = green3;
      if (blue3 > pixbrite) pixbrite = blue3;

      if (amplify == 1.0)                                                  //  amplifier is neutral
      {
         ii = coeff * pixbrite;                                            //  range 0-999 for curve index
         ff = 255.9 * sd->yval[1][ii];                                     //  curve "all" adjustment
         red3 = ff * red3 / pixbrite;                                      //  projected on each RGB color
         green3 = ff * green3 / pixbrite;
         blue3 = ff * blue3 / pixbrite;
      }
      else                                                                 //  use amplifier
      {
         ii = coeff * pixbrite;
         ff = 0.001 * ii;
         ff = ff + amplify * (sd->yval[1][ii] - ff);                       //  amplify (curve - baseline) difference
         if (ff < 0) ff = 0;
         red3 = 255.9 * ff * red3 / pixbrite;
         green3 = 255.9 * ff * green3 / pixbrite;
         blue3 = 255.9 * ff * blue3 / pixbrite;
      }
      
      ii = coeff * red3;                                                   //  add additional RGB adjustments
      if (ii < 0) ii = 0;
      if (ii > 999) ii = 999;
      red3 = 255.9 * sd->yval[2][ii];                                      //  output brightness, 0-255.9

      ii = coeff * green3;
      if (ii < 0) ii = 0;
      if (ii > 999) ii = 999;
      green3 = 255.9 * sd->yval[3][ii];

      ii = coeff * blue3;
      if (ii < 0) ii = 0;
      if (ii > 999) ii = 999;
      blue3 = 255.9 * sd->yval[4][ii];

      if (red3 < 0) red3 = 0;                                              //  stop underflow
      if (green3 < 0) green3 = 0;
      if (blue3 < 0) blue3 = 0;

      maxrgb = red3;                                                       //  stop overflows
      if (green3 > maxrgb) maxrgb = green3;
      if (blue3 > maxrgb) maxrgb = blue3;
      if (maxrgb > 255.9) {
         red3 = red3 * 255.9 / maxrgb;
         green3 = green3 * 255.9 / maxrgb;
         blue3 = blue3 * 255.9 / maxrgb;
      }
      
      //  select area edge blending

      if (sa_stat == 3 && dist < sa_blend) {
         dnew = 1.0 * dist / sa_blend;
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  Return relative RGB illumination values for a light source
//     having a given input temperature of 1000-10000 deg. K
//  5000 K is neutral: all returned factors = 1.0

void blackbodyRGB(int K, float &R, float &G, float &B)
{
   float    kk[19] = { 1.0, 1.5, 2.0, 2.5, 3.0, 3.5, 4.0, 4.5, 5.0, 5.5, 6.0, 6.5, 7.0, 7.5, 8.0, 8.5, 9.0, 9.5, 10.0 };
   float    r1[19] = { 255, 255, 255, 255, 255, 255, 255, 255, 254, 250, 242, 231, 220, 211, 204, 197, 192, 188, 184 };
   float    g1[19] = { 060, 148, 193, 216, 232, 242, 249, 252, 254, 254, 251, 245, 239, 233, 228, 224, 220, 217, 215 };
   float    b1[19] = { 000, 010, 056, 112, 158, 192, 219, 241, 253, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255 };

   static int     ftf = 1;
   static float   r2[10000], g2[10000], b2[10000];

   if (ftf) {                                                              //  initialize
      spline1(19,kk,r1);
      for (int T = 1000; T < 10000; T++)
         r2[T] = spline2(0.001 * T);
      
      spline1(19,kk,g1);
      for (int T = 1000; T < 10000; T++)
         g2[T] = spline2(0.001 * T);
      
      spline1(19,kk,b1);
      for (int T = 1000; T < 10000; T++)
         b2[T] = spline2(0.001 * T);
      
      ftf = 0;
   }
   
   if (K < 1000 || K > 9999) zappcrash("blackbody bad K: %dK",K);
   
   R = r2[K];
   G = g2[K];
   B = b2[K];

   return;
}


/**************************************************************************

   Image Tone Mapping function
   enhance local contrast as opposed to overall contrast

   methodology:
   get brightness gradients for each pixel in 4 directions: SE SW NE NW
   amplify gradients using the edit curve (x-axis range 0-max. gradient)
   integrate 4 new brightness surfaces from the amplified gradients:
     - pixel brightness = prior pixel brightness + amplified gradient
     - the Amplify control varies amplification from zero to overamplified
   new pixel brightness = average from 4 calculated brightness surfaces

***************************************************************************/

float    *Tmap_brmap1;
float    *Tmap_brmap3[4];
int      Tmap_contrast99;
float    Tmap_amplify;

editfunc    EFtonemap;

void     Tmap_initz(zdialog *zd);
int      Tmap_dialog_event(zdialog *zd, cchar *event);
void     Tmap_curvedit(int);
void *   Tmap_thread(void *);


void m_tonemap(GtkWidget *, cchar *)
{
   F1_help_topic = "tone_mapping";

   cchar    *title = ZTX("Tone Mapping");

   EFtonemap.menufunc = m_tonemap;
   EFtonemap.funcname = "tonemap";
   EFtonemap.Farea = 2;                                                    //  select area usable
   EFtonemap.Frestart = 1;                                                 //  restart allowed                 v.13.04
   EFtonemap.threadfunc = Tmap_thread;
   if (! edit_setup(EFtonemap)) return;                                    //  setup: no preview, select area OK

/***
            _____________________________
           |                             |
           |                             |
           |    curve drawing area       |
           |                             |
           |_____________________________| 
            low       contrast       high

            Amplify ========[]=========

            Curve File: [ Open ] [ Save ]
***/

   zdialog *zd = zdialog_new(title,Mwin,Bdone,Bcancel,null);
   EFtonemap.zd = zd;

   zdialog_add_widget(zd,"frame","frame","dialog",0,"expand");
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labcL","hb1",ZTX("low"),"space=4");
   zdialog_add_widget(zd,"label","labcM","hb1",Bcontrast,"expand");
   zdialog_add_widget(zd,"label","labcH","hb1",ZTX("high"),"space=5");

   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"label","labcon","hb2",ZTX("Amplify"),"space=5");
   zdialog_add_widget(zd,"hscale","amplify","hb2","0|1|0.005|0","expand");
   zdialog_add_widget(zd,"label","ampval","hb2",0,"space=5");

   zdialog_add_widget(zd,"hbox","hbcf","dialog",0,"space=8");
   zdialog_add_widget(zd,"label","labcf","hbcf",Bcurvefile,"space=5");
   zdialog_add_widget(zd,"button","loadcurve","hbcf",Bopen,"space=5");
   zdialog_add_widget(zd,"button","savecurve","hbcf",Bsave,"space=5");

   GtkWidget *frame = zdialog_widget(zd,"frame");                          //  set up curve edit
   spldat *sd = splcurve_init(frame,Tmap_curvedit);
   EFtonemap.curves = sd;

   sd->Nspc = 1;
   sd->vert[0] = 0;
   sd->nap[0] = 3;                                                         //  initial curve anchor points
   sd->apx[0][0] = 0.01;
   sd->apy[0][0] = 0.4;
   sd->apx[0][1] = 0.50;
   sd->apy[0][1] = 0.4;
   sd->apx[0][2] = 0.99;
   sd->apy[0][2] = 0.2;

   splcurve_generate(sd,0);                                                //  generate curve data

   Tmap_initz(zd);                                                         //  initialize                   v.13.04
   return;
}


//  initialization for new image file                                      //  v.13.04

void Tmap_initz(zdialog *zd)
{
   int         ii, cc, px, py;
   float       *pix1;
   int         jj, sum, limit, condist[100];
   
   if (Tmap_brmap1) {
      zfree(Tmap_brmap1);                                                   //  free memory
      for (int ii = 0; ii < 4; ii++)
      zfree(Tmap_brmap3[ii]);
   }

   cc = E1pxm->ww * E1pxm->hh * sizeof(float);                             //  allocate brightness map memory
   Tmap_brmap1 = (float *) zmalloc(cc);
   for (ii = 0; ii < 4; ii++)
      Tmap_brmap3[ii] = (float *) zmalloc(cc);

   for (py = 0; py < E1pxm->hh; py++)                                      //  map initial image brightness
   for (px = 0; px < E1pxm->ww; px++)
   {
      ii = py * E1pxm->ww + px;
      pix1 = PXMpix(E1pxm,px,py);
      Tmap_brmap1[ii] = pixbright(pix1);                                   //  pixel brightness 0-255.9
   }

   for (ii = 0; ii < 100; ii++) 
      condist[ii] = 0;

   for (py = 1; py < E1pxm->hh; py++)                                      //  map contrast distribution
   for (px = 1; px < E1pxm->ww; px++)
   {
      ii = py * E1pxm->ww + px;
      jj = 0.388 * fabsf(Tmap_brmap1[ii] - Tmap_brmap1[ii-1]);             //  horiz. contrast, ranged 0 - 99
      condist[jj]++;
      jj = 0.388 * fabsf(Tmap_brmap1[ii] - Tmap_brmap1[ii-E1pxm->ww]);     //  vertical
      condist[jj]++;
   }
   
   sum = 0;
   limit = 0.99 * 2 * (E1pxm->ww-1) * (E1pxm->hh-1);                       //  find 99th percentile contrast

   for (ii = 0; ii < 100; ii++) {
      sum += condist[ii];
      if (sum > limit) break;
   }
   
   Tmap_contrast99 = 255.0 * ii / 100.0;                                   //  0 to 255
   if (Tmap_contrast99 < 4) Tmap_contrast99 = 4;                           //  rescale low-contrast image

   zdialog_resize(zd,300,300);
   zdialog_run(zd,Tmap_dialog_event,"save");                               //  run dialog - parallel

   Tmap_amplify = 0;
   return;
}


//  dialog event and completion callback function

int Tmap_dialog_event(zdialog *zd, cchar *event)
{
   spldat   *sd = EFtonemap.curves;
   char     text[8];
   
   if (strEqu(event,"done")) zd->zstat = 1;                                //  apply and quit               v.13.12
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  v.14.03

   if (zd->zstat)                                                          //  dialog complete
   {
      if (zd->zstat == 1) edit_done(0);
      else edit_cancel(0);
      zfree(Tmap_brmap1);                                                  //  free memory
      for (int ii = 0; ii < 4; ii++)
      zfree(Tmap_brmap3[ii]);
      Tmap_brmap1 = 0;                                                     //  v.13.04
      return 1;
   }

   if (strEqu(event,"blendwidth")) signal_thread();

   if (strEqu(event,"amplify")) {                                          //  slider value
      zdialog_fetch(zd,"amplify",Tmap_amplify);
      sprintf(text,"%.2f",Tmap_amplify);                                   //  numeric feedback
      zdialog_stuff(zd,"ampval",text);
      signal_thread();                                                     //  trigger update thread
   }

   if (strEqu(event,"loadcurve")) {                                        //  load saved curve 
      splcurve_load(sd);
      signal_thread();
      return 0;
   }

   if (strEqu(event,"savecurve")) {                                        //  save curve to file 
      splcurve_save(sd);
      return 0;
   }
   
   return 0;
}


//  this function is called when the curve is edited

void Tmap_curvedit(int)
{
   signal_thread();
   return;
}


//  thread function

void * Tmap_thread(void *)
{
   void * Tmap_wthread1(void *arg);
   void * Tmap_wthread2(void *arg);

   int      ii;
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      for (ii = 0; ii < 4; ii++)                                           //  start working threads1 (must be 4)
         start_wthread(Tmap_wthread1,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion
      
      for (ii = 0; ii < Nwt; ii++)                                         //  start working threads2
         start_wthread(Tmap_wthread2,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion
      
      CEF->Fmods++; 
      CEF->Fsaved = 0;

      if (Tmap_amplify == 0)                                               //  v.13.04
         CEF->Fmods = 0;

      Fpaint2(); 
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  working threads

void * Tmap_wthread1(void *arg)
{
   int         ii, kk, bii, pii, dist = 0;
   int         px, px1, px2, pxinc;
   int         py, py1, py2, pyinc;
   float       b1, b3, xval, yval, grad;
   float       amplify, contrast99;
   spldat      *sd = EFtonemap.curves;

   contrast99 = Tmap_contrast99;                                           //  99th percentile contrast
   contrast99 = 1.0 / contrast99;                                          //  inverted

   amplify = pow(Tmap_amplify,0.5);                                        //  get amplification, 0 to 1 

   bii = *((int *) arg);

   if (bii == 0) {                                                         //  direction SE
      px1 = 1; px2 = E3pxm->ww; pxinc = 1;
      py1 = 1; py2 = E3pxm->hh; pyinc = 1;
      pii = - 1 - E3pxm->ww;
   }
   
   else if (bii == 1) {                                                    //  direction SW
      px1 = E3pxm->ww-2; px2 = 0; pxinc = -1;
      py1 = 1; py2 = E3pxm->hh; pyinc = 1;
      pii = + 1 - E3pxm->ww;
   }
   
   else if (bii == 2) {                                                    //  direction NE
      px1 = 1; px2 = E3pxm->ww; pxinc = 1;
      py1 = E3pxm->hh-2; py2 = 0; pyinc = -1;
      pii = - 1 + E3pxm->ww;
   }
   
   else {   /* bii == 3 */                                                 //  direction NW
      px1 = E3pxm->ww-2; px2 = 0; pxinc = -1;
      py1 = E3pxm->hh-2; py2 = 0; pyinc = -1;
      pii = + 1 + E3pxm->ww;
   }

   for (ii = 0; ii < E3pxm->ww * E3pxm->hh; ii++)                          //  initial brightness map
      Tmap_brmap3[bii][ii] = Tmap_brmap1[ii];

   for (py = py1; py != py2; py += pyinc)                                  //  loop all image pixels
   for (px = px1; px != px2; px += pxinc)
   {
      ii = py * E3pxm->ww + px;

      if (sa_stat == 3) {                                                  //  select area active
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }

      b1 = Tmap_brmap1[ii];                                                //  this pixel brightness
      grad = b1 - Tmap_brmap1[ii+pii];                                     //   - prior pixel --> gradient

      xval = fabsf(grad) * contrast99;                                     //  gradient scaled 0 to 1+
      kk = 1000.0 * xval;
      if (kk > 999) kk = 999;
      yval = 1.0 + 5.0 * sd->yval[0][kk];

      grad = grad * yval;                                                  //  magnified gradient

      b3 = Tmap_brmap3[bii][ii+pii] + grad;                                //  pixel brightness = prior + gradient
      b3 = (1.0 - amplify) * b1 + amplify * b3;                            //  constrain: push b3 toward b1

      if (b3 > 255) b3 = 255;                                              //  constrain
      if (b3 < 1) b3 = 1;

      Tmap_brmap3[bii][ii] = b3;                                           //  new pixel brightness
   }
   
   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


void * Tmap_wthread2(void *arg)
{
   float       *pix1, *pix3;
   int         index, ii, px, py, dist = 0;
   float       b1, b3, bf, f1, f2;
   float       red1, green1, blue1, red3, green3, blue3;

   index = *((int *) arg);

   for (py = index; py < E3pxm->hh; py += Nwt)                             //  loop all image pixels
   for (px = 0; px < E3pxm->ww; px++)
   {
      ii = py * E3pxm->ww + px;

      if (sa_stat == 3) {                                                  //  select area active 
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  pixel is outside area
      }

      pix1 = PXMpix(E1pxm,px,py);                                          //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                          //  output pixel

      b1 = Tmap_brmap1[ii];                                                //  initial pixel brightness
      b3 = Tmap_brmap3[0][ii] + Tmap_brmap3[1][ii]                         //  new brightness = average of four
         + Tmap_brmap3[2][ii] + Tmap_brmap3[3][ii];                        //    calculated brightness surfaces

      bf = 0.25 * b3 / (b1 + 1);                                           //  brightness ratio
      
      red1 = pix1[0];                                                      //  input RGB
      green1 = pix1[1];
      blue1 = pix1[2];
      
      red3 = bf * red1;                                                    //  output RGB
      if (red3 > 255.9) red3 = 255.9;
      green3 = bf * green1;
      if (green3 > 255.9) green3 = 255.9;
      blue3 = bf * blue1;
      if (blue3 > 255.9) blue3 = 255.9;

      if (sa_stat == 3 && dist < sa_blend) {                               //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         red3 = f1 * red3 + f2 * red1;
         green3 = f1 * green3 + f2 * green1;
         blue3 = f1 * blue3 + f2 * blue1;
      }
      
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                               //  not executed, stop g++ warning
}


/**************************************************************************/

//  adjust brightness distribution by flattening and/or expanding range

namespace adjust_brdist_names
{
   float    flatten, deband;                                               //  flatten/deband values 0 - 100%
   float    darken, brighten;                                              //  darken/brighten values, 0 - 100%
   float    flatten_dist[1000];                                            //  calculated flattened distribution

   int    dialog_event(zdialog* zd, cchar *event);
   void * thread(void *);
   void * wthread(void *);

   editfunc    EFbrightdist;
}


//  menu function

void m_adjust_brdist(GtkWidget *, cchar *)                                 //  combine flatten and expand      v.13.10
{
   using namespace adjust_brdist_names;

   cchar  *title = ZTX("Adjust Brightness Distribution");
   F1_help_topic = "adjust_brdist";

   EFbrightdist.menufunc = m_adjust_brdist;
   EFbrightdist.funcname = "brightdist";
   EFbrightdist.FprevReq = 1;                                              //  preview
   EFbrightdist.Farea = 2;                                                 //  select area usable
   EFbrightdist.Frestart = 1;                                              //  restart allowed
   EFbrightdist.threadfunc = thread;
   if (! edit_setup(EFbrightdist)) return;                                 //  setup edit

/***
          ______________________________________
         |    Adjust Brightness Distribution    |
         |                                      |
         | Flatten  =========[]============ NN  |
         |  Deband  ============[]========= NN  |
         |                                      |
         |  Darken  ===[]================== NN  |
         | Brighten ==[]=================== NN  |
         |                                      |
         |                     [Done] [Cancel]  |
         |______________________________________|

***/

   zdialog *zd = zdialog_new(title,Mwin,Bdone,Bcancel,null);
   EFbrightdist.zd = zd;
   
   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5");
   zdialog_add_widget(zd,"hbox","hb2","dialog",0,"space=5");
   zdialog_add_widget(zd,"vbox","vb11","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb12","hb1",0,"homog|expand");
   zdialog_add_widget(zd,"vbox","vb13","hb1",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb21","hb2",0,"homog|space=3");
   zdialog_add_widget(zd,"vbox","vb22","hb2",0,"homog|expand");
   zdialog_add_widget(zd,"vbox","vb23","hb2",0,"homog|space=3");

   zdialog_add_widget(zd,"label","labflatten","vb11",ZTX("Flatten"));
   zdialog_add_widget(zd,"label","labdeband","vb11",ZTX("Deband"));
   zdialog_add_widget(zd,"label","labdarken","vb21",ZTX("Darken"));
   zdialog_add_widget(zd,"label","labbrighten","vb21",ZTX("Brighten"));

   zdialog_add_widget(zd,"hscale","flatten","vb12","0|100|1|0","expand");
   zdialog_add_widget(zd,"hscale","deband","vb12","0|100|1|0","expand");
   zdialog_add_widget(zd,"hscale","darken","vb22","0|100|0.2|0","expand");
   zdialog_add_widget(zd,"hscale","brighten","vb22","0|100|0.2|0","expand");

   zdialog_add_widget(zd,"label","flattenval","vb13","0");
   zdialog_add_widget(zd,"label","debandval","vb13","0");
   zdialog_add_widget(zd,"label","darkenval","vb23","0");
   zdialog_add_widget(zd,"label","brightenval","vb23","0");

   zdialog_resize(zd,300,0);
   zdialog_run(zd,dialog_event,"save");                                    //  run dialog - parallel

   flatten = deband = 0;                                                   //  initz. parameters
   darken = brighten = 0;

   m_show_brdist(0,0);                                                     //  popup brightness histogram

   return;
}


//  dialog event and completion function

int adjust_brdist_names::dialog_event(zdialog *zd, cchar *event)
{
   using namespace adjust_brdist_names;

   char     text[8];

   if (strEqu(event,"done")) zd->zstat = 1;                                //  apply and quit               v.13.12
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  v.14.03

   if (zd->zstat)
   {
      if (zd->zstat == 1) edit_done(0);                                    //  done
      else edit_cancel(0);                                                 //  cancel or destroy
      m_show_brdist(0,"kill");
      return 1;
   }

   if (strEqu(event,"blendwidth"))                                         //  select area blendwidth change
      signal_thread();

   if (strEqu(event,"flatten")) {
      zdialog_fetch(zd,"flatten",flatten);                                 //  get slider values
      sprintf(text,"%.1f",flatten);                                        //    and update numeric feedback 
      zdialog_stuff(zd,"flattenval",text);
      signal_thread();
   }
   
   if (strEqu(event,"deband")) {
      zdialog_fetch(zd,"deband",deband);
      sprintf(text,"%.1f",deband);
      zdialog_stuff(zd,"debandval",text);
      signal_thread();
   }

   if (strEqu(event,"darken")) {
      zdialog_fetch(zd,"darken",darken);
      sprintf(text,"%.1f",darken);
      zdialog_stuff(zd,"darkenval",text);
      signal_thread();
   }

   if (strEqu(event,"brighten")) {
      zdialog_fetch(zd,"brighten",brighten);
      sprintf(text,"%.1f",brighten);
      zdialog_stuff(zd,"brightenval",text);
      signal_thread();
   }

   return 1;
}


//  adjust brightness distribution thread function

void * adjust_brdist_names::thread(void *)
{
   using namespace adjust_brdist_names;

   int         px, py, ii, npix, ww, hh;
   float       bright1;
   float       *pix1;

   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request
      
      ww = E1pxm->ww;                                                      //  bugfix  v.13.11.1
      hh = E1pxm->hh;
      
      for (ii = 0; ii < 1000; ii++)                                        //  clear brightness distribution data
         flatten_dist[ii] = 0;

      if (sa_stat == 3)                                                    //  process selected area
      {
         for (ii = npix = 0; ii < ww * hh; ii++)
         {
            if (! sa_pixmap[ii]) continue;
            py = ii / ww;
            px = ii - py * ww;
            pix1 = PXMpix(E1pxm,px,py);
            bright1 = 1000.0 / 256.0 * pixbright(pix1);
            flatten_dist[int(bright1)]++;
            npix++;
         }
         
         for (ii = 1; ii < 1000; ii++)                                     //  cumulative brightness distribution
            flatten_dist[ii] += flatten_dist[ii-1];                        //   0 ... npix

         for (ii = 0; ii < 1000; ii++)
            flatten_dist[ii] = flatten_dist[ii]                            //  multiplier per brightness level
                             / npix * 1000.0 / (ii + 1);
      }

      else                                                                 //  process whole image
      {
         for (py = 0; py < hh; py++)                                       //  compute brightness distribution
         for (px = 0; px < ww; px++)
         {
            pix1 = PXMpix(E1pxm,px,py);
            bright1 = 1000.0 / 256.0 * pixbright(pix1);
            flatten_dist[int(bright1)]++;
         }
         
         for (ii = 1; ii < 1000; ii++)                                     //  cumulative brightness distribution
            flatten_dist[ii] += flatten_dist[ii-1];                        //   0 ... (ww * hh)

         for (ii = 0; ii < 1000; ii++)
            flatten_dist[ii] = flatten_dist[ii]                            //  multiplier per brightness level 
                             / (ww * hh) * 1000.0 / (ii + 1);
      }
      
      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker thread per processor core
         start_wthread(wthread,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion

      if (! flatten && ! darken && ! brighten)                             //  no modifications
         CEF->Fmods = 0;
      else {
         CEF->Fmods++;                                                     //  image modified, not saved
         CEF->Fsaved = 0;
      }
      
      Fpaint2();                                                           //  v.14.04
   }

   return 0;                                                               //  not executed, stop g++ warning
}


//  worker thread for each CPU processor core

void * adjust_brdist_names::wthread(void *arg)
{
   using namespace adjust_brdist_names;

   int         index = *((int *) (arg));
   int         px, py, ii, dist = 0;
   float       *pix1, *pix3;
   float       fold, fnew, dold, dnew, cmax;
   float       red1, green1, blue1, red3, green3, blue3;
   float       bright1, bright2, deband2;
   float       dark, bright, b1, b3, bf, f1, f2;
   
   //  flatten brightness distribution

   for (py = index; py < E1pxm->hh; py += Nwt)                             //  flatten brightness distribution
   for (px = 0; px < E1pxm->ww; px++)
   {
      if (sa_stat == 3) {                                                  //  select area active
         ii = py * E1pxm->ww + px;   
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  outside pixel
      }

      pix1 = PXMpix(E1pxm,px,py);                                          //  input pixel
      pix3 = PXMpix(E3pxm,px,py);                                          //  output pixel
         
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];

      bright1 = 0.25 * red1 + 0.65 * green1 + 0.10 * blue1;                //  input pixel brightness 0-255.9
      bright1 *= 1000.0 / 256.0;                                           //  0-999
      bright2 = flatten_dist[int(bright1)];                                //  output adjustment factor

      red3 = bright2 * red1;                                               //  fully flattened brightness
      green3 = bright2 * green1;
      blue3 = bright2 * blue1;

      deband2 = (1000.0 - bright1) / 1000.0;                               //  bright1 = 0 to 1000  >>  deband2 = 1 to 0
      deband2 += (1.0 - deband2) * (1.0 - 0.01 * deband);                  //  deband2 = 1 to 1  >>  1 to 0

      fnew = 0.01 * flatten;                                               //  how much to flatten, 0 to 1
      fnew = deband2 * fnew;                                               //  attenuate flatten of brighter pixels
      fold = 1.0 - fnew;                                                   //  how much to retain, 1 to 0

      red3 = fnew * red3 + fold * red1;                                    //  blend new and old brightness
      green3 = fnew * green3 + fold * green1;
      blue3 = fnew * blue3 + fold * blue1;

      if (sa_stat == 3 && dist < sa_blend) {                               //  select area is active,
         dnew = 1.0 * dist / sa_blend;                                     //    blend changes over sa_blend
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      cmax = red3;                                                         //  stop overflow, keep color balance
      if (green3 > cmax) cmax = green3;
      if (blue3 > cmax) cmax = blue3;
      if (cmax > 255.9) {
         cmax = 255.9 / cmax;
         red3 = red3 * cmax;
         green3 = green3 * cmax;
         blue3 = blue3 * cmax;
      }

      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }
   
   //  expand brightness distribution

   dark = 0.01 * darken * 255.9;                                           //  clipping brightness levels
   bright = (1.0 - 0.01 * brighten) * 255.9;                               //  dark = 0-255.9  bright = 255.9-0
   
   if (dark >= bright) {                                                   //  inputs mean 100% clipping
      dark = bright - 0.01;                                                //    retain 1-2% of range          v.13.10
      bright += 0.01;
      if (dark < 0) dark = 0;
      if (bright > 255.9) bright = 255.9;
   }

   for (py = index; py < E1pxm->hh; py += Nwt)                             //  expand brightness distribution
   for (px = 0; px < E1pxm->ww; px++)
   {
      ii = py * E3pxm->ww + px;

      if (sa_stat == 3) {                                                  //  select area active
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  pixel is outside area
      }

      pix1 = pix3 = PXMpix(E3pxm,px,py);                                   //  input and output pixel          v.13.10

      b1 = pixbright(pix1);                                                //  0-255.9

      if (b1 < dark)                                                       //  clip dark pixels
         b3 = 0;
      else if (b1 > bright)                                                //  clip bright pixels
         b3 = bright;
      else b3 = b1;
      
      if (b3 > dark) b3 = b3 - dark;                                       //  expand the rest
      b3 = b3 * (255.9 / (bright - dark));

      bf = b3 / (b1 + 0.1);                                                //  brightness ratio
      
      red1 = pix1[0];                                                      //  input RGB 
      green1 = pix1[1];
      blue1 = pix1[2];
      
      red3 = bf * red1;                                                    //  output RGB
      green3 = bf * green1;
      blue3 = bf * blue1;
      
      if (red3 > 255.9) red3 = 255.9;
      if (green3 > 255.9) green3 = 255.9;
      if (blue3 > 255.9) blue3 = 255.9;
      
      if (sa_stat == 3 && dist < sa_blend) {                               //  select area is active,
         f1 = 1.0 * dist / sa_blend;                                       //    blend changes over sa_blend
         f2 = 1.0 - f1;
         red3 = f1 * red3 + f2 * red1;
         green3 = f1 * green3 + f2 * green1;
         blue3 = f1 * blue3 + f2 * blue1;
      }
      
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  ramp brightness across image, vertical and horizontal gradients

editfunc    EFbrramp;
int         brramp_spc;                                                    //  current spline curves (2 at a time)

void m_brightramp(GtkWidget *, cchar *)                                    //  upgrade for 4 colors
{
   int    brramp_dialog_event(zdialog* zd, cchar *event);
   void   brramp_curvedit(int spc);
   void * brramp_thread(void *);
   void   brramp_init();

   F1_help_topic = "brightness_ramp";

   EFbrramp.menufunc = m_brightramp;
   EFbrramp.funcname = "bright-ramp";
   EFbrramp.FprevReq = 1;                                                  //  use preview
   EFbrramp.Farea = 2;                                                     //  select area usable
   EFbrramp.threadfunc = brramp_thread;
   if (! edit_setup(EFbrramp)) return;                                     //  setup edit
   
/***
          _________________________________________
         |                                         |
         |                                         |
         |           curve edit area               |
         |                                         |
         |_________________________________________|

            (o) all  (o) red  (o) green  (o) blue                          //  colors: all, red, green, blue
            Curve File: [Open]  [Save]                                     //  each has vert. and horiz. curve
                                   [Done]  [Cancel]    

***/

   zdialog *zd = zdialog_new(ZTX("Ramp brightness across image"),Mwin,Bdone,Bcancel,null);
   EFbrramp.zd = zd;

   zdialog_add_widget(zd,"hbox","hb1","dialog",0,"space=5|expand");
   zdialog_add_widget(zd,"vbox","vb1","hb1");
   zdialog_add_widget(zd,"vbox","vb2","hb1",0,"expand");
   zdialog_add_widget(zd,"label","lmax","vb1","+","space=3");
   zdialog_add_widget(zd,"label","lspace","vb1",0,"expand");
   zdialog_add_widget(zd,"label","lmin","vb1","‒","space=3");
   zdialog_add_widget(zd,"label","lspace","vb1");
   zdialog_add_widget(zd,"frame","frame","vb2",0,"expand");

   zdialog_add_widget(zd,"hbox","hb2","vb2");
   zdialog_add_widget(zd,"label","lmin","hb2","‒","space=3");
   zdialog_add_widget(zd,"label","lspace","hb2",0,"expand");
   zdialog_add_widget(zd,"label","lmax","hb2","+","space=3");
   zdialog_add_widget(zd,"hbox","hbrgb","dialog",0,"space=3");
   zdialog_add_widget(zd,"radio","all","hbrgb",Ball,"space=5");
   zdialog_add_widget(zd,"radio","red","hbrgb",Bred,"space=5");
   zdialog_add_widget(zd,"radio","green","hbrgb",Bgreen,"space=5");
   zdialog_add_widget(zd,"radio","blue","hbrgb",Bblue,"space=5");
   zdialog_add_widget(zd,"hbox","hbcf","dialog",0,"space=3");
   zdialog_add_widget(zd,"label","labcf","hbcf",Bcurvefile,"space=8");
   zdialog_add_widget(zd,"button","loadcurves","hbcf",Bopen,"space=5");
   zdialog_add_widget(zd,"button","savecurves","hbcf",Bsave,"space=5");

   GtkWidget *frame = zdialog_widget(zd,"frame");
   spldat *sd = splcurve_init(frame,brramp_curvedit);
   EFbrramp.curves = sd;
   
   brramp_init();                                                          //  set up initial flat curves

   sd->Nspc = 2;                                                           //  only curves 0-1 are shown
   brramp_spc = 2;                                                         //  current curves: all (2-3)
   
   zdialog_stuff(zd,"all",1);                                              //  stuff default selection, all

   zdialog_resize(zd,260,350);
   zdialog_run(zd,brramp_dialog_event,"save");                             //  run dialog - parallel

   return;
}


//  initialize all curves to flat

void brramp_init()
{
   spldat *sd = EFbrramp.curves;

   for (int ii = 0; ii < 10; ii++)                                         //  loop curves: current (0-1), all (2-3),
   {                                                                       //    red (4-5), green (6-7), blue (8-9)
      sd->nap[ii] = 2;                                                     //  horizontal curve ii
      sd->vert[ii] = 0;
      sd->apx[ii][0] = 0.01;
      sd->apx[ii][1] = 0.99;
      sd->apy[ii][0] = sd->apy[ii][1] = 0.5;
      splcurve_generate(sd,ii);

      ii++;
      sd->nap[ii] = 2;                                                     //  vertical curve ii+1
      sd->vert[ii] = 1;
      sd->apx[ii][0] = 0.01;
      sd->apx[ii][1] = 0.99;
      sd->apy[ii][0] = sd->apy[ii][1] = 0.5;
      splcurve_generate(sd,ii);
   }
   
   return;
}


//  dialog event and completion callback function

int brramp_dialog_event(zdialog *zd, cchar *event)
{
   spldat      *sd = EFbrramp.curves;
   int         ii, jj, nn, Fupdate = 0;
   spldat      sdtemp;

   if (strEqu(event,"done")) zd->zstat = 1;                                //  apply and quit               v.13.12
   if (strEqu(event,"enter")) zd->zstat = 1;                               //  v.14.03

   if (zd->zstat) {                                                        //  dialog complete
      if (zd->zstat == 1) edit_done(0);
      else edit_cancel(0);
      return 1;
   }

   if (strEqu(event,"blendwidth")) signal_thread();

   if (strstr("all red green blue",event)) {                               //  new choice of curve
      zdialog_fetch(zd,event,ii);
      if (! ii) return 0;                                                  //  ignore button OFF events

      ii = strcmpv(event,"all","red","green","blue",null);
      brramp_spc = ii = 2 * ii;                                            //  2, 4, 6, 8

      sd->nap[0] = sd->nap[ii];                                            //  copy active curves ii/ii+1 to curves 0/1
      sd->nap[1] = sd->nap[ii+1];
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[0][jj] = sd->apx[ii][jj];
         sd->apy[0][jj] = sd->apy[ii][jj];
      }
      for (jj = 0; jj < sd->nap[1]; jj++) {
         sd->apx[1][jj] = sd->apx[ii+1][jj];
         sd->apy[1][jj] = sd->apy[ii+1][jj];
      }

      Fupdate++;
   }

   if (strEqu(event,"loadcurves"))                                         //  load 8 saved curves from file
   {
      sdtemp.Nspc = 8;
      sdtemp.drawarea = 0;
      nn = splcurve_load(&sdtemp);
      if (nn != 1) return 0;

      for (ii = 0; ii < 8; ii++) {                                         //  load curves 0-7 to curves 2-9
         sd->vert[ii+2] = sdtemp.vert[ii];
         sd->nap[ii+2] = sdtemp.nap[ii];
         for (jj = 0; jj < sdtemp.nap[ii]; jj++) {
            sd->apx[ii+2][jj] = sdtemp.apx[ii][jj];
            sd->apy[ii+2][jj] = sdtemp.apy[ii][jj];
         }
         splcurve_generate(sd,ii+2);
      }

      ii = brramp_spc;                                                     //  current active curves: 2, 4, 6, 8

      sd->nap[0] = sd->nap[ii];                                            //  copy 2 active curves to curves 0-1
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[0][jj] = sd->apx[ii][jj];
         sd->apy[0][jj] = sd->apy[ii][jj];
      }

      sd->nap[1] = sd->nap[ii+1];
      for (jj = 0; jj < sd->nap[1]; jj++) {
         sd->apx[1][jj] = sd->apx[ii+1][jj];
         sd->apy[1][jj] = sd->apy[ii+1][jj];
      }

      Fupdate++;
   }

   if (strEqu(event,"savecurves"))                                         //  save 8 curves to file
   {
      sdtemp.Nspc = 8;
      for (ii = 0; ii < 8; ii++) {
         sdtemp.vert[ii] = sd->vert[ii+2];
         sdtemp.nap[ii] = sd->nap[ii+2];
         for (jj = 0; jj < sd->nap[ii+2]; jj++) {
            sdtemp.apx[ii][jj] = sd->apx[ii+2][jj];
            sdtemp.apy[ii][jj] = sd->apy[ii+2][jj];
         }
      }

      splcurve_save(&sdtemp);
   }

   if (Fupdate)                                                            //  curve has changed
   {
      splcurve_generate(sd,0);                                             //  regenerate curves 0-1
      splcurve_generate(sd,1);
      
      ii = brramp_spc;                                                     //  active curves

      sd->nap[ii] = sd->nap[0];                                            //  copy curves 0-1 to active curves
      for (jj = 0; jj < sd->nap[0]; jj++) {
         sd->apx[ii][jj] = sd->apx[0][jj];
         sd->apy[ii][jj] = sd->apy[0][jj];
      }

      sd->nap[ii+1] = sd->nap[1];
      for (jj = 0; jj < sd->nap[1]; jj++) {
         sd->apx[ii+1][jj] = sd->apx[1][jj];
         sd->apy[ii+1][jj] = sd->apy[1][jj];
      }

      for (jj = 0; jj < 1000; jj++) 
      {
         sd->yval[ii][jj] = sd->yval[0][jj];
         sd->yval[ii+1][jj] = sd->yval[1][jj];
      }

      gtk_widget_queue_draw(sd->drawarea);                                 //  draw curve

      signal_thread();                                                     //  trigger image update
   }

   return 1;
}


//  this function is called when a curve is edited

void brramp_curvedit(int spc)
{
   int      ii = brramp_spc, jj;
   spldat   *sd = EFbrramp.curves;                                         //  active curves: 2, 4, 6, 8 (and +1)

   sd->nap[ii] = sd->nap[0];                                               //  copy curves 0-1 to active curves
   for (jj = 0; jj < sd->nap[0]; jj++) {
      sd->apx[ii][jj] = sd->apx[0][jj];
      sd->apy[ii][jj] = sd->apy[0][jj];
   }

   sd->nap[ii+1] = sd->nap[1];
   for (jj = 0; jj < sd->nap[1]; jj++) {
      sd->apx[ii+1][jj] = sd->apx[1][jj];
      sd->apy[ii+1][jj] = sd->apy[1][jj];
   }

   for (jj = 0; jj < 1000; jj++) {
      sd->yval[ii][jj] = sd->yval[0][jj];
      sd->yval[ii+1][jj] = sd->yval[1][jj];
   }

   if (brramp_spc == 2) {                                                  //  "all" curve was edited
      for (ii = 4; ii < 10; ii += 2) {                                     //  fix R/G/B curves to match
         sd->nap[ii] = sd->nap[2];
         for (jj = 0; jj < sd->nap[2]; jj++) {
            sd->apx[ii][jj] = sd->apx[2][jj];
            sd->apy[ii][jj] = sd->apy[2][jj];
         }
         for (jj = 0; jj < 1000; jj++)
            sd->yval[ii][jj] = sd->yval[2][jj];
      }
      for (ii = 5; ii < 10; ii += 2) {
         sd->nap[ii] = sd->nap[3];
         for (jj = 0; jj < sd->nap[3]; jj++) {
            sd->apx[ii][jj] = sd->apx[3][jj];
            sd->apy[ii][jj] = sd->apy[3][jj];
         }
         for (jj = 0; jj < 1000; jj++)
            sd->yval[ii][jj] = sd->yval[3][jj];
      }
   }

   signal_thread();
   return;
}


//  brramp thread function

void * brramp_thread(void *arg)
{
   void * brramp_wthread(void *);
   
   while (true)
   {
      thread_idle_loop();                                                  //  wait for work or exit request

      for (int ii = 0; ii < Nwt; ii++)                                     //  start worker threads
         start_wthread(brramp_wthread,&Nval[ii]);
      wait_wthreads();                                                     //  wait for completion
      
      Fpaint2();                                                           //  update window
      CEF->Fmods++;                                                        //  image3 modified
      CEF->Fsaved = 0;
   }

   return 0;                                                               //  not executed, stop g++ warning
}


void * brramp_wthread(void *arg)                                           //  worker thread function
{
   int         index = *((int *) arg);
   int         ii, jj, kk, dist = 0, px3, py3;
   float       *pix1, *pix3;
   float       red1, green1, blue1, maxrgb;
   float       red3, green3, blue3;
   float       dispx, dispy;
   float       hramp[3], vramp[3], tramp[3];
   float       spanw, spanh, dold, dnew;
   spldat      *sd = EFbrramp.curves;

   if (sa_stat == 3) {                                                     //  if select area active, ramp
      spanw = sa_maxx - sa_minx;                                           //    brightness over enclosing rectangle
      spanh = sa_maxy - sa_miny;
   }
   else {
      spanw = E3pxm->ww;                                                   //  else over entire image
      spanh = E3pxm->hh;
   }   

   for (py3 = index; py3 < E3pxm->hh; py3 += Nwt)                          //  loop output pixels
   for (px3 = 0; px3 < E3pxm->ww; px3++)
   {
      if (sa_stat == 3) {                                                  //  select area active 
         ii = py3 * E3pxm->ww + px3;
         dist = sa_pixmap[ii];                                             //  distance from edge
         if (! dist) continue;                                             //  pixel outside area
         dispx = (px3 - sa_minx) / spanw;
         dispy = (py3 - sa_miny) / spanh;
      }
      else {
         dispx = px3 / spanw;                                              //  left > right = 0 to 1
         dispy = py3 / spanh;                                              //  top > bottom = 0 to 1
      }
      
      kk = 1000 * dispx;                                                   //  conv. dispx 0-1 to 0-1000
      if (kk > 999) kk = 999;

      for (ii = 0, jj = 4; ii < 3; ii++, jj+=2)                            //  horz. curves for dispx and R/G/B
         hramp[ii] = sd->yval[jj][kk] - 0.5;                               //  scale to -0.5 to +0.5
      
      kk = 1000 * dispy;                                                   //  conv. dispy
      if (kk > 999) kk = 999;

      for (ii = 0, jj = 5; ii < 3; ii++, jj+=2)                            //  vert. curves
         vramp[ii] = sd->yval[jj][kk] - 0.5;

      for (ii = 0; ii < 3; ii++)                                           //  total R/G/B change
         tramp[ii] = 1.0 + hramp[ii] + vramp[ii];                          //  scale to 0.0 to 2.0

      pix1 = PXMpix(E1pxm,px3,py3);                                        //  input pixel
      red1 = pix1[0];
      green1 = pix1[1];
      blue1 = pix1[2];
      
      red3 = red1 * tramp[0];                                              //  change R/G/B
      green3 = green1 * tramp[1];
      blue3 = blue1 * tramp[2];

      maxrgb = red3;
      if (green3 > maxrgb) maxrgb = green3;
      if (blue3 > maxrgb) maxrgb = blue3;
      
      if (maxrgb > 255.9) {                                                //  stop overflow
         red3 = red3 * 255.9 / maxrgb;
         green3 = green3 * 255.9 / maxrgb;
         blue3 = blue3 * 255.9 / maxrgb;
      }

      if (sa_stat == 3 && dist < sa_blend) {                               //  blend changes over blendwidth
         dnew = 1.0 * dist / sa_blend;
         dold = 1.0 - dnew;
         red3 = dnew * red3 + dold * red1;
         green3 = dnew * green3 + dold * green1;
         blue3 = dnew * blue3 + dold * blue1;
      }

      pix3 = PXMpix(E3pxm,px3,py3);                                        //  output pixel
      pix3[0] = red3;
      pix3[1] = green3;
      pix3[2] = blue3;
   }

   exit_wthread();
   return 0;                                                               //  not executed, avoid gcc warning
}


/**************************************************************************/

//  Select area and edit in parallel
//  Current edit function is applied to areas painted with the mouse.
//  Mouse can be weak or strong, and edits are applied incrementally.
//  method: 
//    entire image is a select area with all pixel edge distance = 0 (outside area)
//    blendwidth = 10000 (infinite)
//    pixels painted with mouse have increasing edge distance to amplify edits

int   paint_edits_radius;
int   paint_edits_cpower;
int   paint_edits_epower;

void m_paint_edits(GtkWidget *, cchar *)                                   //  menu function
{
   int   paint_edits_dialog_event(zdialog *, cchar *event);                //  dialog event function
   void  paint_edits_mousefunc();                                          //  mouse function

   cchar    *title = ZTX("Paint Edits");
   cchar    *helptext1 = ZTX("Press F1 for help");
   cchar    *helptext2 = ZTX("Edit function must be active");
   int      yn, cc;

   F1_help_topic = "paint_edits";

   if (! curr_file) return;                                                //  no image
   
   if (sa_stat) {
      yn = zmessageYN(Mwin,ZTX("Select area cannot be kept.\n"             //  v.13.04
                               "Continue?"));
      if (! yn) return;
      sa_unselect();                                                       //  unselect area
      zdialog_free(zdsela);
   }

   if (CEF && CEF->Fpreview) edit_fullsize();                              //  use full-size image
   
/***
    ____________________________________________
   |           Press F1 for help                |
   |       Edit Function must be active         |
   | mouse radius [___|v]                       |
   | power:  center [___|v]  edge [___|v]       |
   | [reset area]                               |
   |                                  [done]    |
   |____________________________________________|
   
***/

   zdsela = zdialog_new(title,Mwin,Bdone,null);
   zdialog_add_widget(zdsela,"label","labhelp1","dialog",helptext1,"space=5");
   zdialog_add_widget(zdsela,"label","labhelp2","dialog",helptext2);
   zdialog_add_widget(zdsela,"label","labspace","dialog");
   zdialog_add_widget(zdsela,"hbox","hbr","dialog",0,"space=3");
   zdialog_add_widget(zdsela,"label","labr","hbr",ZTX("mouse radius"),"space=5");
   zdialog_add_widget(zdsela,"spin","radius","hbr","2|500|1|50");
   zdialog_add_widget(zdsela,"hbox","hbt","dialog",0,"space=3");
   zdialog_add_widget(zdsela,"label","labtc","hbt",ZTX("power:  center"),"space=5");
   zdialog_add_widget(zdsela,"spin","center","hbt","0|100|1|50");
   zdialog_add_widget(zdsela,"label","labte","hbt",Bedge,"space=5");
   zdialog_add_widget(zdsela,"spin","edge","hbt","0|100|1|0");
   zdialog_add_widget(zdsela,"hbox","hbra","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"button","reset","hbra",ZTX("reset area"),"space=5");
   
   paint_edits_radius = 50;
   paint_edits_cpower = 50;
   paint_edits_epower = 0;

   cc = Fpxb->ww * Fpxb->hh * sizeof(uint16);                              //  allocate sa_pixmap[] for new area
   sa_pixmap = (uint16 *) zmalloc(cc);
   memset(sa_pixmap,0,cc);                                                 //  edge distance = 0 for all pixels

   sa_minx = 0;                                                            //  enclosing rectangle
   sa_maxx = Fpxb->ww;
   sa_miny = 0;
   sa_maxy = Fpxb->hh;

   sa_Npixel = Fpxb->ww * Fpxb->hh;
   sa_stat = 3;                                                            //  area status = complete
   sa_mode = mode_image;                                                   //  area mode = whole image
   sa_calced = 1;                                                          //  edge calculation complete
   sa_blend = 10000;                                                       //  "blend width"
   sa_fww = Fpxb->ww;                                                      //  valid image dimensions
   sa_fhh = Fpxb->hh;
   areanumber++;                                                           //  next sequential number

   zdialog_run(zdsela,paint_edits_dialog_event,"save");                    //  run dialog - parallel
   return;
}


//  Adjust whole image area to increase edit power for pixels within the mouse radius
//  sa_pixmap[*]  = 0 = never touched by mouse
//                = 1 = minimum edit power (barely painted)
//                = sa_blend = maximum edit power (edit fully applied)

int paint_edits_dialog_event(zdialog *zd, cchar *event)
{
   void  paint_edits_mousefunc();                                          //  mouse function
   int      cc;
   
   if (zd->zstat)                                                          //  done or cancel
   {
      freeMouse();                                                         //  disconnect mouse function
      zdialog_free(zdsela);                                                //  kill dialog
      sa_unselect();                                                       //  unselect area
      return 0;
   }

   if (sa_stat != 3) return 1;                                             //  area gone
   if (! sa_validate()) return 1;                                          //  area invalid for curr. image file  v.12.12

   if (strEqu(event,"focus")) {                                            //  toggle mouse capture
      if (CEF) takeMouse(paint_edits_mousefunc,0);
      else freeMouse();                                                    //  disconnect mouse
   }

   if (strEqu(event,"radius"))
      zdialog_fetch(zd,"radius",paint_edits_radius);                       //  set mouse radius

   if (strEqu(event,"center"))
      zdialog_fetch(zd,"center",paint_edits_cpower);                       //  set mouse center power

   if (strEqu(event,"edge"))
      zdialog_fetch(zd,"edge",paint_edits_epower);                         //  set mouse edge power
   
   if (strEqu(event,"reset")) {
      sa_unselect();                                                       //  unselect current area if any
      cc = Fpxb->ww * Fpxb->hh * sizeof(uint16);                           //  allocate sa_pixmap[] for new area
      sa_pixmap = (uint16 *) zmalloc(cc);
      memset(sa_pixmap,0,cc);                                              //  edge distance = 0 for all pixels

      sa_minx = 0;                                                         //  enclosing rectangle
      sa_maxx = Fpxb->ww;
      sa_miny = 0;
      sa_maxy = Fpxb->hh;

      sa_Npixel = Fpxb->ww * Fpxb->hh;
      sa_stat = 3;                                                         //  area status = complete
      sa_mode = mode_image;                                                //  area mode = whole image
      sa_calced = 1;                                                       //  edge calculation complete
      sa_blend = 10000;                                                    //  "blend width"
      sa_fww = Fpxb->ww;                                                   //  valid image dimensions
      sa_fhh = Fpxb->hh;
      areanumber++;                                                        //  next sequential number
   }
   
   return 1;
}


//  mouse function - adjust edit strength for areas within mouse radius
//  "edge distance" is increased for more strength, decreased for less

void paint_edits_mousefunc()
{
   int      ii, px, py, rx, ry, mrect;
   int      radius, radius2, cpower, epower;
   float    rad, rad2, power;

   if (! CEF) return;                                                      //  no active edit
   if (sa_stat != 3) return;                                               //  area gone?

   radius = paint_edits_radius;                                            //  pixel selection radius
   radius2 = radius * radius;
   cpower = paint_edits_cpower;
   epower = paint_edits_epower;

   mrect = 2 * radius;
   draw_mousearc(Mxposn,Myposn,mrect,mrect,0);                             //  show mouse selection circle

   if (LMclick || RMclick)                                                 //  mouse click, process normally
      return;

   if (Mbutton != 1 && Mbutton != 3)                                       //  button released
      return;
   
   Mxdrag = Mydrag = 0;                                                    //  neutralize drag                 v.12.11

   for (rx = -radius; rx <= radius; rx++)                                  //  loop every pixel in radius
   for (ry = -radius; ry <= radius; ry++)
   {
      rad2 = rx * rx + ry * ry;
      if (rad2 > radius2) continue;                                        //  outside radius
      px = Mxposn + rx;
      py = Myposn + ry;
      if (px < 0 || px > E1pxm->ww-1) continue;                            //  off the image edge
      if (py < 0 || py > E1pxm->hh-1) continue;

      ii = E1pxm->ww * py + px;
      rad = sqrt(rad2);
      power = cpower + rad / radius * (epower - cpower);                   //  power at pixel radius
      
      if (Mbutton == 1)                                                    //  left mouse button
      {                                                                    //  increase edit power
         sa_pixmap[ii] += 2.0 * power;                                     //  make paint edit 2x faster       v.14.04
         if (sa_pixmap[ii] > sa_blend) sa_pixmap[ii] = sa_blend;
      }

      if (Mbutton == 3)                                                    //  right mouse button
      {                                                                    //  weaken edit power
         if (sa_pixmap[ii] <= power) sa_pixmap[ii] = 0;                    //  erase is half as fast as paint
         else sa_pixmap[ii] -= power;
      }
   }

   sa_minx = Mxposn - radius;                                              //  set temp. smaller area around mouse
   if (sa_minx < 0) sa_minx = 0;
   sa_maxx = Mxposn + radius;
   if (sa_maxx > E1pxm->ww) sa_maxx = E1pxm->ww;
   sa_miny = Myposn - radius;
   if (sa_miny < 0) sa_miny = 0;
   sa_maxy = Myposn + radius;
   if (sa_maxy > E1pxm->hh) sa_maxy = E1pxm->hh;
   sa_Npixel = (sa_maxx - sa_minx) * (sa_maxy - sa_miny);

   zdialog_send_event(CEF->zd,"blendwidth");                               //  notify edit dialog

   draw_mousearc(Mxposn,Myposn,mrect,mrect,0);                             //  show mouse selection circle

   sa_minx = 0;                                                            //  restore whole image area 
   sa_maxx = E1pxm->ww;
   sa_miny = 0;
   sa_maxy = E1pxm->hh;
   sa_Npixel = E1pxm->ww * E1pxm->hh;
   
   return;
}


/**************************************************************************/

//  Use the image brightness or color values to leverage subsequent edits.
//  Method:
//  Select the whole image as an area.
//  Set "edge distance" 1 to 999 from pixel brightness or RGB color.
//  Set "blend width" to 999.
//  Edit function coefficient = edge distance / blend width.

spldat   *leveds_curve;
int      leveds_type, leveds_color;
int      leveds_ptype, leveds_pcolor;
float    *leveds_lever = 0;

void m_lever_edits(GtkWidget *, cchar *)
{
   int    leveds_event(zdialog *, cchar *event);                           //  dialog event and completion func
   void   leveds_curve_update(int spc);                                    //  curve update callback function

   cchar    *title = ZTX("Leverage Edits");
   cchar    *legend = ZTX("Edit Function Amplifier");

   F1_help_topic = "lever_edits";

   if (zdsela) return;                                                     //  dialog already active
   if (! curr_file) return;                                                //  no image
   if (! CEF) {                                                            //  edit func must be active
      zmessageACK(Mwin,0,ZTX("Edit function must be active"));
      return;
   }

   if (CEF->Fpreview) edit_fullsize();                                     //  use full-size image

/***
                  Edit Function Amplifier
             ------------------------------------------
            |                                          |
            |                                          |
            |           curve drawing area             |
            |                                          |
            |                                          |
             ------------------------------------------
             minimum                            maximum

             [+++]  [---]  [+ -]  [- +]  [+-+]  [-+-]
             (o) Brightness  (o) Contrast
             (o) All  (o) Red  (o) Green  (o) Blue
             Curve File: [ Open ] [ Save ]
                                              [ Done ]
***/

   zdsela = zdialog_new(title,Mwin,Bdone,null);

   zdialog_add_widget(zdsela,"label","labt","dialog",legend);
   zdialog_add_widget(zdsela,"frame","fr1","dialog",0,"expand");
   zdialog_add_widget(zdsela,"hbox","hba","dialog");
   zdialog_add_widget(zdsela,"label","labda","hba",ZTX("minimum"),"space=5");
   zdialog_add_widget(zdsela,"label","space","hba",0,"expand");
   zdialog_add_widget(zdsela,"label","labba","hba",ZTX("maximum"),"space=5");
   zdialog_add_widget(zdsela,"hbox","hbb","dialog",0,"space=10");
   zdialog_add_widget(zdsela,"button","b +++","hbb","+++","space=3");
   zdialog_add_widget(zdsela,"button","b ---","hbb","‒ ‒ ‒","space=3");
   zdialog_add_widget(zdsela,"button","b +-", "hbb"," + ‒ ","space=3");
   zdialog_add_widget(zdsela,"button","b -+", "hbb"," ‒ + ","space=3");
   zdialog_add_widget(zdsela,"button","b +-+","hbb","+ ‒ +","space=3");
   zdialog_add_widget(zdsela,"button","b -+-","hbb","‒ + ‒","space=3");

   zdialog_add_widget(zdsela,"hbox","hbbr1","dialog");
   zdialog_add_widget(zdsela,"radio","bright","hbbr1",Bbrightness,"space=5");
   zdialog_add_widget(zdsela,"radio","contrast","hbbr1",Bcontrast,"space=5");
   zdialog_add_widget(zdsela,"hbox","hbbr2","dialog");
   zdialog_add_widget(zdsela,"radio","all","hbbr2",Ball,"space=5");
   zdialog_add_widget(zdsela,"radio","red","hbbr2",Bred,"space=5");
   zdialog_add_widget(zdsela,"radio","green","hbbr2",Bgreen,"space=5");
   zdialog_add_widget(zdsela,"radio","blue","hbbr2",Bblue,"space=5");

   zdialog_add_widget(zdsela,"hbox","hbcf","dialog",0,"space=5");
   zdialog_add_widget(zdsela,"label","labcf","hbcf",Bcurvefile,"space=5");
   zdialog_add_widget(zdsela,"button","loadcurve","hbcf",Bopen,"space=5");
   zdialog_add_widget(zdsela,"button","savecurve","hbcf",Bsave,"space=5");

   GtkWidget *frame = zdialog_widget(zdsela,"fr1");                        //  setup for curve editing
   spldat *sd = splcurve_init(frame,leveds_curve_update);
   leveds_curve = sd;

   sd->Nspc = 1;
   sd->vert[0] = 0;
   sd->nap[0] = 3;                                                         //  initial curve anchor points
   sd->apx[0][0] = 0.01;
   sd->apy[0][0] = 0.5;
   sd->apx[0][1] = 0.50;
   sd->apy[0][1] = 0.5;
   sd->apx[0][2] = 0.99;
   sd->apy[0][2] = 0.5;
   splcurve_generate(sd,0);                                                //  generate curve data

   zdialog_stuff(zdsela,"bright",1);                                       //  type leverage = brightness
   zdialog_stuff(zdsela,"contrast",0);
   zdialog_stuff(zdsela,"all",1);                                          //  color used = all
   zdialog_stuff(zdsela,"red",0);
   zdialog_stuff(zdsela,"green",0);
   zdialog_stuff(zdsela,"blue",0);
   leveds_type = 1;
   leveds_color = 1;
   leveds_ptype = 0;
   leveds_pcolor = 0;

   sa_unselect();                                                          //  unselect current area if any

   int cc = E1pxm->ww * E1pxm->hh * sizeof(uint16);                        //  allocate sa_pixmap[] for area
   sa_pixmap = (uint16 *) zmalloc(cc);

   sa_minx = 0;                                                            //  enclosing rectangle
   sa_maxx = E1pxm->ww;
   sa_miny = 0;
   sa_maxy = E1pxm->hh;

   sa_Npixel = E1pxm->ww * E1pxm->hh;
   sa_stat = 3;                                                            //  area status = complete
   sa_mode = mode_image;                                                   //  area mode = whole image
   sa_calced = 1;                                                          //  edge calculation complete
   sa_blend = 999;                                                         //  "blend width" = 999
   sa_fww = E1pxm->ww;                                                     //  valid image dimensions       v.13.05
   sa_fhh = E1pxm->hh;
   areanumber++;                                                           //  next sequential number

   if (leveds_lever) zfree(leveds_lever);
   cc = E1pxm->ww * E1pxm->hh * sizeof(float);                             //  allocate memory for lever    v.13.05
   leveds_lever = (float *) zmalloc(cc);

   zdialog_resize(zdsela,0,360);
   zdialog_run(zdsela,leveds_event,"save");                                //  run dialog - parallel
   leveds_event(zdsela,"init");                                            //  initialize default params
   return;
}


//  dialog event and completion function

int leveds_event(zdialog *zd, cchar *event)
{
   int         ii, kk, pixdist, rowcc;
   float       px, py, xval, yval, lever;
   float       *pixel0, *pixel1, *pixel2, *pixel3, *pixel4;
   spldat      *sd = leveds_curve;
   
   if (zd->zstat) {                                                        //  done, kill dialog
      sa_unselect();                                                       //  delete area                  v.12.11
      zdialog_free(zdsela);
      zfree(sd);                                                           //  free curve edit memory
      zfree(leveds_lever);
      leveds_lever = 0;
      return 0;
   }
   
   if (! sa_validate() || sa_stat != 3) {                                  //  select area gone             v.12.12
      zdialog_free(zdsela);
      zfree(sd);                                                           //  free curve edit memory
      zfree(leveds_lever);
      leveds_lever = 0;
      return 0;
   }
   
   if (strEqu(event,"loadcurve")) {                                        //  load saved curve 
      splcurve_load(sd);
      if (CEF && CEF->zd) zdialog_send_event(CEF->zd,"blendwidth");        //  notify edit dialog
      return 0;
   }

   if (strEqu(event,"savecurve")) {                                        //  save curve to file
      splcurve_save(sd);
      return 0;
   }

   ii = strcmpv(event,"bright","contrast",null);                           //  new lever type
   if (ii >= 1 && ii <= 2) leveds_type = ii;

   ii = strcmpv(event,"all","red","green","blue",null);                    //  new lever color
   if (ii >= 1 && ii <= 4) leveds_color = ii;

   if (leveds_type != leveds_ptype || leveds_color != leveds_pcolor)       //  test for change
   {
      leveds_ptype = leveds_type;
      leveds_pcolor = leveds_color;

      for (int ipy = 1; ipy < E1pxm->hh-1; ipy++)                          //  v.13.05
      for (int ipx = 1; ipx < E1pxm->ww-1; ipx++)
      {
         pixel0 = PXMpix(E1pxm,ipx,ipy);                                   //  target pixel to measure
         lever = 0;

         if (leveds_type == 1)                                             //  lever type = brightness
         {
            if (leveds_color == 1)
               lever = 0.333 * (pixel0[0] + pixel0[1] + pixel0[2]);        //  use all colors
            else {
               ii = leveds_color - 2;                                      //  use single color
               lever = pixel0[ii];
            }
         }

         else if (leveds_type == 2)                                        //  lever type = contrast
         {
            rowcc = E1pxm->ww * 3;                                         //  image row span

            pixel1 = pixel0 + 3;                                           //  4 pixels right, left, down, up
            pixel2 = pixel0 - 3;
            pixel3 = pixel0 + rowcc;
            pixel4 = pixel0 - rowcc;

            if (leveds_color == 1)                                         //  use all colors
            {
               lever  = fabsf(pixel0[0] - pixel1[0]) + fabsf(pixel0[0] - pixel2[0]) 
                      + fabsf(pixel0[0] - pixel3[0]) + fabsf(pixel0[0] - pixel4[0]);
               lever += fabsf(pixel0[1] - pixel1[1]) + fabsf(pixel0[1] - pixel2[1]) 
                      + fabsf(pixel0[1] - pixel3[1]) + fabsf(pixel0[1] - pixel4[1]);
               lever += fabsf(pixel0[2] - pixel1[2]) + fabsf(pixel0[2] - pixel2[2]) 
                      + fabsf(pixel0[2] - pixel3[2]) + fabsf(pixel0[2] - pixel4[2]);
               lever = lever / 12.0; 
            }
            else                                                           //  use single color
            {
               ii = leveds_color - 2;
               lever  = fabsf(pixel0[ii] - pixel1[ii]) + fabsf(pixel0[ii] - pixel2[ii]) 
                      + fabsf(pixel0[ii] - pixel3[ii]) + fabsf(pixel0[ii] - pixel4[ii]);
               lever = lever / 4.0;
            }
         }

         lever = lever / 1000.0;                                           //  scale 0.0 to 0.999
         lever = pow(lever,0.3);                                           //  log scale: 0.1 >> 0.5, 0.8 >> 0.94

         ii = ipy * E1pxm->ww + ipx;                                       //  save lever for each pixel
         leveds_lever[ii] = lever;
      }
   }

   if (strnEqu(event,"b ",2)) {                                            //  button to move entire curve
      for (ii = 0; ii < sd->nap[0]; ii++) {
         px = sd->apx[0][ii];
         py = sd->apy[0][ii];
         if (strEqu(event,"b +++")) py += 0.1;
         if (strEqu(event,"b ---")) py -= 0.1;
         if (strEqu(event,"b +-"))  py += 0.1 - 0.2 * px;
         if (strEqu(event,"b -+"))  py -= 0.1 - 0.2 * px;
         if (strEqu(event,"b +-+")) py -= 0.05 - 0.2 * fabsf(px-0.5);
         if (strEqu(event,"b -+-")) py += 0.05 - 0.2 * fabsf(px-0.5);
         if (py > 1) py = 1;
         if (py < 0) py = 0;
         sd->apy[0][ii] = py;
      }
      event = "edit";
   }

   if (strEqu(event,"edit")) {
      splcurve_generate(sd,0);                                             //  regenerate the curve
      gtk_widget_queue_draw(sd->drawarea);
   }

   if (! CEF) return 0;                                                    //  no edit function active

   for (int ipy = 1; ipy < E1pxm->hh-1; ipy++)                             //  v.13.05
   for (int ipx = 1; ipx < E1pxm->ww-1; ipx++)
   {
      pixel0 = PXMpix(E1pxm,ipx,ipy);                                      //  target pixel to measure
      ii = ipy * E1pxm->ww + ipx;
      lever = leveds_lever[ii];                                            //  leverage to apply, 0.0 to 0.999
      xval = lever;                                                        //  curve x-value, 0.0 to 0.999
      kk = 1000 * xval;
      if (kk > 999) kk = 999;
      yval = sd->yval[0][kk];                                              //  y-value, 0 to 0.999
      pixdist = 1 + 998 * yval;                                            //  pixel edge distance, 1 to 999
      sa_pixmap[ii] = pixdist;
   }
   
   if (CEF->zd) zdialog_send_event(CEF->zd,"blendwidth");                  //  notify edit dialog
   return 0;
}


//  this function is called when curve is edited using mouse

void  leveds_curve_update(int)
{
   if (! zdsela) return;
   leveds_event(zdsela,"edit");
   return;
}



