/*
 * $Id: arch_gen_cpu_x86.c,v 1.88 2011-01-20 14:20:17 vrsieh Exp $ 
 *
 * Copyright (C) 2007-2011 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_BIOS_POST_CODE	0

#include "config.h"
#include "compiler.h"

#include <assert.h>
#include "fixme.h"
#include <stdio.h>
#include <string.h>

#include "glue-log.h"
#include "glue-main.h"
#include "glue-shm.h"

#include "exec-all.h"
#include "arch_gen_cpu_x86_state.h"
#include "arch_gen_cpu_x86_kernel_fast.h"
#include "arch_gen_cpu_x86_core.h"
#include "arch_gen_cpu_x86_mmu.h"
#include "arch_gen_cpu_x86_reg.h"
#include "arch_gen_cpu_x86_seg.h"
#include "qemu/cpu_jit.h"

#define COMP SNAME

#ifdef CONFIG_CPU_SOCKET_ISA
static void
NAME_(ior)(uint16_t port, unsigned int bs, uint16_t *valp)
{
	uint8_t val8;

	switch (bs) {
	case 0x1:
		sig_isa_bus_inb(env->port_bus, env, &val8, port + 0);
		*valp = val8 << 0;
		break;
	case 0x2:
		sig_isa_bus_inb(env->port_bus, env, &val8, port + 1);
		*valp = val8 << 8;
		break;
	case 0x3:
		sig_isa_bus_inw(env->port_bus, env, valp, port + 0);
		break;
	}
}

static void
NAME_(iow)(uint16_t port, unsigned int bs, uint16_t val)
{
	switch (bs) {
	case 0x1:
		sig_isa_bus_outb(env->port_bus, env, val >> 0, port + 0);
		break;
	case 0x2:
		sig_isa_bus_outb(env->port_bus, env, val >> 8, port + 1);
		break;
	case 0x3:
		sig_isa_bus_outw(env->port_bus, env, val, port + 0);
		break;
	}
}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
static void
NAME_(ior)(uint16_t port, unsigned int bs, uint32_t *valp)
{
	*valp = -1;
	if (unlikely(sig_host_bus_ior(env->port_bus, env, port, bs, valp) != 0)) {
		sig_host_bus_type_addr(env->port_bus, env,
				0, SIG_HOST_BUS_IOR, port);
		/* delay... */
		sig_host_bus_read_data(env->port_bus, env,
				bs, valp);
	}
}

static void
NAME_(iow)(uint16_t port, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_iow(env->port_bus, env, port, bs, val) != 0)) {
		sig_host_bus_type_addr(env->port_bus, env,
				0, SIG_HOST_BUS_IOW, port);
		/* delay... */
		sig_host_bus_write_data(env->port_bus, env,
				bs, val);
	}
}

#elif defined(CONFIG_CPU_SOCKET_775)
static void
NAME_(ior)(uint16_t port, unsigned int bs, uint64_t *valp)
{
	*valp = -1;
	if (unlikely(sig_host775_bus_ior(env->port_bus, env, port, bs, valp) != 0)) {
		sig_host775_bus_type_addr(env->port_bus, env,
				0, SIG_HOST775_BUS_IOR, port);
		/* delay... */
		sig_host775_bus_read_data(env->port_bus, env,
				bs, valp);
	}
}

static void
NAME_(iow)(uint16_t port, unsigned int bs, uint64_t val)
{
	if (unlikely(sig_host775_bus_iow(env->port_bus, env, port, bs, val) != 0)) {
		sig_host775_bus_type_addr(env->port_bus, env,
				0, SIG_HOST775_BUS_IOW, port);
		/* delay... */
		sig_host775_bus_write_data(env->port_bus, env,
				bs, val);
	}
}
#else
#error "Unknown socket."
#endif

uint8_t
NAME_(inb)(uint16_t port)
{
	uint8_t val8;

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	uint16_t val16;

	NAME_(ior)(port & ~1, 1 << (port & 1), &val16);
	val8 = (val16 >> ((port & 1) * 8)) & 0xff;

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val32;

	NAME_(ior)(port & ~3, 1 << (port & 3), &val32);
	val8 = (val32 >> ((port & 3) * 8)) & 0xff;

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

	NAME_(ior)(port & ~7, 1 << (port & 7), &val64);
	val8 = (val64 >> ((port & 7) * 8)) & 0xff;

#else
#error "Unknown socket."
#endif

	return val8;
}

uint16_t
NAME_(inw)(uint16_t port)
{
	uint16_t val16;

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	if (0 < (port & 1)) {
		uint16_t val0;
		uint16_t val1;

		NAME_(ior)((port & ~1) + 0, (0x3 << (port & 3)) & 0x3, &val0);
		NAME_(ior)((port & ~1) + 2, (0x3 << (port & 3)) >> 2, &val1);
		val16 = val0 | (val1 << 8);

	} else {
		NAME_(ior)(port, 0x3, &val16);
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	if (2 < (port & 3)) {
		uint32_t val0;
		uint32_t val1;

		NAME_(ior)((port & ~3) + 0, (0x3 << (port & 3)) & 0xf, &val0);
		NAME_(ior)((port & ~3) + 4, (0x3 << (port & 3)) >> 4, &val1);
		val16 = (val0 >> ((port & 3) * 8))
		      | (val1 << ((4 - (port & 3)) * 8));

	} else {
		uint32_t val32;

		NAME_(ior)(port & ~3, 3 << (port & 3), &val32);
		val16 = (val32 >> ((port & 3) * 8)) & 0xffff;
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	if (6 < (port & 7)) {
		uint64_t val0;
		uint64_t val1;

		NAME_(ior)((port & ~7) + 0, (0x3 << (port & 7)) & 0xff, &val0);
		NAME_(ior)((port & ~7) + 8, (0x3 << (port & 7)) >> 8, &val1);
		val16 = (val0 >> ((port & 7) * 8))
		      | (val1 << ((8 - (port & 7)) * 8));

	} else {
		uint64_t val64;

		NAME_(ior)(port & ~7, 3 << (port & 7), &val64);
		val16 = (val64 >> ((port & 7) * 8)) & 0xffff;
	}
#else
#error "Unknown socket."
#endif

	return val16;
}

#if 80386 <= CONFIG_CPU
uint32_t
NAME_(inl)(uint16_t port)
{
	uint32_t val32;

#if defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	if (0 < (port & 3)) {
		uint32_t val0;
		uint32_t val1;

		NAME_(ior)((port & ~3) + 0, (0xf << (port & 3)) & 0xf, &val0);
		NAME_(ior)((port & ~3) + 4, (0xf << (port & 3)) >> 4, &val1);
		val32 = (val0 >> ((port & 3) * 8))
		      | (val1 << ((4 - (port & 3)) * 8));

	} else {
		NAME_(ior)(port, 0xf, &val32);
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	if (4 < (port & 7)) {
		uint64_t val0;
		uint64_t val1;

		NAME_(ior)((port & ~7) + 0, (0xf << (port & 7)) & 0xff, &val0);
		NAME_(ior)((port & ~7) + 8, (0xf << (port & 7)) >> 8, &val1);
		val32 = (val0 >> ((port & 7) * 8))
		      | (val1 << ((8 - (port & 7)) * 8));

	} else {
		uint64_t val64;

		NAME_(ior)(port & ~7, 0xf << (port & 7), &val64);
		val32 = (val64 >> ((port & 7) * 8)) & 0xffffffff;
	}
#else
#error "Unknown socket."
#endif

	return val32;
}
#endif /* 80386 <= CONFIG_CPU */

void
NAME_(outb)(uint8_t value, uint16_t port)
{
#if DEBUG_BIOS_POST_CODE
        if (port == 0x0080
         || port == 0x0300) {
                /* BIOS Post code. */
                fprintf(stderr, "BIOS: Post code 0x%02x.\n", value);

        }
#endif
        if (port == 0xffff) {
                /* System BIOS / VGA BIOS output port. */
                fprintf(stderr, "%c", value);
                return;
        }

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	uint16_t val16;

	val16 = value << ((port & 1) * 8);
	NAME_(iow)(port & ~1, 1 << (port & 1), val16);
	
#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
        uint32_t val32;

        val32 = value << ((port & 3) * 8);
        NAME_(iow)(port & ~3, 1 << (port & 3), val32);

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
        uint64_t val64;

        val64 = (uint64_t) value << ((port & 7) * 8);
        NAME_(iow)(port & ~7, 1 << (port & 7), val64);
#else
#error "Unknown socket."
#endif
}

void
NAME_(outw)(uint16_t value, uint16_t port)
{
#if DEBUG_BIOS_POST_CODE
        if (port == 0x0080
         || port == 0x0300) {
                /* BIOS Post code. */
                fprintf(stderr, "BIOS: Post code 0x%04x.\n", value);
        }
#endif

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	if (0 < (port & 1)) {
                unsigned char value0 = (value >> 0) & 0xff;
                unsigned char value8 = (value >> 8) & 0xff;

                fprintf(stderr, "%s: WARNING: outw->outb port 0x%04x "
                                "value 0x%x\n", __FUNCTION__, port, value);

                NAME_(outb)(value0, port + 0);
                NAME_(outb)(value8, port + 1);
		
	} else {
                NAME_(iow)(port, 3, value);
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
        uint32_t val32;

        if (2 < (port & 3)) {
                unsigned char value0 = (value >> 0) & 0xff;
                unsigned char value8 = (value >> 8) & 0xff;

                fprintf(stderr, "%s: WARNING: outw->outb port 0x%04x "
                                "value 0x%x\n", __FUNCTION__, port, value);

                NAME_(outb)(value0, port + 0);
                NAME_(outb)(value8, port + 1);

        } else {
                val32 = value << ((port & 3) * 8);
                NAME_(iow)(port & ~3, 3 << (port & 3), val32);
        }

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
        uint64_t val64;

        if (6 < (port & 3)) {
                unsigned char value0 = (value >> 0) & 0xff;
                unsigned char value8 = (value >> 8) & 0xff;

                fprintf(stderr, "%s: WARNING: outw->outb port 0x%04x "
                                "value 0x%x\n", __FUNCTION__, port, value);

                NAME_(outb)(value0, port + 0);
                NAME_(outb)(value8, port + 1);

        } else {
                val64 = (uint64_t) value << ((port & 7) * 8);
                NAME_(iow)(port & ~7, 3 << (port & 7), val64);
        }
#else
#error "Unknown socket."
#endif
}

#if 80386 <= CONFIG_CPU
void
NAME_(outl)(uint32_t value, uint16_t port)
{
#if DEBUG_BIOS_POST_CODE
        if (port == 0x0080
         || port == 0x0300) {
                /* BIOS Post code. */
                fprintf(stderr, "BIOS: Post code 0x%08x.\n", value);
        }
#endif

#if defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
        if (0 < (port & 3)) {
                uint32_t val0;
                uint32_t val1;

                fprintf(stderr, "%s: WARNING: outl->outw port 0x%04x "
                        "value 0x%08lx\n", __FUNCTION__, port, (long)value);

                val0 = value << ((port & 3) * 8);
                val1 = value << ((4 - (port & 3)) * 8);

                NAME_(iow)(port & ~3, (0xf << (port & 3)) & 0xf, val0);
                NAME_(iow)((port & ~3) + 4, 0xf >> (4 - (port & 3)), val1);

        } else {
                NAME_(iow)(port & ~3, 0xf, value);
        }

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
        if (4 < (port & 7)) {
                uint64_t val0;
                uint64_t val1;

                fprintf(stderr, "%s: WARNING: outl->outw port 0x%04x "
                        "value 0x%08lx\n", __FUNCTION__, port, (long)value);

                val0 = (uint64_t) value << ((port & 7) * 8);
                val1 = (uint64_t) value << ((8 - (port & 7)) * 8);

                NAME_(iow)(port & ~7, (0xf << (port & 7)) & 0xff, val0);
                NAME_(iow)((port & ~7) + 8, 0xf >> (8 - (port & 7)), val1);

        } else {
		uint64_t val64;

                val64 = (uint64_t) value << ((port & 7) * 8);
                NAME_(iow)(port & ~7, 0xf << (port & 7), val64);
        }
#else
#error "Unknown socket."
#endif
}
#endif /* 80386 <= CONFIG_CPU */

#if defined(CONFIG_CPU_SOCKET_ISA)
void
NAME_(mr)(Paddr addr, unsigned int bs, uint16_t *valp)
{
	uint8_t val8;

	switch (bs) {
	case 0x1:
		sig_isa_bus_readb(env->port_bus, env, addr + 0, &val8);
		*valp = val8 << 0;
		break;
	case 0x2:
		sig_isa_bus_readb(env->port_bus, env, addr + 1, &val8);
		*valp = val8 << 8;
		break;
	case 0x3:
		sig_isa_bus_readw(env->port_bus, env, addr + 0, valp);
		break;
	}
}

static void
NAME_(mw)(Paddr addr, unsigned int bs, uint16_t val)
{
	switch (bs) {
	case 0x1:
		sig_isa_bus_writeb(env->port_bus, env, addr + 0, val >> 0);
		break;
	case 0x2:
		sig_isa_bus_writeb(env->port_bus, env, addr + 1, val >> 8);
		break;
	case 0x3:
		sig_isa_bus_writew(env->port_bus, env, addr + 0, val >> 0);
		break;
	}
}

static void
NAME_(mx)(Paddr addr, unsigned int bs, uint16_t *valp)
{
	NAME_(mr)(addr, bs, valp);
}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
void
NAME_(mr)(unsigned long addr, unsigned int bs, uint32_t *valp)
{
	if (unlikely(sig_host_bus_mr(env->port_bus, env,
			env->smm, addr, bs, valp) != 0)) {
		sig_host_bus_type_addr(env->port_bus, env,
				env->smm, SIG_HOST_BUS_MR, addr);
		/* delay */
		sig_host_bus_read_data(env->port_bus, env,
				bs, valp);
	}
}

static void
NAME_(mw)(unsigned long addr, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_mw(env->port_bus, env,
			env->smm, addr, bs, val) != 0)) {
		sig_host_bus_type_addr(env->port_bus, env,
				env->smm, SIG_HOST_BUS_MW, addr);
		/* delay */
		sig_host_bus_write_data(env->port_bus, env,
				bs, val);
	}
}

static void
NAME_(mx)(unsigned long addr, unsigned int bs, uint32_t *valp)
{
	if (unlikely(sig_host_bus_mr(env->port_bus, env,
			env->smm, addr, bs, valp) != 0)) {
		sig_host_bus_type_addr(env->port_bus, env,
				env->smm, SIG_HOST_BUS_MR, addr);
		/* delay */
		sig_host_bus_read_data(env->port_bus, env,
				bs, valp);
	}
}

#elif defined(CONFIG_CPU_SOCKET_775)
void
NAME_(mr)(unsigned long addr, unsigned int bs, uint64_t *valp)
{
	if (unlikely(sig_host775_bus_mr(env->port_bus, env,
			env->smm, addr, bs, valp) != 0)) {
		sig_host775_bus_type_addr(env->port_bus, env,
				env->smm, SIG_HOST775_BUS_MR, addr);
		/* delay */
		sig_host775_bus_read_data(env->port_bus, env,
				bs, valp);
	}
}

static void
NAME_(mw)(unsigned long addr, unsigned int bs, uint64_t val)
{
	if (unlikely(sig_host775_bus_mw(env->port_bus, env,
			env->smm, addr, bs, val) != 0)) {
		sig_host775_bus_type_addr(env->port_bus, env,
				env->smm, SIG_HOST775_BUS_MW, addr);
		/* delay */
		sig_host775_bus_write_data(env->port_bus, env,
				bs, val);
	}
}

static void
NAME_(mx)(unsigned long addr, unsigned int bs, uint64_t *valp)
{
	if (unlikely(sig_host775_bus_mr(env->port_bus, env,
			env->smm, addr, bs, valp) != 0)) {
		sig_host775_bus_type_addr(env->port_bus, env,
				env->smm, SIG_HOST775_BUS_MR, addr);
		/* delay */
		sig_host775_bus_read_data(env->port_bus, env,
				bs, valp);
	}
}
#else
#error "Unknown socket."
#endif

static uint32_t
NAME_(mr_data_b)(Paddr pa)
{
	assert(! (pa & 0));

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	uint16_t val16;

	NAME_(mr)(pa & ~1, 1 << (pa & 1), &val16);

	return (val16 >> ((pa & 1) * 8)) & 0xff;

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val32;

	NAME_(mr)(pa & ~3, 1 << (pa & 3), &val32);

	return (val32 >> ((pa & 3) * 8)) & 0xff;

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

	NAME_(mr)(pa & ~7, 1 << (pa & 7), &val64);

	return (val64 >> ((pa & 7) * 8)) & 0xff;
#else
#error "Unknown socket."
#endif
}

static uint32_t
NAME_(mr_data_w)(Paddr pa)
{
	assert(! (pa & 1));

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	uint16_t val16;

	NAME_(mr)(pa & ~1, 3, &val16);

	return val16;

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val32;

	NAME_(mr)(pa & ~3, 3 << (pa & 3), &val32);

	return (val32 >> ((pa & 3) * 8)) & 0xffff;

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

	NAME_(mr)(pa & ~7, 3 << (pa & 7), &val64);

	return (val64 >> ((pa & 7) * 8)) & 0xffff;
#else
#error "Unknown socket."
#endif
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
/*forward*/ static int
NAME_(apic_read)(unsigned long pa, void *to, unsigned long len);
#endif

uint32_t
NAME_(mr_data_l)(Paddr pa)
{
	assert(! (pa & 3));

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	fixme();

	return 0;

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val32;

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! (pa & 0xf)) {
		if (NAME_(apic_read)(pa, &val32, 4) == 0) {
			return val32;
		}
	}
#endif

	NAME_(mr)(pa & ~3, 0xf << (pa & 3), &val32);

	return (val32 >> ((pa & 3) * 8)) & 0xffffffff;

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! (pa & 0xf)) {
		uint32_t val32;

		if (NAME_(apic_read)(pa, &val32, 4) == 0) {
			return val32;
		}
	}
#endif

	NAME_(mr)(pa & ~7, 0xf << (pa & 7), &val64);

	return (val64 >> ((pa & 7) * 8)) & 0xffffffff;
#else
#error "Unknown socket."
#endif
}

static void
NAME_(mw_data_b)(Paddr pa, uint32_t val)
{
	assert(! (pa & 0));

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	val <<= (pa & 1) * 8;

	NAME_(mw)(pa & ~1, 1 << (pa & 1), val);

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	val <<= (pa & 3) * 8;

	NAME_(mw)(pa & ~3, 1 << (pa & 3), val);

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

	val64 = ((uint64_t) val) << ((pa & 7) * 8);
	NAME_(mw)(pa & ~7, 1 << (pa & 7), val64);
#else
#error "Unknown socket."
#endif
}

void
NAME_(mw_data_w)(Paddr pa, uint32_t val)
{
	assert(! (pa & 1));

#if CONFIG_CPU < 80386
	/* 16 Bit Data Bus */
	NAME_(mw)(pa, 3, val);

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	val <<= (pa & 3) * 8;

	NAME_(mw)(pa & ~3, 3 << (pa & 3), val);

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

	val64 = ((uint64_t) val) << ((pa & 7) * 8);
	NAME_(mw)(pa & ~7, 3 << (pa & 7), val64);
#else
#error "Unknown socket."
#endif
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
/*forward*/ static int
NAME_(apic_write)(unsigned long pa, const void *from, unsigned long len);
#endif

void
NAME_(mw_data_l)(Paddr pa, uint32_t val)
{
	assert(! (pa & 3));

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	fixme();

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	val <<= (pa & 3) * 8;

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! (pa & 0xf)) {
		if (NAME_(apic_write)(pa, &val, 4) == 0) {
			return;
		}
	}
#endif

	NAME_(mw)(pa & ~3, 0xf << (pa & 3), val);

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! (pa & 0xf)) {
		if (NAME_(apic_write)(pa, &val, 4) == 0) {
			return;
		}
	}
#endif

	val64 = ((uint64_t) val) << ((pa & 7) * 8);
	NAME_(mw)(pa & ~7, 0xf << (pa & 7), val64);
#else
#error "Unknown socket."
#endif
}

static uint32_t
NAME_(mx_code_b)(Paddr pa)
{
	assert(! (pa & 0));

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	uint16_t val;

	NAME_(mx)(pa & ~1, 1 << (pa & 1), &val);

	return (val >> ((pa & 1) * 8)) & 0xff;

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val;

	NAME_(mx)(pa & ~3, 1 << (pa & 3), &val);

	return (val >> ((pa & 3) * 8)) & 0xff;

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val;

	NAME_(mx)(pa & ~7, 1 << (pa & 7), &val);

	return (val >> ((pa & 7) * 8)) & 0xff;
#else
#error "Unknown socket."
#endif
}

static uint32_t
NAME_(mx_code_w)(Paddr pa)
{
	assert(! (pa & 1));

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	uint16_t val;

	NAME_(mx)(pa, 3, &val);

	return val;

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val;

	NAME_(mx)(pa & ~3, 3 << (pa & 3), &val);

	return (val >> ((pa & 3) * 8)) & 0xffff;

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val;

	NAME_(mx)(pa & ~7, 3 << (pa & 7), &val);

	return (val >> ((pa & 7) * 8)) & 0xffff;
#else
#error "Unknown socket."
#endif
}

static uint32_t
NAME_(mx_code_l)(Paddr pa)
{
	assert(! (pa & 3));

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	fixme();

	return 0;

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val;

	NAME_(mx)(pa & ~3, 0xf << (pa & 3), &val);

	return (val >> ((pa & 3) * 8)) & 0xffffffff;

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val;

	NAME_(mx)(pa & ~7, 0xf << (pa & 7), &val);

	return (val >> ((pa & 7) * 8)) & 0xffffffff;
#else
#error "Unknown socket."
#endif
}

static void
NAME_(mw_code_b)(Paddr pa, uint32_t val)
{
	NAME_(tb_invalidate_phys_page_fast)(pa, 1);
	NAME_(mw_data_b)(pa, val);
}

static void
NAME_(mw_code_w)(Paddr pa, uint32_t val)
{
	NAME_(tb_invalidate_phys_page_fast)(pa, 2);
	NAME_(mw_data_w)(pa, val);
}

static void
NAME_(mw_code_l)(Paddr pa, uint32_t val)
{
	NAME_(tb_invalidate_phys_page_fast)(pa, 4);
	NAME_(mw_data_l)(pa, val);
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
/*forward*/ static int
NAME_(apic_map)(unsigned long pa);
#endif

static int
NAME_(map_r)(Paddr paddr, char **haddr_p)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	/* First try internal APIC. */
	if (! NAME_(apic_map)(paddr)) {
		*haddr_p = NULL;
		return 0;
	}
#endif

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* Try isa bus components. */
	char *haddr_dummy;

	if (! sig_isa_bus_map(env->port_bus, env,
			paddr, haddr_p, &haddr_dummy)) {
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* Try host bus components. */
	if (! sig_host_bus_map_r_check(env->port_bus, env, env->smm, paddr)
	 && ! sig_host_bus_map_r(env->port_bus, env,
	 		env->smm, paddr, haddr_p)) {
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* Try host bus components. */
	if (! sig_host775_bus_map_r_check(env->port_bus, env, env->smm, paddr)
	 && ! sig_host775_bus_map_r(env->port_bus, env,
	 		env->smm, paddr, haddr_p)) {
		return 0;
	}
#else
#error "Unknown socket."
#endif

	/* Region unknown => not mapped. */
	*haddr_p = NULL;
	return 1;
}

static int
NAME_(map_w)(Paddr paddr, char **haddr_p)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	/* First try internal APIC. */
	if (! NAME_(apic_map)(paddr)) {
		*haddr_p = NULL;
		return 0;
	}
#endif

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* Try isa bus components. */
	char *haddr_dummy;

	if (! sig_isa_bus_map(env->port_bus, env,
			paddr, &haddr_dummy, haddr_p)) {
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* Try host bus components. */
	if (! sig_host_bus_map_w_check(env->port_bus, env, env->smm, paddr)
	 && ! sig_host_bus_map_w(env->port_bus, env,
			env->smm, paddr, haddr_p)) {
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* Try host bus components. */
	if (! sig_host775_bus_map_w_check(env->port_bus, env, env->smm, paddr)
	 && ! sig_host775_bus_map_w(env->port_bus, env,
			env->smm, paddr, haddr_p)) {
		return 0;
	}
#else
#error "Unknown socket."
#endif

	/* Region unknown => not mapped. */
	*haddr_p = NULL;
	return 1;
}

static int
NAME_(map_x)(Paddr paddr, char **haddr_p)
{
#if CONFIG_CPU < 80386
#endif

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	/* First try internal APIC. */
	if (! NAME_(apic_map)(paddr)) {
		*haddr_p = NULL;
		return 0;
	}
#endif

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* Try isa bus components. */
	char *haddr_dummy;

	if (! sig_isa_bus_map(env->port_bus, env,
			paddr, haddr_p, &haddr_dummy)) {
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* Try host bus components. */
	if (! sig_host_bus_map_x_check(env->port_bus, env, env->smm, paddr)
	 && ! sig_host_bus_map_x(env->port_bus, env,
			env->smm, paddr, haddr_p)) {
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* Try host bus components. */
	if (! sig_host775_bus_map_x_check(env->port_bus, env, env->smm, paddr)
	 && ! sig_host775_bus_map_x(env->port_bus, env,
			env->smm, paddr, haddr_p)) {
		return 0;
	}
#else
#error "Unknown socket."
#endif

	/* Region unknown => not mapped. */
	*haddr_p = NULL;
	return 1;
}

static void
NAME_(ack)(uint8_t *vecp)
{
#if defined(CONFIG_CPU_SOCKET_ISA)
	sig_isa_bus_ack(env->port_bus, env, vecp);

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	sig_host_bus_inta_addr(env->port_bus, env);
	sig_host_bus_inta_data(env->port_bus, env, vecp);

#elif defined(CONFIG_CPU_SOCKET_775)
	sig_host775_bus_inta_addr(env->port_bus, env);
	sig_host775_bus_inta_data(env->port_bus, env, vecp);
#else
#error "Unknown socket."
#endif
}

#include "arch_gen_cpu_x86_apic.c"
#include "arch_gen_cpu_x86_core.c"
#include "arch_gen_cpu_x86_io.c"
#include "arch_gen_cpu_x86_mmu.c"
#include "arch_gen_cpu_x86_reg.c"

#if 80386 <= CONFIG_CPU
static void
NAME_(a20m_set)(void *_env, unsigned int a20_state)
{
	struct CPUState *oldenv = env;

	env = (struct CPUState *) _env;

	NAME_(mmu_a20m_set)(a20_state);

	env = oldenv;
}
#endif /* 80386 <= CONFIG_CPU */

#if defined(CONFIG_CPU_SOCKET_ISA)
/* Nothing needed */

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
static int
NAME_(map_r_check)(void *_env, unsigned int smm, uint32_t pa)
{
	// fprintf(stderr, "%s\n", __FUNCTION__);
	return 0; /* Always ok for now. */
}

static int
NAME_(map_w_check)(void *_env, unsigned int smm, uint32_t pa)
{
	struct CPUState *oldenv = env;

	env = (struct CPUState *) _env;

	// fprintf(stderr, "%s 0x%08lx\n", __FUNCTION__, pa);
	NAME_(tb_invalidate_phys_page_range)(pa, pa + 4096);

	env = oldenv;
	return 0; /* Always ok for now. */
}

static int
NAME_(map_x_check)(void *_env, unsigned int smm, uint32_t pa)
{
	// fprintf(stderr, "%s\n", __FUNCTION__);
	return 0; /* Always ok for now. */
}

static void
NAME_(unmap)(void *_env, uint32_t pa, uint32_t len)
{
	struct CPUState *oldenv = env;

	env = (struct cpu *) _env;

	NAME_(mmu_unmap_range)(pa, len);

	env = oldenv;
}

#elif defined(CONFIG_CPU_SOCKET_775)
static int
NAME_(map_r_check)(void *_env, unsigned int smm, uint64_t pa)
{
	// fprintf(stderr, "%s\n", __FUNCTION__);
	return 0; /* Always ok for now. */
}

static int
NAME_(map_w_check)(void *_env, unsigned int smm, uint64_t pa)
{
	struct CPUState *oldenv = env;

	env = (struct CPUState *) _env;

	// fprintf(stderr, "%s 0x%08lx\n", __FUNCTION__, pa);
	NAME_(tb_invalidate_phys_page_range)(pa, pa + 4096);

	env = oldenv;
	return 0; /* Always ok for now. */
}

static int
NAME_(map_x_check)(void *_env, unsigned int smm, uint64_t pa)
{
	// fprintf(stderr, "%s\n", __FUNCTION__);
	return 0; /* Always ok for now. */
}

static void
NAME_(unmap)(void *_env, uint64_t pa, uint64_t len)
{
	struct CPUState *oldenv = env;

	env = (struct cpu *) _env;

	NAME_(mmu_unmap_range)(pa, len);

	env = oldenv;
}
#else
#error "Unknown socket."
#endif

static void
NAME_(lint0_set)(void *_env, unsigned int val)
{
	struct CPUState *oldenv = env;

	assert(val == 0 || val == 1);

	env = (struct cpu *) _env;

	NAME_(apic_lint0_set)(val);

	env = oldenv;
}

static void
NAME_(lint1_set)(void *_env, unsigned int val)
{
	struct CPUState *oldenv = env;

	assert(val == 0 || val == 1);

	env = (struct cpu *) _env;

	NAME_(apic_lint1_set)(val);

	env = oldenv;
}

#if 80386 <= CONFIG_CPU
static void
NAME_(smi_set)(void *_env, unsigned int val)
{
	struct CPUState *oldenv = env;

	env = (struct cpu *) _env;

	assert(val == 0 || val == 1);

	NAME_(apic_smi_set)(val);

	env = oldenv;
}
#endif /* 80386 <= CONFIG_CPU */

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
static void
NAME_(eoi_receive)(void *_env, uint8_t vector)
{
	struct CPUState *oldenv = env;

	NAME_(apic_eoi_receive)(vector);

	env = oldenv;
}

static void
NAME_(msg0_receive)(
	void *_env,
	unsigned int destination_mode,
	unsigned int delivery_mode,
	unsigned int level,
	unsigned int trigger_mode,
	uint8_t vector,
	uint8_t destination
)
{
	struct CPUState *oldenv = env;

	env = (struct cpu *) _env;

	NAME_(apic_msg0_receive)(destination_mode, delivery_mode, level,
			trigger_mode, vector, destination);

	env = oldenv;
}

static void
NAME_(status0_receive)(void *_env, unsigned int status)
{
	struct CPUState *oldenv = env;

	NAME_(apic_status0_receive)(status);

	env = oldenv;
}

static void
NAME_(msg1_receive)(void *_env, uint8_t processor_priority)
{
	struct CPUState *oldenv = env;

	NAME_(apic_msg1_receive)(processor_priority);

	env = oldenv;
}

static void
NAME_(status1_receive)(void *_env, unsigned int status)
{
	struct CPUState *oldenv = env;

	NAME_(apic_status1_receive)(status);

	env = oldenv;
}
#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT */

static void
NAME_(power_set)(void *_env, unsigned int val)
{
	struct CPUState *oldenv = env;

	env = (struct CPUState *) _env;

	NAME_(core_power_set)(val);

	env = oldenv;
}

static void
NAME_(n_reset_set)(void *_env, unsigned int n_val)
{
	struct CPUState *oldenv = env;

	env = (struct CPUState *) _env;

	NAME_(core_n_reset_set)(n_val);

	env = oldenv;
}

#if 80386 < CONFIG_CPU
static void
NAME_(n_init_set)(void *_env, unsigned int n_val)
{
	struct CPUState *oldenv = env;

	env = (struct CPUState *) _env;

	NAME_(core_n_init_set)(n_val);

	env = oldenv;
}
#endif /* 80386 <= CONFIG_CPU */

static void
NAME_(n_ignne_set)(void *_env, unsigned int n_val)
{
	struct CPUState *oldenv = env;

	env = (struct CPUState *) _env;

	NAME_(core_n_ignne_set)(n_val);

	env = oldenv;
}

void *
NAME_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_power,
	struct sig_boolean *port_n_reset,
#if 80386 < CONFIG_CPU
	struct sig_boolean *port_n_init,
#endif
	struct sig_boolean_or *port_irq,
	struct sig_boolean_or *port_nmi,
#if 80386 <= CONFIG_CPU
	struct sig_boolean *port_smi,
#endif
	struct sig_boolean_or *port_n_ferr,
	struct sig_boolean *port_n_ignne,
#if 80386 <= CONFIG_CPU
	struct sig_boolean *port_a20,
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	struct sig_icc_bus *port_icc,
#endif
#if defined(CONFIG_CPU_SOCKET_ISA)
	struct sig_isa_bus_main *port_bus
#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	struct sig_host_bus_main *port_bus
#elif defined(CONFIG_CPU_SOCKET_775)
	struct sig_host775_bus_main *port_bus
#else
#error "Unknown socket."
#endif
)
{
	/* APIC */
	static const struct sig_boolean_or_funcs lint0_func = {
		.set = NAME_(lint0_set),
	};
	static const struct sig_boolean_or_funcs lint1_func = {
		.set = NAME_(lint1_set),
	};
#if 80386 <= CONFIG_CPU
	static const struct sig_boolean_funcs smi_func = {
		.set = NAME_(smi_set),
	};
#endif /* 80386 <= CONFIG_CPU */
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	static const struct sig_icc_bus_funcs icc_funcs = {
		.eoi = NAME_(eoi_receive),
		.msg0 = NAME_(msg0_receive),
		.status0 = NAME_(status0_receive),
		.msg1 = NAME_(msg1_receive),
		.status1 = NAME_(status1_receive),
	};
#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT */

	/* Core */
	static const struct sig_boolean_funcs power_funcs = {
		.set = NAME_(power_set),
	};
	static const struct sig_boolean_funcs n_reset_funcs = {
		.set = NAME_(n_reset_set),
	};
#if 80386 < CONFIG_CPU
	static const struct sig_boolean_funcs n_init_funcs = {
		.set = NAME_(n_init_set),
	};
#endif
	static const struct sig_boolean_funcs n_ignne_funcs = {
		.set = NAME_(n_ignne_set),
	};

#if 80386 <= CONFIG_CPU
	/* MMU */
	static const struct sig_boolean_funcs a20_func = {
		.set = NAME_(a20m_set),
	};
#endif

	/* I/O */
#if CONFIG_CPU < 80386
	/* FIXME */
#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	static const struct sig_host_bus_main_funcs main_funcs = {
		.map_r_check = NAME_(map_r_check),
		.map_w_check = NAME_(map_w_check),
		.map_x_check = NAME_(map_x_check),
		.unmap = NAME_(unmap),
	};
#elif defined(CONFIG_CPU_SOCKET_775)
	static const struct sig_host775_bus_main_funcs main_funcs = {
		.map_r_check = NAME_(map_r_check),
		.map_w_check = NAME_(map_w_check),
		.map_x_check = NAME_(map_x_check),
		.unmap = NAME_(unmap),
	};
#endif
	static unsigned int nr = 0;
	struct cpu *cpssp;

	/* Switch on host optimization. */
#if USE_KFAUM && 0	/* FIXME */
	NAME_(kfaum_init)();
#endif

	cpssp = shm_alloc(sizeof(*cpssp), SHM_CODE);
	assert(cpssp);

	/* Configuration */
	/* FIXME */
	cpssp->apic_cluster_id = 1;
	cpssp->apic_arbitration_id = nr++;

	cpssp->process.inst_hz = CONFIG_CPU_FREQ;

	NAME_(apic_create)(cpssp);

	cpssp->a20_mask = 0xffffffff;

	cpssp->sig_power = port_power;
	cpssp->sig_n_reset = port_n_reset;
#if 80386 < CONFIG_CPU
	cpssp->sig_n_init = port_n_init;
#endif
	cpssp->sig_irq = port_irq;
	cpssp->sig_nmi = port_nmi;
#if 80386 <= CONFIG_CPU
	cpssp->sig_smi = port_smi;
#endif
	cpssp->sig_n_ferr = port_n_ferr;
	cpssp->sig_n_ignne = port_n_ignne;
#if 80386 <= CONFIG_CPU
	cpssp->sig_a20 = port_a20;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	cpssp->icc_bus = port_icc;
#endif
	cpssp->port_bus = port_bus;

	NAME_(mmu_init)(cpssp);
	NAME_(jit_buffer_init)(cpssp);
	NAME_(jit_compile_init)(cpssp);
	NAME_(io_init)(cpssp);
	NAME_(apic_init)(cpssp);

	/* Output signals. */
	sig_boolean_or_connect_out(cpssp->sig_n_ferr, cpssp, 0);

	/* Cycle signals. */
#if defined(CONFIG_CPU_SOCKET_ISA)
	/* FIXME */
#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	sig_host_bus_main_connect(cpssp->port_bus, cpssp, &main_funcs);
#elif defined(CONFIG_CPU_SOCKET_775)
	sig_host775_bus_main_connect(cpssp->port_bus, cpssp, &main_funcs);
#else
#error "Unknown socket."
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	sig_icc_bus_connect(cpssp->icc_bus, cpssp, &icc_funcs);
#endif

	/* Input signals. */
	sig_boolean_connect_in(cpssp->sig_power, cpssp, &power_funcs);
	sig_boolean_connect_in(cpssp->sig_n_reset, cpssp, &n_reset_funcs);
#if 80386 < CONFIG_CPU /* FIXME */
	sig_boolean_connect_in(cpssp->sig_n_init, cpssp, &n_init_funcs);
#endif

	sig_boolean_or_connect_in(cpssp->sig_irq, cpssp, &lint0_func);
	sig_boolean_or_connect_in(cpssp->sig_nmi, cpssp, &lint1_func);
#if 80386 <= CONFIG_CPU
	sig_boolean_connect_in(cpssp->sig_smi, cpssp, &smi_func);
#endif

	sig_boolean_connect_in(cpssp->sig_n_ignne, cpssp, &n_ignne_funcs);

#if 80386 <= CONFIG_CPU
	sig_boolean_connect_in(cpssp->sig_a20, cpssp, &a20_func);
#endif

	sched_process_init(&cpssp->process, NAME_(step), cpssp);

	return cpssp;
}

void
NAME_(destroy)(void *_cpssp)
{
	struct cpu *cpssp = _cpssp;

	NAME_(apic_destroy)(cpssp);

	shm_free(cpssp, sizeof(*cpssp));
}

#undef DEBUG_BIOS_POST_CODE
