/* $Id: pci.h,v 1.20 2009-01-28 13:57:50 potyra Exp $ 
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Note: this is the file that every PCI-"Card" in FAUmachine will want
 * to include. It contains lots of helpful macros as well as the definition
 * of the pci_irq_set function */
 
#ifndef __PCI_H_INCLUDED
#define __PCI_H_INCLUDED

#include <inttypes.h>

#include <assert.h>


#define	PCI_BUS(val)	((val >> 16) & 0xff)
#define	PCI_UNIT(val)	((val >> 11) & 0x1f)
#define	PCI_DEV(val)	PCI_UNIT(val)
#define	PCI_FUNC(val)	((val >> 8) & 0x7)
#define PCI_ADDR(bus,dev,func)	(((bus) << 16 ) | ((dev) << 11) | ((func) << 8))

#define PCI_ADDRMASK	0x00ffff00

/* These can be used as PIN in pci_irq_set, or in the PCI_INTERRUPT_PIN
 * configspace field */
#define PCI_INT_A	1
#define PCI_INT_B	2
#define PCI_INT_C	3
#define PCI_INT_D	4

/* These are the offsets of various things in the configuration space of a
 * card.
 * Please note: PCI is generally little endian.
 * From <linux/pci.h> */
#define PCI_VENDOR_ID		0x00	/* 16 bits */
#define PCI_DEVICE_ID		0x02	/* 16 bits */
#define PCI_COMMAND		0x04	/* 16 bits */
#define PCI_STATUS		0x06	/* 16 bits */
#define PCI_CLASS_REVISION	0x08	/* High 24 bits are class, low 8 */
					/* revision */
#define PCI_REVISION_ID		0x08	/* Revision ID */
#define PCI_CLASS_PROG		0x09	/* Reg. Level Programming Interface */
#define PCI_CLASS_DEVICE	0x0a	/* Device class */
#define PCI_CACHE_LINE_SIZE	0x0c	/* 8 bits */
#define PCI_LATENCY_TIMER	0x0d	/* 8 bits */
#define PCI_HEADER_TYPE		0x0e	/* 8 bits */
#define PCI_BIST		0x0f	/* 8 bits */
#define PCI_BASE_ADDRESS_0	0x10	/* 32 bits */
#define PCI_BASE_ADDRESS_1	0x14	/* 32 bits [htype 0,1 only] */
#define PCI_BASE_ADDRESS_2	0x18	/* 32 bits [htype 0 only] */
#define PCI_BASE_ADDRESS_3	0x1c	/* 32 bits */
#define PCI_BASE_ADDRESS_4	0x20	/* 32 bits */
#define PCI_BASE_ADDRESS_5	0x24	/* 32 bits */
/* Header type 0 (normal devices) */
#define PCI_SUBSYSTEM_VENDOR_ID	0x2c
#define PCI_SUBSYSTEM_ID	0x2e
#define PCI_ROM_ADDRESS		0x30	/* Bits 31..11 are address, 10..1 reserved */
#define  PCI_ROM_ADDRESS_ENABLE	0x01
#define PCI_ROM_ADDRESS_MASK	(~0x7ffUL)
#define PCI_CAPABILITY_LIST	0x34	/* Offset of first capability list entry */
/* 0x35-0x3b are reserved */
#define PCI_INTERRUPT_LINE	0x3c	/* 8 bits */
#define PCI_INTERRUPT_PIN	0x3d	/* 8 bits */
#define PCI_MIN_GNT		0x3e	/* 8 bits */
#define PCI_MAX_LAT		0x3f	/* 8 bits */

/* Possible values (ored together) for PCI_COMMAND */
#define  PCI_COMMAND_IO		0x1	/* Enable response in I/O space */
#define  PCI_COMMAND_MEMORY	0x2	/* Enable response in Memory space */
#define  PCI_COMMAND_MASTER	0x4	/* Enable bus mastering */
#define  PCI_COMMAND_SPECIAL	0x8	/* Enable response to special cycles */
#define  PCI_COMMAND_INVALIDATE	0x10	/* Use memory write and invalidate */
#define  PCI_COMMAND_VGA_PALETTE 0x20	/* Enable palette snooping */
#define  PCI_COMMAND_PARITY	0x40	/* Enable parity checking */
#define  PCI_COMMAND_WAIT	0x80	/* Enable address/data stepping */
#define  PCI_COMMAND_SERR	0x100	/* Enable SERR */
#define  PCI_COMMAND_FAST_BACK	0x200	/* Enable back-to-back writes */

/* Possible values (ored together) for PCI_STATUS */
#define  PCI_STATUS_CAP_LIST	0x10    /* Support Capability List */
#define  PCI_STATUS_66MHZ	0x20    /* Support 66 Mhz PCI 2.1 bus */
#define  PCI_STATUS_UDF		0x40    /* Support User Definable Features [obsolete] */
#define  PCI_STATUS_FAST_BACK	0x80    /* Accept fast-back to back */
#define  PCI_STATUS_PARITY	0x100   /* Detected parity error */
#define  PCI_STATUS_DEVSEL_MASK	0x600   /* DEVSEL timing */
#define  PCI_STATUS_DEVSEL_FAST	0x000
#define  PCI_STATUS_DEVSEL_MEDIUM 0x200
#define  PCI_STATUS_DEVSEL_SLOW	0x400
#define  PCI_STATUS_SIG_TARGET_ABORT	0x800	/* Set on target abort */
#define  PCI_STATUS_REC_TARGET_ABORT	0x1000	/* Master ack of " */
#define  PCI_STATUS_REC_MASTER_ABORT	0x2000	/* Set on master abort */
#define  PCI_STATUS_SIG_SYSTEM_ERROR	0x4000	/* Set when we drive SERR */
#define  PCI_STATUS_DETECTED_PARITY	0x8000	/* Set on parity error */

/* Macros for PCI Header Types */
#define  PCI_HEADER_TYPE_NORMAL			0x00
#define  PCI_HEADER_TYPE_BRIDGE			0x01
#define  PCI_HEADER_TYPE_CARDBUS		0x02
#define  PCI_HEADER_TYPE_MULTIFUNCDEVICE	0x80

/* Macros concerning the various PCI_BASE_ADDRESSes */
#define  PCI_BASE_ADDRESS_SPACE		0x01	/* 0 = memory, 1 = I/O */
#define  PCI_BASE_ADDRESS_SPACE_IO	0x01
#define  PCI_BASE_ADDRESS_SPACE_MEMORY	0x00
#define  PCI_BASE_ADDRESS_MEM_TYPE_MASK	0x06
#define  PCI_BASE_ADDRESS_MEM_TYPE_32	0x00	/* 32 bit address */
#define  PCI_BASE_ADDRESS_MEM_TYPE_1M	0x02	/* Below 1M [obsolete] */
#define  PCI_BASE_ADDRESS_MEM_TYPE_64	0x04	/* 64 bit address */
#define  PCI_BASE_ADDRESS_MEM_PREFETCH	0x08	/* prefetchable? */
#define  PCI_BASE_ADDRESS_MEM_MASK	(~0x0fUL)
#define  PCI_BASE_ADDRESS_IO_MASK	(~0x03UL)

/* Some PCI IDs from <linux/pci_ids.h> */
/* Base classes (high byte of PCI_CLASS_DEVICE) */
#define PCI_BASE_CLASS_STORAGE		0x01
#define PCI_BASE_CLASS_NETWORK		0x02
#define PCI_BASE_CLASS_DISPLAY		0x03
#define PCI_BASE_CLASS_MULTIMEDIA	0x04
#define PCI_BASE_CLASS_MEMORY		0x05
#define PCI_BASE_CLASS_BRIDGE		0x06
#define PCI_BASE_CLASS_COMMUNICATION	0x07
#define PCI_BASE_CLASS_SYSTEM		0x08
#define PCI_BASE_CLASS_INPUT		0x09
#define PCI_BASE_CLASS_DOCKING		0x0a
#define PCI_BASE_CLASS_PROCESSOR	0x0b
#define PCI_BASE_CLASS_SERIAL		0x0c
#define PCI_BASE_CLASS_INTELLIGENT	0x0e
#define PCI_BASE_CLASS_SATELLITE	0x0f
#define PCI_BASE_CLASS_CRYPT		0x10
#define PCI_BASE_CLASS_SIGNAL_PROCESSING 0x11
#define PCI_BASE_CLASS_OTHERS		0xff

/* Values for PCI_CLASS_DEVICE */
#define PCI_CLASS_NOT_DEFINED		0x0000
#define PCI_CLASS_NOT_DEFINED_VGA	0x0001
#define PCI_CLASS_STORAGE_SCSI		0x0100
#define PCI_CLASS_STORAGE_IDE		0x0101
#define PCI_CLASS_STORAGE_FLOPPY	0x0102
#define PCI_CLASS_NETWORK_ETHERNET	0x0200
#define PCI_CLASS_DISPLAY_VGA		0x0300
#define PCI_CLASS_DISPLAY_XGA		0x0301
#define PCI_CLASS_DISPLAY_3D		0x0302
#define PCI_CLASS_MULTIMEDIA_VIDEO	0x0400
#define PCI_CLASS_MULTIMEDIA_AUDIO	0x0401
#define PCI_CLASS_BRIDGE_HOST		0x0600
#define PCI_CLASS_BRIDGE_ISA		0x0601
#define PCI_CLASS_BRIDGE_PCI		0x0604
#define PCI_CLASS_BRIDGE_OTHER		0x0680
#define PCI_CLASS_COMMUNICATION_SERIAL	0x0700
#define PCI_CLASS_INPUT_KEYBOARD	0x0900
#define PCI_CLASS_INPUT_MOUSE		0x0902
#define PCI_CLASS_SERIAL_FIREWIRE	0x0c00
#define PCI_CLASS_SERIAL_USB		0x0c03

/* Some Vendor and Device IDs */
#define PCI_VENDOR_ID_INTEL		0x8086
#define PCI_DEVICE_ID_INTEL_82557	0x1229

/* Every PCI card has between 64 and 256 Bytes of so called config-space,
 * containing memory / io / ROM-addresses, the device type, etc.
 * It is recommended that you save this space as an uint32_t array, as
 * most fields in it are 32 bit. For those that aren't, you can use the
 * following functions to access bytes or words from it. This is intended to
 * make porting to non-little-endian architectures easier.
 */
static inline uint8_t
pci_byte_of_long(uint32_t l, int addr)
{
	return (uint8_t)(l >> (addr << 3));
}

static inline uint16_t
pci_word_of_long(uint32_t l, int addr)
{
	return (uint16_t)(l >> (addr << 3));
}

/* pci_(byte|word)_into_long
 * These functions set one byte or word inside a long value.
 * l    pointer to the unsigned long
 * val  value that byte/word will be set to
 * addr offset in long - for a byte this can be 0-3, for a word 0 or 2!
 */
static inline void
pci_byte_into_long(uint32_t *l, uint8_t val, uint8_t addr)
{
	assert(addr < 4);
	
	addr <<= 3;

	*l &= ~(0xffU << addr);
	*l |= val << addr;
}

static inline void
pci_word_into_long(uint32_t *l, uint16_t val, uint8_t addr)
{
	assert((addr == 0) || (addr == 2));

	addr <<= 3;

	*l &= ~(0xffffU << addr);
	*l |= val << addr;
}

/* the setconfig functions. These are just more comfortable wrappers around
 * pci_(word|byte)_into_long. */
static inline void
pci_setconfigb(uint32_t * cs, uint8_t offset, uint8_t val)
{
	pci_byte_into_long(&cs[offset >> 2], val, offset & 3);
}

static inline void
pci_setconfigw(uint32_t * cs, uint8_t offset, uint16_t val)
{
	pci_word_into_long(&cs[offset >> 2], val, offset & 3);
}

static inline void
pci_setconfigl(uint32_t * cs, uint8_t offset, uint32_t val)
{
	cs[offset >> 2] = val;
}

/* getconfig functions. Same as with setconfig... */
static inline uint8_t
pci_getconfigb(const uint32_t * cs, uint8_t offset)
{
	return pci_byte_of_long(cs[offset >> 2], offset & 3);
}

static inline uint16_t
pci_getconfigw(const uint32_t * cs, uint8_t offset)
{
	return pci_word_of_long(cs[offset >> 2], offset & 3);
}

static inline uint32_t
pci_getconfigl(const uint32_t * cs, uint8_t offset)
{
	return cs[offset >> 2];
}

/* The following function helps you with requesting I/O or Memory space.
 * Either BIOS or OS assign resources to the card. They find out how much the
 * card needs by writing all-1s to the BASE_ADDRESS_X configspace registers.
 * The card then nulls the bits it needs. For example, if you wanted 256
 * IOPorts, you would return 0xFFFFFF00 after the BIOS wrote 0xFFFFFFFF into
 * the right BASE_ADDRESS_X register. This nice little function helps you do
 * the calculation.
 *  val     the value the BIOS wrote to you (needn't be 0xff...)
 *  req     how many I/O-ports or how much Memory-space your card needs.
 *          This has to be a power of 2!
 *  flags   Additional flags that get ored into the result. This has to
 *          contain at least PCI_BASE_ADDRESS_SPACE_MEMORY (to request mem) or
 *          PCI_BASE_ADDRESS_SPACE_IO (to request IO Ports).
 */
static inline uint32_t
pci_requestspace(uint32_t val, uint32_t req, uint32_t flags)
{
	if ((flags & PCI_BASE_ADDRESS_SPACE) == PCI_BASE_ADDRESS_SPACE_MEMORY) {
		val &= PCI_BASE_ADDRESS_MEM_MASK;
	} else {
		val &= PCI_BASE_ADDRESS_IO_MASK;
	}
	val &= ~(req - 1);
	val |= flags;
	return val;
}

/* This is the function that should be used by PCI cards to generate
 * IRQs.
 *  addr   is the usual coded device address (use PCI_ADDR macro to generate
 *         it)
 *  pin    is the interrupt line to use (each pci slot has 4, PCI_INT_A to D)
 *  state  is 1 to signal an interrupt or 0 to take the signal back (after a
 *         driver acknowledged it)
 */
extern void pci_set_irq(unsigned long addr, int pin, int state);

#endif /* __PCI_H_INCLUDED */
