/*
 * $Id: keyboard.c,v 1.69 2009-10-06 11:47:00 sand Exp $ 
 *
 * Copyright (C) 2006-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue-main.h"

#include "keyboard.h"

struct cpssp {
	/* Ports */
	struct sig_ps2_main *port_ps2;
	struct sig_boolean *port_opt_num_led;
	struct sig_boolean *port_opt_caps_led;
	struct sig_boolean *port_opt_scroll_led;
	int state_power;
	int state_clkrunning;

	unsigned int scan_code_set;

	uint8_t buffer[8];
	unsigned int head;
	unsigned int tail;

	uint8_t outbuf[16];
	unsigned int outhead;
	unsigned int outtail;
	unsigned int outcount;
	unsigned int outrunning;

	unsigned char keyboard_command;

	unsigned long long delay;
	unsigned long long repeat;

	unsigned int status_left_shift;
	unsigned int status_right_shift;
	unsigned int status_left_control;
	unsigned int status_right_control;
	unsigned int status_left_alt;
	unsigned int status_right_alt;
	unsigned int status_caps_lock;
	unsigned int status_num_lock;
	unsigned int status_scroll_lock;

	int repeated_key;
};

/* Keyboard replies */
#define KBD_REPLY_POR           0xAA /* Power on reset */
#define KBD_REPLY_ACK           0xFA /* Command ACK */
#define KBD_REPLY_RESEND        0xFE /* Command NACK, send the cmd again */

/* Keyboard commands */
/* see PC-Hardware 5th revision, pp. 1038 */
#define KBD_CMD_EX_ENABLE       0xea /* Enable extra keys */
#define KBD_CMD_SET_LEDS        0xed /* Set keyboard leds */
#define KBD_CMD_ECHO            0xee
#define KBD_CMD_SSCANSET        0xf0 /* Set scancode table */
#define KBD_CMD_GETID           0xf2 /* Get identity string */
#define KBD_CMD_SET_RATE        0xf3 /* Set typematic rate */
#define KBD_CMD_ENABLE          0xf4 /* Enable scanning */
#define KBD_CMD_DISABLE         0xf5 /* Disable scanning */

#define KBD_CMD_STD_ENABLE      0xf6 /* Set standard settings and enable */
#define KBD_CMD_SAK_T		0xf7 /* Set all keys typematic */
#define KBD_CMD_SAK_MB		0xf8 /* Set all keys make/break */
#define KBD_CMD_SAK_M		0xf9 /* Set all keys make */
#define KBD_CMD_SAK_TMB		0xfa /* Set all keys typematic/make/break */
#define KBD_CMD_SOK_T		0xfb /* Set one key typematic */
#define KBD_CMD_SOK_MB		0xfc /* Set one key make/break */
#define KBD_CMD_SOK_M		0xfd /* Set one key make */

#define KBD_CMD_RESEND		0xfe /* Resend */
#define KBD_CMD_RESET		0xff /* Reset */

static const struct {
	enum { UNUSED = 0, NORMAL, SPECIAL, CURSOR, KP_DIV, SYSRQ, PAUSE } type;
	uint8_t set1;
	uint8_t set2;
	uint8_t set3;
} table[0x80] = {
 [0x00] = { UNUSED                    },
 [0x01] = { NORMAL,  0x01, 0x76, 0x08 }, /* Esc */
 [0x02] = { NORMAL,  0x02, 0x16, 0x16 }, /* 1 / ! */
 [0x03] = { NORMAL,  0x03, 0x1e, 0x1e }, /* 2 / @ */
 [0x04] = { NORMAL,  0x04, 0x26, 0x26 }, /* 3 / # */
 [0x05] = { NORMAL,  0x05, 0x25, 0x25 }, /* 4 / $ */
 [0x06] = { NORMAL,  0x06, 0x2e, 0x2e }, /* 5 / % */
 [0x07] = { NORMAL,  0x07, 0x36, 0x36 }, /* 6 / ^ */
 [0x08] = { NORMAL,  0x08, 0x3d, 0x3d }, /* 7 / & */
 [0x09] = { NORMAL,  0x09, 0x3e, 0x3e }, /* 8 / * */
 [0x0a] = { NORMAL,  0x0a, 0x46, 0x46 }, /* 9 / ( */
 [0x0b] = { NORMAL,  0x0b, 0x45, 0x45 }, /* 0 / ) */
 [0x0c] = { NORMAL,  0x0c, 0x4e, 0x4e }, /* - / _ */
 [0x0d] = { NORMAL,  0x0d, 0x55, 0x55 }, /* = / + */
 [0x0e] = { NORMAL,  0x0e, 0x66, 0x66 }, /* BackSpace */
 [0x0f] = { NORMAL,  0x0f, 0x0d, 0x0d }, /* Tab */

 [0x10] = { NORMAL,  0x10, 0x15, 0x15 }, /* q */
 [0x11] = { NORMAL,  0x11, 0x1d, 0x1d }, /* w */
 [0x12] = { NORMAL,  0x12, 0x24, 0x24 }, /* e */
 [0x13] = { NORMAL,  0x13, 0x2d, 0x2d }, /* r */
 [0x14] = { NORMAL,  0x14, 0x2c, 0x2c }, /* t */
 [0x15] = { NORMAL,  0x15, 0x35, 0x35 }, /* y */
 [0x16] = { NORMAL,  0x16, 0x3c, 0x3c }, /* u */
 [0x17] = { NORMAL,  0x17, 0x43, 0x43 }, /* i */
 [0x18] = { NORMAL,  0x18, 0x44, 0x44 }, /* o */
 [0x19] = { NORMAL,  0x19, 0x4d, 0x4d }, /* p */
 [0x1a] = { NORMAL,  0x1a, 0x54, 0x54 }, /* [ / { */
 [0x1b] = { NORMAL,  0x1b, 0x5b, 0x5b }, /* ] / } */
 [0x1c] = { NORMAL,  0x1c, 0x5a, 0x5a }, /* Return */
 [0x1d] = { NORMAL,  0x1d, 0x14, 0x11 }, /* Left-Control */
 [0x1e] = { NORMAL,  0x1e, 0x1c, 0x1c }, /* a */
 [0x1f] = { NORMAL,  0x1f, 0x1b, 0x1b }, /* s */

 [0x20] = { NORMAL,  0x20, 0x23, 0x23 }, /* d */
 [0x21] = { NORMAL,  0x21, 0x2b, 0x2b }, /* f */
 [0x22] = { NORMAL,  0x22, 0x34, 0x34 }, /* g */
 [0x23] = { NORMAL,  0x23, 0x33, 0x33 }, /* h */
 [0x24] = { NORMAL,  0x24, 0x3b, 0x3b }, /* j */
 [0x25] = { NORMAL,  0x25, 0x42, 0x42 }, /* k */
 [0x26] = { NORMAL,  0x26, 0x4b, 0x4b }, /* l */
 [0x27] = { NORMAL,  0x27, 0x4c, 0x4c }, /* ; / : */
 [0x28] = { NORMAL,  0x28, 0x52, 0x52 }, /* ' / " */
 [0x29] = { NORMAL,  0x29, 0x0e, 0x0e }, /* ` / ~ */
 [0x2a] = { NORMAL,  0x2a, 0x12, 0x12 }, /* Left-Shift */
 [0x2b] = { NORMAL,  0x2b, 0x5d, 0x5d }, /* \ / | */
 [0x2c] = { NORMAL,  0x2c, 0x1a, 0x1a }, /* z */
 [0x2d] = { NORMAL,  0x2d, 0x22, 0x22 }, /* x */
 [0x2e] = { NORMAL,  0x2e, 0x21, 0x21 }, /* c */
 [0x2f] = { NORMAL,  0x2f, 0x2a, 0x2a }, /* v */

 [0x30] = { NORMAL,  0x30, 0x32, 0x32 }, /* b */
 [0x31] = { NORMAL,  0x31, 0x31, 0x31 }, /* n */
 [0x32] = { NORMAL,  0x32, 0x3a, 0x3a }, /* m */
 [0x33] = { NORMAL,  0x33, 0x41, 0x41 }, /* , / < */
 [0x34] = { NORMAL,  0x34, 0x49, 0x49 }, /* . / > */
 [0x35] = { NORMAL,  0x35, 0x4a, 0x4a }, /* / / ? */
 [0x36] = { NORMAL,  0x36, 0x59, 0x59 }, /* Right-Shift */
 [0x37] = { NORMAL,  0x37, 0x7c, 0x7e }, /* Keypad-* */
 [0x38] = { NORMAL,  0x38, 0x11, 0x19 }, /* Left-Alt */
 [0x39] = { NORMAL,  0x39, 0x29, 0x29 }, /* Space */
 [0x3a] = { NORMAL,  0x3a, 0x58, 0x14 }, /* CapsLock */
 [0x3b] = { NORMAL,  0x3b, 0x05, 0x07 }, /* F1 */
 [0x3c] = { NORMAL,  0x3c, 0x06, 0x0f }, /* F2 */
 [0x3d] = { NORMAL,  0x3d, 0x04, 0x17 }, /* F3 */
 [0x3e] = { NORMAL,  0x3e, 0x0c, 0x1f }, /* F4 */
 [0x3f] = { NORMAL,  0x3f, 0x03, 0x27 }, /* F5 */

 [0x40] = { NORMAL,  0x40, 0x0b, 0x2f }, /* F6 */
 [0x41] = { NORMAL,  0x41, 0x02, 0x37 }, /* F7 */
 [0x42] = { NORMAL,  0x42, 0x0a, 0x3f }, /* F8 */
 [0x43] = { NORMAL,  0x43, 0x01, 0x47 }, /* F9 */
 [0x44] = { NORMAL,  0x44, 0x09, 0x4f }, /* F10 */
 [0x45] = { NORMAL,  0x45, 0x77, 0x76 }, /* NumLock */
 [0x46] = { NORMAL,  0x46, 0x7e, 0x5f }, /* ScollLock */
 [0x47] = { NORMAL,  0x47, 0x6c, 0x6c }, /* Keypad-7 */
 [0x48] = { NORMAL,  0x48, 0x75, 0x75 }, /* Keypad-8 */
 [0x49] = { NORMAL,  0x49, 0x7d, 0x7d }, /* Keypad-9 */
 [0x4a] = { NORMAL,  0x4a, 0x7b, 0x84 }, /* Keypad-- */
 [0x4b] = { NORMAL,  0x4b, 0x6b, 0x6b }, /* Keypad-4 */
 [0x4c] = { NORMAL,  0x4c, 0x73, 0x73 }, /* Keypad-5 */
 [0x4d] = { NORMAL,  0x4d, 0x74, 0x74 }, /* Keypad-6 */
 [0x4e] = { NORMAL,  0x4e, 0x79, 0x7c }, /* Keypad-+ */
 [0x4f] = { NORMAL,  0x4f, 0x69, 0x69 }, /* Keypad-1 */

 [0x50] = { NORMAL,  0x50, 0x72, 0x72 }, /* Keypad-2 */
 [0x51] = { NORMAL,  0x51, 0x7a, 0x7a }, /* Keypad-3 */
 [0x52] = { NORMAL,  0x52, 0x70, 0x70 }, /* Keypad-0 */
 [0x53] = { NORMAL,  0x53, 0x71, 0x71 }, /* Keypad-. */
 [0x54] = { UNUSED                    },
 [0x55] = { UNUSED                    },
 [0x56] = { NORMAL,  0x56, 0x61, 0x13 }, /* PC102: < / > / | */
 [0x57] = { NORMAL,  0x57, 0x78, 0x56 }, /* F11 */
 [0x58] = { NORMAL,  0x58, 0x07, 0x5e }, /* F12 */
 [0x59] = { UNUSED                    },
 [0x5a] = { UNUSED                    },
 [0x5b] = { UNUSED                    },
 [0x5c] = { UNUSED                    },
 [0x5d] = { UNUSED                    },
 [0x5e] = { UNUSED                    },
 [0x5f] = { UNUSED                    },

 [0x60] = { SPECIAL, 0x1c, 0x5a, 0x79 }, /* Keypad-Enter */
 [0x61] = { SPECIAL, 0x1d, 0x14, 0x58 }, /* Right-Control */
 [0x62] = { KP_DIV,  0x00, 0x00, 0x77 }, /* Keypad-/ */
 [0x63] = { SYSRQ,   0x00, 0x00, 0x57 }, /* SysRq / PrtSc */
 [0x64] = { SPECIAL, 0x38, 0x11, 0x39 }, /* Right-Alt */
 [0x65] = { UNUSED                    },
 [0x66] = { CURSOR,  0x47, 0x6c, 0x6e }, /* Home */
 [0x67] = { CURSOR,  0x48, 0x75, 0x63 }, /* Up */
 [0x68] = { CURSOR,  0x49, 0x7d, 0x6f }, /* PgUp */
 [0x69] = { CURSOR,  0x4b, 0x6b, 0x61 }, /* Left */
 [0x6a] = { CURSOR,  0x4d, 0x74, 0x6a }, /* Right */
 [0x6b] = { CURSOR,  0x4f, 0x69, 0x65 }, /* End */
 [0x6c] = { CURSOR,  0x50, 0x72, 0x60 }, /* Down */
 [0x6d] = { CURSOR,  0x51, 0x7a, 0x6d }, /* PgDown */
 [0x6e] = { CURSOR,  0x52, 0x70, 0x67 }, /* Insert */
 [0x6f] = { CURSOR,  0x53, 0x71, 0x64 }, /* Delete */

 [0x70] = { UNUSED                    },
 [0x71] = { UNUSED                    },
 [0x72] = { UNUSED                    },
 [0x73] = { UNUSED                    },
 [0x74] = { UNUSED                    },
 [0x75] = { UNUSED                    },
 [0x76] = { UNUSED                    },
 [0x77] = { PAUSE,   0x00, 0x00, 0x62 }, /* Break / Pause */
 [0x78] = { UNUSED                    },
 [0x79] = { UNUSED                    },
 [0x7a] = { UNUSED                    },
 [0x7b] = { UNUSED                    },
 [0x7c] = { UNUSED                    },
 [0x7d] = { SPECIAL, 0x5b, 0x1f, /*?*/}, /* Left-Windows */ /* FIXME */
 [0x7e] = { SPECIAL, 0x5c, 0x27, /*?*/}, /* Right-Windows */ /* FIXME */
 [0x7f] = { SPECIAL, 0x5d, 0x2f, /*?*/}, /* Menu */ /* FIXME */
};

static void
kbd_reply2(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t byte;

	assert(cpssp->outrunning);
	assert(cpssp->outcount);

	byte = cpssp->outbuf[cpssp->outtail];
	cpssp->outtail = (cpssp->outtail + 1) % sizeof(cpssp->outbuf);
	cpssp->outcount--;
	cpssp->outrunning = 0;

	sig_ps2_send(cpssp->port_ps2, cpssp, byte);

	if (cpssp->outcount
	 && cpssp->state_clkrunning) {
		/* Sending/receiving a byte using 9600 Baud takes about 1ms. */
		cpssp->outrunning = 1;
		time_call_after(TIME_HZ / 1000, kbd_reply2, cpssp);
	}
}

static void
kbd_reply(struct cpssp *cpssp, unsigned char byte)
{
	if (! cpssp->outcount
	 && cpssp->state_clkrunning) {
		/* Sending/receiving a byte using 9600 Baud takes about 1ms. */
		cpssp->outrunning = 1;
		time_call_after(TIME_HZ / 1000, kbd_reply2, cpssp);
	}

	cpssp->outbuf[cpssp->outhead] = byte;
	cpssp->outhead = (cpssp->outhead + 1) % sizeof(cpssp->outbuf);
	cpssp->outcount++;
}

static void
kbd_reset(struct cpssp *cpssp)
{
	time_call_delete(kbd_reply2, cpssp);

	cpssp->scan_code_set = 2;

	cpssp->head = 0;
	cpssp->tail = 0;

	cpssp->outhead = 0;
	cpssp->outtail = 0;
	cpssp->outcount = 0;
	cpssp->outrunning = 0;

	cpssp->repeat = (TIME_HZ * 91) / 1000;
	cpssp->delay = (TIME_HZ * 500) / 1000;
}

static void
kbd_led_update(struct cpssp *cpssp)
{
	sig_boolean_set(cpssp->port_opt_scroll_led, cpssp,
			cpssp->status_scroll_lock);
	sig_boolean_set(cpssp->port_opt_num_led, cpssp,
			cpssp->status_num_lock);
	sig_boolean_set(cpssp->port_opt_caps_led, cpssp,
			cpssp->status_caps_lock);
}

static void
_kbd_command(struct cpssp *cpssp, uint8_t value)
{
	/*
	 * See PC Hardware (5. Auflage).
	 */
	if (cpssp->keyboard_command == 0) {
		switch (value) {
		case 0x00:	/* ? */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;

		case 0x05:	/* ? */
			kbd_reply(cpssp, KBD_REPLY_RESEND);
			break;

		case KBD_CMD_EX_ENABLE: /* 0xea */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			cpssp->keyboard_command = value;
			break;

		case KBD_CMD_SET_LEDS: /* 0xed */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			cpssp->keyboard_command = value;
			break;

		case KBD_CMD_ECHO: /* 0xee */
			kbd_reply(cpssp, KBD_CMD_ECHO);
			break;

		case KBD_CMD_SSCANSET: /* 0xf0 */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			cpssp->keyboard_command = value;
			break;

		case KBD_CMD_GETID: /* 0xf2 */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			if (0) {
				/* AT keyboard */
			} else {
				/* MF-II keyboard */
				kbd_reply(cpssp, 0xab);
				kbd_reply(cpssp, 0x83);
			}
			break;

		case KBD_CMD_SET_RATE: /* 0xf3 */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			cpssp->keyboard_command = value;
			break;

		case KBD_CMD_ENABLE: /* 0xf4 */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;

		case KBD_CMD_DISABLE: /* 0xf5 */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;

		case KBD_CMD_STD_ENABLE: /* 0xf6 */
			/* FIXME */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;

		case KBD_CMD_SAK_T: /* 0xf7 */
			/* FIXME */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;

		case KBD_CMD_SAK_MB: /* 0xf8 */
			/* FIXME */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;

		case KBD_CMD_SAK_M: /* 0xf9 */
			/* FIXME */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;

		case KBD_CMD_SAK_TMB: /* 0xfa */
			/* FIXME */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;

		case KBD_CMD_RESET: /* 0xff */
			kbd_reset(cpssp);
			kbd_reply(cpssp, KBD_REPLY_ACK);
			kbd_reply(cpssp, KBD_REPLY_POR);
			break;

		default:
			fprintf(stderr, "WARNING: keyboard: got command 0x%02x\n",
					value);
			kbd_reply(cpssp, KBD_REPLY_ACK);
			break;
		}

	} else {
		switch (cpssp->keyboard_command) {
		case KBD_CMD_EX_ENABLE: /* 0xea */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			/* ... - FIXME VOSSI */
			break;

		case KBD_CMD_SET_LEDS: /* 0xed */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			cpssp->status_caps_lock = (value >> 2) & 1;
			cpssp->status_num_lock = (value >> 1) & 1;
			cpssp->status_scroll_lock = (value >> 0) & 1;
			kbd_led_update(cpssp);
			break;

		case KBD_CMD_SSCANSET: /* 0xf0 */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			switch (value) {
			case 0: /* Read current setting. */
				kbd_reply(cpssp, cpssp->scan_code_set);
				break;

			case 1:
			case 2:
			case 3: /* Set scancode table. */
				if (value != 1
				 && value != 2) {
					/* FIXME */
					fprintf(stderr, "WARNING: scan_code_set set to %d.\n", value);
					value = 2;
				}
				cpssp->scan_code_set = value;
				break;

			default: /* Bad Parameter */
				/* What should we do here? - FIXME VOSSI */
				break;
			}
			break;

		case KBD_CMD_SET_RATE: /* 0xf3 */
			/* set repeat rate */
			kbd_reply(cpssp, KBD_REPLY_ACK);
			/* calculate delay and repeat rates */
			cpssp->delay = (TIME_HZ * 250 * (((value >> 5) & 0x03) + 1))
					/ 1000;
			cpssp->repeat = ((1 << ((value >> 3) & 0x03))
					* ((value & 0x07) + 8) * TIME_HZ) / 240;
			break;

		default:
			assert(0);
		}

		cpssp->keyboard_command = 0;
	}
}

static void
kbd_power_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_power = val;

	if (val) {
		kbd_reset(cpssp);

	} else {
		cpssp->status_scroll_lock = 0;
		cpssp->status_num_lock = 0;
		cpssp->status_caps_lock = 0;

		kbd_led_update(cpssp);
	}
}

static void
kbd_recv2(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	uint8_t byte;

	byte = cpssp->buffer[cpssp->tail];
	cpssp->tail = (cpssp->tail + 1) % sizeof(cpssp->buffer);

	_kbd_command(cpssp, byte);

	if (cpssp->tail != cpssp->head) {
		/* Sending/receiving a byte using 9600 Baud takes about 1ms. */
		time_call_after(TIME_HZ / 1000, kbd_recv2, cpssp);
	}
}

static void
kbd_recv(void *_cpssp, uint8_t byte)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->buffer[cpssp->head] = byte;
	cpssp->head = (cpssp->head + 1) % sizeof(cpssp->buffer);
	
	/* Sending/receiving a byte using 9600 Baud takes about 1msec. */
	time_call_after(TIME_HZ / 1000, kbd_recv2, cpssp);
}

static void
kbd_clkrunning(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;
	
	cpssp->state_clkrunning = val;

	if (! cpssp->state_clkrunning) {
		if (cpssp->outrunning) {
			/* Cancel running transmission. */
			time_call_delete(kbd_reply2, cpssp);
			cpssp->outrunning = 0;
		}
	} else {
		if (! cpssp->outrunning
		 && cpssp->outcount) {
			/* Start new transmission. */
			cpssp->outrunning = 1;
			time_call_after(TIME_HZ / 1000, kbd_reply2, cpssp);
		}
	}
}

static void
keyboard_generate(struct cpssp *cpssp, int key, int press)
{
	switch (table[key].type) {
	case NORMAL:
		/* Standard Key */
	standard:;
		switch (cpssp->scan_code_set) {
		case 1:
			if (press) {
				kbd_reply(cpssp, table[key].set1);
			} else {
				kbd_reply(cpssp, table[key].set1 | 0x80);
			}
			break;
		case 2:
			if (! press) {
				kbd_reply(cpssp, 0xf0);
			}
			kbd_reply(cpssp, table[key].set2);
			break;
		case 3:
			if (! press) {
				kbd_reply(cpssp, 0xf0);
			}
			kbd_reply(cpssp, table[key].set3);
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		break;

	case SPECIAL:
		/* Special Key */
		switch (cpssp->scan_code_set) {
		case 1:
			kbd_reply(cpssp, 0xe0);
			if (press) {
				kbd_reply(cpssp, table[key].set1);
			} else {
				kbd_reply(cpssp, table[key].set1 | 0x80);
			}
			break;
		case 2:
			kbd_reply(cpssp, 0xe0);
			if (! press) {
				kbd_reply(cpssp, 0xf0);
			}
			kbd_reply(cpssp, table[key].set2);
			break;
		case 3:
			goto standard;
		default:
			assert(0); /* Cannot happen. */
		}
		break;

	case CURSOR:
		/* Cursor Key */
		switch (cpssp->scan_code_set) {
		case 1:
			if ((cpssp->status_left_shift
			  || cpssp->status_right_shift)
			 && cpssp->status_num_lock) {
			base_case_1:;
				kbd_reply(cpssp, 0xe0);
				if (press) {
					kbd_reply(cpssp, table[key].set1);
				} else {
					kbd_reply(cpssp, table[key].set1 | 0x80);
				}
			} else if (cpssp->status_left_shift
				|| cpssp->status_right_shift) {
				if (press) {
					if (cpssp->status_left_shift) {
						kbd_reply(cpssp, 0xe0);
						kbd_reply(cpssp, 0x2a | 0x80);
					}
					if (cpssp->status_right_shift) {
						kbd_reply(cpssp, 0xe0);
						kbd_reply(cpssp, 0x36 | 0x80);
					}
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, table[key].set1);
				} else {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, table[key].set1 | 0x80);
					if (cpssp->status_right_shift) {
						kbd_reply(cpssp, 0xe0);
						kbd_reply(cpssp, 0x36);
					}
					if (cpssp->status_left_shift) {
						kbd_reply(cpssp, 0xe0);
						kbd_reply(cpssp, 0x2a);
					}
				}
			} else if (cpssp->status_num_lock) {
				if (press) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x2a);
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, table[key].set1);
				} else {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, table[key].set1 | 0x80);
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x2a | 0x80);
				}
			} else {
				goto base_case_1;
			}
			break;
		case 2:
			if ((cpssp->status_left_shift
			  || cpssp->status_right_shift)
			 && cpssp->status_num_lock) {
			base_case_2:;
				kbd_reply(cpssp, 0xe0);
				if (! press) {
					kbd_reply(cpssp, 0xf0);
				}
				kbd_reply(cpssp, table[key].set2);
			} else if (cpssp->status_left_shift
				|| cpssp->status_right_shift) {
				if (press) {
					if (cpssp->status_left_shift) {
						kbd_reply(cpssp, 0xe0);
						kbd_reply(cpssp, 0xf0);
						kbd_reply(cpssp, 0x12);
					}
					if (cpssp->status_right_shift) {
						kbd_reply(cpssp, 0xe0);
						kbd_reply(cpssp, 0xf0);
						kbd_reply(cpssp, 0x59);
					}
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, table[key].set2);
				} else {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xf0);
					kbd_reply(cpssp, table[key].set2);
					if (cpssp->status_right_shift) {
						kbd_reply(cpssp, 0xe0);
						kbd_reply(cpssp, 0x59);
					}
					if (cpssp->status_left_shift) {
						kbd_reply(cpssp, 0xe0);
						kbd_reply(cpssp, 0x12);
					}
				}
			} else if (cpssp->status_num_lock) {
				if (press) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x12);
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, table[key].set2);
				} else {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xf0);
					kbd_reply(cpssp, table[key].set2);
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xf0);
					kbd_reply(cpssp, 0x12);
				}
			} else {
				goto base_case_2;
			}
			break;
		case 3:
			goto standard;
		default:
			assert(0); /* Cannot happen. */
		}
		break;

	case KP_DIV:
		/* Keypad-/ */
		switch (cpssp->scan_code_set) {
		case 1:
			if (press) {
				if (cpssp->status_left_shift) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xaa);
				}
				if (cpssp->status_right_shift) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xb6);
				}
				kbd_reply(cpssp, 0xe0);
				kbd_reply(cpssp, 0x35);
			} else {
				kbd_reply(cpssp, 0xe0);
				kbd_reply(cpssp, 0x35 | 0x80);
				if (cpssp->status_right_shift) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x36);
				}
				if (cpssp->status_left_shift) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x2a);
				}
			}
			break;
		case 2:
			if (press) {
				if (cpssp->status_left_shift) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xf0);
					kbd_reply(cpssp, 0x12);
				}
				if (cpssp->status_right_shift) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xf0);
					kbd_reply(cpssp, 0x59);
				}
				kbd_reply(cpssp, 0xe0);
				kbd_reply(cpssp, 0x4a);
			} else {
				kbd_reply(cpssp, 0xe0);
				kbd_reply(cpssp, 0xf0);
				kbd_reply(cpssp, 0x4a);
				if (cpssp->status_right_shift) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x59);
				}
				if (cpssp->status_left_shift) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x12);
				}
			}
			break;
		case 3:
			goto standard;
		default:
			assert(0); /* Cannot happen. */
		}
		break;

	case SYSRQ:
		/* PrtSc / SysRq */
		switch (cpssp->scan_code_set) {
		case 1:
			if (cpssp->status_left_alt
			 || cpssp->status_right_alt) {
				if (press) {
					kbd_reply(cpssp, 0x54);
				} else {
					kbd_reply(cpssp, 0x54 | 0x80);
				}
			} else if (cpssp->status_left_shift
				|| cpssp->status_right_shift
			        || cpssp->status_left_control
				|| cpssp->status_right_control) {
				kbd_reply(cpssp, 0xe0);
				if (press) {
					kbd_reply(cpssp, 0x37);
				} else {
					kbd_reply(cpssp, 0x37 | 0x80);
				}
			} else {
				if (press) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x2a);
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x37);
				} else {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x37 | 0x80);
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x2a | 0x80);
				}
			}
			break;
		case 2:
			if (cpssp->status_left_alt
			 || cpssp->status_right_alt) {
				if (! press) {
					kbd_reply(cpssp, 0xf0);
				}
				kbd_reply(cpssp, 0x84);
			} else if (cpssp->status_left_shift
				|| cpssp->status_right_shift
			        || cpssp->status_left_control
				|| cpssp->status_right_control) {
				kbd_reply(cpssp, 0xe0);
				if (! press) {
					kbd_reply(cpssp, 0xf0);
				}
				kbd_reply(cpssp, 0x7c);
			} else {
				if (press) {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x12);
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0x7c);
				} else {
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xf0);
					kbd_reply(cpssp, 0x7c);
					kbd_reply(cpssp, 0xe0);
					kbd_reply(cpssp, 0xf0);
					kbd_reply(cpssp, 0x12);
				}
			}
			break;
		case 3:
			goto standard;
		default:
			assert(0); /* Cannot happen. */
		}
		break;

	case PAUSE:
		/* Pause */
		switch (cpssp->scan_code_set) {
		case 1:
			if (cpssp->status_left_control
			 || cpssp->status_right_control) {
				kbd_reply(cpssp, 0xe0);
				kbd_reply(cpssp, 0x46);
				kbd_reply(cpssp, 0xe0);
				kbd_reply(cpssp, 0x46 | 0x80);
			} else {
				kbd_reply(cpssp, 0xe1);
				kbd_reply(cpssp, 0x1d);
				kbd_reply(cpssp, 0x45);
				kbd_reply(cpssp, 0xe1);
				kbd_reply(cpssp, 0x1d | 0x80);
				kbd_reply(cpssp, 0x45 | 0x80);
			}
			break;
		case 2:
			if (! press) {
				kbd_reply(cpssp, 0xf0);
			}
			kbd_reply(cpssp, 0x62);
			break;
		case 3:
			goto standard;
		default:
			assert(0); /* Cannot happen. */
		}
		break;

	default:
		assert(0); /* Cannot happen. */
	}
}

static void
keyboard_repeat(void * _cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	keyboard_generate(cpssp, cpssp->repeated_key, 1);

	time_call_after(cpssp->repeat, keyboard_repeat, cpssp);
}

static int
keyboard_is_repeated(int key)
{
	return key != 0x2a /* Left-Shift */
	    && key != 0x36 /* Right-Shift */
	    && key != 0x3a /* CapsLock */
	    && key != 0x1d /* Left-Control */
	    && key != 0x61 /* Right-Control */
	    && key != 0x38 /* Left-Alt */
	    && key != 0x64 /* Right-Alt */
	    && key != 0x45 /* NumLock */
	    && key != 0x46 /* ScrollLock (?) */
	    && key != 0x7d /* Left-Windows */
	    && key != 0x7e /* Right-Windows */
	    && key != 0x77;/* Pause / Break */
}

static void
keyboard_event(void *_cpssp, int key, int press)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	assert(key < 128);

	if (! cpssp->state_power) {
		return;
	}

	if (keyboard_is_repeated(key)) {
		time_call_delete(keyboard_repeat, cpssp);
	}

	if (key == 0x2a) cpssp->status_left_shift = press;
	if (key == 0x36) cpssp->status_right_shift = press;
	if (key == 0x1d) cpssp->status_left_control = press;
	if (key == 0x61) cpssp->status_right_control = press;
	if (key == 0x38) cpssp->status_left_alt = press;
	if (key == 0x64) cpssp->status_right_alt = press;

	keyboard_generate(cpssp, key, press);

	if (press
	 && keyboard_is_repeated(key)) {
		cpssp->repeated_key = key;
		time_call_after(cpssp->delay, keyboard_repeat, cpssp);
	}
}

#define func(nr) \
static void \
keyboard_key ## nr ## _set(void *_cpssp, unsigned int val) \
{ \
	struct cpssp *cpssp = (struct cpssp *) _cpssp; \
	\
	keyboard_event(cpssp, nr, val); \
}

func(0) func(1) func(2) func(3)
func(4) func(5) func(6) func(7)
func(8) func(9) func(10) func(11)
func(12) func(13) func(14) func(15)
func(16) func(17) func(18) func(19)
func(20) func(21) func(22) func(23)
func(24) func(25) func(26) func(27)
func(28) func(29) func(30) func(31)
func(32) func(33) func(34) func(35)
func(36) func(37) func(38) func(39)
func(40) func(41) func(42) func(43)
func(44) func(45) func(46) func(47)
func(48) func(49) func(50) func(51)
func(52) func(53) func(54) func(55)
func(56) func(57) func(58) func(59)
func(60) func(61) func(62) func(63)
func(64) func(65) func(66) func(67)
func(68) func(69) func(70) func(71)
func(72) func(73) func(74) func(75)
func(76) func(77) func(78) func(79)
func(80) func(81) func(82) func(83)
func(84) func(85) func(86) func(87)
func(88) func(89) func(90) func(91)
func(92) func(93) func(94) func(95)
func(96) func(97) func(98) func(99)
func(100) func(101) func(102) func(103)
func(104) func(105) func(106) func(107)
func(108) func(109) func(110) func(111)
func(112) func(113) func(114) func(115)
func(116) func(117) func(118) func(119)
func(120) func(121) func(122) func(123)
func(124) func(125) func(126) func(127)

#undef func

void *
keyboard_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_ps2 *port_ps2,
	struct sig_boolean *port_key0,
	struct sig_boolean *port_key1,
	struct sig_boolean *port_key2,
	struct sig_boolean *port_key3,
	struct sig_boolean *port_key4,
	struct sig_boolean *port_key5,
	struct sig_boolean *port_key6,
	struct sig_boolean *port_key7,
	struct sig_boolean *port_key8,
	struct sig_boolean *port_key9,
	struct sig_boolean *port_key10,
	struct sig_boolean *port_key11,
	struct sig_boolean *port_key12,
	struct sig_boolean *port_key13,
	struct sig_boolean *port_key14,
	struct sig_boolean *port_key15,
	struct sig_boolean *port_key16,
	struct sig_boolean *port_key17,
	struct sig_boolean *port_key18,
	struct sig_boolean *port_key19,
	struct sig_boolean *port_key20,
	struct sig_boolean *port_key21,
	struct sig_boolean *port_key22,
	struct sig_boolean *port_key23,
	struct sig_boolean *port_key24,
	struct sig_boolean *port_key25,
	struct sig_boolean *port_key26,
	struct sig_boolean *port_key27,
	struct sig_boolean *port_key28,
	struct sig_boolean *port_key29,
	struct sig_boolean *port_key30,
	struct sig_boolean *port_key31,
	struct sig_boolean *port_key32,
	struct sig_boolean *port_key33,
	struct sig_boolean *port_key34,
	struct sig_boolean *port_key35,
	struct sig_boolean *port_key36,
	struct sig_boolean *port_key37,
	struct sig_boolean *port_key38,
	struct sig_boolean *port_key39,
	struct sig_boolean *port_key40,
	struct sig_boolean *port_key41,
	struct sig_boolean *port_key42,
	struct sig_boolean *port_key43,
	struct sig_boolean *port_key44,
	struct sig_boolean *port_key45,
	struct sig_boolean *port_key46,
	struct sig_boolean *port_key47,
	struct sig_boolean *port_key48,
	struct sig_boolean *port_key49,
	struct sig_boolean *port_key50,
	struct sig_boolean *port_key51,
	struct sig_boolean *port_key52,
	struct sig_boolean *port_key53,
	struct sig_boolean *port_key54,
	struct sig_boolean *port_key55,
	struct sig_boolean *port_key56,
	struct sig_boolean *port_key57,
	struct sig_boolean *port_key58,
	struct sig_boolean *port_key59,
	struct sig_boolean *port_key60,
	struct sig_boolean *port_key61,
	struct sig_boolean *port_key62,
	struct sig_boolean *port_key63,
	struct sig_boolean *port_key64,
	struct sig_boolean *port_key65,
	struct sig_boolean *port_key66,
	struct sig_boolean *port_key67,
	struct sig_boolean *port_key68,
	struct sig_boolean *port_key69,
	struct sig_boolean *port_key70,
	struct sig_boolean *port_key71,
	struct sig_boolean *port_key72,
	struct sig_boolean *port_key73,
	struct sig_boolean *port_key74,
	struct sig_boolean *port_key75,
	struct sig_boolean *port_key76,
	struct sig_boolean *port_key77,
	struct sig_boolean *port_key78,
	struct sig_boolean *port_key79,
	struct sig_boolean *port_key80,
	struct sig_boolean *port_key81,
	struct sig_boolean *port_key82,
	struct sig_boolean *port_key83,
	struct sig_boolean *port_key84,
	struct sig_boolean *port_key85,
	struct sig_boolean *port_key86,
	struct sig_boolean *port_key87,
	struct sig_boolean *port_key88,
	struct sig_boolean *port_key89,
	struct sig_boolean *port_key90,
	struct sig_boolean *port_key91,
	struct sig_boolean *port_key92,
	struct sig_boolean *port_key93,
	struct sig_boolean *port_key94,
	struct sig_boolean *port_key95,
	struct sig_boolean *port_key96,
	struct sig_boolean *port_key97,
	struct sig_boolean *port_key98,
	struct sig_boolean *port_key99,
	struct sig_boolean *port_key100,
	struct sig_boolean *port_key101,
	struct sig_boolean *port_key102,
	struct sig_boolean *port_key103,
	struct sig_boolean *port_key104,
	struct sig_boolean *port_key105,
	struct sig_boolean *port_key106,
	struct sig_boolean *port_key107,
	struct sig_boolean *port_key108,
	struct sig_boolean *port_key109,
	struct sig_boolean *port_key110,
	struct sig_boolean *port_key111,
	struct sig_boolean *port_key112,
	struct sig_boolean *port_key113,
	struct sig_boolean *port_key114,
	struct sig_boolean *port_key115,
	struct sig_boolean *port_key116,
	struct sig_boolean *port_key117,
	struct sig_boolean *port_key118,
	struct sig_boolean *port_key119,
	struct sig_boolean *port_key120,
	struct sig_boolean *port_key121,
	struct sig_boolean *port_key122,
	struct sig_boolean *port_key123,
	struct sig_boolean *port_key124,
	struct sig_boolean *port_key125,
	struct sig_boolean *port_key126,
	struct sig_boolean *port_key127,
	struct sig_boolean *port_opt_num_led,
	struct sig_boolean *port_opt_caps_led,
	struct sig_boolean *port_opt_scroll_led
)
{
	static const struct sig_boolean_funcs power_funcs = {
		.set = kbd_power_set,
	};
	static const struct sig_ps2_main_funcs funcs = {
		.recv = kbd_recv,
		.clkrunning = kbd_clkrunning,
	};

#define func(nr) \
	static const struct sig_boolean_funcs key ## nr ## _funcs = { \
		.set = keyboard_key ## nr ## _set, \
	}
	func(0); func(1); func(2); func(3);
	func(4); func(5); func(6); func(7);
	func(8); func(9); func(10); func(11);
	func(12); func(13); func(14); func(15);
	func(16); func(17); func(18); func(19);
	func(20); func(21); func(22); func(23);
	func(24); func(25); func(26); func(27);
	func(28); func(29); func(30); func(31);
	func(32); func(33); func(34); func(35);
	func(36); func(37); func(38); func(39);
	func(40); func(41); func(42); func(43);
	func(44); func(45); func(46); func(47);
	func(48); func(49); func(50); func(51);
	func(52); func(53); func(54); func(55);
	func(56); func(57); func(58); func(59);
	func(60); func(61); func(62); func(63);
	func(64); func(65); func(66); func(67);
	func(68); func(69); func(70); func(71);
	func(72); func(73); func(74); func(75);
	func(76); func(77); func(78); func(79);
	func(80); func(81); func(82); func(83);
	func(84); func(85); func(86); func(87);
	func(88); func(89); func(90); func(91);
	func(92); func(93); func(94); func(95);
	func(96); func(97); func(98); func(99);
	func(100); func(101); func(102); func(103);
	func(104); func(105); func(106); func(107);
	func(108); func(109); func(110); func(111);
	func(112); func(113); func(114); func(115);
	func(116); func(117); func(118); func(119);
	func(120); func(121); func(122); func(123);
	func(124); func(125); func(126); func(127);

#undef func

	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	/* Call */
	cpssp->port_ps2 = port_ps2->main;
	sig_ps2_main_connect(port_ps2->main, cpssp, &funcs);

	/* In */
	sig_boolean_connect_in(port_ps2->p5V, cpssp, &power_funcs);

#define func(nr) \
	sig_boolean_connect_in(port_key ## nr, cpssp, &key ## nr ## _funcs)

	func(0); func(1); func(2); func(3);
	func(4); func(5); func(6); func(7);
	func(8); func(9); func(10); func(11);
	func(12); func(13); func(14); func(15);
	func(16); func(17); func(18); func(19);
	func(20); func(21); func(22); func(23);
	func(24); func(25); func(26); func(27);
	func(28); func(29); func(30); func(31);
	func(32); func(33); func(34); func(35);
	func(36); func(37); func(38); func(39);
	func(40); func(41); func(42); func(43);
	func(44); func(45); func(46); func(47);
	func(48); func(49); func(50); func(51);
	func(52); func(53); func(54); func(55);
	func(56); func(57); func(58); func(59);
	func(60); func(61); func(62); func(63);
	func(64); func(65); func(66); func(67);
	func(68); func(69); func(70); func(71);
	func(72); func(73); func(74); func(75);
	func(76); func(77); func(78); func(79);
	func(80); func(81); func(82); func(83);
	func(84); func(85); func(86); func(87);
	func(88); func(89); func(90); func(91);
	func(92); func(93); func(94); func(95);
	func(96); func(97); func(98); func(99);
	func(100); func(101); func(102); func(103);
	func(104); func(105); func(106); func(107);
	func(108); func(109); func(110); func(111);
	func(112); func(113); func(114); func(115);
	func(116); func(117); func(118); func(119);
	func(120); func(121); func(122); func(123);
	func(124); func(125); func(126); func(127);

#undef func

	/* Out */
	cpssp->port_opt_num_led = port_opt_num_led;
	cpssp->port_opt_caps_led = port_opt_caps_led;
	cpssp->port_opt_scroll_led = port_opt_scroll_led;

	return cpssp;
}

void
keyboard_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	free(cpssp);
}
