/* $Id: floppy_gen_floppydrive.c,v 1.51 2009-04-25 15:12:55 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include "fixme.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue-log.h"
#include "glue-main.h"
#include "glue-shm.h"
#include "glue-storage.h"

#include "floppy_gen_floppydrive.h"

#define COMP "floppy_gen_floppydrive"

struct cpssp {
	/*
	 * Config
	 */
	unsigned int model; /* FIXME */
	unsigned int unit; /* FIXME */
#if 0
	unsigned int type;
#endif

	/*
	 * Ports
	 */
	struct sig_shugart_bus *shugart_bus;
	struct sig_boolean_or *shugart_bus_trk0;
	struct sig_boolean_or *shugart_bus_dskchg;
	struct sig_boolean *port_opt_busy_led;

	/*
	 * State
	 */
	char path[1024 + 1]; /* Media #, Path, '\0' */
	struct storage media;

	unsigned int select;	/* Is drive selected? */
	unsigned int motor;	/* Is motor spinning? */
	unsigned int hds;	/* Selected head */
	unsigned int changed;	/* Disk changed? */
	unsigned int wp;	/* Write protected? */

	unsigned char heads;	/* number of sides */
	unsigned char tracks;	/* tracks per side */
	unsigned char sectors;	/* sectors per track */
	
	enum {
		DRIVE_NONE,
		DRIVE_READ,
		DRIVE_READ_DONE,
		DRIVE_WRITE,
		DRIVE_WRITE_DONE,
	} op;			/* Operation */
	unsigned char track;	/* track position */
	unsigned int angle;	/*
				 * 0: ID sector 1 (Index hole = 1)
				 * 1: Data sector 1
				 * 2: ID sector 2
				 * 2: Data sector 2
				 * ...
				 * 2*N+0: ID sector N+1
				 * 2*N+1: Data sector N+1
				 */

	unsigned char buffer[16*1024];
	int bufsize;
};

/*forward*/ static void
floppy_gen_floppydrive_process(struct cpssp *cpssp);

static void
floppy_gen_floppydrive_step(struct cpssp *cpssp)
{
again:	;
	sig_shugart_bus_index_set(cpssp->shugart_bus, cpssp, cpssp->angle == 0);

	switch (cpssp->op) {
	case DRIVE_READ:
		if (cpssp->angle % 2 == 0) {
			/* Read ID. */
			/*
			 * This really will read the sector.
			 * But we need to know whether the sector exists.
			 */
			floppy_gen_floppydrive_process(cpssp);
			return;

		} else {
			/* Read data. */
			static unsigned char zero[512] = { 0, 0 /* ... */ };

			cpssp->angle = (cpssp->angle + 1) % 36;
			cpssp->op = DRIVE_READ_DONE;

			if (cpssp->bufsize < 0) {
				/* Read error. */
				assert(0);
			} else if (cpssp->bufsize == 0) {
				/* No media. */
				sig_shugart_bus_readdata(cpssp->shugart_bus, cpssp,
						zero, sizeof(zero));
			} else {
				/* Data read. */
				sig_shugart_bus_readdata(cpssp->shugart_bus, cpssp,
						cpssp->buffer, cpssp->bufsize);
			}
		}
		break;

	case DRIVE_WRITE:
		if (cpssp->angle % 2 == 0) {
			/* Write ID. */
			/*
			 * We don't write any IDs.
			 */
			/* Nothing to do... */

			cpssp->angle = (cpssp->angle + 1) % 36;
			cpssp->op = DRIVE_WRITE_DONE;

		} else {
			/* Write data. */
			floppy_gen_floppydrive_process(cpssp);
			return;
		}
		break;

	default:
		assert(0);
	}

	switch (cpssp->op) {
	case DRIVE_READ_DONE:
	case DRIVE_WRITE_DONE:
		cpssp->op = DRIVE_NONE;
		break;
	case DRIVE_READ:
	case DRIVE_WRITE:
		goto again;
	default:
		assert(0);
	}
}

static void
floppy_gen_floppydrive_interrupt(struct cpssp *cpssp, unsigned int retval)
{
	switch (cpssp->op) {
	case DRIVE_READ:
		cpssp->op = DRIVE_READ_DONE;
		cpssp->angle = (cpssp->angle + 1) % 36;

		cpssp->bufsize = retval;

		if (cpssp->bufsize < 0) {
			/* Read error. */
			assert(0);
		} else if (cpssp->bufsize == 0) {
			/* No media. */
			sig_shugart_bus_readid(cpssp->shugart_bus, cpssp,
					0, 0);
		} else {
			/* ID read. */
			sig_shugart_bus_readid(cpssp->shugart_bus, cpssp,
					cpssp->track, cpssp->angle / 2 + 1);
		}
		break;

	case DRIVE_WRITE:
		/* No feedback. */
		cpssp->op = DRIVE_WRITE_DONE;
		cpssp->angle = (cpssp->angle + 1) % 36;
		break;

	default:
		assert(0); /* Cannot happen. */
	}

	switch (cpssp->op) {
	case DRIVE_READ_DONE:
	case DRIVE_WRITE_DONE:
		cpssp->op = DRIVE_NONE;
		break;
	case DRIVE_READ:
	case DRIVE_WRITE:
		floppy_gen_floppydrive_step(cpssp);
		break;
	default:
		assert(0);
	}
}

static void
floppy_gen_floppydrive_change(void *_cpssp, const char *path)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	/*
	 * Get new media path.
	 */
	strcpy(cpssp->path, path);

	/*
	 * Close old image (if any).
	 */
	if (cpssp->media.path[0]) {
		storage_close(&cpssp->media);
	}

	/*
	 * Change path.
	 */
	storage_change(&cpssp->media, cpssp->path);

	/*
	 * Open new image (if any).
	 */
	if (cpssp->media.path[0]) {
		if (storage_open(&cpssp->media, 1) < 0) {
			/* Try read-only. */
			if (storage_open(&cpssp->media, 0) < 0) {
				cpssp->media.path[0] = '\0';
			} else {
				cpssp->wp = 1;
			}
		} else {
			cpssp->wp = 0;
		}
	}

	cpssp->changed = 1;
	sig_boolean_or_set(cpssp->shugart_bus_dskchg, cpssp,
			cpssp->changed && cpssp->select);

	cpssp->angle = 0;
	cpssp->op = DRIVE_NONE;
}

static void
floppy_gen_floppydrive_select_set(void *s, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	cpssp->select = val;

	sig_boolean_or_set(cpssp->shugart_bus_trk0, cpssp,
			val && cpssp->track == 0);
	sig_boolean_or_set(cpssp->shugart_bus_dskchg, cpssp,
			val && cpssp->changed);
	if(cpssp->select) {
		sig_shugart_bus_wp_set(cpssp->shugart_bus, cpssp, cpssp->wp);
	}
}

static void
floppy_gen_floppydrive_motor_set(void *s, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	cpssp->motor = val;

	sig_boolean_set(cpssp->port_opt_busy_led, cpssp, val);
}

static void
floppy_gen_floppydrive_hds_set(void *s, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	cpssp->hds = val;
}

static void
floppy_gen_floppydrive_step_in(void *s)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (! cpssp->select) {
		return;
	}

	if (cpssp->track < 79) {
		cpssp->track++;
	}

	if(cpssp->changed && cpssp->media.path[0]) {
		cpssp->changed = 0;
		sig_boolean_or_set(cpssp->shugart_bus_dskchg, cpssp, 0);
	}
	sig_boolean_or_set(cpssp->shugart_bus_trk0, cpssp, cpssp->track == 0);
}

static void
floppy_gen_floppydrive_step_out(void *s)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (! cpssp->select) {
		return;
	}

	if (0 < cpssp->track) {
		cpssp->track--;
	}

	if(cpssp->changed && cpssp->media.path[0]) {
		cpssp->changed = 0;
		sig_boolean_or_set(cpssp->shugart_bus_dskchg, cpssp, 0);
	}
	sig_boolean_or_set(cpssp->shugart_bus_trk0, cpssp, cpssp->track == 0);
}

static void
floppy_gen_floppydrive_read_start(void *s)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (! cpssp->select) {
		return;
	}

	cpssp->bufsize = 512; /* FIXME VOSSI */
	switch (cpssp->op) {
	case DRIVE_NONE:
		cpssp->op = DRIVE_READ;
		floppy_gen_floppydrive_step(cpssp);
		break;
	case DRIVE_READ_DONE:
	case DRIVE_WRITE_DONE:
		cpssp->op = DRIVE_READ;
		break;
	default:
		assert(0);
	}
}

static void
floppy_gen_floppydrive_writedata(
	void *s,
	unsigned char *buf,
	unsigned int bufsize
)
{
	struct cpssp *cpssp = (struct cpssp *) s;

	if (! cpssp->select) {
		return;
	}

	memcpy(cpssp->buffer, buf, bufsize);
	cpssp->bufsize = bufsize;
	switch (cpssp->op) {
	case DRIVE_NONE:
		cpssp->op = DRIVE_WRITE;
		floppy_gen_floppydrive_step(cpssp);
		break;
	case DRIVE_READ_DONE:
	case DRIVE_WRITE_DONE:
		cpssp->op = DRIVE_WRITE;
		break;
	default:
		assert(0);
	}
}

void
floppy_gen_floppydrive_init(
	unsigned int nr,
	struct sig_power_device *port_power,
	struct sig_shugart_bus *port_shugart,
	struct sig_boolean *port_opt_busy_led,
	struct sig_string *port_change
)
{
	static const struct sig_shugart_bus_funcs shugart_funcs = {
		.motor_set = floppy_gen_floppydrive_motor_set,
		.select_set = floppy_gen_floppydrive_select_set,
		.hds_set = floppy_gen_floppydrive_hds_set,
		.step_in = floppy_gen_floppydrive_step_in,
		.step_out = floppy_gen_floppydrive_step_out,
		.read_start = floppy_gen_floppydrive_read_start,
		.writedata = floppy_gen_floppydrive_writedata,
	};
	static const struct sig_string_funcs change_funcs = {
		.set = floppy_gen_floppydrive_change,
	};
	struct cpssp *cpssp;
	
	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	cpssp->shugart_bus = port_shugart;
	sig_shugart_bus_connect_drive(cpssp->shugart_bus, cpssp,
			cpssp->unit, &shugart_funcs);
	cpssp->shugart_bus_trk0 = port_shugart->trk0;
	sig_boolean_or_connect_out(cpssp->shugart_bus_trk0, cpssp, 0);
	cpssp->shugart_bus_dskchg = port_shugart->dskchg;
	sig_boolean_or_connect_out(cpssp->shugart_bus_dskchg, cpssp, 0);

	cpssp->port_opt_busy_led = port_opt_busy_led;
	sig_boolean_connect_out(port_opt_busy_led, cpssp, 0);

	sig_string_connect(port_change, cpssp, &change_funcs);

	storage_init(&cpssp->media);
}

static unsigned int
floppy_gen_floppydrive_process_rw_sector(struct cpssp *cpssp, unsigned int wflag)
{
	unsigned long pos;
	unsigned int ret;

	pos = (((cpssp->track * 2) + cpssp->hds) * cpssp->sectors + cpssp->angle / 2)
			* 512;

	if(!wflag || !cpssp->wp) {
		ret = storage_read_write(wflag, &cpssp->media,
				cpssp->buffer, pos, cpssp->bufsize);
	} else {
		ret = cpssp->bufsize;
	}
	return ret;
}

static void
floppy_gen_floppydrive_process(struct cpssp *cpssp)
{
	unsigned int out;

	if (cpssp->media.path[0]) {
		switch (cpssp->op) {
		case DRIVE_READ:
			out = floppy_gen_floppydrive_process_rw_sector(cpssp, IO_READ);
			break;
		case DRIVE_WRITE:
			out = floppy_gen_floppydrive_process_rw_sector(cpssp, IO_WRITE);
			break;
		default:
			assert(0);
		}
	} else {
		/* No media. */
		out = 0;
	}

	floppy_gen_floppydrive_interrupt(cpssp, out);
}

extern unsigned char floppydrive[4]; /* FIXME */

void
floppy_gen_floppydrive_create(
	unsigned int nr,
	const char *name,
	const char *model,
	const char *unit
)
{
	static unsigned int count = 0;
	static unsigned int unit_default = 0;
	struct cpssp *cpssp;

	shm_create(COMP, nr, sizeof(*cpssp));
	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	if (model == NULL
	 || atoi(model) == -1) {
		cpssp->model = 4;
	} else {
		cpssp->model = strtoul(model, NULL, 0);
	}
	floppydrive[count++] = cpssp->model; /* FIXME */
	if (unit == NULL
	 || atoi(unit) == -1) {
		cpssp->unit = unit_default;
		unit_default++;
	} else {
		cpssp->unit = strtoul(unit, NULL, 0);
	}

	cpssp->hds = 0;
	cpssp->track = 0;
	cpssp->angle = 0;
	cpssp->changed = 1;
	cpssp->wp = 0;

	cpssp->heads = 2;
	cpssp->tracks = 80;
	cpssp->sectors = 18;

	storage_create(&cpssp->media,
		"",	/* No media inserted, yet. */
		0,	/* No write access needed, yet. */
		(const char *) 0, /* No image to copy. */
		0,	/* Image size to create. */
		512,	/* Blocksize. */
		0,	/* Don't create image. */
		0,	/* No cow. */
		0	/* No sync. */
	);

	shm_unmap(cpssp, sizeof(*cpssp));
}

void
floppy_gen_floppydrive_destroy(unsigned int nr)
{
	struct cpssp *cpssp;

	cpssp = shm_map(COMP, nr, sizeof(*cpssp), 0);

	storage_destroy(&cpssp->media);

	shm_unmap(cpssp, sizeof(*cpssp));
	shm_destroy(COMP, nr);
}
