/* $Id: sig_cs.c,v 1.15 2009-01-27 17:06:41 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "fixme.h"

#include "glue-shm.h"

#include "sig_cs.h"

int
sig_cs_readb(
	struct sig_cs *b,
	void *s,
	uint8_t *valp,
	unsigned long addr
)
{
	unsigned int nr;
	int (*func)(void *, uint8_t *, unsigned long);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			*valp = 0xff;
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->readb;
		if (func != 0 && func(b->member[nr].s, valp, addr) == 0) {
			return 0;
		}
	}
}

int
sig_cs_readw(
	struct sig_cs *b,
	void *s,
	uint16_t *valp,
	unsigned long addr
)
{
	unsigned int nr;
	int (*func)(void *, uint16_t *, unsigned long);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			*valp = 0xffff;
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->readw;
		if (func != 0 && func(b->member[nr].s, valp, addr) == 0) {
			return 0;
		}
	}
}

int
sig_cs_readl(
	struct sig_cs *b,
	void *s,
	uint32_t *valp,
	unsigned long addr
)
{
	unsigned int nr;
	int (*func)(void *, uint32_t *, unsigned long);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			*valp = 0xffffffff;
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->readl;
		if (func != 0 && func(b->member[nr].s, valp, addr) == 0) {
			return 0;
		}
	}
}

int
sig_cs_writeb(
	struct sig_cs *b,
	void *s,
	uint8_t val,
	unsigned long addr
)
{
	unsigned int nr;
	int (*func)(void *, uint8_t, unsigned long);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->writeb;
		if (func != 0 && func(b->member[nr].s, val, addr) == 0) {
			return 0;
		}
	}
}

int
sig_cs_writew(
	struct sig_cs *b,
	void *s,
	uint16_t val,
	unsigned long addr
)
{
	unsigned int nr;
	int (*func)(void *, uint16_t, unsigned long);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->writew;
		if (func != 0 && func(b->member[nr].s, val, addr) == 0) {
			return 0;
		}
	}
}

int
sig_cs_writel(
	struct sig_cs *b,
	void *s,
	uint32_t val,
	unsigned long addr
)
{
	unsigned int nr;
	int (*func)(void *, uint32_t, unsigned long);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->writel;
		if (func != 0 && func(b->member[nr].s, val, addr) == 0) {
			return 0;
		}
	}
}

int
sig_cs_read(
	struct sig_cs *b,
	void *s,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	unsigned int nr;
	int (*func)(void *, uint32_t, unsigned int, uint32_t *);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->read;
		if (func
		 && func(b->member[nr].s, addr, bs, valp) == 0) {
			return 0;
		}
	}
}

int
sig_cs_write(
	struct sig_cs *b,
	void *s,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	unsigned int nr;
	int (*func)(void *, uint32_t, unsigned int, uint32_t);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->write;
		if (func
		 && func(b->member[nr].s, addr, bs, val) == 0) {
			return 0;
		}
	}
}

int
sig_cs_map(
	struct sig_cs *b,
	void *s,
	unsigned long addr,
	unsigned int len,
	char **haddr_mr_p,
	char **haddr_mw_p
)
{
	unsigned int nr;
	int (*func)(void *, unsigned long, unsigned int, char **, char **);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->map;
		if (func
		 && func(b->member[nr].s, addr, len,
				haddr_mr_p, haddr_mw_p) == 0) {
			return 0;
		}
	}
}

int
sig_cs_unmap(struct sig_cs *b, void *s, unsigned long pa, unsigned long len)
{
	unsigned int nr;
	int (*func)(void *, unsigned long, unsigned long);

	for (nr = 0; ; nr++) {
		if (nr == b->nmembers) {
			return 1;
		}
		if (b->member[nr].s == s) {
			continue;
		}
		func = b->member[nr].f->unmap;
		if (func && func(b->member[nr].s, pa, len) == 0) {
			return 0;
		}
	}
}

void
sig_cs_connect(
	struct sig_cs *b,
	void *s,
	const struct sig_cs_funcs *f
)
{
	assert(b);
	assert(b->nmembers < sizeof(b->member) / sizeof(b->member[0]));

	b->member[b->nmembers].s = s;
	b->member[b->nmembers].f = f;
	b->nmembers++;
}

struct sig_cs *
sig_cs_init(const char *name, int nr)
{
	struct sig_cs *bus;

	bus = shm_map(name, nr, sizeof(*bus), 0);

	bus->nmembers = 0;

	return bus;
}

static int
sig_cs_s0_readb(
	void *_f,
	uint8_t *valp,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_readb(f->s1, f, valp, addr);
}

static int
sig_cs_s0_readw(
	void *_f,
	uint16_t *valp,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_readw(f->s1, f, valp, addr);
}

static int
sig_cs_s0_readl(
	void *_f,
	uint32_t *valp,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_readl(f->s1, f, valp, addr);
}

static int
sig_cs_s0_writeb(
	void *_f,
	uint8_t val,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_writeb(f->s1, f, val, addr);
}

static int
sig_cs_s0_writew(
	void *_f,
	uint16_t val,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_writew(f->s1, f, val, addr);
}

static int
sig_cs_s0_writel(
	void *_f,
	uint32_t val,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_writel(f->s1, f, val, addr);
}

static int
sig_cs_s0_read(
	void *_f,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_read(f->s1, f, addr, bs, valp);
}

static int
sig_cs_s0_write(
	void *_f,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_write(f->s1, f, addr, bs, val);
}

static int
sig_cs_s0_map(
	void *_f,
	unsigned long addr,
	unsigned int len,
	char **haddr_mr_p,
	char **haddr_mw_p
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_map(f->s1, f, addr, len, haddr_mr_p, haddr_mw_p);
}

static int
sig_cs_s0_unmap(
	void *_f,
	unsigned long addr,
	unsigned long len
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_unmap(f->s1, f, addr, len);
}

static int
sig_cs_s1_readb(
	void *_f,
	uint8_t *valp,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_readb(f->s0, f, valp, addr);
}

static int
sig_cs_s1_readw(
	void *_f,
	uint16_t *valp,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_readw(f->s0, f, valp, addr);
}

static int
sig_cs_s1_readl(
	void *_f,
	uint32_t *valp,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_readl(f->s0, f, valp, addr);
}

static int
sig_cs_s1_writeb(
	void *_f,
	uint8_t val,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_writeb(f->s0, f, val, addr);
}

static int
sig_cs_s1_writew(
	void *_f,
	uint16_t val,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_writew(f->s0, f, val, addr);
}

static int
sig_cs_s1_writel(
	void *_f,
	uint32_t val,
	unsigned long addr
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_writel(f->s0, f, val, addr);
}

static int
sig_cs_s1_read(
	void *_f,
	uint32_t addr,
	unsigned int bs,
	uint32_t *valp
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_read(f->s0, f, addr, bs, valp);
}

static int
sig_cs_s1_write(
	void *_f,
	uint32_t addr,
	unsigned int bs,
	uint32_t val
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_write(f->s0, f, addr, bs, val);
}

static int
sig_cs_s1_map(
	void *_f,
	unsigned long addr,
	unsigned int len,
	char **haddr_mr_p,
	char **haddr_mw_p
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_map(f->s0, f, addr, len, haddr_mr_p, haddr_mw_p);
}

static int
sig_cs_s1_unmap(
	void *_f,
	unsigned long addr,
	unsigned long len
)
{
	struct sig_cs_merge *f = (struct sig_cs_merge *) _f;

	return sig_cs_unmap(f->s0, f, addr, len);
}

struct sig_cs_merge *
sig_cs_merge(
	struct sig_cs *s0,
	struct sig_cs *s1
)
{
	static const struct sig_cs_funcs s0_funcs = {
		.readb = sig_cs_s0_readb,
		.readw = sig_cs_s0_readw,
		.readl = sig_cs_s0_readl,
		.writeb = sig_cs_s0_writeb,
		.writew = sig_cs_s0_writew,
		.writel = sig_cs_s0_writel,

		.read = sig_cs_s0_read,
		.write = sig_cs_s0_write,

		.map = sig_cs_s0_map,
		.unmap = sig_cs_s0_unmap,
	};
	static const struct sig_cs_funcs s1_funcs = {
		.readb = sig_cs_s1_readb,
		.readw = sig_cs_s1_readw,
		.readl = sig_cs_s1_readl,
		.writeb = sig_cs_s1_writeb,
		.writew = sig_cs_s1_writew,
		.writel = sig_cs_s1_writel,

		.read = sig_cs_s1_read,
		.write = sig_cs_s1_write,

		.map = sig_cs_s1_map,
		.unmap = sig_cs_s1_unmap,
	};
	struct sig_cs_merge *m;

	m = malloc(sizeof(*m));
	assert(m);

	m->s0 = s0;
	sig_cs_connect(s0, m, &s0_funcs);
	m->s1 = s1;
	sig_cs_connect(s1, m, &s1_funcs);

	return m;
}

void
sig_cs_split(struct sig_cs_merge *m)
{
	fixme();
}

void
sig_cs_create(const char *name, int nr)
{
	shm_create(name, nr, sizeof(struct sig_cs));
}

void
sig_cs_destroy(const char *name, int nr)
{
	shm_destroy(name, nr);
}
