/* $Id: glue-audio_file.c,v 1.6 2009-01-27 17:06:40 potyra Exp $ 
 *
 * file writing audio output plugin for FAUmachine
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>

#include "glue-audio_internal.h"

static ao_info_t info = 
{
	"File writing audio output",
	"file",
	"Matthias Sand",
	""
};

LIBAO_EXTERN(file);

static const struct _RiffWaveChunk {
	uint32_t chunkID;
	int32_t  fileSize;
	uint32_t riffType;
} riffWaveChunk = {
	.chunkID  = 0x46464952,	/* 'RIFF' */
	.fileSize = 4,
	.riffType = 0x45564157,	/* 'WAVE' */
};

static const struct _FormatChunk {
	uint32_t chunkID;
	int32_t  chunkSize;
	int16_t  wFormatTag;
	uint16_t wChannels;
	uint32_t dwSamplesPerSec;
	uint32_t dwAvgBytesPerSec;
	uint16_t wBlockAlign;
	uint16_t wBitsPerSample;
} formatChunk = {
	.chunkID          = 0x20746d66,	/* 'fmt ' */
	.chunkSize        = 16,
	.wFormatTag       = 0x0001,	/* PCM */ 
	.wChannels        = GLUE_AUDIO_CHANNELS,
	.dwSamplesPerSec  = GLUE_AUDIO_RATE,
	.dwAvgBytesPerSec = GLUE_AUDIO_RATE
			    * GLUE_AUDIO_CHANNELS
			    * GLUE_AUDIO_FORMAT_BYTES,
	.wBlockAlign      = GLUE_AUDIO_CHANNELS
			    * GLUE_AUDIO_FORMAT_BYTES,
	.wBitsPerSample   = GLUE_AUDIO_FORMAT_BITS,
};

static const struct _DataChunkHeader {
	uint32_t chunkID;
	int32_t  chunkSize;
} dataChunkHeader = {
	.chunkID   = 0x61746164,	/* 'data' */
	.chunkSize = 0,
};

#define DATA_OFFSET sizeof(struct _RiffWaveChunk)	\
		    + sizeof(struct _FormatChunk)	\
		    + sizeof(struct _DataChunkHeader)

#define BUFFERSIZE 65536

static FILE *fp = NULL;

static void
update_filesize(void)
{
	int32_t total_size;
	int32_t size;

	total_size = ftell(fp);

	size = total_size - 8;
	fseek(fp, 4, SEEK_SET);
	fwrite(&size, sizeof(size), 1, fp);

	size = total_size - DATA_OFFSET;
	fseek(fp, DATA_OFFSET - 4, SEEK_SET);
	fwrite(&size, sizeof(size), 1, fp);

	fseek(fp, 0, SEEK_END);
}

/* open & setup audio device
 * return: 1=success 0=fail */
static int
init(void)
{
	static int auto_nr = 0;
	char path[32];
	unsigned int nr;

	nr = auto_nr;
	do {
		auto_nr = (auto_nr + 1) % 1000;
		sprintf(path, "audio-%03d.wav", auto_nr);
		if (access(path, F_OK) != 0) {
			break;
		}
	} while (nr != auto_nr);

	if (nr == auto_nr) {
		return 0;
	}
	nr = auto_nr;

	sprintf(path, "audio-%03d.wav", nr);

	fp = fopen(path, "w");

	if (fp == NULL) {
		return 0;
	}

	fwrite(&riffWaveChunk, sizeof(struct _RiffWaveChunk), 1, fp);
	fwrite(&formatChunk, sizeof(struct _FormatChunk), 1, fp);
	fwrite(&dataChunkHeader, sizeof(struct _DataChunkHeader), 1, fp);
	update_filesize();

	return 1;
}

/* close audio device */
static void
uninit(int immed)
{
	if (fp) {
		fclose(fp);
		fp = NULL;
	}
}

/* return: how many bytes can be played without blocking */
static int
get_space()
{
	return BUFFERSIZE;
}

/* plays 'len' bytes of 'data'
 * return: number of bytes played */
static int
play(void *data, int len)
{
	assert(fp);

	fwrite(data, 1, len, fp);
	update_filesize();

	return len;
}

/* return: delay in seconds between first and last sample */
static float
get_delay(void)
{
	return 0.0;
}
