/* $Id: kbd.c,v 1.61 2009-01-28 14:37:58 potyra Exp $ 
 *
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "build_config.h"
#include "compiler.h"

#define DEBUG	0

#define KBD_CONTROL_KBDINT	0x01
#define KBD_CONTROL_AUXINT	0x02
#define KBD_CONTROL_KBDDIS	0x10
#define KBD_CONTROL_AUXDIS	0x20
#define KBD_CONTROL_XLATE	0x40

#define KBD_CCMD_WRITE_MODE	0x60
#define KBD_CCMD_PLSE		0xF0 /* pulse: bits 3-0 select which bit */

/* ==================== RUNTIME_RM ==================== */
#if defined(RUNTIME_RM)

CODE16;

#include "assert.h"
#include "stdio.h"
#include "kbd.h"
#include "io.h"
#include "var.h"
#include "const.h"
#include "segment.h"
#include "ptrace.h"
#include "kbd.h"

int
kbd_keyavailable(void)
{
	return (var_get(kbd_head) != var_get(kbd_tail));
}

static short
dequeue_key(unsigned char *code, unsigned char *ascii, short incr)
{
	unsigned short tail;
	unsigned short head;

	tail = var_get(kbd_tail);
	head = var_get(kbd_head);

	if (tail == head) {
		*code = '\0';
		*ascii = '\0';
		return 0;
	}

	*ascii = get_byte(VAR_SEG, tail + 0);
	*code  = get_byte(VAR_SEG, tail + 1);

	if (incr) {
		tail += 2;
		if (var_get(kbd_bend) <= tail) {
			tail = var_get(kbd_bstart);
		}
		var_put(kbd_tail, tail);
	}

	return 1;
}

unsigned short
kbd_getkey(char removekey)
{
	unsigned char code;
	unsigned char ascii;
	unsigned short keyval;

	dequeue_key(&code, &ascii, removekey);

	keyval = ((unsigned short) code << 8) | ascii;

	return keyval;
}

void
kbd_system_reset(void)
{
		outb(KBD_CCMD_PLSE | 0x0e, 0x64);
		/* NOT REACHED */
}

static short
enqueue_key(unsigned char code, unsigned char ascii)
{
	unsigned short tail;
	unsigned short tmp;
	unsigned short head;

	tail = var_get(kbd_tail);
	head = var_get(kbd_head);
	tmp = head;
	head += 2;
	if (var_get(kbd_bend) <= head) {
		head = var_get(kbd_bstart);
	}

	if (head == tail) {
		/* Beep */
		/* FIXME VOSSI */
		return 0;
	}

	put_byte(VAR_SEG, tmp + 0, ascii);
	put_byte(VAR_SEG, tmp + 1, code);

	var_put(kbd_head, head);

	return 1;
}

/*
 * KBD:
 *
 * Get keycode from buffer.
 * Keycode is removed from buffer.
 *
 * In:	AH	= 00h
 *
 * Out:	AL	= ASCII code
 *	AH	= scan code
 */
static void
bios_16_00xx(struct regs *regs)
{
	/* Wait while no key in buffer. */
	while (! kbd_keyavailable()) {
		asm volatile (
			"sti\n\t"
			"hlt\n\t"
			"cli\n\t"
		);
	}

	/*
	 * Remove keystroke from buffer.
	 */
	AX = kbd_getkey(1);
}

/*
 * KBD:
 *
 * Check for keycode in buffer.
 * Keycode is *not* removed from buffer.
 *
 * In:	AH	= 01h
 *
 * Out:	AL	= ASCII character
 * 	AH	= scan code
 * 	F
 * 	 zero	= 0: key available; 1: no key available
 */
static void
bios_16_01xx(struct regs *regs)
{
	if (! kbd_keyavailable()) {
		/*
		 * No keystroke available.
		 */
		F |= 1 << 6;	/* Set zero flag. */

	} else {
		/*
		 * Peek keystroke.
		 */
		AX = kbd_getkey(0);
		F &= ~(1 << 6);	/* Clear zero flag. */
	}
}

/*
 * KBD: Get shift key states.
 *
 * In:	AH	= 02h
 *
 * Out:	AL
 * 	 Bit0	= Right shift key pressed
 * 	 Bit1	= Left shift key pressed
 * 	 Bit2	= Ctrl key pressed
 * 	 Bit3	= Alt key pressed
 * 	 Bit4	= ScrollLock mode active
 * 	 Bit5	= NumLock mode active
 * 	 Bit6	= CapsLock mode active
 * 	 Bit7	= Insert mode active
 */
static void
bios_16_02xx(struct regs *regs)
{
	AL = var_get(kbd_shift0);
}

/*
 * Set keyboard rate.
 */
static void
bios_16_03xx(struct regs *regs)
{
	/* FIXME VOSSI */
}

/*
 * Store key-stroke into buffer.
 */
static void
bios_16_05xx(struct regs *regs)
{
	if (! enqueue_key(CH, CL)) {
		AL = 1;
	} else {
		AL = 0;
	}
}

/*
 * KBD:
 *
 * get keyboard functionality
 *
 * In:	AH = 09h
 *
 * Out:	AL = supported keyboard functions
 *	 Bit0	= INT 16/AX=0300h supported
 *	 Bit1	= INT 16/AX=0304h supported
 *	 Bit2	= INT 16/AX=0305h supported
 *	 Bit3	= INT 16/AX=0306h supported
 *	 Bit4	= INT 16/AH=0Ah supported
 *	 Bit5	= INT 16/AH=10h-12h supported (enhanced keyboard support)
 *	 Bit6	= INT 16/AH=20h-22h supported (122-key keyboard support)
 *	 Bit7	= reserved
 */
static void
bios_16_09xx(struct regs *regs)
{
	AL = 0x20; /* we support 0x10-12 */
}

/*
 * KBD:
 *
 * Get enhanced keycode.
 *
 * In:	AH = 10h
 *
 * Out:	AL	= ASCII code
 *	AH	= scan code
 */
static void
bios_16_10xx(struct regs *regs)
{
	/* FIXME VOSSI */
	bios_16_00xx(regs);
}

static void
bios_16_11xx(struct regs *regs)
{
	if (! kbd_keyavailable()) {
		/*
		 * No keystroke available.
		 */
		F |= 1 << 6;	/* Set zero flag. */

	} else {
		/*
		 * Peek keystroke.
		 */
		AX = kbd_getkey(0);
		F &= ~(1 << 6);	/* Clear zero flag. */
	}
}

/*
 * KBD:
 *
 * Get extended shift states.
 *
 * In:	AH = 12h
 *
 * Out:	AL
 *	 Bit0	= right Shift pressed
 *	 Bit1	= left Shift pressed
 *	 Bit2	= left or right Ctrl pressed
 *	 Bit3	= left or right Alt pressed
 *	 Bit4	= ScrollLock mode active
 *	 Bit5	= NumLock mode active
 *	 Bit6	= CapsLock mode active
 *	 Bit7	= Insert mode active
 * 	AH
 *	 Bit0	= left Ctrl pressed
 *	 Bit1	= left Alt pressed
 *	 Bit2	= right Ctrl pressed
 *	 Bit3	= right Alt pressed
 *	 Bit4	= ScrollLock pressed
 *	 Bit5	= NumLock pressed
 *	 Bit6	= CapsLock pressed
 *	 Bit7	= SysRq pressed
 */
static void
bios_16_12xx(struct regs *regs)
{
	AL = var_get(kbd_shift0);
	AH = 0; /* FIXME MARCEL */
}

C_ENTRY void
bios_16_xxxx(struct regs *regs)
{
	if (AH == 0x00) {
		bios_16_00xx(regs);
	} else if (AH == 0x01) {
		bios_16_01xx(regs);
	} else if (AH == 0x02) {
		bios_16_02xx(regs);
	} else if (AH == 0x03) {
		bios_16_03xx(regs);
	} else if (AH == 0x05) {
		bios_16_05xx(regs);
	} else if (AH == 0x09) {
		bios_16_09xx(regs);
	} else if (AH == 0x10) {
		bios_16_10xx(regs);
	} else if (AH == 0x11) {
		bios_16_11xx(regs);
	} else if (AH == 0x12) {
		bios_16_12xx(regs);

	} else if (AH == 0x25) {
		/* Keyboard reset. */
		/* Reset mappings to default. */
		/* Do nothing... FIXME VOSSI */

	} else if (AH == 0x92) {
		/* keyboard capability check called by DOS 5.0+ keyb */
		AH = 0x80;	/* function int16 ah=0x10-0x12 supported */

	} else if (AH == 0xA2) {
		/* 122 keys capability check called by DOS 5.0+ keyb */
		/* don't change AH: function int16 ah=0x20-0x22 NOT supported */

	} else {
		dprintf("kbd: unsupported function %02x\n", AH);
	}
}

C_ENTRY void
bios_09(void)
{
	/* no special key - translation table */
	static CONST char kbd_lc[] = {
	/* 00 */	0, '\033', '1', '2',
	/* 04 */	'3', '4', '5', '6',
	/* 08 */	'7', '8', '9', '0',
	/* 0c */	'-', '=', '\b', '\t',
	/* 10 */	'q', 'w', 'e', 'r',
	/* 14 */	't', 'y', 'u', 'i',
	/* 18 */	'o', 'p', '[', ']',
	/* 1c */	'\r', -1, 'a', 's',
	/* 20 */	'd', 'f', 'g', 'h',
	/* 24 */	'j', 'k', 'l', ';',
	/* 28 */	'\'', '`', -1, '\\',
	/* 2c */	'z', 'x', 'c', 'v',
	/* 30 */	'b', 'n', 'm', ',',
	/* 34 */	'.', '/', -1, '*',
	/* 38 */	-1, ' ', 0, 0,
	/* 3c */	0, 0, 0, 0,
	/* 40 */	0, 0, 0, 0,
	/* 44 */	0, 0, 0, 0,
	/* 48 */	0, 0, '-', 0,
	/* 4c */	0, 0, '+', 0,
	/* 50 */	0, 0, 0, 0,
	};
	/* shift key - translation table */
	static CONST char kbd_uc[] = {
	/* 00 */	0, 27, '!', '@',
	/* 04 */	'#', '$', 37, 0x5e,
	/* 08 */	'&', '*', '(', ')',
	/* 0c */	'_', '+', 0x08, 0,
	/* 10 */	'Q', 'W', 'E', 'R',
	/* 14 */	'T', 'Y', 'U', 'I',
	/* 18 */	'O', 'P', '{', '}',
	/* 1c */	0x0d, -1, 'A', 'S',
	/* 20 */	'D', 'F', 'G', 'H',
	/* 24 */	'J', 'K', 'L', ':',
	/* 28 */	'"', 0x7e, -1, '|',
	/* 2c */	'Z', 'X', 'C', 'V',
	/* 30 */	'B', 'N', 'M', '<',
	/* 34 */	'>', '?', -1, 0,
	/* 38 */	-1, ' ', -1, 0,
	/* 3c */	0, 0, 0, 0,
	/* 40 */	0, 0, 0, 0,
	/* 44 */	0, 0, 0, 0,
	/* 48 */	0, 0, '-', 0,
	/* 4c */	0, 0, '+', 0,
	/* 50 */	0, 0, 0, 0,
	};
	/* ctrl key - translation table */
	static CONST char kbd_ctrl[] = {
	/* 00 */	0, 0, 0, 0,
	/* 04 */	0, 0, 0, 0,
	/* 08 */	0, 0, 0, 0,
	/* 0c */	0, 0, 0, 0,
	/* 10 */	0x11, 0x17, 0x05, 0x12,
	/* 14 */	0x14, 0x19, 0x15, 0x09,
	/* 18 */	0x0f, 0x10, 0x1b, 0x1d,
	/* 1c */	0, 0, 0x01, 0x13,
	/* 20 */	0x04, 0x06, 0x07, 0x08,
	/* 24 */	0x0a, 0x0b, 0x0c, 0,
	/* 28 */	0, 0, 0, 0,
	/* 2c */	0x1a, 0x18, 0x03, 0x16,
	/* 30 */	0x02, 0x0e, 0x0d, 0,
	/* 34 */	0, 0, 0, 0,
	/* 38 */	0, 0
	};

	unsigned char code;
	unsigned char ascii;
	uint16_t ax;
	uint16_t done;

	code = inb(0x60);

	eoi();

	if (code == 0xe0) {
		var_put(mf2_state, var_get(mf2_state) | 0x01);
		
		/* We will get another IRQ immediately with part 2 of the
		 * scancode! */
		return;
	}

	/* Allow for keyboard intercept. */
	ax = code;
	asm (
		"stc\n\t"
		"int $0x15\n\t"
		"jnc 1f\n\t"
		"movw $0, %1\n\t"
		"jmp 2f\n"
	"1:\n\t"
		"movw $1, %1\n"
	"2:\n\t"
		: "=a" (ax), "=r" (done)
		: "0" ((uint16_t) (0x4f00 | code))
	);
	if (done) {
		return;
	}

#if DEBUG
	dprintf("kbd: put: %02x\n", code);
#endif
	if (code == 0x36) {
		/* Right shift pressed. */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x01);

	} else if (code == (0x36 | 0x80)) {
		/* Right shift released. */
		var_put(kbd_shift0, var_get(kbd_shift0) & ~0x01);

	} else if (code == 0x2a) {
		/* Left shift pressed. */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x02);

	} else if (code == (0x2a | 0x80)) {
		/* Left shift released. */
		var_put(kbd_shift0, var_get(kbd_shift0) & ~0x02);

	} else if (code == 0x1d) {
		/* Control pressed. */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x04);

	} else if (code == (0x1d | 0x80)) {
		/* Control released. */
		var_put(kbd_shift0, var_get(kbd_shift0) & ~0x04);

	} else if (code == 0x38) {
		/* Alt pressed. */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x08);

	} else if (code == (0x38 | 0x80)) {
		/* Alt released. */
		var_put(kbd_shift0, var_get(kbd_shift0) & ~0x08);

	} else if (code == 0x46) {
		/* scroll lock on */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x10);

	} else if (code == 0x45) {
		/* num lock on */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x20);

	} else if (code == 0x3a) {
		/* caps lock on */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x40);

	} else if (code == 0x52) {
		/* insert on */
		var_put(kbd_shift0, var_get(kbd_shift0) | 0x80);

	} else if (code & 0x80) {
		/* Key released. */

	} else if (/* 0 <= code && */ code < sizeof(kbd_lc)) {
		if (var_get(kbd_shift0) & 0x08) {
			/* ALT */
			ascii = 0;	/* FIXME VOSSI */

		} else if (var_get(kbd_shift0) & 0x04) {
			/* Control */
			ascii = const_get(kbd_ctrl[code]);

		} else if (var_get(kbd_shift0) & 0x03) {
			/* Shift */
			ascii = const_get(kbd_uc[code]);

		} else {
			/* No modifier pressed. */
			ascii = const_get(kbd_lc[code]);
		}

		enqueue_key(code, ascii);
		
	} else {
		/* just forward returned values and set ascii to 0 - FIXME MARCEL */
		enqueue_key(code, 0);
	}

	var_put(mf2_state, var_get(mf2_state) & ~0x01);
}

#endif /* RUNTIME_RM */
/* ==================== REAL-MODE INIT ==================== */
#ifdef INIT_RM

CODE16;

#include "var.h"
#include "io.h"

void
kbd_init(void)
{
	var_put(kbd_shift0, 0);
	var_put(kbd_shift1, 0);
	var_put(kbd_numpad, 0);
	var_put(ctrlbrk, 0);
	var_put(led_stat, 0);
	var_put(mf2_state, 0x10);

	var_put(kbd_head, (unsigned short) (long) &VAR->kbd_buf[0]);
	var_put(kbd_tail, (unsigned short) (long) &VAR->kbd_buf[0]);
	var_put(kbd_bstart, (unsigned short) (long) &VAR->kbd_buf[0]);
	var_put(kbd_bend, (unsigned short) (long) &VAR->kbd_buf[sizeof(VAR->kbd_buf)]);

	/*
	 * set keyboard mode to
	 * keyboard interrupts enabled
	 * mouse disabled
	 * keycode translation on
	 */
	outb(KBD_CCMD_WRITE_MODE, 0x64);
	outb(KBD_CONTROL_KBDINT | KBD_CONTROL_AUXDIS | KBD_CONTROL_XLATE, 0x60);
}

#endif /* INIT_RM */
