"""

Interface to ogg / vorbis (currently ogginfo and oggenc)

"""

from tools import which, cmdoutput, TRUE, FALSE
from log4py import Logger, LOGLEVEL_NORMAL
from string import find, split, lower, join, strip, atoi, atof

VERSION_1 = 0x0100             # 1.0
VERSION_1_PRE = 0x0090         # 0.9

class ogginfo:

    __ogginfo_data = {}

    def __init__(self, filename, loglevel = LOGLEVEL_NORMAL):
        """ Initialize the class and read the file information. """
        self.__ogginfo_logger = Logger().get_instance(self)
        self.__ogginfo_logger.set_loglevel(loglevel)
        self.__ogginfo_command = which("ogginfo")
        if (self.__ogginfo_command == ""):
            self.__ogginfo_logger.debug("ogginfo executable not found.")
        else:
            self.__ogginfo_version = self.__ogginfo_get_version()
            self.__ogginfo_logger.debug("ogginfo version %s found" % self.__ogginfo_version)
            self.__ogginfo_read_fileinfo(filename)

    def __ogginfo_get_version(self):
        """ get the version information by checking ogginfo -h. """
        command = "%s -h" % self.__ogginfo_command
        output = cmdoutput(command, TRUE)
        if (output[0] == "filename=-h"):
            return VERSION_1_PRE
        else:
            return VERSION_1

    def __ogginfo_read_fileinfo_version_pre_1(self, filename):
        """ Read all available information for vorbis-tools pre1 versions. """
        command = "%s \"%s\"" % (self.__ogginfo_command, filename)
        self.__ogginfo_logger.debug("Executing '%s'" % command)
        output = cmdoutput(command, strip = TRUE)
        for intCounter in range(len(output)):
            if (find(output[intCounter], "=") != -1):
                splitted = split(output[intCounter], "=")
                self.__ogginfo_data[strip(lower(splitted[0]))] = strip(splitted[1])

    def __ogginfo_read_fileinfo_version_1(self, filename):
        """ Read all available information for vorbis-tools >= 1.0 versions. """
        command = "%s \"%s\"" % (self.__ogginfo_command, filename)
        self.__ogginfo_logger.debug("Executing '%s'" % command)
        output = cmdoutput(command, strip = TRUE)
        for intCounter in range(len(output)):
            if (find(output[intCounter], ":") != -1):
                splitted = split(output[intCounter], ":")
                self.__ogginfo_data[strip(lower(splitted[0]))] = strip(join(splitted[1:], ":"))

    def __ogginfo_read_fileinfo(self, filename):
        """ Read all available information for a given file. """
        if (self.__ogginfo_version == VERSION_1):
            self.__ogginfo_read_fileinfo_version_1(filename)
        else:
            self.__ogginfo_read_fileinfo_version_pre_1(filename)

    def track_length(self):
        """ Returns the track length in seconds. """
        if (self.__ogginfo_data.has_key("playback length")):
            splitted = split(self.__ogginfo_data["playback length"], ":")
            seconds = 0
            for intCounter in range(len(splitted)):
                if (splitted[intCounter][-1] == "m"):
                    seconds = seconds + atoi(splitted[intCounter][:-1]) * 60
                elif (splitted[intCounter][-1] == "s"):
                    seconds = seconds + atoi(round(atof(splitted[intCounter][:-1])))
            return seconds
        elif (self.__ogginfo_data.has_key("total_length")):
            return atof(self.__ogginfo_data["total_length"])

    def average_bitrate(self):
        """ Returns the average bitrate in kbps. """
        bitrate = None
        if (self.__ogginfo_data.has_key("average bitrate")):
            if (self.__ogginfo_data["average bitrate"][-4:] == "kbps"):
                bitrate = atof(strip(self.__ogginfo_data["average bitrate"][:-4]))
        elif (self.__ogginfo_data.has_key("bitrate_average")):
            bitrate = atoi(self.__ogginfo_data["bitrate_average"])
        return bitrate

    def version(self):
        return self.__ogginfo_version

class oggenc:

    source = None
    target = None
    quiet = FALSE
    quality = 3

    def __init__(self, loglevel = LOGLEVEL_NORMAL):
        """ Initialize the oggencoder. """
        self.__oggenc_logger = Logger().get_instance(self)
        self.__oggenc_logger.set_loglevel(loglevel)
        self.__oggenc_command = which("oggenc")

    def available(self):
        return (self.__oggenc_command != "")

    def command_line(self):
        """ Return the command line for the given arguments. """
        if ((self.source == None) or (self.target == None)):
            return None
        else:
            command_line = self.__oggenc_command
            command_line = "%s --quality=%d" % (command_line, self.quality)
            if (self.quiet):
                command_line = "%s --quiet" % command_line
            if (find(self.target, " ") != -1):
                self.target = "\"%s\"" % self.target
            command_line = "%s -o %s" % (command_line, self.target)
            if (find(self.source, " ") != -1):
                self.source = "\"%s\"" % self.source
            command_line = "%s %s" % (command_line, self.source)
            return command_line

def test():
    ogginfo_test = ogginfo("41_30secOgg-q0.ogg")
    print "Version (hex): %s" % ogginfo_test.version()
    print "Track length (sec.): %f" % ogginfo_test.track_length()
    print "Average bitrate: %f" % ogginfo_test.average_bitrate()

    oggenc_test = oggenc()
    oggenc_test.quality = 5
    oggenc_test.source = "-"
    oggenc_test.target = "output.ogg"
    print oggenc_test.command_line()

if (__name__ == "__main__"):
    test()
