"""
Implementation of an Editor demo for the Traits User Interface.

The demo shows each of the standard Traits editors on a separate
page of a wizard.  The 2-4 styles of each wizard are shown
together on the page for comparison purposes.  In cases where
Editors with significant variations (e.g. TextEditor and
RangeEditor) are shown on multiple pages.
"""

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import wx

from enthought.enable2.traits.api import RGBAColor
from enthought.enable2.traits.ui.wx.enable_rgba_color_editor import \
    EnableRGBAColorEditor

from enthought.traits.api \
    import Trait, HasTraits, Bool, Str, Int, Range, Enum, Password, Color, \
           List, RGBColor, Font, File, Directory, Code, \
           Tuple, Button

from enthought.traits.ui.api \
    import View, Group, Item, ButtonEditor, \
           CheckListEditor, ImageEnumEditor, SetEditor

#-------------------------------------------------------------------------------
#  Constants:
#-------------------------------------------------------------------------------

# This list of image names (with the standard suffix "_origin") is used to
# construct an image enumeration trait to demonstrate the ImageEnumEditor.

image_list = [ 'top left', 'top right', 'bottom left', 'bottom right' ]

#-------------------------------------------------------------------------------
#  Classes:
#-------------------------------------------------------------------------------

class SampleClass ( HasTraits ):
    """ This Sample class is used to demonstrate the InstanceEditor demo.
    """

    #--------------------------------------------------------------------------
    # The actual attributes don't matter here; we just need an assortment
    # to demonstrate the InstanceEditor's capabilities.
    #--------------------------------------------------------------------------

    name             = Str
    occupation       = Str
    age              = Range( 21,65 )
    registered_voter = Bool

    #--------------------------------------------------------------------------
    # The InstanceEditor uses whatever view is defined for the class.  The
    # default view lists the fields alphabetically, so it's best to define one
    # explicitly.
    #--------------------------------------------------------------------------

    view = View( 'name', 'occupation', 'age', 'registered_voter' )


class EditorDemo ( HasTraits ):
    """ This class specifies the details of the Trait Editor demo.

    First, all the sample traits to be edited are defined, then a hierarchy
    of Groups is created to specify the details of the editor demo display.
    In most cases the actual editor to be used is not specified explicitly;
    instead, the default for the trait type is used.  [It is worth noting that
    two exceptions to this (ButtonEditor and ChecklistEditor) are specified
    when the corresponding trait is defined, while another (EnableRGBAEditor)
    is specified during the definition of the group hierarchy.  This lack of
    symmetry may not be desirable.]

    """

    # To demonstrate any given Trait editor, an appropriate Trait is required.
    boolean_trait    = Bool
    enumeration      = Enum( 'aa', 'bb', 'cc', 'dd', 'xx' )
    string_trait     = Str( "sample string" )
    int_trait        = Int( 1 )
    password         = Password
    small_int_range  = Range( 1, 16 )
    medium_int_range = Range( 1, 25 )
    large_int_range  = Range( 1, 150 )
    float_range      = Range( 0.0, 150.0 )
    play_list        = List( Str, [ "The Merchant of Venice", "Hamlet",
                                    "MacBeth" ] )
    color_trait      = Color
    rgba_color_trait = RGBAColor
    font_trait       = Font
    code_sample      = Code( 'import sys\n\nsys.print("hello world!")' )
    file_trait       = File
    directory_trait  = Directory
    compound_trait   = Trait( 1, Range( 1, 6 ), 'a', 'b', 'c', 'd', 'e', 'f' )
    tuple_trait      = Tuple( Color, Range( 1, 4 ), Str )
    fire_event       = Button( 'Click Me' )
    checklist_1      = List( editor = CheckListEditor(
                                   values = [ 'one', 'two', 'three', 'four' ],
                                   cols   = 4 ) )
    checklist_2      = List( editor = CheckListEditor(
                                   values=[ 'one', 'two', 'three', 'four' ],
                                   cols = 2 ) )
    ordered_set      = List( editor = SetEditor(
                                 values  = ['apples', 'berries', 'cantaloupe' ],
                                 ordered = True,
                                 left_column_title  = 'Available Fruit',
                                 right_column_title = 'Fruit Bowl' ) )
    unordered_set    = List( editor = SetEditor(
                               values = [ 'kumquats', 'pomegranates', 'kiwi' ],
                               left_column_title  = 'Available Fruit',
                               right_column_title = 'Exotic Fruit Bowl' ) )
    sample_instance  = Trait( SampleClass() )
    image_from_list  = Trait( editor = ImageEnumEditor( values = image_list,
                                                        suffix = '_origin',
                                                        cols   = 4,
                                                        klass  = SampleClass ),
                              *image_list )

    # Groups are used to define the demo display - one top-level group per
    # editor, showing each of the up to four styles available for the editor.
    # (In most of these cases, the editor is the default for the trait type.)

    # BooleanEditor display (for Bool traits):
    bool_group = Group( Group( " " ),
                        Group( "15", 'boolean_trait', "15",
                               style = 'simple',
                               label = 'Simple:',
                               show_border = True ),
                        Group( " " ),
                        Group( "15", 'boolean_trait', "15",
                               style = 'custom',
                               label = 'Custom:',
                               show_border = True ),
                        Group( " " ),
                        Group( "15", 'boolean_trait', "15",
                               style = 'text',
                               label = 'Text:',
                               show_border = True ),
                        Group( " " ),
                        Group( "15", 'boolean_trait', "15",
                               style = 'readonly',
                               label = 'Read Only:',
                               show_border = True ),
                        label = 'BooleanEditor interface' )

    # EnumEditor display (for Enum traits):
    enum_group = Group( Group(" "),
                        Group('15', 'enumeration', '15',
                              style='simple',
                              label='Simple:',
                              show_border=True),
                        Group(" "),
                        Group('enumeration',
                              style='custom',
                              label='Custom:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'enumeration', '15',
                              style='text',
                              label='Text:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'enumeration', '15',
                              style='readonly',
                              label='Read Only:',
                              show_border=True),
                        label='EnumEditor interface' )

    # TextEditor display without multi-line capability (for various traits):
    text_int_group = Group( Group(" "),
                            Group('15', 'int_trait', '15',
                                  style='simple',
                                  label='Simple:',
                                  show_border=True),
                            Group(" "),
                            Group('15', 'int_trait', '15',
                                  style='custom',
                                  label='Custom:',
                                  show_border=True),
                            Group(" "),
                            Group('15', 'int_trait', '15',
                                  style='text',
                                  label='Text:',
                                  show_border=True),
                            Group(" "),
                            Group('15', 'int_trait', '15',
                                  style='readonly',
                                  label='Read Only:',
                                  show_border=True),
                            label='TextEditor interface (Int trait)' )

    # TextEditor display with multi-line capability (for various traits):
    text_str_group = Group( Group(" "),
                            Group('15', 'string_trait', '15',
                                  style='simple',
                                  label='Simple:',
                                  show_border=True),
                            Group(" "),
                            Group('15', 'string_trait', '15',
                                  style='custom',
                                  label='Custom:',
                                  show_border=True),
                            Group(" "),
                            Group('15', 'string_trait', '15',
                                  style='text',
                                  label='Text:',
                                  show_border=True),
                            Group(" "),
                            Group('15', 'string_trait', '15',
                                  style='readonly',
                                  label='Read Only:',
                                  show_border=True),
                            label='TextEditor interface (Str trait)' )

    # TextEditor display with secret typing capability (for Password traits):
    text_pass_group = Group( Group(" "),
                             Group('15', 'password', '15',
                                   style='simple',
                                   label='Simple:',
                                   show_border=True),
                             Group(" "),
                             Group('15', 'password', '15',
                                   style='custom',
                                   label='Custom:',
                                   show_border=True),
                             Group(" "),
                             Group('15', 'password', '15',
                                   style='text',
                                   label='Text:',
                                   show_border=True),
                             Group(" "),
                             Group('15', 'password', '15',
                                   style='readonly',
                                   label='Read Only:',
                                   show_border=True),
                             label='TextEditor interface (Password trait)' )

    # RangeEditor display for narrow integer Range traits (< 17 wide):
    int_range_group1 = Group( Group(" "),
                              Group('10', 'small_int_range', '10',
                                    style='simple',
                                    label='Simple:',
                                    show_border=True),
                              Group(" "),
                              Group('small_int_range',
                                    style='custom',
                                    label='Custom:',
                                    show_border=True),
                              Group(" "),
                              Group('15', 'small_int_range', '15',
                                    style='text',
                                    label='Text:',
                                    show_border=True),
                              Group(" "),
                              Group('15', 'small_int_range', '15',
                                    style='readonly',
                                    label='Read Only:',
                                    show_border=True),
                              label="RangeEditor interface" +
                                          "(Int Range: under 17 wide)" )

    # RangeEditor display for medium-width integer Range traits (17 - 100):
    int_range_group2 = Group( Group(" "),
                              Group('10', 'medium_int_range', '10',
                                    style='simple',
                                    label='Simple:',
                                    show_border=True),
                              Group(" "),
                              Group('10', 'medium_int_range', '10',
                                    style='custom',
                                    label='Custom:',
                                    show_border=True),
                              Group(" "),
                              Group('15', 'medium_int_range', '15',
                                    style='text',
                                    label='Text:',
                                    show_border=True),
                              Group(" "),
                              Group('15', 'medium_int_range', '15',
                                    style='readonly',
                                    label='Read Only:',
                                    show_border=True),
                              label='RangeEditor interface' +
                                          '(Int Range: 17 to 100 wide)' )

    # RangeEditor display for wide integer Range traits (> 100):
    int_range_group3 = Group( Group(" "),
                              Group('15', 'large_int_range', '15',
                                    style='simple',
                                    label='Simple:',
                                    show_border=True),
                              Group(" "),
                              Group('15', 'large_int_range', '15',
                                    style='custom',
                                    label='Custom:',
                                    show_border=True),
                              Group(" "),
                              Group('15', 'large_int_range', '15',
                                    style='text',
                                    label='Text:',
                                    show_border=True),
                              Group(" "),
                              Group('15', 'large_int_range', '15',
                                    style='readonly',
                                    label='Read Only:',
                                    show_border=True),
                              label='RangeEditor interface' +
                                          '(Int Range: over 100 wide)' )

    # RangeEditor display for float Range traits:
    float_range_group = Group( Group(" "),
                               Group('10', 'float_range', '10',
                                     style='simple',
                                     label='Simple:',
                                     show_border=True),
                               Group(" "),
                               Group('10', 'float_range', '10',
                                     style='custom',
                                     label='Custom:',
                                     show_border=True),
                               Group(" "),
                               Group('15', 'float_range', '15',
                                     style='text',
                                     label='Text:',
                                     show_border=True),
                               Group(" "),
                               Group('15', 'float_range', '15',
                                     style='readonly',
                                     label='Read Only:',
                                     show_border=True),
                               label='RangeEditor interface (Float Range)' )

    # ListEditor display (for List traits to be modified):
    list_group = Group( Group(" "),
                        Group('10', 'play_list', '10',
                              style='simple',
                              label='Simple:',
                              show_border=True),
                        Group(" "),
                        Group('play_list',
                              style='custom',
                              label='Custom:',
                              show_border=True),
                        Group(" "),
                        Group('play_list',
                              style='text',
                              label='Text:',
                              show_border=True),
                        Group(" "),
                        Group('play_list',
                              style='readonly',
                              label='Read Only:',
                              show_border=True),
                        label='ListEditor interface' )

    # ColorEditor display (for Color traits):
    color_group = Group( Group(" "),
                         Group('15', 'color_trait', '15',
                               style='simple',
                               label='Simple:',
                               show_border=True),
                         Group(" "),
                         Group('color_trait',
                               style='custom',
                               label='Custom:',
                               show_border=True),
                         Group(" "),
                         Group('15', 'color_trait', '15',
                               style='text',
                               label='Text:',
                               show_border=True),
                         Group(" "),
                         Group('15' ,'color_trait', '15',
                               style='readonly',
                               label='Read Only:',
                               show_border=True),
                         label='ColorEditor interface (also RGBColorEditor)' )

    # RGBAColorEditor display (for RGBAColor traits):
    rgba_group = Group( Group(" "),
                        Group('15', 'rgba_color_trait', '15',
                              style='simple',
                              label='Simple:',
                              show_border=True),
                        Group(" "),
                        Group('rgba_color_trait',
                              style='custom',
                              label='Custom:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'rgba_color_trait','15',
                              style='text',
                              label='Text:',
                              show_border=True),
                        Group(" "),
                        Group('15','rgba_color_trait','15',
                              style='readonly',
                              label='Read Only:',
                              show_border=True),
                        label='RGBAColorEditor interface' )

    # EnableRGBAColorEditor display (alternate editor for RGBAColor traits):
    enable_group = Group( Group(" "),
                          Group('15', Item('rgba_color_trait',
                                           editor=EnableRGBAColorEditor()),'15',
                                style='simple',
                                label='Simple:',
                                show_border=True),
                          Group(" "),
                          Group( Item('rgba_color_trait',
                                      editor=EnableRGBAColorEditor()),
                                 style='custom',
                                 label='Custom:',
                                 show_border=True),
                          Group(" "),
                          Group('15', Item('rgba_color_trait',
                                           editor=EnableRGBAColorEditor()),'15',
                                style='text',
                                label='Text:',
                                show_border=True),
                          Group(" "),
                          Group('15', Item('rgba_color_trait',
                                           editor=EnableRGBAColorEditor()), '15',
                                style='readonly',
                                label='Read Only:',
                                show_border=True),
                          label='EnableRGBAColorEditor interface' )

    # FontEditor display (for Font traits):
    font_group = Group( Group(" "),
                        Group('15', 'font_trait', '15',
                              style='simple',
                              label='Simple:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'font_trait', '15',
                              style='custom',
                              label='Custom:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'font_trait', '15',
                              style='text',
                              label='Text:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'font_trait', '15',
                              style='readonly',
                              label='Read Only:',
                              show_border=True),
                        label='FontEditor interface' )

    # CodeEditor display (for Code traits):
    code_group = Group( Group(" "),
                        Group('code_sample',
                              style='simple',
                              label='Simple:',
                              show_border=True),
                        Group(" "),
                        Group('code_sample',
                              style='custom',
                              label='Custom:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'code_sample', '15',
                              style='text',
                              label='Text:',
                              show_border=True),
                        Group(" "),
                        Group('code_sample',
                              style='readonly',
                              label='Read Only:',
                              show_border=True),
                        label='CodeEditor interface' )

    # FileEditor display (for File traits):
    file_group = Group( Group(" "),
                        Group('15', 'file_trait', '15',
                              style='simple',
                              label='Simple:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'file_trait', '15',
                              style='custom',
                              label='Custom:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'file_trait', '15',
                              style='text',
                              label='Text:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'file_trait', '15',
                              style='readonly',
                              label='Read Only:',
                              show_border=True),
                        label='FileEditor interface' )

    # DirectoryEditor display (for Directory traits):
    dir_group = Group( Group(" "),
                       Group('15', 'directory_trait', '15',
                             style='simple',
                             label='Simple:',
                             show_border=True),
                       Group(" "),
                       Group('15', 'directory_trait', '15',
                             style='custom',
                             label='Custom:',
                             show_border=True),
                       Group(" "),
                       Group('15', 'directory_trait', '15',
                             style='text',
                             label='Text:',
                             show_border=True),
                       Group(" "),
                       Group('15', 'directory_trait', '15',
                             style='readonly',
                             label='Read Only:',
                             show_border=True),
                       label='DirectoryEditor interface' )

    # CompoundEditor display (for Compound traits):
    comp_group = Group( Group(" "),
                        Group('compound_trait',
                              style='simple',
                              label='Simple:',
                              show_border=True),
                        Group(" "),
                        Group('compound_trait',
                              style='custom',
                              label='Custom:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'compound_trait', '15',
                              style='text',
                              label='Text:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'compound_trait', '15',
                              style='readonly',
                              label='Read Only:',
                              show_border=True),
                        label='CompoundEditor interface' )

    # TupleEditor display (for Tuple traits):
    tuple_group = Group( Group(" "),
                         Group('tuple_trait',
                               style='simple',
                               label='Simple:',
                               show_border=True),
                         Group(" "),
                         Group('tuple_trait',
                               style='custom',
                               label='Custom:',
                               show_border=True),
                         Group(" "),
                         Group('15', 'tuple_trait', '15',
                               style='text',
                               label='Text:',
                               show_border=True),
                         Group(" "),
                         Group('15', 'tuple_trait', '15',
                               style='readonly',
                               label='Read Only:',
                               show_border=True),
                         label='TupleEditor interface' )

    # ButtonEditor display (for Event traits):
    event_group = Group( Group(" "),
                         Group('15', 'fire_event', '15',
                               style='simple',
                               label='Simple:',
                               show_border=True),
                         Group(" "),
                         Group('15', 'fire_event', '15',
                               style='custom',
                               label='Custom:',
                               show_border=True),
                         label='ButtonEditor interface' )

    # CheckListEditor display (for List traits? 4-column format):
    check_group = Group( Group(" "),
                         Group('15', 'checklist_1', '15',
                               style='simple',
                               label='Simple:',
                               show_border=True),
                         Group(" "),
                         Group('15', 'checklist_1', '15',
                               style='custom',
                               label='Custom:',
                               show_border=True),
                         Group(" "),
                         Group('15', 'checklist_1', '15',
                               style='text',
                               label='Text:',
                               show_border=True),
                         Group(" "),
                         Group('15', 'checklist_1', '15',
                               style='readonly',
                               label='Read Only:',
                               show_border=True),
                         label='CheckListEditor interface' )

    # CheckListEditor display (for List traits? 2-column format):
    check_group2 = Group( Group(" "),
                          Group('15', 'checklist_2', '15',
                                style='simple',
                                label='Simple:',
                                show_border=True),
                          Group(" "),
                          Group('15', 'checklist_2', '15',
                                style='custom',
                                label='Custom:',
                                show_border=True),
                          Group(" "),
                          Group('15', 'checklist_2', '15',
                                style='text',
                                label='Text:',
                                show_border=True),
                          Group(" "),
                          Group('15', 'checklist_2', '15',
                                style='readonly',
                                label='Read Only:',
                                show_border=True),
                          label='CheckListEditor interface ' +
                                       '(column width variation)' )

    # SetEditor display (to create an ordered subset of a given set):
    os_group = Group( Group(" "),
                      Group('15', 'ordered_set', '15',
                            style='simple',
                            label='Simple',
                            show_border=True,
                            show_labels=False),
                      label='SetEditor interface (ordered=True)' )

    # SetEditor display (to create an unordered subset of a given set):
    uos_group = Group( Group(" "),
                       Group('15', 'unordered_set', '15',
                            style='simple',
                            label='Simple',
                            show_border=True,
                            show_labels=False),
                      label='SetEditor interface (ordered=False)' )

    # InstanceEditor display (for Instance traits):
    inst_group = Group( Group(" "),
                        Group('15', 'sample_instance', '15',
                              style='simple',
                              label='Simple:',
                              show_border=True),
                        Group(" "),
                        Group('sample_instance',
                              style='custom',
                              label='Custom:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'sample_instance', '15',
                              style='text',
                              label='Text:',
                              show_border=True),
                        Group(" "),
                        Group('15', 'sample_instance', '15',
                              style='readonly',
                              label='Read Only:',
                              show_border=True),
                        label='InstanceEditor interface' )

    # ImageEnumEditor display (for selecting one from a list of images):
    img_group = Group( Group(" "),
                       Group('10', 'image_from_list', '10',
                             style='simple',
                             label='Simple:',
                             show_border=True),
                       Group(" "),
                       Group('10', 'image_from_list', '10',
                             style='custom',
                             label='Custom:',
                             show_border=True),
                       Group(" "),
                       Group('10', 'image_from_list', '10',
                             style='text',
                             label='Text:',
                             show_border=True),
                       Group(" "),
                       Group('10', 'image_from_list', '10',
                             style='readonly',
                             label='Read Only:',
                             show_border=True),
                       label='ImageEnumEditor interface' )


    # In the actual demo, defined by this View, the editor display groups appear
    # alphabetically, one per wizard page or dialogue tab:
    view1 = View( bool_group,
                  event_group,
                  check_group,
                  check_group2,
                  code_group,
                  color_group,
                  comp_group,
                  dir_group,
                  enable_group,
                  enum_group,
                  file_group,
                  font_group,
                  img_group,
                  inst_group,
                  list_group,
                  int_range_group1,
                  int_range_group2,
                  int_range_group3,
                  float_range_group,
                  rgba_group,
                  os_group,
                  uos_group,
                  text_int_group,
                  text_str_group,
                  text_pass_group,
                  tuple_group )


# Code to run demo:
if __name__ == '__main__':
    EditorDemo().configure_traits( kind = 'wizard' )

