"""
Demonstration of a viewport class.
"""

import time
import wx
import pdb

from enthought.traits.api import Any, Event, Float, Instance, Int, Property, true
from enthought.enable import BaseContainer, Component, Container, FilledContainer
from enthought.enable import Scrolled
from enthought.enable.wx import Window
from enthought.enable.enable_traits import bounds_trait, grey_color_trait, red_color_trait
from enthought.enable.traits.api import RGBAColor
from enthought.kiva import Font
from enthought.kiva.backend_image import Image


font = Font(face_name="Arial")

class Box(Component):
    color = red_color_trait

    delay = Float(0.50)

    def _draw(self, gc):
        if self.event_state == "clicked":
            print "waiting %0.4f seconds... " % self.delay,
            time.sleep(self.delay)
            print "done."

            gc.save_state()
            gc.set_fill_color(self.color_)
            gc.rect(*self.bounds)
            gc.fill_path()
            gc.restore_state()
        else:
            gc.save_state()
            gc.set_stroke_color(self.color_)
            gc.set_fill_color(self.color_)
            gc.set_line_width(1.0)
            gc.rect(*self.bounds)
            gc.stroke_path()

            gc.set_font(font)
            x,y,dx,dy = self.bounds
            tx, ty, tdx, tdy = gc.get_text_extent(str(self.delay))
            gc.set_text_position(x+dx/2-tdx/2, y+dy/2-tdy/2)
            gc.show_text(str(self.delay))
            gc.restore_state()

    def normal_left_down(self, event):
        print "clicked!"
        self.event_state = "clicked"
        event.handled = True
        self.redraw()

    def clicked_left_up(self, event):
        self.event_state = "normal"
        event.handled = True
        self.redraw()

class Viewport(Component):

    # The container we are "viewing"
    canvas_container = Instance(BaseContainer)

    # The bounds of the viewport, relative to the canvas_container
    view_bounds = bounds_trait

    draw_border = true
    border_color = RGBAColor((0.0, 1.0, 0.0, 1.0))
    empty_color = RGBAColor((0.75, 0.75, 0.75, 1.0))

    # These are proxies for our canvas_container
    min_width = Property(Float)
    min_height = Property(Float)

    x = Property
    y = Property

    def __init__(self, canvas, bounds=None, view_bounds=None):
        """
        Create a viewport for the given canvas.  "bounds" defines the location
        and dimensions of the viewport within its parent container; "view_bounds"
        defines the position and dimensions of the area of canvas_container
        to display.
        """
        if bounds is not None:
            # Don't initialize this via traits with a kw arg to the parent
            # Component constructor, since we don't have a good default to
            # pass in if bounds is None.
            self.bounds = bounds
        self.canvas_container = canvas
        if view_bounds is None:
            self.view_bounds = (0, 0, bounds[2], bounds[3])
        else:
            self.view_bounds = view_bounds
        Component.__init__(self)

    def _draw(self, gc):
        gc.save_state()

        if self.draw_border:
            gc.set_stroke_color(self.border_color_)
            gc.set_fill_color(self.empty_color_)
            gc.set_line_width(1.0)
            gc.rect(*self.bounds)
            gc.draw_path()

        sb = self.bounds
        vb = self.view_bounds
        cb = self.canvas_container.bounds

        # The viewport's origin in its container is (bounds[0], bounds[1])
        # relative to the input GC origin.  The viewport's origin is the same
        # as the viewport origin on the canvas.
        # The canvas origin is then offset (-view_bounds[0], -view_bounds[1])
        # from the viewport origin.
        gc.translate_ctm(sb[0]-vb[0], sb[1]-vb[1])

        # Scale the CTM so that the viewport's dimensions onto its canvas_container
        # map into its actual size (in self.bounds) on its parent.
        # TODO: not implemented yet - confirm the order of transformations when
        #       scaling and translating.
        if (sb[2] != vb[2]) or (sb[3] != vb[3]):
            raise NotImplementedError, "Scaled viewports are not implemented yet."

        # The clipping rectangle in the now-translated GC is just the view_bounds.
        gc.clip_to_rect(*self.view_bounds)

        # Tell the container to actually draw itself.
        self.canvas_container._draw(gc)
        gc.restore_state()

    def _view_bounds_changed(self, old, newval):
        self.redraw()

    def _components_at(self, x, y):
        # map x,y -> viewport space
        newx = x - self.bounds[0]
        newy = y - self.bounds[1]
        # map viewport space -> canvas_space
        newx += self.view_bounds[0]
        newy += self.view_bounds[1]
        results = [self] + self.canvas_container._components_at(newx, newy)
        return results

    #---------------------------------------------------------------------
    # Demo-related panning action
    #---------------------------------------------------------------------

    def normal_right_down(self, event):
        self.event_state = "panning"
        self.pointer = "hand"
        self._original_xy = (event.x, event.y)

    def panning_right_up(self, event):
        self._original_xy = None
        self.event_state = "normal"
        self.pointer = "arrow"

    def panning_mouse_move(self, event):
        x,y,dx,dy = self.view_bounds
        newx = x - (event.x - self._original_xy[0])
        newy = y - (event.y - self._original_xy[1])
        self._original_xy = (event.x, event.y)
        self.view_bounds = (newx, newy, dx, dy)

    #---------------------------------------------------------------------
    # Property getters and setters
    #---------------------------------------------------------------------

    def _get_min_width(self):
        #return self.canvas_container.min_width
        return self.view_bounds[2]

    def _get_min_height(self):
        #return self.canvas_container.min_height
        return self.view_bounds[3]

    def _get_max_width(self):
        return self.view_bounds[2]

    def _get_max_height(self):
        return self.view_bounds[3]

    def _get_x(self):
        return self.view_bounds[0]

    def _set_x(self, x):
        print "setting x:", x
        ignore, y, dx, dy = self.view_bounds
        self.view_bounds = x, y, dx, dy

    def _get_y(self):
        return self.view_bounds[1]

    def _set_y(self, y):
        print "setting y:", y
        x, ignore, dx, dy = self.view_bounds
        self.view_bounds = x, y, dx, dy


class TestContainer(Container):
    fgcolor = RGBAColor((0.0, 0.75, 1.0, 1.0))

    image_file_name = "deepfield.jpg"

    _image = Any

    def __init__(self, *args, **kw):
        Container.__init__(self, *args, **kw)
        self._image = Image("deepfield.jpg")

    def _draw_container(self, gc):
        gc.save_state()
        gc.set_stroke_color(self.fgcolor_)
        #print "canvas container bounds:", self.bounds
        # TODO: look at CTM and investigate what the Scrolled() is doing to it.
        gc.draw_image(self._image)
        gc.rect(*self.bounds)
        gc.stroke_path()
        gc.restore_state()

    def _draw ( self, gc ):
        self._draw_container( gc )
        gc.save_state()
        gc.clip_to_rect(*self.bounds)
        for component in self.components:
            # call _draw() instead of draw() to get around some window._update_region
            # related issues
            component._draw( gc )
        gc.restore_state()

    def _components_at ( self, x, y ):
        result = [ self ]
        for component in self.components:
            result.extend( component.components_at( x, y ) )
        return result



class EnableWindowFrame ( wx.Frame ):
    def __init__ ( self, canvas, viewport, *args, **kw ):
        wx.Frame.__init__( *(self,) + args, **kw )
        sizer = wx.BoxSizer( wx.HORIZONTAL )
        self.viewport_window = Window( self, -1, component = viewport )
        self.canvas_window = Window( self, -1, component = canvas )
        sizer.Add( self.canvas_window.control, 1, wx.EXPAND )
        sizer.Add( self.viewport_window.control, 1, wx.EXPAND )
        self.SetSizer( sizer )
        self.SetAutoLayout( True )
        self.Show( True )


def main():
    app = wx.PySimpleApp()
    times_and_bounds = { 0.5 : (60,200,100,100),
                            0.33 : (240,200,100,100),
                            0.25: (60,50,100,100),
                            0.10: (240,50,100,100) }

    # Create the canvas
    canvas = TestContainer(min_width=600, min_height=600,
                          max_width=600, max_height=600)
    for delay, bounds in times_and_bounds.items():
        box = Box()
        canvas.add(box)
        box.bounds = bounds
        box.delay = delay

    # create the view and the scrolled container
    view = Viewport(canvas, bounds=(100,100,300,300))

    view_parent = FilledContainer(view, bg_color=(0.53,0.81,0.98,1.0),
                                  bounds=(0,0,500,500))

    frame = EnableWindowFrame(Scrolled(canvas), view_parent, None, -1,
                            "Viewport test", size=wx.Size(1250,650))
    app.SetTopWindow(frame)
    frame.Show(True)
    app.MainLoop()


if __name__ == "__main__":
    main()