#-------------------------------------------------------------------------------
#
#  Defines the concrete top-level Enable 'Window' class for the wxPython GUI 
#  toolkit, based on the kiva agg driver.
#
#  Written by: David C. Morrill
#
#  Date: 09/25/2003
#
#  (c) Copyright 2003 by Enthought, Inc.
#
#  Classes defined: Window
#
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
#  Debug/Testing flags:
#-------------------------------------------------------------------------------

# Gotcha! enabled:
try:
    import enthought.gotcha as gotcha
    have_gotcha = True
except:
    have_gotcha = False

#-------------------------------------------------------------------------------
#  Imports:
#-------------------------------------------------------------------------------

import wx

WidgetClass = wx.Window

import time

from enthought.enable.base       import GraphicsContextEnable, send_event_to
from enthought.enable.events     import MouseEvent, KeyEvent, DragEvent
from enthought.enable.window     import AbstractWindow
from enthought.enable.component  import Component

from enthought.traits.api            import Trait, Instance
from enthought.traits.ui.wx.menu import MakeMenu

try:
    from enthought.util.wx.drag_and_drop import PythonDropTarget
except:
    PythonDropTarget = None

#-------------------------------------------------------------------------------
#  Constants:
#-------------------------------------------------------------------------------

# Number of pixels to scroll at a time:
scroll_incr = 16

# Map from pointer shape name to pointer shapes:
pointer_map = {
   'arrow':             wx.CURSOR_ARROW,
   'right arrow':       wx.CURSOR_RIGHT_ARROW,
   'blank':             wx.CURSOR_BLANK,
   'bullseye':          wx.CURSOR_BULLSEYE,
   'char':              wx.CURSOR_CHAR,
   'cross':             wx.CURSOR_CROSS,
   'hand':              wx.CURSOR_HAND,
   'ibeam':             wx.CURSOR_IBEAM,
   'left button':       wx.CURSOR_LEFT_BUTTON,
   'magnifier':         wx.CURSOR_MAGNIFIER,
   'middle button':     wx.CURSOR_MIDDLE_BUTTON,
   'no entry':          wx.CURSOR_NO_ENTRY,
   'paint brush':       wx.CURSOR_PAINT_BRUSH,
   'pencil':            wx.CURSOR_PENCIL,
   'point left':        wx.CURSOR_POINT_LEFT,
   'point right':       wx.CURSOR_POINT_RIGHT,
   'question arrow':    wx.CURSOR_QUESTION_ARROW, 
   'right button':      wx.CURSOR_RIGHT_BUTTON, 
   'size top':          wx.CURSOR_SIZENS, 
   'size bottom':       wx.CURSOR_SIZENS, 
   'size left':         wx.CURSOR_SIZEWE, 
   'size right':        wx.CURSOR_SIZEWE, 
   'size top right':    wx.CURSOR_SIZENESW, 
   'size bottom left':  wx.CURSOR_SIZENESW, 
   'size top left':     wx.CURSOR_SIZENWSE, 
   'size bottom right': wx.CURSOR_SIZENWSE, 
   'sizing':            wx.CURSOR_SIZING, 
   'spray can':         wx.CURSOR_SPRAYCAN,
   'wait':              wx.CURSOR_WAIT, 
   'watch':             wx.CURSOR_WATCH,
   'arrow wait':        wx.CURSOR_ARROWWAIT
}

# Map from wxPython special key names into Enable key names:
key_map = {
    wx.WXK_BACK:      'Backspace',
    wx.WXK_TAB:       'Tab',
    wx.WXK_RETURN:    'Enter',
    wx.WXK_ESCAPE:    'Esc',
    wx.WXK_DELETE:    'Delete',
    wx.WXK_START:     'Start',
    wx.WXK_LBUTTON:   'Left Button',
    wx.WXK_RBUTTON:   'Right Button',
    wx.WXK_CANCEL:    'Cancel',
    wx.WXK_MBUTTON:   'Middle Button',
    wx.WXK_CLEAR:     'Clear',
    wx.WXK_SHIFT:     'Shift',
    wx.WXK_CONTROL:   'Control',
    wx.WXK_MENU:      'Menu',
    wx.WXK_PAUSE:     'Pause',
    wx.WXK_CAPITAL:   'Capital',
    wx.WXK_PRIOR:     'Page Up',
    wx.WXK_NEXT:      'Page Down',
    wx.WXK_END:       'End',
    wx.WXK_HOME:      'Home',
    wx.WXK_LEFT:      'Left',
    wx.WXK_UP:        'Up',
    wx.WXK_RIGHT:     'Right',
    wx.WXK_DOWN:      'Down',
    wx.WXK_SELECT:    'Select',
    wx.WXK_PRINT:     'Print',
    wx.WXK_EXECUTE:   'Execute',
    wx.WXK_SNAPSHOT:  'Snapshot',
    wx.WXK_INSERT:    'Insert',
    wx.WXK_HELP:      'Help',
    wx.WXK_NUMPAD0:   'Numpad 0',
    wx.WXK_NUMPAD1:   'Numpad 1',
    wx.WXK_NUMPAD2:   'Numpad 2',
    wx.WXK_NUMPAD3:   'Numpad 3',
    wx.WXK_NUMPAD4:   'Numpad 4',
    wx.WXK_NUMPAD5:   'Numpad 5',
    wx.WXK_NUMPAD6:   'Numpad 6',
    wx.WXK_NUMPAD7:   'Numpad 7',
    wx.WXK_NUMPAD8:   'Numpad 8',
    wx.WXK_NUMPAD9:   'Numpad 9',
    wx.WXK_MULTIPLY:  'Multiply',
    wx.WXK_ADD:       'Add',
    wx.WXK_SEPARATOR: 'Separator',
    wx.WXK_SUBTRACT:  'Subtract',
    wx.WXK_DECIMAL:   'Decimal',
    wx.WXK_DIVIDE:    'Divide',
    wx.WXK_F1:        'F1',
    wx.WXK_F2:        'F2',
    wx.WXK_F3:        'F3',
    wx.WXK_F4:        'F4',
    wx.WXK_F5:        'F5',
    wx.WXK_F6:        'F6',
    wx.WXK_F7:        'F7',
    wx.WXK_F8:        'F8',
    wx.WXK_F9:        'F9',
    wx.WXK_F10:       'F10',
    wx.WXK_F11:       'F11',
    wx.WXK_F12:       'F12',
    wx.WXK_F13:       'F13',
    wx.WXK_F14:       'F14',
    wx.WXK_F15:       'F15',
    wx.WXK_F16:       'F16',
    wx.WXK_F17:       'F17',
    wx.WXK_F18:       'F18',
    wx.WXK_F19:       'F19',
    wx.WXK_F20:       'F20',
    wx.WXK_F21:       'F21',
    wx.WXK_F22:       'F22',
    wx.WXK_F23:       'F23',
    wx.WXK_F24:       'F24',
    wx.WXK_NUMLOCK:   'Num Lock',
    wx.WXK_SCROLL:    'Scroll Lock'
}
    
# Reusable instance to avoid constructor/destructor overhead:
wx_rect = wx.Rect( 0, 0, 0, 0 )

# Default 'fake' start event for wxPython based drag operations:
default_start_event = MouseEvent()

#-------------------------------------------------------------------------------
#  'EnableTimer' class:
#
#  Note: This class maintains a 'work list' of scheduled components, where 
#  each item in the list has the form: [ component, interval, timer_pop_time ]
#
#-------------------------------------------------------------------------------

# To conserve system resources, there is only one 'timer' per program:
system_timer = None
 
class EnableTimer ( wx.Timer ): 
    
    #---------------------------------------------------------------------------
    #  Initialize the object:
    #---------------------------------------------------------------------------
    
    def __init__ ( self ):
        wx.Timer.__init__( self )
        self._work_list = []
        
    #---------------------------------------------------------------------------
    #  Schedule a timer event for a specified component:
    #---------------------------------------------------------------------------
    
    def schedule ( self, component, interval ):
        work_list = self._work_list
        if len( work_list ) == 0:
            self.Start( 5 )
        for i, item in enumerate( work_list ):
            if component is item[0]:
                del work_list[i]
                break
        self.reschedule( component, interval )

    #---------------------------------------------------------------------------
    #  Reshedule a recurring timer event for a component:
    #---------------------------------------------------------------------------

    def reschedule ( self, component, interval ):
        pop_time  = time.time() + interval
        new_item  = [ component, interval, pop_time ]
        work_list = self._work_list
        for i, item in enumerate( work_list ):
            if pop_time < item[2]:
                work_list.insert( i, new_item )
                break
        else:
            work_list.append( new_item )

    #---------------------------------------------------------------------------
    #  Cancel any pending timer events for a component:
    #---------------------------------------------------------------------------
    
    def cancel ( self, component ):
        work_list = self._work_list
        for i, item in enumerate( work_list ):
            if component is item[0]:
                del work_list[i]
                if len( work_list ) == 0:
                    self.Stop()
                break
        return (len( work_list ) != 0)        
        
    #---------------------------------------------------------------------------
    #  Handle a timer 'pop' event:
    #---------------------------------------------------------------------------
     
    # This routine is for performance testing purposes:
    def Notify ( self ):
        if have_gotcha:
            gotcha.profile( self._timed_Notify )
        else:
            self._timed_Notify()
        
    def _timed_Notify ( self ):
        now       = time.time()
        work_list = self._work_list
        n         = len( work_list )
        i         = 0
        while (i < n) and (now >= work_list[i][2]):
            i += 1
        if i > 0:
            reschedule = work_list[:i]
            del work_list[:i]
            for component, interval, ignore in reschedule:
                self.reschedule( component, interval )
                component.timer = True
        
#-------------------------------------------------------------------------------
#  'Window' class:
#-------------------------------------------------------------------------------

class Window ( AbstractWindow ):

    # Screen scroll increment amount:

    scroll_incr = ( wx.SystemSettings_GetMetric( wx.SYS_SCREEN_Y )
                    or 768 ) / 20

    # Width/Height of standard scrollbars:
    scrollbar_dx = wx.SystemSettings_GetMetric( wx.SYS_VSCROLL_X )
    scrollbar_dy = wx.SystemSettings_GetMetric( wx.SYS_HSCROLL_Y )

    # Reference to the actual wxPython window:
    control      = Instance( wx.Window )
    
    #---------------------------------------------------------------------------
    #  Initialize the object:
    #---------------------------------------------------------------------------
    
    def __init__ ( self, parent, wid = -1, pos = wx.DefaultPosition,
                   size = wx.DefaultSize, **traits ):
        # Create the delegate: 
        self.control = control = WidgetClass( parent, wid, pos, size,
                                     style = wx.CLIP_CHILDREN | wx.WANTS_CHARS )
        AbstractWindow.__init__( self, **traits )
        self._timer          = None
        self._mouse_captured = False
        self._dc             = None

        # Set up the 'erase background' event handler:
        wx.EVT_ERASE_BACKGROUND( control, self._on_erase_background )
 
        # Set up the 'paint' event handler:
        wx.EVT_PAINT( control, self._paint )
        wx.EVT_SIZE(  control, self._on_size )
        
        # Set up mouse event handlers:
        wx.EVT_LEFT_DOWN(     control, self._on_left_down )
        wx.EVT_LEFT_UP(       control, self._on_left_up )
        wx.EVT_LEFT_DCLICK(   control, self._on_left_dclick )
        wx.EVT_MIDDLE_DOWN(   control, self._on_middle_down )
        wx.EVT_MIDDLE_UP(     control, self._on_middle_up )
        wx.EVT_MIDDLE_DCLICK( control, self._on_middle_dclick )
        wx.EVT_RIGHT_DOWN(    control, self._on_right_down )
        wx.EVT_RIGHT_UP(      control, self._on_right_up )
        wx.EVT_RIGHT_DCLICK(  control, self._on_right_dclick )
        wx.EVT_MOTION(        control, self._on_mouse_move )
        wx.EVT_ENTER_WINDOW(  control, self._on_window_enter )
        wx.EVT_LEAVE_WINDOW(  control, self._on_window_leave )
        wx.EVT_MOUSEWHEEL(    control, self._on_mouse_wheel )
        
        # Handle key up/down events:
        wx.EVT_KEY_DOWN( control, self._on_key_updown )
        wx.EVT_KEY_UP(   control, self._on_key_updown )
        wx.EVT_CHAR(     control, self._on_char )
        
        # Attempt to allow wxPython drag and drop events to be mapped to
        # Enable drag events:
        if PythonDropTarget is not None:
           control.SetDropTarget( PythonDropTarget( self ) ) 
           self._drag_over = []
 
    #---------------------------------------------------------------------------
    #  Handle keyboard keys changing their up/down state:
    #---------------------------------------------------------------------------
           
    def _on_key_updown ( self, event ):

        k = event.GetKeyCode()
        t = event.GetEventType()

        if k == wx.WXK_SHIFT:
            self.shift_pressed = (t == wx.wxEVT_KEY_DOWN)
        elif k == wx.WXK_ALT:
            self.alt_pressed   = (t == wx.wxEVT_KEY_DOWN)
        elif k == wx.WXK_CONTROL:
            self.ctrl_pressed  = (t == wx.wxEVT_KEY_DOWN)

        event.Skip()
 
    #---------------------------------------------------------------------------
    #  Handle keyboard keys being pressed:
    #---------------------------------------------------------------------------
           
    def _on_char ( self, event ):

        focus_owner = self.focus_owner
        if focus_owner is not None:
            control_down = event.ControlDown()
            key_code     = event.GetKeyCode()
            if control_down and (1 <= key_code <= 26):
                key = chr( key_code + 96 )
            else:
                key = key_map.get( key_code )
                if key is None:
                    if key_code >= 0 and key_code < 256:
                        key = chr( key_code )
            focus_owner.key = KeyEvent( 
                                  character    = key,
                                  alt_down     = event.AltDown(),
                                  control_down = control_down,
                                  shift_down   = event.ShiftDown(),
                                  x            = event.GetX(),
                                  y            = self._flip_y( event.GetY() ),
                                  event        = event )
                                        
        else:
            event.Skip()
        
    #---------------------------------------------------------------------------
    #  Convert from a Kiva to a wxPython y coordinate: 
    #---------------------------------------------------------------------------
    
    def _flip_y ( self, y ):
        return int( self._size[1] - 1 - y )
   
    #---------------------------------------------------------------------------
    #  Erase background event handler:
    #---------------------------------------------------------------------------

    def _on_erase_background ( self, event ):
        pass

    #---------------------------------------------------------------------------
    #  Size event handler:
    #---------------------------------------------------------------------------
        
    def _on_size ( self, event ):
        dx, dy = self.control.GetSizeTuple()
        self.component.bounds = ( 0.0, 0.0, float( dx ), float( dy ) )  
        self.control.Refresh()
    

    #---------------------------------------------------------------------------
    #  Capture all future mouse events:
    #---------------------------------------------------------------------------
    
    def _capture_mouse ( self ):
        if not self._mouse_captured:
            self._mouse_captured = True
            self.control.CaptureMouse()    
   
    #---------------------------------------------------------------------------
    #  Release the mouse capture:
    #---------------------------------------------------------------------------
    
    def _release_mouse ( self ):
        if self._mouse_captured:
            self._mouse_captured = False
            self.control.ReleaseMouse()
    
    #---------------------------------------------------------------------------
    #  Convert a GUI toolkit mouse event into a MouseEvent:
    #---------------------------------------------------------------------------
    
    def _create_mouse_event ( self, event ):
        if event is not None:
            x           = event.GetX()
            y           = event.GetY()
            mouse_wheel = ((event.GetLinesPerAction() * 
                            event.GetWheelRotation()) / 
                            (event.GetWheelDelta() or 1))
            # Note: The following code fixes a 'bug' in wxPython that returns
            # 'mouse_wheel' events in screen coordinates, rather than window
            # coordinates:
            if mouse_wheel != 0:
               x, y = self.control.ScreenToClientXY( x, y )
            return MouseEvent( x            = x,
                               y            = self._flip_y( y ),
                               alt_down     = event.AltDown(),    
                               control_down = event.ControlDown(),
                               shift_down   = event.ShiftDown(),
                               left_down    = event.LeftIsDown(),
                               middle_down  = event.MiddleIsDown(),
                               right_down   = event.RightIsDown(),
                               mouse_wheel  = mouse_wheel )
                               
        # If no event specified, make one up:
        x, y = wx.GetMousePosition()
        x, y = self.control.ScreenToClientXY( x, y )
        return MouseEvent( x            = x,
                           y            = self._flip_y( y ),
                           alt_down     = self.alt_pressed,    
                           control_down = self.ctrl_pressed,
                           shift_down   = self.shift_pressed,
                           left_down    = False,
                           middle_down  = False,
                           right_down   = False,
                           mouse_wheel  = 0 )
       
    #---------------------------------------------------------------------------
    #  Create a Kiva graphics context of a specified size:  
    #---------------------------------------------------------------------------       
    
    def _create_gc ( self, size, pix_format = "bgra32" ):
        gc = GraphicsContextEnable( size, pix_format = pix_format )
        gc._clip_bounds = []   ### TEMPORARY ###
        return gc
        
    #---------------------------------------------------------------------------
    #  Request a redraw of the window:
    #---------------------------------------------------------------------------
    
    def _redraw ( self, coordinates ):
        if coordinates is None:
            if self.control:
                self.control.Refresh( False )
        else:
            xl, yb, xr, yt = coordinates
            rect = wx_rect
            rect.SetX( int( xl ) )
            rect.SetY( int( self._flip_y( yt - 1 ) ) )
            rect.SetWidth(  int( xr - xl ) )
            rect.SetHeight( int( yt - yb ) )
            if self.control:
                self.control.Refresh( False, rect )
            
    #---------------------------------------------------------------------------
    #  Get the appropriate gc size for a specified component size:
    #---------------------------------------------------------------------------
    
    def _get_gc_size ( self, size ):
        return size
        #dx, dy = size
        #return ( ((int( dx ) + scroll_incr - 1 ) / scroll_incr) * scroll_incr,
        #         ((int( dy ) + scroll_incr - 1 ) / scroll_incr) * scroll_incr)
        
    #---------------------------------------------------------------------------
    #  Set the size of the associated component:
    #---------------------------------------------------------------------------
    
    def _set_component_size ( self, size ):
        if self.control:
            self.control.SetVirtualSize( size )
            # Don't rely on the _on_size handler to do this.  
            self.component.bounds = ( 0.0, 0.0, size[0], size[1] )  
    
    #---------------------------------------------------------------------------
    #  Get the size of the underlying toolkit control:
    #---------------------------------------------------------------------------
    
    def _get_control_size ( self ):
        result = None
        if self.control:
            result = self.control.GetSizeTuple()

        return result
    
    #---------------------------------------------------------------------------
    #  Do a GUI toolkit specific screen update:
    #---------------------------------------------------------------------------

    def _window_paint ( self, event ):
        # NOTE: This should do an optimal update based on the value of the 
        # self._update_region, but Kiva does not currently support this:
        
        control = self.control
        wdc     = control._dc = wx.PaintDC( control )

        self._gc.pixel_map.draw_to_wxwindow( control, 0, 0 )

        # Draw the 'cursor', if necessary:
        cx = self._cursor_x
        cy = self._cursor_y
        if (cx is not None) or (cy is not None):
            dx, dy = control.GetSizeTuple()
            color  = self._cursor_color
            wdc.SetPen( wx.Pen( wx.Colour( int( 255 * color[0] ), 
                                           int( 255 * color[1] ), 
                                           int( 255 * color[2] ) ) ) )
            if cx is not None:
                x, y, dy = cx
                wdc.DrawLine( x, y, x, y + dy - 1 )
            if cy is not None:
                y, x, dx = cy
                wdc.DrawLine( x, y, x + dx - 1, y )
                
        control._dc = None

    #---------------------------------------------------------------------------
    #  Set the current pointer (i.e. cursor) shape:
    #---------------------------------------------------------------------------
        
    def _set_pointer ( self, pointer ):
        ptr = pointer_map[ pointer ]
        if type( ptr ) is int:
            pointer_map[ pointer ] = ptr = wx.StockCursor( ptr )
        self.control.SetCursor( ptr )
        
    #---------------------------------------------------------------------------
    #  Set the current tooltip for the window:
    #---------------------------------------------------------------------------
        
    def _set_tooltip ( self, tooltip ):
        wx.ToolTip_Enable( False )
        self.control.SetToolTip( wx.ToolTip( tooltip ) )
        wx.ToolTip_Enable( True )
        
    #---------------------------------------------------------------------------
    #  Check to see if the 'tooltip' infomation needs to be updated:
    #---------------------------------------------------------------------------

    def _tooltip_check ( self, component ):
        tooltip = component.tooltip
        if tooltip != self._tooltip:
            self._tooltip = tooltip
            self.control.SetToolTipString( tooltip )

    #---------------------------------------------------------------------------
    #  Set up or cancel a timer for a specified component:
    #---------------------------------------------------------------------------
        
    def _set_timer_interval ( self, component, interval ):
        global system_timer
        if interval is None:
            if ((system_timer is not None) and 
                (not system_timer.cancel( component ))):
                system_timer = None
        else:
            if system_timer is None:
                system_timer = EnableTimer()
            system_timer.schedule( component, interval )
            
    #---------------------------------------------------------------------------
    #  Sets the keyboard focus to this window:  
    #---------------------------------------------------------------------------
                        
    def _set_focus ( self ):
        self.control.SetFocus()
        
    #---------------------------------------------------------------------------
    #  Check to see if the 'cursor' needs to be updated on the screen:
    #---------------------------------------------------------------------------

    def _cursor_check ( self, cx = None, cy = None, color = None ):
        if cx is not None:
            x, y, dy = map( int, cx )
            cx       = ( x, self._flip_y( y + dy - 1 ), dy )
        if cy is not None:
            y, x, dx = map( int, cy )
            cy = ( self._flip_y( y ), x, dx )
        scx           = self._cursor_x
        scy           = self._cursor_y
        color_changed = (self._cursor_color != color)
        if color_changed or (scx != cx):
            if scx is not None:
                self.control.Refresh( False, 
                                      wx.Rect( scx[0], scx[1], 1, scx[2] ) )
            if cx is not None:
                self.control.Refresh( False, 
                                      wx.Rect( cx[0], cx[1], 1, cx[2] ) )
            self._cursor_x = cx
        if color_changed or (scy != cy):
            if scy is not None:
                self.control.Refresh( False, 
                                      wx.Rect( scy[1], scy[0], scy[2], 1 ) )
            if cy is not None:
                self.control.Refresh( False, 
                                      wx.Rect( cy[1], cy[0], cy[2], 1 ) )
            self._cursor_y = cy
        self._cursor_color = color
        
    #---------------------------------------------------------------------------
    #  Handle wxPython drag and drop events:
    #---------------------------------------------------------------------------
        
    def wx_dropped_on ( self, x, y, drag_object, drag_result ):
        # Process the 'dropped_on' event for the object(s) it was dropped on:
        y             = self._flip_y( y )
        components_at = self.component.components_at( x, y )
        drag_event    = DragEvent( x           = x,
                                   y           = y,
                                   x0          = 0.0,
                                   y0          = 0.0,
                                   copy        = drag_result != wx.DragMove,
                                   components  = [ drag_object ],
                                   start_event = default_start_event )
        index = send_event_to( components_at, 'dropped_on', drag_event )

        # Send all the runner-ups a 'drag_leave' consolation prize:
        drag_over = self._drag_over
        for component in components_at[ 0: index ]:
            if component in drag_over:
                component.drag_leave = drag_event 
        self._drag_over = []
        if drag_event.handled:
            return drag_result
        return wx.DragNone

    def wx_drag_over ( self, x, y, drag_object, drag_result ):
        y = self._flip_y( y )
        drag_over_event = DragEvent( x    = x,
                                     y    = y,
                                     x0   = 0.0,
                                     y0   = 0.0,
                                     copy = drag_result != wx.DragMove,
                                     components  = [ drag_object ],
                                     start_event = default_start_event )
        new_drag_over = []
        cur_drag_over = self._drag_over
        for component in self.component.components_at( x, y ):
            new_drag_over.append( component )
            if component in cur_drag_over:
                cur_drag_over.remove( component )
                component.drag_over = drag_over_event
            else:
                component.drag_enter = drag_over_event
        for component in cur_drag_over:
            component.drag_leave = drag_over_event
        self._drag_over = new_drag_over
        if drag_over_event.handled:
            return drag_result
        return wx.DragNone
        
    def wx_drag_leave ( self, drag_object ):
        if len( self._drag_over ) > 0:
            drag_over_event = DragEvent( x    = 0.0,
                                         y    = 0.0,
                                         x0   = 0.0,
                                         y0   = 0.0,
                                         copy = False,
                                         components  = [ drag_object ],
                                         start_event = default_start_event )
            for component in self._drag_over:
                component.drag_leave = drag_over_event
            self._drag_over = []
            
    #---------------------------------------------------------------------------
    #  Create a wxMenu from a string description:
    #---------------------------------------------------------------------------
        
    def create_menu ( self, menu_definition, owner ):
        return MakeMenu( menu_definition, owner, True, self.control )
        
    #---------------------------------------------------------------------------
    #  Pop-up a wxMenu at a specified location:
    #---------------------------------------------------------------------------
    
    def popup_menu ( self, menu, x, y ):
        self.control.PopupMenuXY( menu.menu, int(x), int( self._flip_y(y) ) ) 
        
#-------------------------------------------------------------------------------
#  'WindowComponent' class:  
#-------------------------------------------------------------------------------


if wx.__version__.startswith("2.6"):
    WxWindowTrait = Trait(None, wx.WindowPtr)
elif wx.__version__.startswith("2.8"):
    WxWindowTrait = Trait(None, wx.Window)
else:
    raise RuntimeError("Unsupported wxPython version.")

class WindowComponent ( Component ):
    
    #---------------------------------------------------------------------------
    #  Trait definitions:
    #---------------------------------------------------------------------------

    component = WxWindowTrait
    
    #---------------------------------------------------------------------------
    #  Initialize the object:
    #---------------------------------------------------------------------------
    
    def __init__ ( self, component = None, **traits ):
        Component.__init__( self, **traits )
        if component is not None:
            self.component  = component
            size            = component.GetAdjustedBestSize()
            self.min_width  = size.GetWidth()
            self.min_height = size.GetHeight()
            self._container_changed( self.container )
            
    #---------------------------------------------------------------------------
    #  Determine if it is valid to manipulate the wxWindow component:
    #---------------------------------------------------------------------------
            
    def _is_valid ( self ):
        return ((self.component is not None) and 
                isinstance( self.container.window, Window ))
                
    #---------------------------------------------------------------------------
    #  Set the correct visibility for the wxWindows component:
    #---------------------------------------------------------------------------
                
    def _set_visibility ( self ):
        self.component.Show( 
                 isinstance( self.container.window, Window ) and self.visible )
        
    #---------------------------------------------------------------------------
    #  Handle the container being changed:
    #---------------------------------------------------------------------------
        
    def _container_changed ( self, container ):
        if self.component is not None:
            self._set_visibility()
            self._bounds_changed( self.bounds )
        
    #---------------------------------------------------------------------------
    #  Handle the bounds of the component being changed: 
    #---------------------------------------------------------------------------
    
    def _bounds_changed ( self, bounds ):
        if self.component is not None:
            x, y, dx, dy = bounds
            comp = self.component
            comp.SetDimensions( 
                 int( x ), int( comp.GetParent()._flip_y( y + dy - 1.0 ) ), 
                 int( dx ), int( dy ) )
            comp.Refresh()
                       
    #---------------------------------------------------------------------------
    #  Generate any additional components that contain a specified (x,y) point:
    #---------------------------------------------------------------------------
       
    def _components_at ( self, x, y ):
        return [ self ]
        
    #---------------------------------------------------------------------------
    #  Draw the component in a specified graphics context if it needs it:
    #---------------------------------------------------------------------------
        
    def draw ( self, gc ):
        if self._drawable and self.window._needs_redraw( self.bounds ):
            comp = self.component
            if comp is not None:
                comp.Show( self.visible )
                comp.Refresh()
                
